#############################################################################
##
#W  scanmtx.gi     share packages 'atlasrep' and 'meataxe'      Thomas Breuer
##
#H  @(#)$Id: scanmtx.gi,v 1.10 2001/03/27 16:09:01 gap Exp $
##
#Y  Copyright (C)  2001,  Lehrstuhl D fuer Mathematik,  RWTH Aachen,  Germany
##
##  Whenever this file is changed in one of the packages
##  `atlasrep' or `meataxe',
##  do not forget to update the corresponding file in the other package!
##
##  This file contains the implementation part of the interface routines for
##  reading and writing {\MeatAxe} text format and straight line programs
##  used in the {\ATLAS} of Group Representations.
##
if IsBound( Revision ) then
  Revision.( "atlasrep/gap/scanmtx_gi" ) :=
    "@(#)$Id: scanmtx.gi,v 1.10 2001/03/27 16:09:01 gap Exp $";
  Revision.( "cmeataxe/gap/scanmtx_gi" ) :=
    "@(#)$Id: scanmtx.gi,v 1.10 2001/03/27 16:09:01 gap Exp $";
fi;


#############################################################################
##
#V  FFLists
##
InstallFlushableValue( FFLists, [] );


#############################################################################
##
#F  FFList( <F> )
##
##  (This program was originally written by Meinolf Geck.)
##
InstallGlobalFunction( FFList, function( F )

    local sizeF,
          p,
          dim,
          elms,
          root,
          powers,
          i,
          pow;

    sizeF:= Size( F );

    if not IsBound( FFLists[ sizeF ] ) then

      p:= Characteristic( F );
      dim:= DegreeOverPrimeField( F );
      elms:= List( Cartesian( List( [ 1 .. dim ], i -> [ 0 .. p-1 ] ) ),
                   Reversed );
      root:= PrimitiveRoot( F );
      pow:= One( root );
      powers:= [ pow ];
      for i in [ 2 .. dim ] do
        pow:= pow * root;
        powers[i]:= pow;
      od;
      FFLists[ sizeF ]:= elms * powers;

    fi;

    return FFLists[ sizeF ];
end );


#############################################################################
##
#F  CMeatAxeFileHeaderInfo( <string> )
##
InstallGlobalFunction( "CMeatAxeFileHeaderInfo", function( string )

    local lline, header, pos, line, degree;

    # Remove a comment part if necessary.
    pos:= Position( string, '#' );
    if pos <> fail then
      string:= string{ [ 1 .. pos ] };
    fi;

    # Split the header string, and convert the entries to integers.
    lline:= List( SplitString( string, "", " \n" ), Int );
    if fail in lline then

      # If the header is valid then it is of new type;
      # remove everything before a leading `"'.
      header:= SplitString( string, "", " \n" );
      pos:= First( header, x -> not IsEmpty( x ) and x[1] = '\"' );
      if pos <> fail then
        header:= header{ [ pos .. Length( header ) ] };
      fi;

      if Length( header ) = 2 and header[1] = "permutation" then

        line:= [ 12, 1,, 1 ];
        if 7 < Length( header[2] ) and header[2]{ [ 1 .. 7 ] } = "degree=" then
          degree:= Int( header[2]{ [ 8 .. Length( header[2] ) ] } );
          line[3]:= degree;
        fi;

      elif Length( header ) = 4 and header[1] = "matrix" then

        line:= [ 6 ];
        if 6 < Length( header[2] ) and header[2]{ [ 1 .. 6 ] } = "field=" then
          line[2]:= Int( header[2]{ [ 7 .. Length( header[2] ) ] } );
          if IsInt( line[2] ) and line[2] < 10 then
            line[1]:= 1;
          fi;
        fi;
        if 5 < Length( header[3] ) and header[3]{ [ 1 .. 5 ] } = "rows=" then
          line[3]:= Int( header[3]{ [ 6 .. Length( header[3] ) ] } );
        fi;
        if 5 < Length( header[4] ) and header[4]{ [ 1 .. 5 ] } = "cols=" then
          line[4]:= Int( header[4]{ [ 6 .. Length( header[4] ) ] } );
        fi;

      fi;

      if fail in line or Number( line ) <> 4 then
        Info( InfoCMeatAxe, 1,
              "corrupted (new) MeatAxe file header" );
        return fail;
      fi;

    else

      # The header is of old type, consisting of four integers.
      if Length( lline ) = 3 and lline[1] = 12 then

        # We may be dealing with permutations of a degree requiring
        # (at least) six digits, for example the header for a permutation
        # on $100000$ points can be `12     1100000     1'.
        line:= String( lline[2] );
        if line[1] = '1' then
          lline[4]:= lline[3];
          lline[3]:= Int( line{ [ 2 .. Length( line ) ] } );
          lline[2]:= 1;
        fi;

      fi;

      if Length( lline ) <> 4 then
        Info( InfoCMeatAxe, 1,
              "corrupted (old) MeatAxe file header" );
        return fail;
      fi;
      line:= lline;

    fi;

    return line;
end );


#############################################################################
##
#F  ScanMeatAxeFile( <filename>[, <q>] )
#F  ScanMeatAxeFile( <string>[, <q>], "string" )
##
InstallGlobalFunction( ScanMeatAxeFile, function( arg )

    local filename,
          file,
          line,
          q,
          mode,
          degree,
          result,
          string,
          pos,
          pos2,
          offset,
          j,
          imgs,
          i,
          F,
          nrows,
          fflist,
          digits,
          ncols,
          one,
          len,
          row;

    if    Length( arg ) = 1
       or ( Length( arg ) = 2 and IsPosInt( arg[2] ) ) then

      filename:= arg[1];
      file:= InputTextFile( filename );
      if file = fail then
        Error( "cannot create input stream for file <filename>" );
      fi;

      InfoRead1( "#I  reading `", filename, "' started\n" );
      line:= ReadLine( file );
      if line = fail then
        Info( InfoCMeatAxe, 1, "no first line exists" );
        CloseStream( file );
        return fail;
      fi;
      if Length( arg ) = 2 then
        q:= arg[2];
      fi;

    elif    ( Length( arg ) = 2 and IsString( arg[1] ) and arg[2] = "string" )
         or ( Length( arg ) = 3 and IsString( arg[1] ) and IsPosInt( arg[2] )
                                and arg[3] = "string" ) then

      string:= arg[1];
      pos:= Position( string, '\n' );
      if pos = fail then
        return fail;
      fi;
      line:= string{ [ 1 .. pos-1 ] };
      string:= string{ [ pos+1 .. Length( string ) ] };
      if Length( arg ) = 3 then
        q:= arg[2];
      fi;

    else
      Error( "usage: ScanMeatAxeFile( <filename>[, <q>] )" );
    fi;

    # Interpret the first line as file header.
    # From the first line, determine whether a matrix or a permutation
    # is to be read.
    line:= CMeatAxeFileHeaderInfo( line );
    if line = fail then
      Info( InfoCMeatAxe, 1, "corrupted file header" );
      if not IsBound( string ) then
        CloseStream( file );
      fi;
      return fail;
    fi;

    mode:= line[1];

    if mode in [ 2, 12 ] then

      # a permutation, to be converted to a matrix,
      # or a list of permutations
      if not IsBound( string ) then
#T Here we read the whole file at once.
#T A factor of about 2 in space could be saved by reading line by line.
        string:= ReadAll( file );
        CloseStream( file );
        InfoRead1( "#I  reading `", filename, "' done\n" );
      fi;

      # Remove comment parts.
#T not really clever ...
      while '#' in string do
        pos:= Position( string, '#' );
        pos2:= Position( string, '\n', pos );
        if pos2 = fail then
          pos2:= Length( string ) + 1;
        fi;
        string:= Concatenation( string{ [ 1 .. pos-1 ] },
                                string{ [ pos2 .. Length( string ) ] } );
      od;

      # Split the line into substrings representing numbers.
      string:= SplitString( string, "", " \n" );

      if mode = 12 then

        # a list of permutations (in free format)
        degree:= line[3];
        result:= [];
        if Length( string ) <> degree * line[4] then
          Info( InfoCMeatAxe, 1, "corrupted file" );
          return fail;
        fi;
        offset:= 0;
        for j in [ 1 .. line[4] ] do
          imgs:= [];
          for i in [ 1 .. degree ] do
            imgs[i]:= Int( string[ i + offset ] );
          od;
          Add( result, PermList( imgs ) );
          offset:= offset + degree;
        od;

      elif mode = 2 then

        # a permutation, to be converted to a matrix
        # (Note that we cannot leave the task to `PermutationMat'
        # because we admit also non-square results.)
        nrows:= line[3];
        if Length( string ) <> nrows then
          Info( InfoCMeatAxe, 1, "corrupted file" );
          return fail;
        fi;
        F:= GF( line[2] );
        result:= NullMat( nrows, line[4], F );
        one:= One( F );
        for i in [ 1 .. nrows ] do
          result[i][ Int( string[i] ) ]:= one;
        od;

        # Convert the matrix to the compressed representation.
        MakeImmutable( result );
        if IsBound( q ) then
          ConvertToMatrixRep( result, q );
        else
          ConvertToMatrixRep( result );
        fi;

      fi;

    elif mode in [ 1, 3, 4, 5, 6 ] then

      # a matrix, in various similar formats
      F:= GF( line[2] );
      nrows:= line[3];
      ncols:= line[4];
      result:= [];

      # The case of a string that is available in {\GAP} is dealt with
      # in parallel with the case of a file that is read line by line
      # (for space reasons).
      pos:= 1;
      if IsBound( string ) then
        line:= string;
        len:= Length( line );
        if mode <> 1 then

          # Remove comment parts if applicable.
#T not really clever ...
          while '#' in line do
            pos:= Position( line, '#' );
            pos2:= Position( line, '\n', pos );
            if pos2 = fail then
              pos2:= Length( line ) + 1;
            fi;
            line:= Concatenation( line{ [ 1 .. pos-1 ] },
                                  line{ [ pos2 .. Length( line ) ] } );
          od;

          # Split the string into substrings representing numbers.
          line:= SplitString( line, "", " \n" );
          if Length( line ) <> nrows * ncols then
            Info( InfoCMeatAxe, 1, "corrupted file" );
            return fail;
          fi;

        fi;
      else
        len:= 0;
      fi;

      if mode = 1 then

        # a matrix (in fixed format)
        fflist:= FFList( F );
        digits:= "0123456789";

        for i in [ 1 .. nrows ] do

          # Fill the `i'-th row of the matrix.
          row:= [];
          for j in [ 1 .. ncols ] do

            # Get new input if necessary.
            while pos <= len and line[ pos ] in " \n" do
              pos:= pos + 1;
            od;
            if len < pos and IsBound( file ) then
              line:= ReadLine( file );
              if line = fail then
                Info( InfoCMeatAxe, 1, "corrupted file" );
                CloseStream( file );
                return fail;
              fi;
              len:= Length( line );
              if len = 0 then
                Info( InfoCMeatAxe, 1, "corrupted file" );
                CloseStream( file );
                return fail;
              fi;
              pos:= 1;
            fi;

            # Assign the `j'-th entry.
            row[j]:= fflist[ Position( digits, line[ pos ] ) ];
            pos:= pos + 1;

          od;

          # Convert the row into the compact representation.
          ConvertToVectorRep( row );
          result[i]:= row;

        od;

        # For the final check whether all input was processed,
        # ignore trailing whitespace and line breaks.
        while pos <= len and line[ pos ] in " \n" do
          pos:= pos + 1;
        od;

      elif mode = 5 then

        # an integer matrix (in free format), to be reduced mod a prime
        one:= One( F );

        for i in [ 1 .. nrows ] do

          # Fill the `i'-th row of the matrix.
          row:= [];
          for j in [ 1 .. ncols ] do

            # Get new input if necessary.
            if len < pos and IsBound( file ) then

              line:= ReadLine( file );
              if line = fail then
                Info( InfoCMeatAxe, 1, "corrupted file" );
                CloseStream( file );
                return fail;
              fi;

              # Remove a comment part if applicable.
              pos:= Position( line, '#' );
              if pos <> fail then
                line:= line{ [ 1 .. pos-1 ] };
              fi;

              # Split the line into substrings representing numbers.
              line:= SplitString( line, "", " \n" );
              len:= Length( line );
              if len = 0 then
                Info( InfoCMeatAxe, 1, "corrupted file" );
                CloseStream( file );
                return fail;
              fi;
              pos:= 1;

            fi;

            # Assign the `j'-th entry.
            row[j]:= Int( line[ pos ] );
            pos:= pos + 1;

          od;

          # Transfer the row into a finite field vector.
          row:= row * one;

          # Convert the row into the compact representation.
          ConvertToVectorRep( row );
          result[i]:= row;

        od;

      elif mode in [ 3, 4, 6 ] then

        # a matrix (in free format)
        fflist:= FFList( F );

        for i in [ 1 .. nrows ] do

          # Fill the `i'-th row of the matrix.
          row:= [];
          for j in [ 1 .. ncols ] do

            # Get new input if necessary.
            if len < pos and IsBound( file ) then

              line:= ReadLine( file );
              if line = fail then
                Info( InfoCMeatAxe, 1, "corrupted file" );
                CloseStream( file );
                return fail;
              fi;

              # Remove a comment part if applicable.
              pos:= Position( line, '#' );
              if pos <> fail then
                line:= line{ [ 1 .. pos-1 ] };
              fi;

              # Split the line into substrings representing numbers.
              line:= SplitString( line, "", " \n" );
              len:= Length( line );
              if len = 0 then
                Info( InfoCMeatAxe, 1, "corrupted file" );
                CloseStream( file );
                return fail;
              fi;
              pos:= 1;

            fi;

            # Assign the `j'-th entry.
            row[j]:= fflist[ Int( line[ pos ] ) + 1 ];
            pos:= pos + 1;

          od;

          # Convert the row into the compact representation.
          ConvertToVectorRep( row );
          result[i]:= row;

        od;

      fi;

      # Check that exactly the whole input was scanned.
      # (Be aware of possible whitespace, linebreaks, and comments
      # at the end of the file.)
      if IsBound( file ) and pos <> len + 1 then
        Info( InfoCMeatAxe, 1, "corrupted file" );
        CloseStream( file );
        return fail;
      fi;
   #    ReadLine( file ) <> fail ) then
   #    Info( InfoCMeatAxe, 1, "corrupted file" );
   #    if IsBound( file ) then
   #      CloseStream( file );
   #    fi;
   #    return fail;

      # Close the stream if applicable
      if not IsBound( string ) then
        CloseStream( file );
        InfoRead1( "#I  reading `", filename, "' done\n" );
      fi;

      # Convert further.
      MakeImmutable( result );
      if IsBound( q ) then
        ConvertToMatrixRep( result, q );
      else
        ConvertToMatrixRep( result );
      fi;

    else
      Info( InfoCMeatAxe, 1, "unknown mode" );
      return fail;
    fi;

    return result;
end );


#############################################################################
##
#M  MeatAxeString( <mat>, <q> )
##
InstallMethod( MeatAxeString,
    "for matrix over a finite field, and field order",
    [ IsTable and IsFFECollColl, IsPosInt ],
    function( mat, q )

    local nrows,     # number of rows of `mat'
          ncols,     # number of columns of `mat'
          one,       # identity element of the field of matrix entries
          zero,      # zero element of the field of matrix entries
          perm,      # list of perm. images if `mat' is a perm. matrix
          i,         # loop over the rows of `mat'
          noone,     # no `one' found yet in the current row
          row,       # one row of `mat'
          j,         # loop over the columns of `mat'
          mode,      # mode of the {\MeatAxe} string (first header entry)
          str,       # {\MeatAxe} string, result
          fflist,    #
          values,
          linelen,
          nol,
          k;

    # Check that `mat' is rectangular.
    if not IsMatrix( mat ) then
      Error( "<mat> must be a matrix" );
    fi;
    nrows:= Length( mat );
    ncols:= Length( mat[1] );

    # Check that `q' and `mat' are compatible.
    if not IsPrimePowerInt( q ) or ( q mod Characteristic( mat ) <> 0 ) then
      Error( "<q> and the characteristic of <mat> are incompatible" );
    fi;

    # If the matrix is a ``generalized permutation matrix''
    # then construct a string of {\MeatAxe} mode 2.
    one:= One( mat[1][1] );
    zero:= Zero( one );
    perm:= [];
    for i in [ 1 .. nrows ] do
      noone:= true;
      row:= mat[i];
      for j in [ 1 .. ncols ] do
        if row[j] = one then
          if noone then
            perm[i]:= j;
            noone:= false;
          else
            perm:= fail;
            break;
          fi;
        elif row[j] <> zero then
          perm:= fail;
          break;
        fi;
      od;
      if perm = fail then
        break;
      fi;
    od;

    # Start with the header line.
    # We try to keep the files as small as possible by using the (old)
    # mode `1' whenever possible.
    if perm <> fail then
      mode:= "2";
    elif q < 10 then
      mode:= "1";
    else
      mode:= "6";
    fi;
    str:= ShallowCopy( mode );          # mode,
    Append( str, " " );
    Append( str, String( q ) );         # field size,
    Append( str, " " );
    Append( str, String( nrows ) );     # number of rows,
    Append( str, " " );
    Append( str, String( ncols ) );     # number of columns
    Append( str, "\n" );

    # Add the matrix entries.
    fflist:= FFList( GF(q) );
    if mode = "1" then

      # Set the parameters for filling lines in the fixed format
      values:= "0123456789";
      linelen:= 80;
      nol:= Int( ncols / linelen );

      for row in mat do
        i:= 1;
        for j in [ 0 .. nol-1 ] do
          for k in [ 1 .. linelen ] do
            Add( str, values[ Position( fflist, row[i] ) ] );
            i:= i + 1;
          od;
          Add( str, '\n' );
        od;
        for i in [ nol * linelen + 1 .. ncols ] do
          Add( str, values[ Position( fflist, row[i] ) ] );
        od;
        Add( str, '\n' );
      od;

    elif mode = "2" then

      for i in perm do
        Append( str, String( i ) );
        Add( str, '\n' );
      od;

    else

      # free format
      values:= List( [ 0 .. q-1 ], String );
      for row in mat do
        for i in row do
          Append( str, values[ Position( fflist, i ) ] );
          Add( str, '\n' );
        od;
      od;

    fi;

    # Return the result.
    return str;
    end );


#############################################################################
##
#M  MeatAxeString( <perms>, <degree> )
##
InstallMethod( MeatAxeString,
    "for list of permutations, and degree",
    [ IsPermCollection and IsList, IsPosInt ],
    function( perms, degree )

    local str, perm, i;

    # Start with the header line.
    str:= "12 1 ";
    Append( str, String( degree ) );
    Append( str, " " );
    Append( str, String( Length( perms ) ) );
    Append( str, "\n" );

    # Add the images.
    for perm in perms do
      for i in [ 1 .. degree ] do
        Append( str, String( i ^ perm ) );
        Add( str, '\n' );
      od;
    od;

    # Return the result.
    return str;
    end );


#############################################################################
##
#M  MeatAxeString( <perm>, <q>, <dims> )
##
InstallMethod( MeatAxeString,
    "for permutation, field order, and dimensions",
    [ IsPerm, IsPosInt, IsList ],
    function( perm, q, dims )

    local str, i;

    # Start with the header line.
    # (The mode is `2': a permutation, to be converted to a matrix.)
    str:= "2 ";                         # mode,
    Append( str, String( q ) );         # field size,
    Append( str, " " );
    Append( str, String( dims[1] ) );   # number of rows,
    Append( str, " " );
    Append( str, String( dims[2] ) );   # number of columns
    Append( str, "\n" );

    # Add the images.
    for i in [ 1 .. dims[1] ] do
      Append( str, String( i ^ perm ) );
      Add( str, '\n' );
    od;

    # Return the result.
    return str;
    end );


#############################################################################
##
#F  ScanStraightLineProgram( <filename> )
#F  ScanStraightLineProgram( <string>, "string" )
##
InstallGlobalFunction( ScanStraightLineProgram, function( arg )

    local filename,
          strdata,
          file,
          data,
          echo,
          line,
          labels,
          i,
          lines,
          output,
          a, b, c,
          outputs,
          result,
          search;

    # Get and check the input.
    if   Length( arg ) = 1 and IsString( arg[1] ) then
      filename:= arg[1];
    elif Length( arg ) = 2 and IsString( arg[1] ) and arg[2] = "string" then
      strdata:= SplitString( arg[1], "", "\n" );
    else
      Error( "usage: ScanStraightLineProgram( <filename>[, \"string\"] )" );
    fi;

    # Read the data if necessary.
    data:= [];
    echo:= [];
    if IsBound( strdata ) then

      for line in strdata do

        # Omit empty lines and comment lines.
        if   4 < Length( line ) and line{ [ 1 .. 4 ] } = "echo" then
          Add( echo, line );
        elif not IsEmpty( line ) and line[1] <> '#' then
          line:= SplitString( line, "", " \n" );
          if not IsEmpty( line ) then
            Add( data, line );
          fi;
        fi;

      od;

    else

      file:= InputTextFile( filename );
      if file = fail then
        Error( "cannot create input stream for file <filename>" );
      fi;

      InfoRead1( "#I  reading `", filename, "' started\n" );
      line:= ReadLine( file );
      while line <> fail do

        # Omit empty lines and comment lines.
        if   4 < Length( line ) and line{ [ 1 .. 4 ] } = "echo" then
          Add( echo, line );
        elif not IsEmpty( line ) and line[1] <> '#' then
          line:= SplitString( line, "", " \n" );
          if not IsEmpty( line ) then
            Add( data, line );
          fi;
        fi;
        line:= ReadLine( file );

      od;
      CloseStream( file );
      InfoRead1( "#I  reading `", filename, "' done\n" );

    fi;

    # Determine the labels that occur.
    if IsEmpty( data ) then
      Info( InfoCMeatAxe, 1, "empty input of straight line program" );
      return fail;
    fi;
    labels:= [];
    if data[1][1] <> "inp" then

      # There is no `inp' line.
      # The default input is given by the labels `1' and `2'.
      labels:=[ "1", "2" ];

    fi;
    for line in data do
      if line[1] = "inp" then
        if Length( line ) = 2 and Int( line[2] ) <> fail then
          Append( labels, List( [ 1 .. Int( line[2] ) ], String ) );
        elif Int( line[2] ) = Length( line ) - 2 then
          Append( labels, line{ [ 3 .. Length( line ) ] } );
        else
          Info( InfoCMeatAxe, 1, "corrupted line `", line, "'" );
          return fail;
        fi;
      elif line[1] = "pwr" then
        for i in [ 3 .. Length( line ) ] do
          if not line[i] in labels then
            Add( labels, line[i] );
          fi;
        od;
      else
        for i in [ 2 .. Length( line ) ] do
          if not line[i] in labels then
            Add( labels, line[i] );
          fi;
        od;
      fi;
    od;

    # Translate the lines.
    lines:= [];
    output:= [];
    for line in data do
      if line[1] = "oup" or line[1] = "op" then
#T clean this!

        Add( output, line );

      elif line[1] <> "inp" then

        if   line[1] = "mu" and Length( line ) = 4 then
          a:= Position( labels, line[2] );
          b:= Position( labels, line[3] );
          c:= Position( labels, line[4] );
          Add( lines, [ [ a, 1, b, 1 ], c ] );
        elif line[1] = "iv" and Length( line ) = 3 then
          a:= Position( labels, line[2] );
          b:= Position( labels, line[3] );
          Add( lines, [ [ a, -1 ], b ] );
        elif line[1] = "pwr" and Length( line ) = 4 then
          a:= Int( line[2] );
          b:= Position( labels, line[3] );
          c:= Position( labels, line[4] );
          Add( lines, [ [ b, a ], c ] );
        elif line[1] = "cjr" and Length( line ) = 3 then
          a:= Position( labels, line[2] );
          b:= Position( labels, line[3] );
          Add( lines, [ [ b, -1, a, 1, b, 1 ], a ] );
        elif line[1] = "cp" and Length( line ) = 3 then
          a:= Position( labels, line[2] );
          b:= Position( labels, line[3] );
          Add( lines, [ [ a, 1 ], b ] );
        elif line[1] = "cj" and Length( line ) = 4 then
          a:= Int( line[2] );
          b:= Position( labels, line[3] );
          c:= Position( labels, line[4] );
          Add( lines, [ [ b, -1, a, 1, b, 1 ], c ] );
        elif line[1] = "com" and Length( line ) = 4 then
          a:= Int( line[2] );
          b:= Position( labels, line[3] );
          c:= Position( labels, line[4] );
          Add( lines, [ [ a, -1, b, -1, a, 1, b, 1 ], c ] );
        else
          Info( InfoCMeatAxe, 1, "strange line `", line, "'" );
          return fail;
        fi;

      fi;
    od;

    # Specify the output.
    result:= rec();
    if IsEmpty( output ) then

      # The default output is given by the labels `1' and `2'.
      # Note that this is allowed only if these labels really occur.
      if IsSubset( labels, [ "1", "2" ] ) then
        Add( lines, [ [ Position( labels, "1" ), 1 ],
                      [ Position( labels, "2" ), 1 ] ] );
      else
        Info( InfoCMeatAxe, 1, "missing `oup' statement" );
        return fail;
      fi;

    else

      # The `oup' lines list the output labels.
      outputs:= [];
      for line in output do
        if Length( line ) = 2 and Int( line[2] ) <> fail then
          Append( outputs, List( [ 1 .. Int( line[2] ) ],
                          x -> [ Position( labels, String( x ) ), 1 ] ) );
        elif Length( line ) = Int( line[2] ) + 2 then
          Append( outputs, List( line{ [ 3 .. Length( line ) ] },
                          x -> [ Position( labels, x ), 1 ] ) );
        else
          Info( InfoCMeatAxe, 1, "corrupted line `", line, "'" );
          return fail;
        fi;
      od;
      Add( lines, outputs );

      # The straight line program is thought for computing
      # class representatives,
      # and the bijection between the output labels and the class names
      # is given by the `echo' lines.
      # For the line `oup <l> <b1> <b2> ... <bl>',
      # there must be the `echo' line `Classes <n1> <n2> ... <nl>'.
      if not IsEmpty( echo ) then

        # Check that the `echo' lines fit to the `oup' lines.
        echo:= List( echo, line -> SplitString( line, "", "\" \n" ) );
        while     not IsEmpty( echo )
              and LowercaseString( echo[1][2] ) <> "classes" do
          echo:= echo{ [ 2 .. Length( echo ) ] };
        od;

        echo:= List( echo, x -> Filtered( x,
                 y -> y <> "echo" and LowercaseString( y ) <> "classes" ) );
        outputs:= Concatenation( echo );
        output:= Sum( List( output, x -> Int( x[2] ) ), 0 );
        if Length( outputs ) < output then
          Info( InfoCMeatAxe, 1,
                "`oup' and `echo' lines not compatible" );
          return fail;
        fi;
        outputs:= outputs{ [ 1 .. output ] };
        result.outputs:= outputs;

      fi;

    fi;

    # Construct and return the result.
    result.program:= StraightLineProgramNC( lines );
    return result;
end );


#############################################################################
##
#F  AtlasStringOfStraightLineProgram( <prog>[, <outputnames>] )
##
InstallGlobalFunction( AtlasStringOfStraightLineProgram, function( arg )

    local prog,           # straight line program, first argument
          outputnames,    # list of strings, optional second argument
          str,            # string, result
          resused,        # maximal label currently used in the program
          i,
          translateword,  # local function
          line,
          lastresult,
          namline,
          resline,
          inline;

    # Get and check the arguments.
    if   Length( arg ) = 1 and IsStraightLineProgram( arg[1] ) then
      prog:= arg[1];
    elif Length( arg ) = 2 and IsStraightLineProgram( arg[1] )
                           and IsList( arg[2] ) then
      prog:= arg[1];
      outputnames:= arg[2];
    else
      Error( "usage: ",
             "AtlasStringOfStraightLineProgram( <prog>[, <outputnames>] )" );
    fi;

    # Write the line of inputs.
    str:= "inp ";
    resused:= NrInputsOfStraightLineProgram( prog );
    Append( str, String( resused ) );
    Add( str, '\n' );

    # function to translate a word into a series of simple operations
    translateword:= function( word, respos )

      local used,  # maximal label, including intermediate results
            new,
            i;

      if resused < respos then
        resused:= respos;
      fi;
      used:= resused;

      if Length( word ) = 2 then

        # The word describes a simple powering.
        if word[2] = -1 then
          Append( str,
                  Concatenation( "iv ", String( word[1] ),
                                 " ", String( respos ), "\n" ) );
        elif 0 <= word[2] then
          Append( str,
                  Concatenation( "pwr ", String( word[2] ),
                                 " ", String( word[1] ),
                                 " ", String( respos ), "\n" ) );
        else
          used:= used + 1;
          Append( str,
                  Concatenation( "iv ", String( word[1] ),
                                 " ", String( used ), "\n" ) );
          Append( str,
                  Concatenation( "pwr ", String( -word[i] ),
                                 " ", String( used ),
                                 " ", String( respos ), "\n" ) );
        fi;

      else

        # Get rid of the powerings.
        new:= [];
        for i in [ 2, 4 .. Length( word ) ] do
          if word[i] = 1 then
            Add( new, word[ i-1 ] );
          elif 0 < word[i] then
            used:= used + 1;
            Append( str,
                    Concatenation( "pwr ", String( word[i] ),
                                   " ", String( word[ i-1 ] ),
                                   " ", String( used ), "\n" ) );
            Add( new, used );
          else
            used:= used + 1;
            Append( str,
                    Concatenation( "iv ", String( word[ i-1 ] ),
                                   " ", String( used ), "\n" ) );
            if word[i] < -1 then
              used:= used + 1;
              Append( str,
                      Concatenation( "pwr ", String( -word[i] ),
                                     " ", String( used-1 ),
                                     " ", String( used ), "\n" ) );
            fi;
            Add( new, used );
          fi;
        od;

        # Now form the product of the elements in `new'.
        if Length( new ) = 1 then
          if new[1] <> respos then
            Append( str,
                    Concatenation( "pwr 1 ", String( new[1] ),
                                   " ", String( respos ), "\n" ) );
          fi;
        elif Length( new ) = 2 then
          Append( str,
                  Concatenation( "mu ", String( new[1] ),
                                 " ", String( new[2] ),
                                 " ", String( respos ), "\n" ) );
        else
          used:= used + 1;
          Append( str,
                  Concatenation( "mu ", String( new[1] ),
                                 " ", String( new[2] ),
                                 " ", String( used ), "\n" ) );
          for i in [ 3 .. Length( new )-1 ] do
            used:= used + 1;
            Append( str,
                    Concatenation( "mu ", String( used - 1 ),
                                   " ", String( new[i] ),
                                   " ", String( used ), "\n" ) );
          od;
          used:= used + 1;
          Append( str,
                  Concatenation( "mu ", String( used - 1 ),
                                 " ", String( new[ Length( new ) ] ),
                                 " ", String( respos ), "\n" ) );
        fi;

      fi;
    end;

    # Loop over the lines.
    for line in LinesOfStraightLineProgram( prog ) do

      if ForAll( line, IsList ) then

        # The list describes the return values.
        lastresult:= [];
        for i in [ 1 .. Length( line ) ] do
          if Length( line[i] ) = 2 and line[i][2] = 1 then
            Add( lastresult, String( line[i][1] ) );
          else
            resused:= resused + 1;
            translateword( line[i], resused );
            Add( lastresult, String( resused ) );
          fi;
        od;

        if IsBound( outputnames ) then

          if Length( line ) <> Length( outputnames ) then
            Error( "<outputnames> has the wrong length" );
          fi;

          # Write the `echo' statements.
          # (Split the output specifications into lines if necessary.)
          i:= 1;
          namline:= "";
          resline:= "";
          inline:= 0;
          while i <= Length( outputnames ) do
            if    60 < Length( namline ) + Length( outputnames[i] )
               or 60 < Length( resline ) + Length( lastresult[i] ) then
              Append( str,
                  Concatenation( "echo \"Classes", namline, "\"\n" ) );
              Append( str,
                  Concatenation( "oup ", String( inline ), resline, "\n" ) );
              namline:= "";
              resline:= "";
              inline:= 0;
            fi;
            Add( namline, ' ' );
            Add( resline, ' ' );
            Append( namline, outputnames[i] );
            Append( resline, lastresult[i] );
            inline:= inline + 1;
            i:= i + 1;
          od;
          if inline <> 0 then
            Append( str,
                Concatenation( "echo \"Classes", namline, "\"\n" ) );
            Append( str,
                Concatenation( "oup ", String( inline ), resline, "\n" ) );
          fi;

        elif ForAll( [ 1 .. Length( line ) ], i -> line[i] = [ i, 1 ] ) then

          # Write a short output statement.
          Append( str, "oup " );
          Append( str, String( Length( line ) ) );
          Add( str, '\n' );

        else

          # Write the full output statements.
          i:= 1;
          resline:= "";
          inline:= 0;
          while i <= Length( lastresult ) do
            if 60 < Length( resline ) + Length( lastresult[i] ) then
              Append( str,
                  Concatenation( "oup ", String( inline ), resline, "\n" ) );
              resline:= "";
              inline:= 0;
            fi;
            Add( resline, ' ' );
            Append( resline, lastresult[i] );
            inline:= inline + 1;
            i:= i + 1;
          od;
          if inline <> 0 then
            Append( str,
                Concatenation( "oup ", String( inline ), resline, "\n" ) );
          fi;

        fi;

        # Return the result.
        return str;

      else

        # Separate word and position where to put the result,
        # and translate the line into a sequence of simple steps.
        if ForAll( line, IsInt ) then
          resused:= resused + 1;
          lastresult:= resused;
          translateword( line, resused );
        else
          lastresult:= line[2];
          translateword( line[1], lastresult );
        fi;

      fi;
    od;

    # (If we arrive here then there is exactly one output value.)

    # Write the `echo' statements if applicable.
    # (This isn't really probable, is it?)
    if IsBound( outputnames ) then
      if Length( outputnames ) <> 1 then
        Error( "<outputnames> has the wrong length" );
      fi;
      Append( str,
          Concatenation( "echo \"Classes ", outputnames[1], "\"\n" ) );
    fi;

    # Write the output statement.
    Append( str, "oup 1 " );
    Append( str, String( lastresult ) );
    Add( str, '\n' );

    # Return the result;
    return str;
end );


#############################################################################
##
#E

