##############################################################################
##
#W  up2.gi                         XMOD Package                  Chris Wensley
#W                                                                 & Murat Alp
##
##  Installation file for functions of the XMOD package.
##
#H  @(#)$Id: up2.gi,v 1.2 2002/04/19 gap Exp $
##
#Y  Copyright (C) 1999,2001 University of St. Andrews, North Haugh,
#Y                          St. Andrews, Fife KY16 9SS, Scotland
##
Revision.("xmod/gap/up2_gi") := 
    "@(#)$Id: up2.gi,v 2.001 2002/04/17 gap Exp $";

#############################################################################
##
#M  IsUpMapping
##
InstallMethod( IsUpMapping, "generic method for derivations and sections",
    true, [ IsGeneralMapping ], 0,
function( map )
    return ( HasObject2d( map ) and HasGeneratorImages( map ) );
end );

##############################################################################
##
#M  UpMappingObj( <obj>, <ims> ) . . . . . . . . . . . . . . . make up-mapping
##
InstallMethod( UpMappingObj, "for 2dobject and images", true,
    [ Is2dObject, IsHomogeneousList ], 0,
function( obj, ims )

    local  filter, fam, map, genR, invR, ok, src, rng, aut, name, hom;

    fam := FamilyObj( [ obj, ims ] );
    filter := IsUpMappingObj;
    src := Source( obj );
    rng := Range( obj );
    map := Objectify( NewType( fam, filter ), rec() );
    SetObject2d( map, obj );
    SetGeneratorImages( map, ims );
    SetIsUpMapping( map, true );
    if IsPreXMod( obj ) then
        ok := IsDerivation( map );
        if ( ok and IsXMod( obj ) ) then
            SetIsDerivation( map, true );
        fi;
        genR := StrongGeneratorsStabChain( StabChain( rng ) );
        invR := List( genR, r -> r^(-1) );
    elif IsPreCat1( obj ) then
        genR := GeneratorsOfGroup( rng );
        hom := GroupHomomorphismByImages( rng, src, genR, ims );
        if RespectsMultiplication( hom ) then
##            SetIsGroupHomomorphism( map, true );
        else
            Error( "proposed section not a group homomorphism" );
        fi;
        ok := IsSection( map );
        if ( ok and IsCat1( obj ) ) then
            SetIsSection( map, true );
        fi;
    else
        Error( "2dobject neither an xmod nor a cat1-group" );
    fi;
    return map;
end );

##############################################################################
##
#M  Source( <map> )  . . . . . . . . . . . . . . . . . . source for up-mapping
##
InstallOtherMethod( Source, "generic method for an up-mapping",
    [ IsUpMapping ], 0,
function ( map )
    return Source( Object2d( map ) );
end );

##############################################################################
##
#M  Range( <map> )  . . . . . . . . . . . . . . . . . . . range for up-mapping
##
InstallOtherMethod( Range, "generic method for an up-mapping",
    [ IsUpMapping ], 0,
function ( map )
    return Range( Object2d( map ) );
end );

#############################################################################
##
#M  IsDerivation
##
InstallMethod( IsDerivation, "generic method for derivation of pre-xmod",
    true, [ IsUpMapping ], 0,
function( chi )

    local  im, inv, XM, bdy, R, genR, invR, oneR, ord, S, oneS,
           genrng, rho, imrho, ok, g, i, j, r, s, t, u, v, w, aut, act, pair,
           iso, fp, pres, T, numrels, lenrels, rels, rel, len, triv;

    XM := Object2d( chi );
    im := GeneratorImages( chi );
    S := Source( XM );
    oneS := One( S );
    R := Range( XM );
    oneR := One( R );
    genR := StrongGeneratorsStabChain( StabChain( R ) );
    iso := IsomorphismFpGroupByGenerators( R, genR );
    fp := Range( iso );
    Info( InfoXMod, 3, "iso to fp group : ", iso );
    invR := List( genR, r -> r^(-1) );
    triv := List( genR, r -> oneS );
    if ( GeneratorImages( chi ) = triv ) then
        return true;
    fi;
    genrng := [ 1..Length( genR ) ];
    bdy := Boundary( XM );
    act := XModAction( XM );

    # calculate  chi(r^-1)  for each generator  r
    inv := 0 * genrng;
    for j in genrng do
        r := genR[j];
        s := im[j];
        aut := Image( act, r^(-1) );
        inv[j] := Image( aut, s )^(-1);
    od;
    Info( InfoXMod, 3, "  Images = ", im, ",  inverses = ", inv );

    pres := PresentationFpGroup( fp );
    T := pres!.tietze;
    numrels := T[TZ_NUMRELS];
    rels := T[TZ_RELATORS];
    lenrels := T[TZ_LENGTHS];
    w := oneS;
    v := oneR;
    for i in [1..numrels] do
        rel := rels[i];
        len := lenrels[i];
        for j in Reversed( [1..len] ) do
            g := rel[j];
            if ( g > 0 ) then
                r := genR[g];
                s := im[g];
            else
                r := invR[-g];
                s := inv[-g];
            fi;
            aut := Image( act, v );
            u := Image( aut, s );
            w := u * w;
            v := r * v;
        od;
        if ( w <> oneS ) then
            Info( InfoXMod, 3, "chi(rel) <> One(S)  when rel = ", rel );
            return false;
        fi;
    od;
    SetIsDerivation( chi, true );
    return true;
end );

##############################################################################
##
#M  IsSection                    tests the section axioms for a pre-cat1-group
##
InstallMethod( IsSection, "generic method for section of cat1-group",
    true, [ IsUpMapping ], 0,
function( xi )

    local  C, Crng, idrng, ok, reg;

    if not IsGroupHomomorphism( xi ) then
        return false;
    fi;
    C := Object2d( xi );
    Crng := Range( C );
    if not ( ( Source(xi) = Crng ) and ( Range(xi) = Source(C) ) ) then
        Print( "<xi> not a homomorphism: Range( C ) -> Source( C )\n" );
        return false;
    fi;
    idrng := InclusionMapping( Crng, Crng );   
    ok := ( xi*Tail(C) = idrng );
    reg := IsInjective( xi*Head(C) );
    ## SetIsRegularSection( xi, ok and reg );
    SetIsSection( xi, true );
    return ok;
end );

##############################################################################
##                               Derivations                                ##
##############################################################################

##############################################################################
##
#M  DerivationByImages                                     sets up the mapping
##
InstallMethod( DerivationByImages, "method for an xmod", true,
    [ IsXMod, IsHomogeneousList ], 0,
function( XM, im )

    local  nargs, usage, isder, chi, R, S, genR, ngR, ok;

    S := Source( XM );
    R := Range( XM );
    genR := StrongGeneratorsStabChain( StabChain( R ) );
    ngR := Length( genR );
    if not ( IsList( im ) and ( Length( im ) = ngR )
                   and ForAll( im, x -> ( x in S ) ) ) then
        Error( "<im> must be a list of |genR| elements in S" );
    fi;
    chi := UpMappingObj( XM, im );
    ok := IsDerivation( chi );
    if not ok then 
        return fail;
    else
        return chi;
    fi;
end );

#############################################################################
##
#M  PrintObj( <chi> ) . . . . . . . . . . . . . . . . . . . view a derivation
##
InstallMethod( PrintObj, "method for a derivation", true, [ IsDerivation ], 0,
    function( chi )
    local  obj, iso, R, genR;
    obj := Object2d( chi );
    R := Range( obj );
    genR := StrongGeneratorsStabChain( StabChain( R ) );
    iso := IsomorphismFpGroupByGenerators( R, genR );
    Print( "DerivationByImages( ", R, ", ", Source(obj), ", ",
         genR, ", ", GeneratorImages(chi), " )" );
end );

#############################################################################
##
#M  ViewObj( <chi> ) . . . . . . . . . . . . . . . . . . .  view a derivation
##
InstallMethod( ViewObj, "method for a derivation", true, [ IsDerivation ], 0,
    function( chi )
    local  obj, iso, R, genR;
    obj := Object2d( chi );
    R := Range( obj );
    genR := StrongGeneratorsStabChain( StabChain( R ) );
    iso := IsomorphismFpGroupByGenerators( R, genR );
    Print( "DerivationByImages( ", R, ", ", Source(obj), ", ",
         genR, ", ", GeneratorImages(chi), " )" );
end );

##############################################################################
##
#M  DerivationImage                  image of  r \in R  by the derivation  chi
##
InstallMethod( DerivationImage, "method for a derivation and a group element",
    true, [ IsDerivation, IsObject ], 0,
function( chi, r )

    local  XM, S, R, genR, imchi, elR, elS, ngR, genrng, j, g, s, u, v,
           rpos, spos, ord, P, a, act;

    XM := Object2d( chi );
    S := Source( XM );
    R := Range( XM );
    if not ( r in R ) then
        Error( "second parameter must be an element of chi.source" );
    fi;
    if ( r = One( R ) ) then
        return One( S );
    fi;
    elR := Elements( R );
    elS := Elements( S );
    genR := StrongGeneratorsStabChain( StabChain( R ) );
    imchi := GeneratorImages( chi );
    rpos := Position( elR, r );
    if HasImagePositions( chi ) then
        spos := ImagePositions( chi )[ rpos ];
        return elS[ spos ];
    fi;
    ord := GenerationOrder( R );
    P := GenerationPairs( R );
    j := P[rpos][1];
    g := P[rpos][2];
    if ( j = 1 ) then   # r is a generator
        return imchi[g];
    fi;
    act := XModAction( XM );
    u := imchi[g];
    v := genR[g];
    while ( j > 1 ) do
        g := P[j][2];
        j := P[j][1];
        s := imchi[g];
        a := Image( act, v );
        u := Image( a, s ) * u;
        v := genR[g] * v;
    od;
    return u;
end );

##############################################################################
##
#M  ImagePositions                    returns list of positions of chi(r) in S
##
InstallMethod( ImagePositions, "method for a derivation", true,
    [ IsDerivation ], 0,
function( chi )

    local  XM, R, S, elR, genR, ngR, ord, P, oR, elS, oneS,
           i, j, k, x, y, L, ok, imchi, act, agen, a;

    XM := Object2d( chi );
    R := Range( XM );
    genR := StrongGeneratorsStabChain( StabChain( R ) );
    ngR := Length( genR );
    imchi := GeneratorImages( chi );
    act := XModAction( XM );
    agen := List( genR, r -> Image( act, r ) );
    S := Source( XM );
    elR := Elements( R );
    elS := Elements( S );
    oneS := One( S );
    ord := GenerationOrder( R );
    P := GenerationPairs( R );
    if ( InfoLevel( InfoXMod ) > 2 ) then
        ok := CheckGenerationPairs( R );
        Print( "checking generation pairs: ", ok, "\n" );
    fi;
    oR := Size( R );
    L := 0 * [1..oR];
    L[1] := Position( elS, oneS );
    for k in [2..oR] do
        i := ord[k];
        j := P[i][2];
        x := elS[ L[ P[i][1] ] ];
        a := agen[j];
        y := Image( a, x ) * imchi[j]; 
        L[i] := Position( elS, y );
    od;
    return L;
end );

##############################################################################
##
#M  CompositeDerivation                 Whitehead composite of two derivations
##
InstallMethod( CompositeDerivation, "method for two derivations", true,
    [ IsDerivation, IsDerivation ], 0,
function( chi, chj )

    local  XM, imi, imj, R, genR, numrng, rng, r, k,
           s, si, sj, bdy, bsj, imcomp, comp;

    XM := Object2d( chi );
    R := Range( XM );
    genR := StrongGeneratorsStabChain( StabChain( R ) );
    if not ( Object2d( chj ) = XM ) then
        Error( "<chi>,<chj> must be derivations of the SAME crossed module" );
    fi;
    numrng := Length( genR );
    rng := [ 1..numrng ];
    imi := GeneratorImages( chi );
    imj := GeneratorImages( chj );
    bdy := Boundary( XM );
    imcomp := 0 * rng;
    for k in rng do
        r := genR[k];
        sj := imj[k];
        si := imi[k];
        bsj := Image( bdy, sj );
        s := DerivationImage( chi, bsj );
        imcomp[k] := si * sj * s;
    od;
    comp := DerivationByImages( XM, imcomp );
    return comp;
end );

##############################################################################
##
#M  \*( <chi1>, <chi2> ) . . . . . . . . . . . . . . . . . . for 2 derivations
##
InstallOtherMethod( \*, "for two derivations of crossed modules",
    IsIdenticalObj, [ IsDerivation, IsDerivation ], 0,
function( chi1, chi2 )
    return CompositeDerivation( chi2, chi1 );
end );

##############################################################################
##
#M  IsRegularDerivation               test whether a derivation has an inverse
##
InstallMethod( IsRegularDerivation, "method for derivations", true,
    [ IsDerivation ], 0,
function( chi )
    local  XM, S, bdy, R, genR, genrng, im, imrho, rho, ok;

    XM := Object2d( chi );
    S := Source( XM );
    bdy := Boundary( XM );
    R := Range( XM );
#    genR := GeneratorsOfGroup( R );
    genR := StrongGeneratorsStabChain( StabChain( R ) );
    im := GeneratorImages( chi );
    genrng := [ 1..Length( genR ) ];
    imrho := List( genrng, i -> genR[i] * Image( bdy, im[i] ) );
    rho := GroupHomomorphismByImages( R, R , genR, imrho);
    ok := ( ( rho <> fail ) and IsBijective( rho ) );
    return ok;
end );

#############################################################################
##
#M  InverseDerivations      Finds all semigroup inverses XJ for derivation Xi
##                                                i.e.  XiXjXi=Xi & XjXiXj=Xj
InstallMethod( InverseDerivations, "method for a derivation", true,
    [ IsDerivation ], 0,
function( chi )

    local  XM, x, D, L, size, j, chj, chk, chl, chh, chg, inv;

    XM := Object2d( chi );
    D := AllDerivations( XM );
    L := ImagesList( D );
    size := Length( L );
    inv := [ ];
    for j in [1..size] do
        chj := DerivationByImages( XM, L[j] );
        chk := CompositeDerivation( chi, chj );
        chl := CompositeDerivation( chk, chi );
        chh := CompositeDerivation( chj, chi );
        chg := CompositeDerivation( chh, chj );
        if ( ( GeneratorImages( chi ) = GeneratorImages( chl ) )
            and ( GeneratorImages( chg ) = GeneratorImages( chj ) ) ) then
            Add( inv, Position( L, GeneratorImages( chj ) ) );
        fi;
    od;
    return inv;
end );

##############################################################################
##
#M  ListInverseDerivations               List all inverses for each derivation
##
InstallMethod( ListInverseDerivations, "method for a crossed module", true,
    [ IsXMod ], 0,
function( XM )

    local  L, i, M, size, D, chi, inv;

    D := AllDerivations( XM );
    L := ImagesList( D );
    size := Length( L );
    M := 0 * [1..size];
    if IsBound( D.inverses ) then
        return D.inverses;
    fi;
    for i in [1..size] do
        chi := DerivationByImages( XM, L[i] );
        inv := InverseDerivations( chi );
        M[i] := inv;
    od;
    return M;
    end );

##############################################################################
##                                 Sections                                 ##
##############################################################################

##############################################################################
##
#M  SectionByImages                                   sets up GroupHomByImages
##
InstallMethod( SectionByImages, "method for a cat1-group", true,
    [ IsCat1, IsGroupHomomorphism ], 0,
function( C, xi )

    local  R, G, genR, ngR, nargs, usage, isect, im;

    G := Source( C );
    R := Range( C );
#    genR := GeneratorsOfGroup( R );
    genR := StrongGeneratorsStabChain( StabChain( R ) );
    ngR := Length( genR );
    if not ( ( Source( xi ) = R ) and ( Range( xi ) = G ) ) then
        Error( "require  xi : Range(C) -> Source(C)" );
    fi;
    im := List( genR, r -> Image( xi, r ) );
    SetIsUpMapping( xi, true );
    SetObject2d( xi, C );
    SetGeneratorImages( xi, im );
    isect := IsSection( xi );
    return xi;
end );

###############################################################################
##
#M  PrintObj( <xi> )  . . . . . . . . . . . . . . . . . . . . view a derivation
##
InstallMethod( PrintObj, "method for a section", true, [ IsSection ], 0,
    function( xi )
    local  obj;
    obj := Object2d( xi );
    Print( "SectionByImages( ", Range( obj ), ", ", Source( obj ), ", ",
        GeneratorsOfGroup( Range( obj ) ), ", ", GeneratorImages( xi ), " )" );
end );

################################################################################
##
#M  SectionByDerivation        the cat1-group section determined by a derivation
##
InstallMethod( SectionByDerivation, "method for a derivation", true,
    [ IsDerivation ], 0,
function( chi )

    local  XM, R, genR, ngR, xi, imchi, imxi, i, r, 
           er, eR, eK, eKchi, g, C;

    XM := Object2d( chi );
    R := Range( XM );
    genR := StrongGeneratorsStabChain( StabChain( R ) );
    ngR := Length( genR );
    C := Cat1OfXMod( XM );
    eR := RangeEmbedding( C );
    eK := KernelEmbedding( C );
    imchi := GeneratorImages( chi );
    eKchi := List( imchi, s -> Image( eK, s ) );
    imxi := 0 * [ 1..ngR ];
    for i in [ 1..Length( genR ) ] do
        r := genR[i];
        er := Image( eR, r );
        g := er * eKchi[i];
        imxi[i] := g;
    od;
    ## xi := SectionByImages( C, imxi );   use  NC
    xi := GroupHomomorphismByImages( R, Source(C), genR, imxi );
    xi := SectionByImages( C, xi );
    return xi;
end );

###############################################################################
##
#M  DerivationBySection       construct XMod derivation from cat1-group section
##
InstallMethod( DerivationBySection, "method for a section", true,
    [ IsSection ], 0,
function( xi )

    local  C, imxi, eR, eK, R, genR, ngR, S, XM, imchi, r, er, s, i, chi;

    if not IsSection( xi ) then
        Error( "Parameter must be a section of a cat1-group" );
    fi;
    C := xi.cat1;
    imxi := GeneratorImages( xi );
    XM := XModOfCat1( C );
    S := C.kernel;
    R := Range( C );
#    genR := GeneratorsOfGroup( R );
    genR := StrongGeneratorsStabChain( StabChain( R ) );
    ngR := Length( genR );
    eR := C.embedRange;
    eK := C.embedKernel;
    imchi := 0 * [ 1..ngR ];
    for i in [ 1..Length( genR ) ] do
        r := genR[i];
        er := Image( eR, r );
        s := er^(-1) * imxi[i];
        imchi[i] := PreImagesRepresentative( eK, s );
        Info( InfoXMod, 2, "In xi->chi :- ", [ i, r, er, s] );
    od;
    chi := DerivationByImages( XM, imchi );
    chi.isDerivation := IsDerivation( chi );
    return chi;
end );

##############################################################################
##
#M  CompositeSection                       Whitehead composite of two sections
##
InstallMethod( CompositeSection, "method for two sections", true,
    [ IsSection, IsSection ], 0,
function( xi, xj )

    local  C, R, genR, h, e, hxj, ehxj, xihxj, r, im1, im2, im3,
           numrng, k, imcomp, comp;

    C := Object2d( xi );
    if not ( C = Object2d( xj ) ) then
        Error( "<xi>,<xj> must be sections of the SAME cat1-group" );
    fi;
    R := Range( C );
#    genR := GeneratorsOfGroup( R );
    genR := StrongGeneratorsStabChain( StabChain( R ) );
    h := Head( C );
    e := RangeEmbedding( C );
    hxj := xj * h;
    ehxj := hxj * e;
    xihxj := hxj * xi;
    numrng := Length( genR );
    imcomp := 0 * [1..numrng];
    for k in [1..numrng] do
        r := genR[k];
        im1 := Image( xj, r );
        im2 := Image( ehxj, r )^(-1);
        im3 := Image( xihxj, r );
        imcomp[k] := im1 * im2 * im3;
    od;
    comp := SectionByImages( C, imcomp );
    return comp;
end );


###############################################################################
##                        Monoids of Derivations and Sections                ##
###############################################################################


###############################################################################
##
#M  MonoidOfUpMappingsObj( <obj>, <ims>, <str> )    construct upmappings record
##
InstallMethod( MonoidOfUpMappingsObj, "for a 2dObject", true,
    [ Is2dObject, IsHomogeneousList, IsString ], 0,
function( obj, images, str )

    local  filter, fam, mon;

    fam := FamilyObj( [ obj, images, str ] );
    filter := IsMonoidOfUpMappingsObj;
    mon := Objectify( NewType( fam, filter ), rec() );
    SetIsMonoidOfUpMappings( mon, true );
    SetObject2d( mon, obj );
    SetImagesList( mon, images );
    SetAllOrRegular( mon, str );
    if IsXMod( obj ) then
        SetIsMonoidOfDerivations( mon, true );
    elif IsCat1( obj ) then
        SetIsMonoidOfSections( mon, true );
    else
        Error( "<obj> not an xmod nor a cat1" );
    fi;
    return mon;
end );

##############################################################################
##
#M  ImagesTable                      returns list of lists of DerivationImages
##
InstallMethod( ImagesTable, "method for crossed module derivations", true,
    [ IsMonoidOfDerivations ], 0,
function( D )

    local  str, T, i, chi, L, XM, size;

    XM := Object2d( D );
    L := ImagesList( D );
    size := Length( L );
    T := 0 * [1..size];
    for i in [1..size] do
        chi := DerivationByImages( XM, L[i] );
        T[i] := ImagePositions( chi );
    od;
    return T;
end );

##############################################################################
##
#M  BacktrackDerivationsJ          recursive function for BacktrackDerivations
##
InstallMethod( BacktrackDerivationsJ, "method for a crossed module", true,
    [ IsXMod, IsHomogeneousList, IsHomogeneousList, IsHomogeneousList,
      IsInt, IsString ], 0,
function( XM, subs, imrho, imchi, j, str )
    
    local  Xsrc, onesrc, Xrng, isorng, genrng, derivgen, s, ok, rho, el, bdy,
           k, J, genJ, ord, r, aut, w, t, i, chi;

    Xsrc := Source( XM );
    Xrng := Range( XM );
    onesrc := One( Xsrc );
    genrng := StrongGeneratorsStabChain( StabChain( Xrng ) );
    el := Elements( Xsrc );
    k := Length( genrng );
    bdy := Boundary( XM );
    if ( k < j ) then
        # complete list of images found: if a derivation, add to genimagesList
        imchi := ShallowCopy( imchi );
        derivgen := [ imchi ];
        chi := DerivationByImages( XM, imchi );
        ok := ( ( chi <> fail ) and IsDerivation( chi ) );
        if ( ok and IsXMod( XM ) ) then
            SetIsDerivation( chi, true );
            if ( ok and ( str = "reg" ) ) then
                ok := IsRegularDerivation( chi );
            fi;
        fi;
        if not ok then
            derivgen := [ ];
        fi;
    else
        J := subs[j];
        genJ := GeneratorsOfGroup( J );
        derivgen := [ ];
        for s in el do
            imchi[ j ] := s;
            imrho[ j ] := genJ[j] * Image( bdy, s );
            rho := GroupHomomorphismByImages( J, Xrng, genJ, imrho );
            ok := IsGroupHomomorphism( rho );
            if ok then
                r := genJ[j];
                ord := Order( r );
                w := s;
                t := s;
                aut := Image( XModAction( XM ), r );
                for i in [1..ord-1] do
                    t := Image( aut, t );
                    w := t * w;
                od;
                if ( w <> onesrc ) then
                    ok := false;
                    Info( InfoXMod, 2, "test fails at j,r,s,w = ", [j,r,s,w] );
                fi;
            fi;
            if ok then
                Append( derivgen,
                    BacktrackDerivationsJ( XM, subs, imrho, imchi, j+1, str ) );
            fi;
            imrho := imrho{[1..j]};
        od;
    fi;
    return derivgen;
end );

##############################################################################
##
#M  BacktrackDerivations            recursive construction for all derivations
##
InstallMethod( BacktrackDerivations, "method for a crossed module", true,
    [ IsXMod, IsString ], 0,
function( XM, str )

    local  R, len, genR, subs, images, sorted, derivrec;

    if not ( str in [ "reg", "all" ] ) then
        Error( "Invalid reg|all string" );
    fi;
    R := Range( XM );
    genR := StrongGeneratorsStabChain( StabChain( R ) );
    len := Length( genR );
    subs := List( [1..len], i -> Subgroup( R, genR{[1..i]} ) ); 
    images := BacktrackDerivationsJ( XM, subs, [], [], 1, str );
    derivrec := MonoidOfUpMappingsObj( XM, images, str );
    return derivrec;
end );

##############################################################################
##
#M  RegularDerivations    find all invertible derivations for a crossed module
##
InstallMethod( RegularDerivations, "method for a crossed module", true, 
    [ IsXMod ], 0,
function( XM )

    local  how, ok;

    ok := true;
    return BacktrackDerivations( XM, "reg" );
end );

##############################################################################
##
#M  AllDerivations                   find all derivations for a crossed module
##
InstallMethod( AllDerivations, "method for a crossed module", true, 
    [ IsXMod ], 0,
function( XM )

    local  how, ok;

    ok := true;
    return BacktrackDerivations( XM, "all" );
end );

##############################################################################
##
#M  WhiteheadMonoidTable( XM )                 Table of products of derivations
##
InstallMethod( WhiteheadMonoidTable, "method for a crossed module", true,
    [ IsXMod ], 0,
function( XM )

    local  C, D, L, i, j, chi, images, J, M, size;

    D := AllDerivations( XM );
    L := ImagesList( D );
    size := Length( L );
    M := 0 * [1..size];
    C := 0 * [1..size];
    for i in [1..size] do
        C[i] := DerivationByImages( XM, L[i] );
    od;
    for i in [1..size] do
        J := 0 * [1..size];
        for j in [1..size] do
            chi := CompositeDerivation( C[i], C[j] );
            J[j] := Position( L, GeneratorImages( chi ) );
        od;
        M[i] := J;
    od;
    return M;
end );

###############################################################################
##
#M  WhiteheadGroupTable( XM )           Table of products of regular derivations
##
InstallMethod( WhiteheadGroupTable, "method for a crossed module", true,
    [ IsXMod ], 0,
function( XM )

    local  C, D, L, i, j, chi, images, J, M, reg, len;

    D := RegularDerivations( XM );
    reg := AllOrRegular( D );
    L := ImagesList( D );
    len := Length( L );
    M := 0 * [1..len];
    C := 0 * [1..len];
    for i in [1..len] do
        C[i] := DerivationByImages( XM, L[i] );
    od;
    for i in [1..len] do
        J := 0 * [1..len];
        for j in [1..len] do
            chi := CompositeDerivation( C[i], C[j] );
            J[j] := Position( L, GeneratorImages( chi ) );
        od;
        M[i] := J;
    od;
    return M;
end );

###############################################################################
##
#M  WhiteheadPermGroup( XM )                   perm rep for the Whitehead group
##
##  this needs some more work
##
InstallMethod( WhiteheadPermGroup, "method for a crossed module", true,
    [ IsXMod ], 0,
function( XM )
    local  tab, gens, grp;
    tab := WhiteheadGroupTable( XM );
    gens := List( tab, row -> PermList( row ) );
    ## grp := Group( gens );
    ## gens := StrongGeneratorsStabChain( StabChain( grp ) );
    return Group( gens );
end );

###############################################################################
##
#M  WhiteheadTransMonoid( XM )               trans rep for the Whitehead monoid
##
##  this needs some more work
##
InstallMethod( WhiteheadTransMonoid, "method for a crossed module", true,
    [ IsXMod ], 0,
function( XM )
    local  tab, gens;
    tab := WhiteheadMonoidTable( XM );
    gens := List( tab, row -> Transformation( row ) );
    return Monoid( gens );
end );

#############################################################################
##
#E  up2.gi  . . . . . . . . . . . . . . . . . . . . . . . . . . . . ends here
