#############################################################################
##
#W  util.gi                        XMOD Package                  Chris Wensley
#W                                                                 & Murat Alp
##
##  Installation file for functions of the XMOD package.
##
#H  @(#)$Id: util.gi,v 1.2 2002/04/19 gap Exp $
##
#Y  Copyright (C) 1999,2001 University of St. Andrews, North Haugh,
#Y                          St. Andrews, Fife KY16 9SS, Scotland
##
Revision.("xmod/gap/util_gi") := 
    "@(#)$Id: util.gi,v 2.001 2002/04/17 gap Exp $";

##############################################################################
##
#M  IdentitySubgroup . . . . . . . . . . . . . . . . . . . . . . .  of a group
##
InstallMethod( IdentitySubgroup, "of a group", true, [ IsGroup ], 0,
function( G )

    local  S;

    S := Subgroup( G, [ One( G ) ] );
    if HasName( G ) then
        SetName( S, Concatenation( "id(", Name( G ), ")" ) );
    fi;
    return S;
end );

#############################################################################
##
#M  ZeroMapping                                                   for groups
##
InstallMethod( ZeroMapping, "for groups", true,
              [IsGroup, IsGroup ], 0,
function( G, H )

    local  genG, imH, z, oneH;

    genG := GeneratorsOfGroup( G );
    oneH := One( H );
    imH := List( genG, x -> oneH );
    z := GroupHomomorphismByImages( G, H, genG, imH );
    return z;
end );

#############################################################################
##
#M  InclusionMapping( <G>, <H> )
##
InstallMethod( InclusionMapping, "generic method for subgroup",
               IsIdenticalObj, [ IsGroup, IsGroup ], 0,
function( G, H )

    local  genH, inc, ok;
    if not IsSubgroup( G, H ) then
        Error( "usage: InclusionMapping( G, H );  with  H <= G" );
    fi;
    genH := GeneratorsOfGroup( H );
    inc := GroupHomomorphismByImages( H, G, genH, genH );
    ok := IsInjective( inc );
    return inc;
end );

#############################################################################
##
#M  RestrictionMappingOp( <hom>, <src>, <rng> )
##
## InstallMethod( RestrictionMappingOp, "generic method for hom + 2 subgroups",
InstallMethod( RestrictionMapping, "generic method for hom + 2 subgroups",
               true, [ IsGeneralMapping, IsGroup, IsGroup ], 0,
function( hom, src, rng )

    local  S, R, gensrc, genrng, im, ok, res;

    S := Source( hom );
    R := Range( hom );
    if not ( IsSubgroup( S, src ) and IsSubgroup( R, rng ) ) then
        Error( "usage: RestrictionMappingOp( hom, src, rng ); ",
               "with src <= Source(hom), rng <= Range(hom)" );
    fi;
    gensrc := GeneratorsOfGroup( src );
    im := List( gensrc, g -> Image( hom, g ) );
    if not ForAll( im, r -> ( r in rng ) ) then
        Error( "Image(src) not contained in rng" );
    fi;
    res := GroupGeneralMappingByImages( src, rng, gensrc, im );
    SetIsGroupHomomorphism( res, true );
    return res;
end );

##############################################################################
##
#F  RestrictionMapping( <hom>, <src> ) 
#F  RestrictionMapping( <hom>, <src>, <rng> )
##
#InstallGlobalFunction( RestrictionMapping, function( arg )
#
#    local  nargs, hom, S, R;

#    nargs := Length( arg );
#    if not ( ( nargs > 1 ) and ( nargs < 4 ) ) then
#        Error( "Usage: RestrictionMapping( hom, src [,rng] );" );
#    fi;
#    hom := arg[1];
#    S := arg[2];
#    if ( nargs = 2 ) then
#        R := Range( hom );
#    else
#        R := arg[3];
#    fi;
#    return RestrictionMappingOp( hom, S, R );
#end );

#############################################################################
##
#M  Display( <list> )
##
InstallMethod( Display, "generic method for lists", true, [ IsList ], 0,
function( L )

    local  len, i;

    len := Length( L );
    if ( len < 2 ) then
        Print( L, "\n" );
    else
        Print( "[ " );
        for i in [1..len-1] do
            if IsBound( L[i] ) then
                Print( L[i], ",\n  " );
            else
                Print( "," );
            fi;
        od;
        Print( L[len], "\n  ]\n" );
    fi;
end );

#############################################################################
##
#M  IsEndomorphism( <f> )
##
## (10/04/02) changed to subgroup test, rather than equality
##
InstallMethod( IsEndomorphism,
    "generic method for homomorphisms", true, [ IsGroupHomomorphism ], 0,
function( f )
    return IsSubgroup( Source( f ), Range( f ) ); 
end );

#############################################################################
##
#M  IsAutomorphism( <f> )
##
InstallMethod( IsAutomorphism,
    "generic method for homomorphisms", true, [ IsGroupHomomorphism ], 0,
function( f )
    return ( ( Source( f ) = Range( f ) ) and IsBijective( f ) ); 
end );

#############################################################################
##
#M  IsGroupOfAutomorphisms
##
InstallOtherMethod( IsGroupOfAutomorphisms,
               "generic method for groups and groups",
               true, [ IsGroup, IsGroup ], 0,
function( A, S )

    local  a;

    if not IsGroupOfAutomorphisms( A ) then
        return false;
    fi;
    for a in GeneratorsOfGroup( a ) do
        if not ( ( Source( a ) = S ) and ( Range( a )  = S ) ) then
            return false;
        fi;
    od;
    return true;
end );

#############################################################################
##
#M  RModuleObject( <abgrp>, <act> )
##
InstallMethod( RModuleObject, "for abelian group and group action",
    true, [ IsCommutative, IsGroupHomomorphism ], 0,
function( ab, act )

    local  filter, fam, obj;

    fam := FamilyObj( [ ab, act ] );
    filter := IsRModuleObj;
    obj := Objectify( NewType( fam, filter ), rec() );
    SetRModuleGroup( obj, ab );
    SetRModuleAction( obj, act );
    SetIsRModule( obj, true );
    return obj;
end );

#############################################################################
##
#M  TrivialAction                                                  for groups
##
InstallMethod( TrivialAction, "for groups", true,
              [ IsGroup, IsGroup ], 0,
function( G, H )

    local  triv;

    triv := Group( IdentityMapping( G ) );
    return ZeroMapping( H, triv );
end );

##############################################################################
##
#M  InnerAutomorphismsOfNormalSubgroup       for a group and a normal subgroup
##
InstallMethod( InnerAutomorphismsOfNormalSubgroup,
    "for a group and a normal subgroup", true, [ IsGroup, IsGroup ], 0,
function( G, N )

    local  nargs, id, genG, genN, genA, A, g, conj, name;

    if not IsNormal( G, N ) then
        Error( "Second parameter must be normal subgroup of the first\n" );
    fi;
    if ( ( G = N ) and HasAutomorphismGroup( G ) ) then
        return InnerAutomorphismsAutomorphismGroup( G );
    fi;
    genG := GeneratorsOfGroup( G );
    genN := GeneratorsOfGroup( N );
    id := InclusionMapping( N, N );
    genA := [ ];
    for g in genG do
        conj := InnerAutomorphism( N, g );
        Add( genA, conj );
    od;
    A := Group( genA, id );
    SetIsGroupOfAutomorphisms( A, true );
    return A;
end );

#############################################################################
##
#F  InnerAutomorphismGroup( <G> ) 
#F  InnerAutomorphismGroup( <G>, <N> )
##
InstallGlobalFunction( InnerAutomorphismGroup, function( arg )

    local  nargs, G, N;

    nargs := Length( arg );
    # N not specified
    if ( ( nargs = 1 ) and IsGroup( arg[1] ) ) then
        G := arg[1];
        N := G;
    elif ( ( nargs = 2 ) and IsGroup( G ) and IsGroup( N ) ) then
        G := arg[1];
        N := arg[2];
    else
        Print( "usage: InnerAutomorphismGroup( G );\n" ); 
        Print( "   or: InnerAutomorphismGroup( G, N );\n" );
        return fail;
    fi;
    return InnerAutomorphismsOfNormalSubgroup( G, N );
end );

##############################################################################
##
#M  IsomorphismPermGroup                           for groups of automorphisms
##
InstallMethod( IsomorphismPermGroup, "for group of group automorphisms",
    true, [ IsGroupOfAutomorphisms ], 0,
function( A )

    local  G, a, genA, genG, eG1, eG2, new, num1, num2, ima, P, genP, a2p;

    a := One( A );
    genA := GeneratorsOfGroup( A );
    G := Source( a );
    genG := GeneratorsOfGroup( G );
    # determine the closure of genG under A
    eG2 := genG;
    num1 := 0;
    num2 := Length( genG );
    new := genG;
    while ( num1 <> num2 ) do
        num1 := num2;
        eG1 := eG2;
        for a in genA do
            ima := List( new, g -> Image( a, g ) );
            eG2 := Union( eG2, ima );
        od;
        new := Difference( eG2, eG1 );
        num2 := Length( eG2 );
    od;
    # construct perm group P using action of A on this closure
    P := Operation( A, eG2 );
    genP := GeneratorsOfGroup( P );
    if HasName( G ) then
        if not HasName( A ) then
            if ( HasAutomorphismGroup( G ) 
                 and ( A = AutomorphismGroup( G ) ) ) then
                SetName( A, Concatenation( "Aut(", Name( G ), ")" ) );
            else
                SetName( A, Concatenation( "SubAut(", Name( G ), ")" ) );
            fi;
        fi;
        SetName( P, Concatenation( "Perm", Name( A ) ) );
    fi;
    a2p := OperationHomomorphism( A, P );
    return a2p;
end );

##############################################################################
##
#M  Embedding                              for permutation semidirect products
##
InstallMethod( Embedding, "generic method for perm semidirect products",
    true, [ IsPermGroup and HasSemidirectProductInfo, IsPosInt ], 0,
function( D, i )
    local  info, G, genG, imgs, hom;

    info := SemidirectProductInfo( D );
    if IsBound( info.embeddings[i] ) then
        return info.embeddings[i];
    fi;
    G := info.groups[i];
    genG := GeneratorsOfGroup( G );
    imgs := GeneratorsOfGroup( D ){[info.lenlist[i]+1 .. info.lenlist[i+1]]};
    hom := GroupHomomorphismByImages( G, D, genG, imgs );
    SetIsInjective( hom, true );
    info.embeddings[i] := hom;
    return hom;
end );

##############################################################################
##
#M  Projection                             for permutation semidirect products
##
InstallOtherMethod( Projection, "generic method for perm semidirect products",
    true, [ IsPermGroup and HasSemidirectProductInfo ], 0,
function( D )
    local  info, genD, G, genG, imgs, hom, ker, list;

    info := SemidirectProductInfo( D );
    if not IsBool( info.projections ) then
        return info.projections;
    fi;
    G := info.groups[1];
    genG := GeneratorsOfGroup( G );
    genD := GeneratorsOfGroup( D );
    list := [ info.lenlist[2]+1 .. info.lenlist[3] ];
    imgs := Concatenation( genG, List( list, j -> () ) );
    hom := GroupHomomorphismByImagesNC( D, G, genD, imgs );
    SetIsSurjective( hom, true );
    ker := Subgroup( D, genD{ list } );
    SetKernelOfMultiplicativeGeneralMapping( hom, ker );
    info.projections := hom;
    return hom;
end );

##############################################################################
##
#M  EndomorphismClassObj( <nat>, <iso>, <aut>, <conj> ) . . . . . make a class
##
InstallMethod( EndomorphismClassObj,
  "generic method for an endomorphism class", true,
  [ IsGroupHomomorphism, IsGroupHomomorphism, IsGroupOfAutomorphisms, IsList ],
  0,
function( nat, iso, aut, conj )

    local  filter, fam, class;

    fam := FamilyObj( [ nat, iso, aut, conj ] );
    filter := IsEndomorphismClassObj;
    class := Objectify( NewType( fam, filter ), rec() );
    SetIsEndomorphismClass( class, true );
    SetNaturalHomEndoClass( class, nat );
    SetIsomorphism( class, iso );
    SetAutoGroup( class, aut );
    SetConjugators( class, conj );
    return class;
end );

#############################################################################
##
#M  Display . . . . . . . . . . . . . . . . . . . . . for endomorphism classes
##
InstallMethod( Display, "generic method for endomorphism classes",
    true, [ IsEndomorphismClassObj ], 0,
function( class )
    Print( "class of group endomorphisms with\n" );
    Print( "natural hom: " );
    ViewObj( NaturalHomEndoClass( class ) );
    Print( "\n" );
    Print( "isomorphism: " );
    ViewObj( Isomorphism( class ) );
    Print( "\n" );
    Print( "autogp gens: ", GeneratorsOfGroup( AutoGroup( class ) ), "\n" );
    Print( "conjugators: ", Conjugators( class ), "\n" );
end );

#############################################################################
##
#M  ZeroEndomorphismClass( <G> )
##
InstallMethod( ZeroEndomorphismClass, "generic method for a group",
    true, [ IsGroup ], 0, 
function( G )

    local  Q, nat, iso, aut, conj, idgp;

    Q := Group( One( G ) );
    nat := ZeroMapping( G, Q );
    iso := IdentityMapping( Q );
    aut := Group( iso );
    SetIsAutomorphismGroup( aut, true );
    conj := [ One( G ) ];
    return EndomorphismClassObj( nat, iso, aut, conj );
end );

#############################################################################
##
#M  AutomorphismClass( <G> )
##
InstallMethod( AutomorphismClass, "generic method for a group", true,
    [ IsGroup ], 0, 
function( G )

    local  iso, aut, conj;

    aut := AutomorphismGroup( G );
    iso := InclusionMapping( G, G );
    conj := [ One( G ) ];
    return EndomorphismClassObj( iso, iso, aut, conj );
end );

#############################################################################
##
#M  NontrivialEndomorphismClasses( <G> )
##
InstallMethod( NontrivialEndomorphismClasses, "generic method for a group",
    true, [ IsGroup ], 0, 
function( G )

    local  nargs, valid, case, switch, oldcase, normG, rcosG, N, nnum,
           natG, quotG, genG, oneG, Qnum, ccs, reps, cnum, Q, R, iso, L, im,
           phi, nat, proj, comp, normal, cosets, conj, Cnum, g, gim, zero,
           i, j, k, l, Lnum, aut, idgp, Ecl, Enum, classes, class, name, ok;

    genG := GeneratorsOfGroup( G );
    oneG := One( G );
    idgp := Subgroup( G, [ One(G) ] );
    # determine non-monomorphic endomorphisms by kernel
    normG := NormalSubgroups( G );
    nnum := Length( normG );
    ccs := ConjugacyClassesSubgroups( G );
    cnum :=Length( ccs );
    reps := List( ccs, c -> Representative( c ) );
    rcosG := List( normG, N -> RightCosets( G, N ) );
    natG := List( normG, N -> NaturalHomomorphismByNormalSubgroup( G, N ) );
    quotG := List( natG, n -> Image( n ) );
    normal := List( reps, R -> ( Normalizer( G, R ) = G ) );
    classes := [ ];
    for i in [2..(nnum-1)] do
        N := normG[i];
        Q := quotG[i];
        proj := natG[i];
        aut := AutomorphismGroup( Q );
        for j in [2..(cnum-1)] do
            R := reps[j];
            ok := ( Intersection( N, R ) = idgp );
            iso := IsomorphismGroups( Q, R );
            if not ( iso = fail ) then
                # (unnecessary?) check that this gives a homomorphism
                comp := CompositionMapping( iso, proj );
                im := List( genG, x -> Image( comp, x ) );
                phi := GroupHomomorphismByImages( G, R, genG, im );
                if not IsGroupHomomorphism( phi ) then
                    Error( "phi not a homomorphism" );
                fi;
                if not normal[j] then
                    cosets := RightCosets( G, Normalizer( G, R ) );
                    conj := List( cosets, c -> Representative( c ) );
                else
                    conj := [ oneG ];
                fi;
                class := EndomorphismClassObj( proj, iso, aut, conj );
                Add( classes, class );
            fi;
        od;
    od;
    return classes;
end );

#############################################################################
##
#M  NonIntersectingEndomorphismClasses( <G> )
##
InstallMethod( NonIntersectingEndomorphismClasses,
    "generic method for a group", true, [ IsGroup ], 0, 
function( G )

    local  nargs, valid, case, switch, oldcase, normG, rcosG, N, nnum,
           natG, quotG, genG, oneG, Qnum, ccs, reps, cnum, Q, R, iso, L, im,
           phi, nat, proj, comp, normal, cosets, conj, Cnum, g, gim, zero,
           i, j, k, l, Lnum, aut, idgp, Ecl, Enum, classes, class, name, ok;

    genG := GeneratorsOfGroup( G );
    oneG := One( G );
    idgp := Subgroup( G, [ One(G) ] );
    # determine non-monomorphic endomorphisms by kernel
    normG := NormalSubgroups( G );
    nnum := Length( normG );
    ccs := ConjugacyClassesSubgroups( G );
    cnum :=Length( ccs );
    reps := List( ccs, c -> Representative( c ) );
    rcosG := List( normG, N -> RightCosets( G, N ) );
    natG := List( normG, N -> NaturalHomomorphismByNormalSubgroup( G, N ) );
    quotG := List( natG, n -> Image( n ) );
    normal := List( reps, R -> ( Normalizer( G, R ) = G ) );

    classes := [ ];
    for i in [2..(nnum-1)] do
        N := normG[i];
        Q := quotG[i];
        proj := natG[i];
        aut := AutomorphismGroup( Q );
        for j in [2..(cnum-1)] do
            R := reps[j];
            ok := ( Intersection( N, R ) = idgp );
            if ok then
                iso := IsomorphismGroups( Q, R );
            fi;
            if ( ok and not ( iso = fail ) ) then
                # (unnecessary?) check that this gives a homomorphism
                comp := CompositionMapping( iso, proj );
                im := List( genG, x -> Image( comp, x ) );
                phi := GroupHomomorphismByImages( G, R, genG, im );
                if not IsGroupHomomorphism( phi ) then
                    Error( "phi not a homomorphism" );
                fi;
                if not normal[j] then
                    cosets := RightCosets( G, Normalizer( G, R ) );
                    conj := List( cosets, c -> Representative( c ) );
                else
                    conj := [ oneG ];
                fi;
                class := EndomorphismClassObj( proj, iso, aut, conj );
                Add( classes, class );
            fi;
        od;
    od;
    return classes;
end );

##############################################################################
##
#F  EndomorphismClasses                        finds all homomorphisms  G -> G
##
InstallGlobalFunction( EndomorphismClasses, 
function( arg )
    
    local  nargs, valid, G, case, classes, auts, ends, zero, disj;

    nargs := Length( arg );
    G := arg[1];
    valid := ( IsGroup( G ) and ( nargs = 2 ) and ( arg[2] in [0..3] ) );
    if not valid then
         Print( "\nUsage:  EndomorphismClasses( G [, case] );\n" );
         Print( " choose  case = 1  to include automorphisms & zero,\n" );
         Print( "default  case = 2  to exclude automorphisms & zero,\n" );
         Print( "         case = 3  when  N meet H  trivial,\n" );
         return fail;
    fi;
    if ( Length( arg ) = 1 ) then
        case := 2;
    else
        case := arg[2];
    fi;
    ends := NontrivialEndomorphismClasses( G );
    if ( case = 1 ) then
        zero := ZeroEndomorphismClass( G );
        auts := AutomorphismClass( G );
    fi;
    if ( case = 2 ) then
        classes := ends;
    elif ( case = 1 ) then
        classes := Concatenation( [auts], ends, [zero] );
    else
        classes := NonIntersectingEndomorphismClasses( G );
    fi;
    return classes;
end );

###############################################################################
##
#M  EndomorphismImages           finds all homomorphisms  G -> G  by converting
##                                 EndomorphismClasses into a list of genimages
##
InstallMethod( EndomorphismImages,
    "generic method for a list of endomorphism classes", true, [ IsList ], 0,
function( classes )

    local  clnum, G, genG, Q, autos, rho, psi, phi, L, Lnum, LR, k, l,
           im, g, gim, i, c, nat, iso, aut, conj, cjnum, comp, R;

    if ( ( classes = [] ) or not IsEndomorphismClass( classes[1] ) ) then
        Error( "usage: EndomorphismImages( <list of endo classes> );" );
    fi;
    c := classes[1];
    nat := NaturalHomEndoClass( c );
    G := Source( nat );
    genG := GeneratorsOfGroup( G );
    clnum := Length( classes );
    L := [ ];
    for i in [1..clnum] do
        c := classes[i];
        nat := NaturalHomEndoClass( c );
        iso := Isomorphism( c );
        aut := AutoGroup( c );
        conj := Conjugators( c );
        comp := CompositionMapping( iso, nat );
        R := Image( comp );
        cjnum := Length( conj );
        Q := Image( nat );
        autos := Elements( aut );
        LR := [ ];
        for k in [ 1..Length( autos ) ] do
            rho := autos[k];
            psi := nat * rho * iso;
            im := List( genG, x -> Image( psi, x ) );
            Add( LR, im );
        od;
        Lnum := Length( LR );
        for k in [2..cjnum] do
            g := conj[k];
            for l in [1..Lnum] do
                im := LR[l];
                gim := List( im, x -> x^g );
                Add( LR, gim );
            od;
        od;
        L := Concatenation( L, LR ); 
    od;
    return L;
end );

###############################################################################
##
#M  IdempotentImages          finds all homomorphisms  h : G -> G  with h^2 = h
##                   by converting EndomorphismClasses into a list of genimages
##
InstallMethod( IdempotentImages,
    "generic method for a list of endomorphism classes", true, [ IsList ], 0,
function( classes )

    local  clnum, G, genG, Q, autos, rho, psi, phi, L, Lnum, LR, LR2, k, l,
           im, im2, psi2, g, gim, i, c, nat, iso, aut, conj, cjnum, comp, R;

    if ( ( classes = [] ) or not IsEndomorphismClass( classes[1] ) ) then
        Error( "usage: EndomorphismImages( <list of endo classes> );" );
    fi;
    c := classes[1];
    nat := NaturalHomEndoClass( c );
    G := Source( nat );
    genG := GeneratorsOfGroup( G );
    clnum := Length( classes );
    L := [ ];
    for i in [1..clnum] do
        c := classes[i];
        nat := NaturalHomEndoClass( c );
        iso := Isomorphism( c );
        aut := AutoGroup( c );
        conj := Conjugators( c );
        comp := CompositionMapping( iso, nat );
        R := Image( comp );
        cjnum := Length( conj );
        Q := Image( nat );
        autos := Elements( aut );
        LR := [ ];
        for k in [ 1..Length( autos ) ] do
            rho := autos[k];
            psi := nat * rho * iso;
            im := List( genG, x -> Image( psi, x ) );
            Add( LR, im );
        od;
        LR2 := [ ];
        Lnum := Length( LR );
        for k in [1..cjnum] do
            g := conj[k];
            for l in [1..Lnum] do
                im := LR[l];
                gim := List( im, x -> x^g );
                psi2 := GroupHomomorphismByImages( G, G, genG, gim );
                im2 := List( gim, x -> Image( psi2, x ) );
                if ( gim = im2 ) then
                    Add( LR2, gim );
                fi;
            od;
        od;
        L := Concatenation( L, LR2 ); 
    od;
    return L;
end );

## ???? replace by ElementsInGenerators ???? ##

##############################################################################
##
#M  GenerationOrder         elements of G generated as words in the generators
##
InstallMethod( GenerationOrder, "method for a group", true, [ IsGroup ], 0,
function( G )

    local  oG, eG, ord, ngG, genG, g, i, j, k, P, pos, n, x, y;

#    genG := GeneratorsOfGroup( G );
    genG := StrongGeneratorsStabChain( StabChain( G ) );
    ngG := Length( genG );
    oG := Size( G );
    eG := Elements( G );
    ord := 0 * [1..oG];
    ord[1] := 1;
    P := 0 * [1..oG];
    P[1] := [ 1, 0 ];
    for n in [1..ngG] do
        g := genG[n];
        pos := Position( eG, g );
        P[pos] := [ 1, n ];
        ord[ n+1 ] := pos;
    od;
    n := ngG + 1;
    k := 2;
    while ( k < oG ) do
        i := ord[k];
        x := eG[i];
        for j in [1..ngG] do
            g := genG[j];
            y := x * g;
            pos := Position( eG, y );
            if ( P[pos] = 0 ) then
                P[pos] := [ i, j ];
                n := n+1;
                ord[n] := pos;
            fi;
        od;
        k := k+1;
    od;
    SetGenerationPairs( G, P );
    return ord;
end );

##############################################################################
##
#M  CheckGenerationPairs             G.generationPairs, G.generationOrder ok ?
##
InstallMethod( CheckGenerationPairs, "method for a ", true, [ IsGroup ], 0,
function( G )

    local  eG, oG, genG, P, i, g, x;

    genG := StrongGeneratorsStabChain( StabChain( G ) );
    oG := Size( G );
    eG := Elements( G );
    P := GenerationPairs( G );
    if ( P[1] <> [1,0] ) then
        return false;
    fi;
    for i in [2..oG] do
        x := eG[ P[i][1] ];
        g := genG[ P[i][2] ];
        if ( x*g <> eG[i] ) then
            Info( InfoXMod, 2, x, " * ", g, " <> ", eG[i] );
            return false;
        fi;
    od;
    return true;
end );

#############################################################################
##
#E  util.gi . . . . . . . . . . . . . . . . . . . . . . . . . . . . ends here
