########################################################################
##
#F  AllOneVector( <n> [, <field> ] )
##
##  Return a vector with all ones.
##

InstallMethod(AllOneVector, "length, Field", true, [IsInt, IsField], 0, 
function(n, F) 
    if n <= 0 then
        Error( "AllOneVector: <n> must be a positive integer" );
    fi;
    return List( [ 1 .. n ], x -> One(F) );
end);

InstallOtherMethod(AllOneVector, "length, fieldsize", true, [IsInt, IsInt], 0, 
function(n, q) 
	return AllOneVector(n, GF(q)); 
end); 

InstallOtherMethod(AllOneVector, "length", true, [IsInt], 0, 
function(n) 
	return AllOneVector(n, Rationals); 
end); 


########################################################################
##
#F  AllOneCodeword( <n>, <field> )
##
##  Return a codeword with <n> ones.
##

InstallMethod(AllOneCodeword, "wordlength, field", true, [IsInt, IsField], 0, 
function(n, F) 
    if n <= 0 then
        Error( "AllOneCodeword: <n> must be a positive integer" );
    fi;
    return Codeword( AllOneVector( n, F ), F );
end);

InstallOtherMethod(AllOneCodeword, "wordlength, fieldsize", true, 
	[IsInt, IsInt], 0, 
function(n, q) 
	return AllOneCodeword(n, GF(q)); 
end); 

InstallOtherMethod(AllOneCodeword, "wordlength", true, [IsInt], 0, 
function(n) 
	return AllOneCodeword(n, GF(2)); 
end); 


#############################################################################
##
#F  IntCeiling( <r> )
##
##  Return the smallest integer greater than or equal to r.
##  3/2 => 2,  -3/2 => -1.
##

InstallMethod(IntCeiling, "method for integer", true, [IsInt], 0, 
function(r) 
	# don't round integers 
	return r; 
end); 

InstallMethod(IntCeiling, "method for rational", true, [IsRat], 0, 
function(r) 
	if r > 0 then
		# round positive numbers to smallest integer 
		# greater than r (3/2 => 2)
		return Int(r)+1;
	else
		# round negative numbers to smallest integer
		# greater than r (-3/2 => -1)
		return Int(r);
	fi;
end);


########################################################################
##
#F  IntFloor( <r> ) 
##
##  Return the greatest integer smaller than or equal to r.
##  3/2 => 1, -3/2 => -2.
##

InstallMethod(IntFloor, "method for integer", true, [IsInt], 0, 
function(r) 
	# don't round integers
	return r;
end); 

InstallMethod(IntFloor, "method for rational", true, [IsRat], 0, 
function(r) 
	if r > 0 then
		# round positive numbers to largest integer
		# smaller than r (3/2 => 1)
		return Int(r);
	else
		# round negative numbers to largest integer
		# smaller than r (-3/2 => -2)
		return Int(r-1);
    fi;
end);


########################################################################
##
#F  KroneckerDelta( <i>, <j> )
##
##  Return 1 if i = j,
##         0 otherwise
##

InstallMethod(KroneckerDelta, true, [IsInt, IsInt], 0, 
function ( i, j )
    
    if i = j then
        return 1;
    else
        return 0;
    fi;
    
end);


########################################################################
##
#F  BinaryRepresentation( <elements>, <length> )
##
##  Return a binary representation of an element
##  of GF( 2^k ), where k <= length.
##  
##  The representation is actually the binary 
##  representation of k+1, where k is the exponent 
##  of the element, taken in the field 2^length.  
##  For example, Z(16)^10 = Z(4)^2.  If length = 4, 
##  the binary representation 1011 = 11(base 10) 
##  is returned.  If length = 2, the binary 
##  representation 11 = 3(base 10) is returned. 
##  
##  If elements is a list, then return the binary
##  representation of every element of the list.
##
##  This function is used to make to Gabidulin codes.
##  It is not intended to be a global function, but including
##  it in all five Gabidulin codes is a bit over the top
##
##  Therefore, no error checking is done.
##

BinaryRepresentation := function ( elementlist, length )
    
    local field, i, log, vector, element;
    
    if IsList( elementlist ) then
        return( List( elementlist,
                      x -> BinaryRepresentation( x, length ) ) );
    else
        
        element := elementlist;
        field := Field( element );

        vector := NullVector( length, GF(2) );
    
        if element = Zero(field) then 
            # exception, log is not defined for zeroes
            return vector;
        else
            log := LogFFE( element, Z(2^length) ) + 1; 
        
            for i in [ 1 .. length ] do
                if log >= 2^( length - i ) then
                    vector[ i ] := One(GF(2)); 
                    log := log - 2^( length - i );
                fi;
            od;
        
            return vector;
        fi;
    fi;
end;


########################################################################
##
#F  SortedGaloisFieldElements( <size> )
##
##  Sort the field elements of size <size> according to
##  their log.
##
##  This function is used to make to Gabidulin codes.
##  It is not intended to be a global function, but including
##  it in all five Gabidulin codes is not a good idea.
##

SortedGaloisFieldElements := function ( size )
    
    local field, els, sortlist;
    
    if IsInt( size ) then
        field := GF( size );
    else
        field := size;
        size := Size( field );
    fi;
    
    els := ShallowCopy(AsSSortedList( field ));
    sortlist := NullVector( size );
    # log 1 = 0, so we add one to each log to avoid
    # conflicts with the 0 for zero.

    sortlist := List( els, function( x )
        if x = Zero(field) then
            return 0;
        else
            return LogFFE( x, PrimitiveRoot( field ) ) + 1;
        fi;
        end );

    sortlist{ [ 2 .. size ] } := List( els { [ 2 .. size ] },
                                 x -> LogFFE( x, PrimitiveRoot( field ) ) + 1 );
    SortParallel( sortlist, els );
    
    return els;
end;

