#############################################################################
##
#W  intmat.gi                    Polycyc                         Bettina Eick
##                                                              Werner Nickel
##
##
## Functions for integer matrices 
##

NullspaceIntMat := function( A )
    local hnfm, rels, zero, j;

    # compute hnf
    hnfm := HermiteNormalFormIntegerMatTransforms( A );
    rels := hnfm.rowtrans;
    hnfm := hnfm.normal;

    # get relations
    zero := hnfm[1] * 0;
    j := Position( hnfm, zero );
    if not IsBool( j ) then 
        return rels{[j..Length(rels)]};
    else
        return []; 
    fi;
end;

BasisIntMat := function( A )
    local hnfm, zero, j;
    hnfm := HermiteNormalFormIntegerMat( A );
    zero := hnfm[1] * 0;
    j := Position( hnfm, zero );
    if not IsBool( j ) then
        hnfm := hnfm{[1..j-1]};
    fi;
    return hnfm;
end;

SolutionIntMat := function( A, s )
    local B, N, H;
    B := Concatenation( [s], A );
    N := NullspaceIntMat( B );
    H := HermiteNormalFormIntegerMat( N );
    if H[1][1] = 1 then 
        return -H[1]{[2..Length(H[1])]};
    else
        return fail;
    fi;
end;


##
## Some general functions 
##

#############################################################################
##
#F OnBases( base, mat )
##
OnBases := function( base, mat )
    local new;
    new := base * mat;
    if Length( new ) = 0 then
        return new;
    elif IsFFE( new[1][1] ) then
        TriangulizeMat( new );
    else
        new := TriangulizedIntegerMat( new );
    fi;
    return new;
end;

#############################################################################
##
#F FixedPoints( pts, gens, oper )
##
FixedPoints := function( pts, gens, oper )
    return Filtered( pts, x -> ForAll( gens, y -> oper( x, y ) = x ) );
end;



#############################################################################
##
##  The following functions invert an integer matrix by computing successive
##  approximations p-addically for a given prime p.
##

##
##  Convert a matrix over GF(p) to an integer matrix with entries in
##  the range [(1-p)/2,..,(p-1)/2].
##
IntMatSym := function( M )
    local i, j, p, IM;
    
    p := Characteristic( M[1][1] );
    IM := [];
    for i in [1..Length(M)] do 
        IM[i] := [];
        for j in [1..Length(M[i])] do
            IM[i][j] := IntFFE( M[i][j] );
            if 2 * IM[i][j] + 1 > p then
                IM[i][j] := IM[i][j] - p;
            fi;
        od;
    od;
    return IM;
end;

##
##  Compute the inverse of an integer matrix by p-adic approximations.
##
InverseMatModular := function( T, p )
    
    local   e,       #  identity of GF(p)
            Tp,      #  inverse modulo p
            Tr,      #  the remainder term
            Ti,      #  the inverse mod p^k
            k,       #  iteration variable
            N,       #  the null matrix
            t;       #  
    
    e  := One( GF(p) );
    N  := NullMat( Length(T), Length(T), Integers );
    Tp := (e*T)^-1;
    
    Tr := IdentityMat( Length(T), Integers );
    Ti := N;
    k  := 0;
    while Tr <> N do         # loop invariant: T * Ti + p^k * Tr = I

        t  := IntMatSym( Tp * (e*Tr) );
        Ti := Ti + p^k * t;
        Tr := (Tr - T * t) / p;
        k := k+1;
    od;
    
    return Ti;
end;

InverseIntegerMat := M -> InverseMatModular( M, 251 );

