#############################################################################
##
#W  grpint.gi                  Polycyc                           Bettina Eick
##

#############################################################################
##
#F NormalIntersection( N, U ) . . . . . . . . . . . . . . . . . . . .U \cap N
##
## Note that U must normalize N or N must normalize U.
##
InstallOtherMethod( NormalIntersection, true, [IsPcpGroup, IsPcpGroup], 0,
function( N, U )
    local G, igs, igsN, igsU, n, s, I, id, ls, rs, is, g, d, al, ar, e, tm;

    G    := Parent( N );
    igs  := Igs(G);
    igsN := Igs( N );
    igsU := Igs( U );
    n    := Length( igs );

    # if N or U is trivial
    if Length( igsN ) = 0 or Length( igsU ) = 0 then
        return SubgroupByIgs(G, [] );
    fi;

    # if N or U are equal to G
    if Length( igsN ) = n and ForAll(igsN, x -> LeadingExponent(x) = 1) then 
        return U;
    elif Length(igsU) = n and ForAll(igsU, x -> LeadingExponent(x) = 1) then 
        return N;
    fi;
  
    # if N is a tail
    s := Depth( igsN[1] );
    if Length( igsN ) = n-s+1 and 
       ForAll( igsN, x -> LeadingExponent(x) = 1 ) then
        I := Filtered( igsU, x -> Depth(x) >= s );
        return SubgroupByIgs( G, I );
    fi;

    # otherwise compute
    id := One(G);
    ls := List( igs, x -> id );
    rs := List( igs, x -> id );
    is := List( igs, x -> id );

    for g in igsU do
        d := Depth( g );
        ls[d] := g;
        rs[d] := g;
    od;

    I := [];
    for g in igsN do
        d := Depth( g );
        if ls[d] = id  then
            ls[d] := g;
        else
            Add( I, g );
        fi;
    od;

    # enter the pairs [ u, 1 ] of <I> into [ <ls>, <rs> ]
    for al  in I  do
        ar := id;
        d  := Depth( al );

        # compute sum and intersection
        while al <> id and ls[d] <> id  do
            e := Gcdex( LeadingExponent(ls[d]), LeadingExponent(al) );
            tm := ls[d]^e.coeff1 * al^e.coeff2;
            al := ls[d]^e.coeff3 * al^e.coeff4;
            ls[d] := tm;
            tm := rs[d]^e.coeff1 * ar^e.coeff2;
            ar := rs[d]^e.coeff3 * ar^e.coeff4;
            rs[d] := tm;
            d := Depth( al );
        od;

        # we have a new sum generator
        if al <> id  then
            ls[d] := al;
            rs[d] := ar;
       
        # we have a new intersection generator
        elif ar <> id then
            d := Depth( ar );
            while ar <> id and is[d] <> id  do
                e  := Gcdex(LeadingExponent( is[d] ), LeadingExponent( ar ));
                tm := is[d]^e.coeff1 * ar^e.coeff2;
                ar := is[d]^e.coeff3 * ar^e.coeff4;
                is[d] := tm;
                d  := Depth( ar );
            od;
            if ar <> id  then
                is[d] := ar;
            fi;
        fi;
    od;

    # sum := Filtered( ls, x -> x <> id );
    I := Filtered( is, x -> x <> id );
    return SubgroupByIgs( G, I );
end );

#############################################################################
##
#F AbelianIntersection( baseN, baseU )  . . . . . . . . . . . . . . .U \cap N
##
## N and U are subgroups of a free abelian group given by exponents.
##
AbelianIntersection := function( baseN, baseU )
    local n, s, id, ls, rs, is, g, I, al, ar, d, l1, l2, e, tm; 

    # if N or U is trivial
    if Length( baseN ) = 0 or Length( baseU ) = 0 then return []; fi;
    n := Length( baseN[1] );

    # if N or U are equal to G
    if Length( baseN ) = n then return baseU;
    elif Length( baseU ) = n then return baseN; fi;
  
    # if N is a tail
    s := PositionNonZero( baseN[1] );
    if Length( baseN ) = n-s+1 and 
    ForAll( baseN, x -> x[PositionNonZero(x)] = 1 ) then
        return Filtered( baseU, x -> Depth(x) >= s );
    fi;

    # otherwise compute
    id := List( [1..n], x -> 0 );
    ls := MutableIdentityMat( n );
    rs := MutableIdentityMat( n );
    is := MutableIdentityMat( n );

    for g in baseU do
        d := PositionNonZero( g );
        ls[d] := g;
        rs[d] := g;
    od;

    I := [];
    for g in baseN do
        d := PositionNonZero( g );
        if ls[d] = id  then
            ls[d] := g;
        else
            Add( I, g );
        fi;
    od;

    # enter the pairs [ u, 1 ] of <I> into [ <ls>, <rs> ]
    for al  in I  do
        ar := id;
        d  := Depth( al );

        # compute sum and intersection
        while al <> id and ls[d] <> id  do
            l1 := ls[d][d];
            l2 := al[d];            
            e := Gcdex( l1, l2 );
            tm := e.coeff1 * ls[d] +  e.coeff2 * al;
            al := e.coeff3 * ls[d] +  e.coeff4 * al;
            ls[d] := tm;
            tm := e.coeff1 * rs[d] +  e.coeff2 * ar;
            ar := e.coeff3 * rs[d] +  e.coeff4 * ar;
            rs[d] := tm;
            d := Depth( al );
        od;

        # we have a new sum generator
        if al <> id  then
            ls[d] := al;
            rs[d] := ar;
       
        # we have a new intersection generator
        elif ar <> id then
            d := Depth( ar );
            while ar <> id and is[d] <> id  do
                l1 := is[d][d];
                l2 := ar[d];
                e  := Gcdex( l1, l2 );
                tm := e.coeff1 * is[d] +  e.coeff2 * ar;
                ar := e.coeff3 * is[d] +  e.coeff4 * ar;
                is[d] := tm;
                d  := Depth( ar );
            od;
            if ar <> id  then
                is[d] := ar;
            fi;
        fi;
    od;
    return Filtered( is, x -> x <> id );
end;

