/*
** 1999-12-23 -	Hm, perhaps this is an unfortunate choice of module title. It might bloat,
**		or something. :) Anyway, the purpose of this module is to help handle
**		(toplevel) windows, specifically remembering their sizes and positions.
*/

#include "gentoo.h"

#include <ctype.h>

#include "window.h"

/* ----------------------------------------------------------------------------------------- */

typedef struct {
	guint32		id;		/* Unique identifier for this windata. */
	WinType		type;		/* What type of window? */
	gboolean	persistent;	/* Makes close() actually do a hide(). */
	const gchar	*title;		/* Text for window's title bar. */
	const gchar	*label;		/* Short descriptive label (like "main", "config" etc). */
	gboolean	pos_grab;
	gboolean	size_grab;
	/* Data below here is stored in config file. */
	gint		x, y;		/* Window's opening position. */
	gint		width, height;	/* The desired size of this window. */
	gboolean	pos_use;	/* Should we set use the recorded position? */
	gboolean	pos_update;	/* Update recorded position when window closes? */
	gboolean	size_use;	/* Use recorded size? */
	gboolean	size_update;	/* Update recorded size with current size on close? */
} WinDef;

struct WinInfo {
	MainInfo	*min;		/* The handiness of this pointer cannot be overstated. */
	GList		*windows;	/* List of WinDef:s as per above. There's no rush. */
};

/* ----------------------------------------------------------------------------------------- */

static WinDef *	window_find(const WinInfo *wi, guint32 id);

/* ----------------------------------------------------------------------------------------- */

WinInfo * win_wininfo_new(MainInfo *min)
{
	WinInfo	*wi;

	wi = g_malloc(sizeof *wi);
	wi->min	    = min;
	wi->windows = NULL;

	return wi;
}

/* 1999-12-23 -	Create a new WinInfo, initialized with the "default" (i.e., legacy) windows. */
WinInfo * win_wininfo_new_default(MainInfo *min)
{
	WinInfo	*wi;

	wi = win_wininfo_new(min);

	win_window_new(wi, WIN_MAIN,		WIN_TYPE_SIMPLE_TOPLEVEL, FALSE, "gentoo v" VERSION, "main");
	win_window_pos_grab_set(wi, WIN_MAIN, TRUE);
	win_window_size_set(wi, WIN_MAIN, 764, 932);
	win_window_size_use_set(wi, WIN_MAIN, TRUE);
	win_window_size_update_set(wi, WIN_MAIN, TRUE);
	win_window_size_grab_set(wi, WIN_MAIN, TRUE);

	win_window_new(wi, WIN_CONFIG,		WIN_TYPE_COMPLEX_DIALOG,  TRUE, _("Configure gentoo"), "config");
	win_window_pos_use_set(wi, WIN_CONFIG, TRUE);
	win_window_size_set(wi, WIN_CONFIG, -1, 464);
	win_window_size_use_set(wi, WIN_CONFIG, TRUE);
	win_window_size_update_set(wi, WIN_CONFIG, TRUE);

	win_window_new(wi, WIN_TEXTVIEW,	WIN_TYPE_SIMPLE_DIALOG,	  FALSE, _("Text Viewer"), "textview");
	win_window_pos_use_set(wi, WIN_TEXTVIEW, TRUE);
	win_window_size_set(wi, WIN_TEXTVIEW, 640, 480);
	win_window_size_use_set(wi, WIN_TEXTVIEW, TRUE);
	win_window_size_update_set(wi, WIN_TEXTVIEW, TRUE);

	return wi;
}

/* 1999-12-23 -	Create a copy of <wi>, sharing no memory with it. */
WinInfo * win_wininfo_copy(const WinInfo *wi)
{
	WinInfo	*nwi;
	WinDef	*win;
	GList	*iter;

	nwi = win_wininfo_new(wi->min);

	for(iter = wi->windows; iter != NULL; iter = g_list_next(iter))
	{
		win = iter->data;

		win_window_new(nwi, win->id, win->type, win->persistent, win->title, win->label);
		win_window_pos_grab_set(nwi, win->id, win->pos_grab);
		win_window_pos_set(nwi, win->id, win->x, win->y);
		win_window_pos_use_set(nwi, win->id, win->pos_use);
		win_window_pos_update_set(nwi, win->id, win->pos_update);
		win_window_size_grab_set(nwi, win->id, win->size_grab);
		win_window_size_set(nwi, win->id, win->width, win->height);
		win_window_size_use_set(nwi, win->id, win->size_use);
		win_window_size_update_set(nwi, win->id, win->size_update);
	}
	return nwi;
}

void win_wininfo_destroy(WinInfo *wi)
{
	GList	*iter;

	for(iter = wi->windows; iter != NULL; iter = g_list_next(iter))
		g_free(iter->data);
	g_list_free(wi->windows);
	g_free(wi);
}

/* ----------------------------------------------------------------------------------------- */

enum { SUBFRAME_POSITION, SUBFRAME_SIZE };

static void evt_boolean_clicked(GtkWidget *wid, gpointer user)
{
	gboolean	*flag = user;
	gboolean	*modified;

	*flag = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(wid));
	if((modified = gtk_object_get_user_data(GTK_OBJECT(wid))) != NULL)
		*modified = TRUE;
}

static void evt_integer_changed(GtkWidget *wid, gpointer user)
{
	gint		*value = user;
	gboolean	*modified;

	*value = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(wid));
	if((modified = gtk_object_get_user_data(GTK_OBJECT(wid))) != NULL)
		*modified = TRUE;
}

/* Grab position. We implicitly *know* that the window in question is the main
** gentoo window, and therefore access it directly through the MainInfo. We
** then put the grabbed position into the editing copy of the WinDef, and update
** the spin buttons to match. Of course, we don't forget setting that pesky old
** modified-flag, either. :) What would people think?
*/
static void evt_pos_grab_clicked(GtkWidget *wid, gpointer user)
{
	MainInfo	*min;
	WinDef		*wd = user;

	min = gtk_object_get_data(GTK_OBJECT(wid), "min");
	gdk_window_get_position(min->gui->window->window, &wd->x, &wd->y);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(gtk_object_get_data(GTK_OBJECT(wid), "spin1")), wd->x);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(gtk_object_get_data(GTK_OBJECT(wid), "spin2")), wd->y);
	*((gboolean *) gtk_object_get_data(GTK_OBJECT(wid), "modified")) = TRUE;
}

/* Grab size. Works only for main gentoo window. See comment for pos_grab above. */
static void evt_size_grab_clicked(GtkWidget *wid, gpointer user)
{
	MainInfo	*min;
	WinDef		*wd = user;

	min = gtk_object_get_data(GTK_OBJECT(wid), "min");
	gdk_window_get_size(min->gui->window->window, &wd->width, &wd->height);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(gtk_object_get_data(GTK_OBJECT(wid), "spin1")), wd->width);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(gtk_object_get_data(GTK_OBJECT(wid), "spin2")), wd->height);
	*((gboolean *) gtk_object_get_data(GTK_OBJECT(wid), "modified")) = TRUE;
}

static GtkWidget * subframe_build(const WinInfo *wi, WinDef *win, gint type, gboolean *modified)
{
	const gchar	*ltext[] = { N_("Position"), N_("Size") }, *etext[] = { N_("X"), N_("Y"), N_("Width"), N_("Height") };
	GtkWidget	*label, *frame, *cbtn, *vbox, *table, *spin1, *spin2;
	GtkObject	*adj;
	gint		val1 = 0, val2 = 0;

	frame = gtk_frame_new(_(ltext[type]));
	vbox = gtk_vbox_new(FALSE, 0);
	cbtn = gtk_check_button_new_with_label(_("Set on Open?"));
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(cbtn), (type == SUBFRAME_POSITION) ?
								win_window_pos_use_get(wi, win->id) :
								win_window_size_use_get(wi, win->id));
	gtk_object_set_user_data(GTK_OBJECT(cbtn), modified);
	gtk_signal_connect(GTK_OBJECT(cbtn), "clicked", GTK_SIGNAL_FUNC(evt_boolean_clicked),
								(type == SUBFRAME_POSITION) ? &win->pos_use : &win->size_use);
	gtk_box_pack_start(GTK_BOX(vbox), cbtn, FALSE, FALSE, 0);

	cbtn = gtk_check_button_new_with_label(_("Update on Close?"));
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(cbtn), (type == SUBFRAME_POSITION) ?
								win_window_pos_update_get(wi, win->id) :
								win_window_size_update_get(wi, win->id));
	gtk_object_set_user_data(GTK_OBJECT(cbtn), modified);
	gtk_signal_connect(GTK_OBJECT(cbtn), "clicked", GTK_SIGNAL_FUNC(evt_boolean_clicked),
								(type == SUBFRAME_POSITION) ? &win->pos_update : &win->size_update);
	gtk_box_pack_start(GTK_BOX(vbox), cbtn, FALSE, FALSE, 0);


	table = gtk_table_new(2, 3, FALSE);

	if(type == SUBFRAME_POSITION)
		win_window_pos_get(wi, win->id, &val1, &val2);
	else
		win_window_size_get(wi, win->id, &val1, &val2);

	label  = gtk_label_new(_(etext[2 * type]));
	gtk_table_attach(GTK_TABLE(table), label, 0, 1, 0, 1,  0,0,0,0);
	adj    = gtk_adjustment_new(val1, -1.0, 65535.0, 1.0, 25.0, 25.0);
	spin1  = gtk_spin_button_new(GTK_ADJUSTMENT(adj), 0.0, 0.0);
	gtk_object_set_user_data(GTK_OBJECT(spin1), modified);
	gtk_signal_connect(GTK_OBJECT(spin1), "changed", GTK_SIGNAL_FUNC(evt_integer_changed),
								type == SUBFRAME_POSITION ? &win->x : &win->width);
	gtk_table_attach(GTK_TABLE(table), spin1, 1, 2, 0, 1,  GTK_EXPAND|GTK_FILL,0,0,0);
	label = gtk_label_new(_(etext[2 * type + 1]));
	gtk_table_attach(GTK_TABLE(table), label, 0, 1, 1, 2,  0,0,0,0);
	adj   = gtk_adjustment_new(val2, -1.0, 65535.0, 1.0, 25.0, 25.0);
	spin2  = gtk_spin_button_new(GTK_ADJUSTMENT(adj), 0.0, 0.0);
	gtk_object_set_user_data(GTK_OBJECT(spin2), modified);
	gtk_signal_connect(GTK_OBJECT(spin2), "changed", GTK_SIGNAL_FUNC(evt_integer_changed),
								type == SUBFRAME_POSITION ? &win->y : &win->height);
	gtk_table_attach(GTK_TABLE(table), spin2, 1, 2, 1, 2,  GTK_EXPAND|GTK_FILL,0,0,0);
	if((type == SUBFRAME_POSITION && win->pos_grab) || (type == SUBFRAME_SIZE && win->size_grab))
	{
		GtkWidget	*grab;

		grab = gtk_button_new_with_label(_("Grab"));

		gtk_object_set_data(GTK_OBJECT(grab), "min", wi->min);
		gtk_object_set_data(GTK_OBJECT(grab), "spin1", spin1);
		gtk_object_set_data(GTK_OBJECT(grab), "spin2", spin2);
		gtk_object_set_data(GTK_OBJECT(grab), "modified", modified);
		gtk_signal_connect(GTK_OBJECT(grab), "clicked", GTK_SIGNAL_FUNC((type == SUBFRAME_POSITION) ?
									evt_pos_grab_clicked : evt_size_grab_clicked), win);
		gtk_table_attach(GTK_TABLE(table), grab, 2, 3, 0, 2,  0,0,0,0);
	}
	gtk_box_pack_start(GTK_BOX(vbox), table, FALSE, FALSE, 0);
	gtk_container_add(GTK_CONTAINER(frame), vbox);

	return frame;
}

GtkWidget * win_wininfo_build(const WinInfo *wi, gboolean *modified)
{
	GtkWidget	*vbox, *frame, *hbox, *sframe;
	GList		*iter;
	gchar		tmp[64];
	WinDef		*win;

	vbox = gtk_vbox_new(FALSE, 0);

	for(iter = wi->windows; iter != NULL; iter = g_list_next(iter))
	{
		win = iter->data;

		g_snprintf(tmp, sizeof tmp, "%c%s", toupper(*win->label), win->label + 1);
		frame  = gtk_frame_new(tmp);
		hbox   = gtk_hbox_new(FALSE, 0);
		sframe = subframe_build(wi, win, SUBFRAME_POSITION, modified);
		gtk_box_pack_start(GTK_BOX(hbox), sframe, TRUE, TRUE, 5);
		sframe = subframe_build(wi, win, SUBFRAME_SIZE, modified);
		gtk_box_pack_start(GTK_BOX(hbox), sframe, TRUE, TRUE, 5);
		gtk_container_add(GTK_CONTAINER(frame), hbox);
		gtk_box_pack_start(GTK_BOX(vbox), frame, TRUE, TRUE, 5);
	}

	return vbox;
}

/* ----------------------------------------------------------------------------------------- */

/* 2000-02-24 -	Write WinInfo out to given file <out>, at its current location. */
void win_wininfo_save(const WinInfo *wi, FILE *out)
{
	GList	*iter;
	WinDef	*wd;

	xml_put_node_open(out, "Windows");
	for(iter = wi->windows; iter != NULL; iter = g_list_next(iter))
	{
		wd = iter->data;

		xml_put_node_open(out, "Window");
		xml_put_uinteger(out, "id", wd->id);
		xml_put_integer(out, "x", wd->x);
		xml_put_integer(out, "y", wd->y);
		xml_put_integer(out, "w", wd->width);
		xml_put_integer(out, "h", wd->height);
		xml_put_boolean(out, "pos_use", wd->pos_use);
		xml_put_boolean(out, "pos_update", wd->pos_update);
		xml_put_boolean(out, "size_use", wd->size_use);
		xml_put_boolean(out, "size_update", wd->size_update);
		xml_put_node_close(out, "Window");
	}
	xml_put_node_close(out, "Windows");
}

static void window_load(XmlNode *window, gpointer user)
{
	guint32	id;

	if(xml_get_uinteger(window, "id", &id))
	{
		WinDef	*wd;

		if((wd = window_find(user, id)) != NULL)
		{
			xml_get_integer(window, "x", &wd->x);
			xml_get_integer(window, "y", &wd->y);
			xml_get_integer(window, "w", &wd->width);
			xml_get_integer(window, "h", &wd->height);
			xml_get_boolean(window, "pos_use", &wd->pos_use);
			xml_get_boolean(window, "pos_update", &wd->pos_update);
			xml_get_boolean(window, "size_use", &wd->size_use);
			xml_get_boolean(window, "size_update", &wd->size_update);
		}
	}
}

/* 2000-02-24 -	Fill in <wi> with details for windows, loaded from <node>. */
void win_wininfo_load(WinInfo *wi, XmlNode *node)
{
	xml_node_visit_children(node, window_load, wi);
}

/* ----------------------------------------------------------------------------------------- */

void win_window_new(WinInfo *wi, guint32 id, WinType type, gboolean persistent, const gchar *title, const gchar *label)
{
	WinDef	*wd;

	wd = g_malloc(sizeof *wd);
	wd->id    = id;
	wd->type  = type;
	wd->persistent = persistent;
	wd->title = title;
	wd->label = label;
	wd->pos_grab = FALSE;
	wd->size_grab = FALSE;

	wd->x	   = 32;
	wd->y	   = 32;
	wd->width  = 32;
	wd->height = 32;

	wd->pos_use     = FALSE;
	wd->pos_update  = FALSE;
	wd->size_use    = FALSE;
	wd->size_update = TRUE;

	wi->windows = g_list_append(wi->windows, wd);
}

static WinDef * window_find(const WinInfo *wi, guint32 id)
{
	GList	*iter;

	if(wi == NULL)
		return NULL;

	for(iter = wi->windows; iter != NULL; iter = g_list_next(iter))
	{
		if(((WinDef *) iter->data)->id == id)
			return iter->data;
	}
	return NULL;
}

void win_window_pos_grab_set(const WinInfo *wi, guint32 id, gboolean grab)
{
	WinDef	*wd;

	if((wd = window_find(wi, id)) != NULL)
		wd->pos_grab = grab;
}

void win_window_pos_set(const WinInfo *wi, guint32 id, gint x, gint y)
{
	WinDef	*wd;

	if((wd = window_find(wi, id)) != NULL)
	{
		wd->x = x;
		wd->y = y;
	}
}

gboolean win_window_pos_get(const WinInfo *wi, guint32 id, gint *x, gint *y)
{
	WinDef	*wd;

	if((wd = window_find(wi, id)) != NULL)
	{
		if(x)
			*x = wd->x;
		if(y)
			*y = wd->y;
		return TRUE;
	}
	return FALSE;
}

void win_window_pos_use_set(const WinInfo *wi, guint32 id, gboolean enabled)
{
	WinDef	*wd;

	if((wd = window_find(wi, id)) != NULL)
		wd->pos_use = enabled;
}

gboolean win_window_pos_use_get(const WinInfo *wi, guint32 id)
{
	WinDef	*wd;

	if((wd = window_find(wi, id)) != NULL)
		return wd->pos_use;
	return FALSE;
}

void win_window_pos_update_set(const WinInfo *wi, guint32 id, gboolean enabled)
{
	WinDef	*wd;

	if((wd = window_find(wi, id)) != NULL)
		wd->pos_update = enabled;
}

gboolean win_window_pos_update_get(const WinInfo *wi, guint32 id)
{
	WinDef	*wd;

	if((wd = window_find(wi, id)) != NULL)
		return wd->pos_update;
	return FALSE;
}

void win_window_size_grab_set(const WinInfo *wi, guint32 id, gboolean grab)
{
	WinDef	*wd;

	if((wd = window_find(wi, id)) != NULL)
		wd->size_grab = grab;
}

void win_window_size_set(const WinInfo *wi, guint32 id, gint width, gint height)
{
	WinDef	*wd;

	if((wd = window_find(wi, id)) != NULL)
	{
		wd->width    = width;
		wd->height   = height;
	}
}

gboolean win_window_size_get(const WinInfo *wi, guint32 id, gint *width, gint *height)
{
	WinDef	*wd;

	if((wd = window_find(wi, id)) != NULL)
	{
		if(width)
			*width = wd->width;
		if(height)
			*height = wd->height;
		return TRUE;
	}
	return FALSE;
}

void win_window_size_use_set(const WinInfo *wi, guint32 id, gboolean enabled)
{
	WinDef	*wd;

	if((wd = window_find(wi, id)) != NULL)
		wd->size_use = enabled;
}

gboolean win_window_size_use_get(const WinInfo *wi, guint32 id)
{
	WinDef	*wd;

	if((wd = window_find(wi, id)) != NULL)
		return wd->size_use;
	return FALSE;
}

void win_window_size_update_set(const WinInfo *wi, guint32 id, gboolean enabled)
{
	WinDef	*wd;

	if((wd = window_find(wi, id)) != NULL)
		wd->size_update = enabled;
}

gboolean win_window_size_update_get(const WinInfo *wi, guint32 id)
{
	WinDef	*wd;

	if((wd = window_find(wi, id)) != NULL)
		return wd->size_update;
	return FALSE;
}

/* ----------------------------------------------------------------------------------------- */

void win_window_destroy(WinInfo *wi, guint32 id)
{
	WinDef	*wd;

	if((wd = window_find(wi, id)) != NULL)
	{
		wi->windows = g_list_remove(wi->windows, wd);
		g_free(wd);
	}
}

/* ----------------------------------------------------------------------------------------- */

/* 1999-12-23 -	Open the window described by <key> in <wi>. The window will not be shown yet. */
GtkWidget * win_window_open(const WinInfo *wi, guint32 id)
{
	WinDef	*wd;

	if((wd = window_find(wi, id)) != NULL)
	{
		GtkWidget	*wid = NULL;

		switch(wd->type)
		{
			case WIN_TYPE_SIMPLE_TOPLEVEL:
				wid = gtk_window_new(GTK_WINDOW_TOPLEVEL);
				break;
			case WIN_TYPE_SIMPLE_DIALOG:
				wid = gtk_window_new(GTK_WINDOW_DIALOG);
				break;
			case WIN_TYPE_SIMPLE_POPUP:
				wid = gtk_window_new(GTK_WINDOW_POPUP);
				break;
			case WIN_TYPE_COMPLEX_DIALOG:
				wid = gtk_dialog_new();
				break;
		}
		gtk_object_set_data(GTK_OBJECT(wid), "windef", wd);
		if(!wd->pos_use)
			gtk_window_set_position(GTK_WINDOW(wid), GTK_WIN_POS_MOUSE);
		gtk_widget_realize(wid);
		return wid;
	}
	return NULL;
}

/* 2000-02-19 -	Refresh link between <wi> and <window>. Handy after <wi> has been replaced
**		while window was open, as happens in the config window case.
*/
void win_window_relink(const WinInfo *wi, guint32 id, GtkWidget *window)
{
	WinDef	*wd;

	if((wd = window_find(wi, id)) != NULL)
		gtk_object_set_data(GTK_OBJECT(window), "windef", wd);
}

/* 1999-12-23 -	Use this rather than gtk_widget_show() when you want to the window to appear, since
**		this gives us a chance to set the window's size and position according to preference.
*/
void win_window_show(GtkWidget *window)
{
	WinDef	*wd;

	if(GTK_WIDGET_VISIBLE(window))
		return;

	wd = gtk_object_get_data(GTK_OBJECT(window), "windef");
	if((wd != NULL) && wd->size_use && (wd->width >= 0))
		gtk_window_set_default_size(GTK_WINDOW(window), wd->width, wd->height);

	/* This causes the window to appear at some more or less random position. We then reposition
	** it to the configured place. This is visually distracting, unattractive, and almost s*cks,
	** but unfortunately I haven't been able to figure out how to fix it. If you move the show()
	** call to below the if, it works, but not all the time. :( Tips welcome.
	*/
	gtk_widget_show(window);

	if((wd != NULL) && wd->pos_use && (wd->x >= 0))
	{
		while(gtk_events_pending())
			gtk_main_iteration();
		gtk_widget_set_uposition(GTK_WIDGET(window), wd->x, wd->y);
	}
}

/* 2000-03-04 -	If <a_new> is different from *<a>, update *<a> to <a_new> and return TRUE, else
**		return FALSE. Likewise for <b>, whatever that would mean.
*/
static gboolean update_pair(gint *a, gint *b, gint a_new, gint b_new)
{
	if((*a != a_new) || (*b != b_new))
	{
		*a = a_new;
		*b = b_new;
		return TRUE;
	}
	return FALSE;
}

/* 2000-03-18 -	Do an update for the given window, changing the config as needed. Returns TRUE
**		if the config was indeed changed.
*/
gboolean win_window_update(GtkWidget *win)
{
	WinDef		*wd;

	if((wd = gtk_object_get_data(GTK_OBJECT(win), "windef")) != NULL)
	{
		gboolean	ret = FALSE;
		gint		ta, tb;

		if(wd->pos_update)
		{
			gdk_window_get_root_origin(win->window, &ta, &tb);
			ret |= update_pair(&wd->x, &wd->y, ta, tb);
		}
		if(wd->size_update)
		{
			gdk_window_get_size(win->window, &ta, &tb);
			ret |= update_pair(&wd->width, &wd->height, ta, tb);
		}
		return ret;
	}
	return FALSE;
}

/* 1999-12-23 -	Close the given window. Use this rather than a direct gtk_widget_destroy(), since
**		we need a chance to update the position and size fields if needed. Also, if the
**		window is classified as "persistent", we don't actually destroy it, but only hide
**		it. Returns TRUE if the config was changed, i.e. if one of the update flags were
**		set AND the relevant property actually had changed.
*/
gboolean win_window_close(GtkWidget *win)
{
	WinDef	*wd;

	if((wd = gtk_object_get_data(GTK_OBJECT(win), "windef")) != NULL)
	{
		gboolean	ret;

		ret = win_window_update(win);
		if(wd->persistent)
			gtk_widget_hide(win);
		else
			gtk_widget_destroy(win);
		return ret;
	}
	gtk_widget_destroy(win);
	return FALSE;
}
