/*
 * Copyright (C) 2002 Pascal Haakmat.
 */

#include <stdlib.h>
#include <string.h>
#include "mem.h"
#include "track.h"
#include "cache.h"

void
cache_clear(cache *c) {
    c->low = c->high = 0;
}

void
cache_dump(cache *c) {
    INFO("%10d/%10d(%10d) %p %s\n", 
         c->low,
         c->high,
         c->sz,
         c->data, 
         c->type == CACHE_NULL ? "null cache" : "");
}

cache *
cache_join(cache *c1,
           cache *c2) {
    void *d;
    if(c1->type != c2->type) {
        FAIL("cannot join caches of different types\n");
        return NULL;
    }
    if(c1->type == CACHE_NULL) {
        c1->sz += c2->sz;
        c1->high = c1->sz;
        return c1;
    }
    if(c1->type != CACHE_REAL) {
        FAIL("cannot join caches of type %d\n", c1->type);
        return NULL;
    }
    if(c1->high != c1->sz) {
        FAIL("first cache must be full for join c1: \n");
        cache_dump(c1);
        FAIL("c2: \n");
        cache_dump(c2);
        return NULL;
    }

    //    DEBUG("before join:\n");
    //    DEBUG("c1:\n");
    //    cache_dump(c1);
    //    DEBUG("c2:\n");
    //    cache_dump(c2);
    d = realloc(c1->data, c1->sz + c2->sz);

    if(!d) {
        FAIL("cannot get enough memory to join caches (need %d bytes joined)\n", 
             c1->sz + c2->sz);
        return NULL;
    }

    c1->data = d;
    memcpy(c1->data + c1->sz, c2->data, c2->sz);
    c1->sz += c2->sz;
    c1->high += c2->high;
    //    DEBUG("after join:\n");
    //    DEBUG("c1:\n");
    //    cache_dump(c1);
    return c1;
}

cache *
cache_clone(cache *c,
            size_t off,
            size_t sz) {
    size_t off_old = off, sz_old = sz;
    cache *c2;

    if(sz == 0) {
        FAIL("sz == 0\n");
        abort();
    }

    if(off + sz > c->sz) {
        FAIL("off(%d) + sz(%d) > c->sz(%d)\n", off, sz, c->sz);
        abort();
    }

    c2 = cache_new(c->type, sz);

    if(!c2){ 
        FAIL("cannot create new cache for clone (%d bytes)\n", sz);
        return NULL;
    }

    c2->low = 0;
    c2->high = c2->sz;

    if(c->type == CACHE_NULL)
        return c2;

    DEBUG("requesting from parent: off: %d, sz: %d\n", 
          off, sz);
    cache_find(c,
               c2->data,
               &off,
               &sz);

    if(off != off_old || sz != sz_old) {
        FAIL("cache did not return data %d-%d\n", off, sz);
        cache_destroy(c2);
        return NULL;
    }

    return c2;
}

void 
cache_move(cache *c,
           size_t new_off,
           size_t old_off,
           size_t sz) {
    if(c->type == CACHE_NULL)
        return;

    /* This was for testing. */
    //    if(sz & 1 || new_off & 1 || old_off & 1)
    //        abort();

    if(c->low > old_off)
        c->low -= old_off - new_off;

    if(c->high > old_off)
        c->high -= old_off - new_off;

    if(c->high < 0)
        c->high = 0;
    if(c->low < 0)
        c->low = 0;
    if(c->high > c->sz)
        c->high = c->sz;
    if(c->low > c->high)
        c->low = c->high;
    
    //    DEBUG("new_off: %d, old_off: %d, sz: %d, c->low: %d, c->high: %d\n",
    //          new_off, old_off, sz, c->low, c->high); 

    memmove(c->data + new_off, c->data + old_off, sz);
}

/* Returns zero on success. */

int
cache_resize(cache *c, 
             size_t sz) {
    void *buf;

    //    DEBUG("resizing: cur: %d, req: %d\n", c->sz, sz);

    if(c->type == CACHE_NULL) {
        c->sz = sz;
        c->high = sz;
        return 0;
    }

    //    if(sz & 1)
    //        abort();

    /* Promise that resize always succeeds if the new size is smaller
       than the old size. */

    buf = c->data;
    if(sz > c->sz) {
        buf = realloc(c->data, sz);
        if(!buf) {
            FAIL("could not grow cache (old size: %d, new size: %d)\n",
                 c->sz, sz);
            return 1;
        }
    }
    c->data = buf;
    c->sz = sz;

    if(c->high > sz)
        c->high = sz;

    if(c->low > c->high)
        c->low = c->high;

    return 0;
}

/**
 * Finds some data in the cache. On return the offset and sz variables 
 * contain the actually found data (i.e. actual bytes returned and 
 * the position that they were found at). 
 */
void
cache_find(cache *c,
           void *dst,
           size_t *offset,
           size_t *sz) {
    size_t dst_offset = 0;
    //    DEBUG("request: c.data: %p, c.low: %d, c.high: %d, c.sz: %d, offset: %d, sz: %d\n",
    //          c->data, c->low, c->high, c->sz, *offset, *sz);
    
    /*
    if(!c->data) {
        *sz = 0;
        return;
    }
    */

    /* Not found when:
       - offset equals or beyond c->high:
         offset >= c->high
       - offset + sz is below c->low:
         offset + sz < c->low */
    
    if(*offset >= c->high) {
        *sz = 0;
        return;
    }
    if(*offset + *sz < c->low) {
        *sz = 0;
        return;
    }
    if(*offset < c->low) {
        dst_offset = (c->low - *offset);
        *sz = *sz - (c->low - *offset);
        *offset = c->low;
    }
    if(*offset + *sz > c->high)
        *sz = c->high - *offset;

    //    DEBUG("return: c.data: %p, c.low: %d, c.high: %d, c.sz: %d, offset: %d, sz: %d, dst: %p, dst_offset: %d\n",
    //          c->data, c->low, c->high, c->sz, *offset, *sz, dst, dst_offset);

    if(c->type == CACHE_NULL) 
        memset(dst + dst_offset, '\0', *sz);
    else
        memcpy(dst + dst_offset, c->data + *offset, *sz);
    
}

/* Returns bytes written. */

size_t
cache_fill(cache *c,
           void *src,
           size_t offset,
           size_t sz) {
    if(!c->data)
        return sz;
    
    //    DEBUG("request: c.data: %p, c.low: %d, c.high: %d, c.sz: %d, offset: %d, sz: %d\n",
    //          c->data, c->low, c->high, c->sz, offset, sz);
    
    /* Not filled when:
       - data is already in cache:
         offset >= c->low && offset + sz <= c->high
       - offset exceeds cache size:
         offset >= c->sz
       - if this insertion does not make the cache discontiguous:
         offset + sz < c->low || offset > c->high
       - offset is beyond the cache:
         offset >= c->sz */

    /* Already in cache. */
        
    if(offset >= c->low && offset + sz <= c->high) {
        DEBUG("already in cache\n");
        return sz;
    }

    /* Discontiguous. */

    if((c->low != c->high) && (offset + sz < c->low || offset > c->high)) {
        DEBUG("discontiguous\n");
        return sz;
    }

    /* Offset too big. */

    if(offset >= c->sz) {
        FAIL("offset too big\n");
        return sz;
    }

    /* Size too big to fit in cache. */

    if(offset + sz > c->sz)
        sz = c->sz - offset;
    
    if(c->low == c->high) {
        c->low = offset;
        c->high = offset + sz;
    } else {
        c->low = MIN(c->low, offset);
        c->high = MAX(c->high, offset + sz);
    }

    //    DEBUG("return: c.data: %p, c.low: %d, c.high: %d, c.sz: %d, offset: %d, sz: %d\n",
    //          c->data, c->low, c->high, c->sz, offset, sz);

    memcpy(c->data + offset, src, sz);
    return sz;
}

cache *
cache_new(cache_type type,
          size_t sz) {
    cache *c = mem_alloc(sizeof(cache));
    if(!c) {
        DEBUG("failed to create cache object (%d bytes)\n", sz);
        return NULL;
    }
    c->type = type;
    c->low = 0;
    c->high = 0;
    c->sz = sz;
    c->data = NULL;
    if(type == CACHE_NULL) {
        c->high = sz;
        return c;
    }
    
    c->data = mem_alloc(sz);
    if(!c->data) {
        DEBUG("failed to allocate %d bytes\n", sz);
        mem_free(c);
        return NULL;
    }
    //    DEBUG("allocated %d bytes\n", c->sz);
    return c;
}

void
cache_destroy(cache *c) {
    if(c->data) {
        //        DEBUG("releasing %d bytes\n", c->sz);
        mem_free(c->data);
    }
    c->data = NULL;
    c->low = 0;
    c->high = 0;
    c->sz = 0;
    mem_free(c);
}
