#ifndef MODULE_H
#define MODULE_H

#include <glib.h>
#include <audiofile.h>
#include "config.h"
#include "undo.h"
#include "shell.h"
#include "action.h"

typedef int module_id;

/* Module info. */

typedef struct {
    char *name;
    char *fname;
    void *handle;
    char *error;
} module;

extern module modules[];

#define MODULE_INIT(module, n) \
((module)->name = (n))

/* Module interface. */

/* Required. Called once to during module lifetime to initialize the
 * module. Module should return a structure initialized using the
 * MODULE_INIT() macro or NULL if initialization fails.
 */

module *
module_new();

/* Optional. Called once during module lifetime to destroy the module. */

void
module_destroy();

/* Optional. This function is called when the user performs the "open"
 * action on the module. It should set up the structure in
 * shl->module_state[id]. When this function is not defined, then
 * the module's module_execute() function will be called instead. It
 * is the responsibility of this function to display an interface
 * and dispatch the MODULE_EXECUTE action when necessary.
 */

void
module_open(module_id id,
            shell *shl, 
            int undo);

/* Required if module_open() exists. Should tear down the structure in
 * shl->module_state[id].
 */

void
module_close(struct _module_state *module_state);

/* Required. Invoked when the module should do its job. If the module
 * does not have a module_open() function, then this function is
 * invoked automatically. If the module does have a module_open()
 * function, then the module itself it responsible for invoking this
 * function. The id of the module can be retrieved by using
 * shl->active_module_id. This function should return an action_group
 * that can be applied to reverse the effects of the action, or NULL
 * if that is not possible.
 */

action_group *
module_execute(shell *shl, 
               int undo);

/* Perform the argument "func" on subsequent sections
 * of the given track's frame buffer.
 */

#define ITERATOR_INIT(frame_offset, frame_count) \
    AFframecount iter_read = 0, \
      iter_written = 0, \
      iter_frame_offset = (frame_offset), \
      iter_frame_count = 0, \
      iter_frames_processed = 0, \
      iter_frame_count_old = iter_frame_count = (frame_count); \
    int32_t *int32_frame_bits = malloc(EFFECT_BUF_SIZE * sizeof(int32_t)); \
    int iter_escape = 0;

#define ITERATOR_EXIT() \
    if(int32_frame_bits) \
      free(int32_frame_bits);

#define ITERATOR_ESCAPE() \
    iter_escape = 1;

#define ITERATOR_READER(shl, track) \
    do { \
      iter_read = track_int32_frame_buffer_get((track), \
                                               int32_frame_bits, \
                                               iter_frame_offset, \
                                               MIN(EFFECT_BUF_SIZE, \
                                               iter_frame_count)); \
      if(iter_read <= 0) \
        ITERATOR_ESCAPE(); \
      iter_written = iter_read; \
    } while(0)

#define ITERATOR_SKEL(shl, track, reader, func) \
    do { \
    if(!int32_frame_bits) \
      break; \
    if(gtk_main_iteration_do(FALSE)) \
      break; \
    do { \
      reader; \
      do { \
        func; \
        gtk_progress_set_percentage((shl)->progress,  \
                                    CLAMP((float)iter_frames_processed / \
                                          (float)(iter_frame_count_old), 0, 1)); \
        while(gtk_events_pending()) \
          if(gtk_main_iteration_do(FALSE)) \
            ITERATOR_ESCAPE(); \
      } while(0); \
      iter_frames_processed += iter_read; \
      iter_frame_offset += iter_written; \
      iter_frame_count -= iter_read; \
      if(shl->module_cancel_requested) \
        ITERATOR_ESCAPE(); \
    } while(!iter_escape && iter_read > 0 && iter_frame_count); \
    DEBUG("total: %ld\n", iter_frame_count_old); \
    if(!gtk_main_iteration_do(FALSE)) \
      gtk_progress_set_percentage(shl->progress, 0); \
    } while(0) 

#define ITERATOR(shl, track, func) \
    ITERATOR_SKEL(shl, track, ITERATOR_READER(shl, track), func)

/*
#define ITERATOR(shl, track, func) \
    do { \
    if(!int32_frame_bits) \
      break; \
    if(gtk_main_iteration_do(FALSE)) \
      break; \
    do { \
      iter_read = track_int32_frame_buffer_get((track), \
                                               int32_frame_bits, \
                                               iter_frame_offset, \
                                               MIN(EFFECT_BUF_SIZE, \
                                                   iter_frame_count)); \
      if(iter_read <= 0) \
        ITERATOR_ESCAPE(); \
      if(shl->module_cancel_requested) \
        ITERATOR_ESCAPE(); \
      iter_written = iter_read; \
      do { \
        func; \
        gtk_progress_set_percentage((shl)->progress,  \
                                    CLAMP((float)iter_frames_processed / \
                                          (float)(iter_frame_count_old), 0, 1)); \
        while(gtk_events_pending()) \
          if(gtk_main_iteration_do(FALSE)) \
            ITERATOR_ESCAPE(); \
      } while(0); \
      iter_frames_processed += iter_read; \
      iter_frame_offset += iter_written; \
      iter_frame_count -= iter_read; \
    } while(!iter_escape && iter_read > 0 && iter_frame_count); \
    DEBUG("total: %ld\n", iter_frame_count_old); \
    if(!gtk_main_iteration_do(FALSE)) \
      gtk_progress_set_percentage(shl->progress, 0); \
    } while(0) 
*/

/* ITERATOR2 is flotsam... */

#define ITERATOR2(shl, source, channel_map, func, func2) \
    do { \
    if(!int32_frame_bits) \
      break; \
    if(gtk_main_iteration_do(FALSE)) \
      break; \
    for(iter_channel = 0; iter_channel < (source)->channels; iter_channel++) \
      if((1 << iter_channel) & (channel_map)) \
        iter_channel_count++; \
    do { \
      iter_more_data = iter_channel_count; \
      iter_max_read = 0; \
      iter_max_written = 0; \
      for(iter_channel = 0; \
          !iter_escape && iter_channel < (source)->channels; \
          iter_channel++) { \
        DEBUG("source->channels: %d, iter_channel: %d\n",  \
              (source)->channels, iter_channel); \
        if((1 << iter_channel) & (channel_map)) { \
          DEBUG("channel in map\n"); \
          iter_read = track_int32_frame_buffer_get((source)->tracks[iter_channel], \
                                                   int32_frame_bits, \
                                                   iter_frame_offset, \
                                                   MIN(EFFECT_BUF_SIZE, \
                                                       iter_frame_count)); \
          DEBUG("iter_read: %ld\n", iter_read); \
          if(iter_read > iter_max_read) \
            iter_max_read = iter_read; \
          if(iter_read <= 0) \
            iter_more_data--; \
          if(shl->module_cancel_requested) \
            ITERATOR_ESCAPE(); \
          iter_written = iter_read; \
          if(iter_written > iter_max_written) \
            iter_max_written = iter_written; \
          do { \
            func; \
            gtk_progress_set_percentage((shl)->progress,  \
                                        CLAMP((float)iter_frames_processed / \
                                              (float)(iter_frame_count_old), 0, 1)); \
            while(gtk_events_pending()) \
              if(gtk_main_iteration()) \
                ITERATOR_ESCAPE(); \
          } while(0); \
        } \
      } \
      do { \
        func2; \
        while(gtk_events_pending()) \
          if(gtk_main_iteration()) \
            ITERATOR_ESCAPE(); \
      } while(0); \
      iter_frames_processed += iter_max_read; \
      iter_frame_offset += iter_max_written; \
      iter_frame_count -= iter_max_read; \
      DEBUG("iter_frames_processed: %ld, iter_frame_offset: %ld, iter_frame_count: %ld\n", \
            iter_frames_processed, iter_frame_offset, iter_frame_count); \
    } while(!iter_escape && iter_more_data > 0 && iter_frame_count); \
    DEBUG("total: %ld\n", iter_frame_count_old); \
    if(!gtk_main_iteration_do(FALSE)) \
      gtk_progress_set_percentage(shl->progress, 0); \
    } while(0) 

void
module_exit();

void
module_load(const char *path);

int
module_init();


#endif /* ! MODULE_H */
