/*
 * Copyright (C) 2002 Pascal Haakmat.
 */

#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <glib.h>
#include <math.h>
#include <errno.h>
#include "mem.h"
#include "snd.h"
#include "track.h"
#include "blocklist.h"
#include "cache.h"

AFframecount
track_frame_count(track *tr) {
    return blocklist_frame_count(tr->bl);
}

/* Has no function anymore. */

void
track_frame_cache_fill(track *tr,
                       void *bits,
                       AFframecount frame_offset,
                       AFframecount frame_count) {
    GList *l = NULL;
    cache *c;
    size_t cache_el_sz, src_off, r;
    int err;

    err = blocklist_block_find(tr->bl, &frame_offset, &l);

    if(err) {
        FAIL("could not find block for frame_offset %ld\n", frame_offset);
        return;
    }

    cache_el_sz = tr->frame_width;
    src_off = 0;

    //    DEBUG("request %s: frame_offset: %ld, frame_count: %ld\n", 
    //          is_graph_cache ? "GRAPH" : "FRAME", frame_offset, frame_count);

    while(l && frame_count > 0) {
        c = ((block *)l->data)->frame_cache;
        r = cache_fill(c,
                       bits + src_off,
                       frame_offset * cache_el_sz,
                       frame_count * cache_el_sz);
        src_off += r;
        r /= cache_el_sz;
        frame_count -= r;
        frame_offset = 0;
        l = l->next;
    }
    
    if(!l && frame_count) {
        FAIL("cache fill failed, %ld bytes lost\n", frame_count);
        blocklist_dump(tr->bl);
        abort();
    }
}

void
track_graph_cache_fill(track *tr,
                       void *bits,
                       AFframecount frame_offset,
                       AFframecount frame_count) {
    GList *l = NULL;
    cache *c;
    AFframecount fc_req, fc_total;
    size_t cache_el_sz, src_off, r;
    float hres;
    int err;

    DEBUG("frame_offset: %ld, frame_count: %ld\n", 
          frame_offset, frame_count);
    err = blocklist_block_find(tr->bl, &frame_offset, &l);

    if(err) {
        FAIL("could not find block for frame_offset %ld\n", frame_offset);
        return;
    }

    cache_el_sz = sizeof(_graph_bits_t);
    hres = tr->graph_hres;
    frame_offset /= hres;
    src_off = 0;
    fc_total = 0;

    //    DEBUG("request %s: frame_offset: %ld, frame_count: %ld\n", 
    //          is_graph_cache ? "GRAPH" : "FRAME", frame_offset, frame_count);

    while(l && frame_count > 0) {
        c = ((block *)l->data)->graph_cache;

        fc_req = MIN(((block *)l->data)->frame_count, frame_count);

        r = cache_fill(c,
                       bits + src_off,
                       frame_offset * cache_el_sz,
                       ceil(fc_req / hres) * cache_el_sz);
        fc_total += fc_req;
        src_off = floor(fc_total / hres) * cache_el_sz;
        frame_count -= fc_req;
        frame_offset = 0;
        l = l->next;
    }
    
    if(!l && frame_count) {
        FAIL("cache fill failed, %ld bytes lost\n", frame_count);
        blocklist_dump(tr->bl);
        abort();
    }
}


/* Copies data from the frame caches in the blocks for this track into
 * the given buffer. The close relative for this function is
 * track_graph_cache_find(), it uses the same algorithm but knows how
 * to "blend" graph caches (i.e. how to combine two blocks' peak data
 * properly).
 *
 * In the setup phase we find the block that the requested offset
 * resides in. Then we ask the frame cache for that block to return as
 * much data as possible.
 * 
 * Then we evaluate the cache reply. We need to return a contiguous
 * block that either starts at the requested start and extends to
 * some X, or, a contiguous block that starts at some X and
 * extends to the requested end. This is because cache misses need
 * to be replenished in a single (contiguous) read. So there are
 * seven cases:
 *
 * 1. Nothing found.
 * 2. Found something from the requested start up to requested end.
 * 3. Found something from the requested start up to the end of the cache.
 * 4. Found something from the requested start up to some other X.
 * 5. Found something from some X up to the requested end.
 * 6. Found something from some X up to the end of the cache.
 * 7. Found something from some X up to some Y.
 *
 * In case 1 we simply fail and return nothing.
 *
 * In cases 2, 4 and 5 we have found some data, and also know that
 * this is the last block that we need to look at, so we can
 * return.
 *
 * In case 3 we need to look at the next block(s) and evaluate it
 * similarly to this block until there is no more data to be
 * retrieved (or until we encounter a gap).
 *
 * In case 6 we also need to look at the next block(s), but with
 * the added constraint that any subsequently provided data MUST
 * ultimately end at the requested end, without any intermediate
 * gaps. IOW, in case 6 we "fail_on_gap".
 *
 * Finally in case 7 we have a gap and return nothing.
 *
 * On entry the frame_offset and frame_count parameters specify how
 * many frames are to be retrieved from what offset. On return,
 * frame_offset and frame_count indicate to what extent the request
 * was satisfied. That is the frame_count variable returns the number
 * of bytes that remain to be read and the frame_offset variable
 * returns the position that they should be read at.
 */
void
track_frame_cache_find(track *tr,
                       void *bits,
                       AFframecount *frame_offset,
                       AFframecount *frame_count) {
    AFframecount fo, fc;
    GList *l = NULL;
    cache *c;
    size_t cache_el_sz, src_off, off, start_gap, ct, total_ct;
    int fail_on_gap = 0, has_gap = 0, err;

    fo = *frame_offset;
    fc = *frame_count;

    /* Setup. */

    err = blocklist_block_find(tr->bl, &fo, &l);

    if(err)
        return;

    c = ((block *)l->data)->frame_cache;
    cache_el_sz = tr->frame_width;
    src_off = 0;

    off = fo * cache_el_sz;
    ct = fc * cache_el_sz;
    cache_find(c,
               bits + src_off,
               &off,
               &ct);
    
    /* Case 1: Didn't find anything, no use looking further. */
    
    if(!ct)
        return;

    /* Case 2 & 4: */

    if(off == fo * cache_el_sz &&
       (ct == fc * cache_el_sz ||
        off + ct < ((block *)l->data)->frame_count * cache_el_sz)) {
        *frame_offset += ct / cache_el_sz;
        *frame_count -= ct / cache_el_sz;
        return;
    }

    /* Case 5: No need to adjust frame_offset. */

    if(off + ct == (fo + fc) * cache_el_sz) {
        *frame_count -= ct / cache_el_sz;
        return;
    }

    /* Case 7: Found something, but it does not start at the requested
       beginning and it does not end at the end of the cache buffer
       and it does not end at the requested end -> useless. */
    
    if(off > fo * cache_el_sz &&
       off + ct < ((block *)l->data)->frame_count * cache_el_sz &&
       (off + ct < (fo + fc) * cache_el_sz)) {
        return;
    }

    /* Now only cases 3 and 6 remain. */

    start_gap = off - (fo * cache_el_sz);
    src_off += ct + start_gap;
    total_ct = ct;
    ct = (fc * cache_el_sz) - src_off;
    off = 0;
    l = l->next;

    /* Case 6: fail when returned data does not end at requested end. */

    if(start_gap)
        fail_on_gap = 1;

    while(l && ct > 0) {
        has_gap = 1;
        c = ((block *)l->data)->frame_cache;
        cache_find(c,
                   bits + src_off,
                   &off,
                   &ct);

        if(!ct)
            break;

        if(off)
            break;

        /* Ends at some X, but not at end of cache and not at end of request. */

        if((off + ct < ((block *)l->data)->frame_count * cache_el_sz) &&
           ((fc * cache_el_sz) - (src_off + ct)) > 0)
            if(fail_on_gap)
                break;

        src_off += ct;
        total_ct += ct;
        ct = (fc * cache_el_sz) - src_off;
        off = 0;
        l = l->next;
        has_gap = 0;
    }

    if(fail_on_gap && has_gap)
        return;

    /* Case 3: adjust frame_offset. */

    if(!start_gap)
        *frame_offset += total_ct / cache_el_sz;

    *frame_count -= total_ct / cache_el_sz;
}

/* Almost identical to track_graph_cache_find above. It is a bit more
 * complicated though because it has to account for the fact that a
 * block may contain less frames than a a single graph_bits_t
 * describes, in which case the graph caches for consecutive blocks
 * need to be "blended". 
 */

void
track_graph_cache_find(track *tr,
                       void *bits,
                       AFframecount *frame_offset,
                       AFframecount *frame_count) {
    AFframecount fo, frames_read, frames_left, frames_total_read;
    GList *l = NULL;
    cache *c;
    size_t cache_el_sz, src_off, off, start_gap, ct;
    int fail_on_gap = 0, has_gap = 0, spill = 0, err = 0;
    float frame_divider;

    fo = *frame_offset;
    frames_left = *frame_count;

    /* Setup. */
    
    err = blocklist_block_find(tr->bl, &fo, &l);

    if(!l)
        return;

    c = ((block *)l->data)->graph_cache;
    frame_divider = tr->graph_hres;
    cache_el_sz = sizeof(_graph_bits_t);
    src_off = 0;

    off = floor(fo / frame_divider) * cache_el_sz;
    ct = ceil(MIN(((block *)l->data)->frame_count, frames_left) / frame_divider) * cache_el_sz;
    cache_find(c,
               bits + src_off,
               &off,
               &ct);
    
    /* Calculate how many frames (as opposed to peak cache elements)
       were actually returned. */

    frames_read = (ct / cache_el_sz) * frame_divider;
    if(((off + ct) / cache_el_sz) * frame_divider > ((block *)l->data)->frame_count) 
        frames_read -= 
            ((((off + ct) / cache_el_sz) * frame_divider) - ((block *)l->data)->frame_count);

    /* Case 1: Didn't find anything, no use looking further. */
    
    if(!ct)
        return;

    /* Case 2 & 4: */

    if(off == floor(fo / frame_divider) * cache_el_sz &&
       (frames_read == frames_left || 
        ((off / cache_el_sz) * frame_divider) + frames_read < ((block *)l->data)->frame_count)) {
        *frame_offset += frames_read;
        *frame_count -= frames_read;
        return;
    }

    /* Case 5: No need to adjust frame_offset. */

    if((floor(off / cache_el_sz) * frame_divider) + frames_read == 
       (floor(fo / frame_divider) * cache_el_sz) + frames_left) {
        *frame_count -= frames_read;
        return;
    }

    /* Case 7: Found something, but it does not start at the requested
       beginning and it does not end at the end of the cache buffer
       and it does not end at the requested end -> useless. */
    
    if(off > floor(fo / frame_divider) * cache_el_sz &&
       ((off / cache_el_sz) * frame_divider) + frames_read < ((block *)l->data)->frame_count &&
       ((off / cache_el_sz) * frame_divider) + frames_read < 
       (floor(fo / frame_divider) * cache_el_sz) + frames_left) {
        return;
    }
    
    /* Now only cases 3 and 6 remain. */

    /* Intermezzo.

       The number of frames in a block is not always a neat multiple
       of the number of frames represented by a single graph_bits unit
       (pair of low/high peaks). The block code guarantees that the
       graph cache is always "too big", so the maximum error that we
       can get because of this (in the extreme case where a block
       contains just 1 frame) is GRAPH_BITS_HRES - 1 frames. When we
       need to stitch multiple blocks together this rapidly adds up.
       So we need a compensation mechanism for that. We do this by
       keeping track of how many frames each block contains/describes,
       then advancing the index into the graph cache by
       sizeof(graph_bits_t) at each multiple of 128 (i.e. the
       graph_hres). 

       At this point we know we need to get another block and stitch
       things together. In case we did not read a nice 128 multiple, then
       we know we need to recalculate the final peak element. The 
       spill variable tracks this. */

    start_gap = off - (floor(fo / frame_divider) * cache_el_sz);
    spill = frames_read % (int)frame_divider;
    src_off = start_gap + (ct - (ceil(spill / frame_divider) * cache_el_sz));
    frames_total_read = frames_read;
    frames_left -= frames_read;

    l = l->next;

    //    DEBUG("frames_read: %ld, frames_left: %ld, spill: %d, new ct: %d, src_off: %d\n",
    //          frames_read, frames_left, spill, ct, src_off);

    /* Case 6: fail when subsequent caches cannot satisfy the request
       up to requested end (i.e. throw away all our work). */

    if(start_gap)
        fail_on_gap = 1;

    while(l && frames_left > 0) {

        /* FIXME: in case we haven't read 128 frames yet we fudge the
           offset to overwrite the previous peak element, thus
           destroying some peak data. We should instead combine the
           last peak element from the previous buffer with the first
           peak element from the current buffer and calculate a new
           high/low. */

        has_gap = 1;
        c = ((block *)l->data)->graph_cache;
        off = 0;
        ct = ceil(MIN(((block *)l->data)->frame_count, frames_left) / frame_divider) * cache_el_sz;
        cache_find(c,
                   bits + src_off,
                   &off,
                   &ct);

        if(off)
            break;

        if(!ct)
            break;

        frames_read = (ct / cache_el_sz) * frame_divider;
        if(frames_read > MIN(((block *)l->data)->frame_count, frames_left)) 
            frames_read = MIN(((block *)l->data)->frame_count, frames_left);

        /* Ends at some X, but not at end of cache and not at end of request. */

        if(frames_read < ((block *)l->data)->frame_count &&
           frames_read < frames_left) 
            if(fail_on_gap)
                break;

        spill = frames_read % (int)frame_divider;
        frames_total_read += frames_read;
        frames_left -= frames_read;
        src_off = floor(frames_total_read / frame_divider) * cache_el_sz;

        //        DEBUG("frames_read: %ld, frames_left: %ld, spill: %d, new ct: %d, src_off: %d\n",
        //              frames_read, frames_left, spill, ct, src_off);

        l = l->next;
        has_gap = 0;
    }

    //    DEBUG("fell out of loop: off: %d, ct: %d, l: %p, frames_read: %ld, frames_left: %ld, spill: %d, src_off: %d, frames_total_read: %ld\n",
    //          off, ct, l, frames_read, frames_left, spill, src_off, frames_total_read);

    if(fail_on_gap && has_gap)
        return;

    /* Case 3: adjust frame_offset. */
    
    if(!start_gap)
        *frame_offset += frames_total_read;

    *frame_count -= frames_total_read;
}

/* Copies the requested range from the track into the supplied buffer. 
 */

AFframecount
track_frame_buffer_get(track *tr,
                       _frame_bits frame_bits,
                       AFframecount frame_offset,
                       AFframecount frame_count) {
    AFframecount tr_ct, frame_count_old = frame_count;

    rwlock_rlock(&tr->rwl);

    /* Clamp frame count. */

    tr_ct = track_frame_count(tr);
    if(frame_offset + frame_count > tr_ct) {
        frame_count = tr_ct - frame_offset;
        frame_count_old = frame_count;
        if(frame_count <= 0) {
            rwlock_runlock(&tr->rwl);
            return 0;
        }
    }

    track_frame_cache_find(tr,
                           frame_bits,
                           &frame_offset,
                           &frame_count);
    
    rwlock_runlock(&tr->rwl);

    if(!frame_count) 
        return frame_count_old;

    return 0;
}

/* Retrieves the requested frames from the frame buffer, scales them
 * according to the given scale factor, and delivers the result in the
 * supplied buffer. There is no explicit error condition for this
 * function. A return value of 0 may either mean there are no more
 * frames or that some memory could not be allocated. 
 */

AFframecount
track_graph_buffer_read(track *tr, 
                        _graph_bits graph_bits,
                        AFframecount frame_offset,
                        AFframecount frame_count,
                        float hres) {
    _frame_bits fb;
    AFframecount r;
    size_t fb_sz = tr->frame_width * frame_count;

    fb = mem_alloc(fb_sz);
    if(!fb) 
        return 0;

    r = track_frame_buffer_get(tr, 
                               fb, 
                               frame_offset, 
                               frame_count);
    
    if(r <= 0) {
        mem_free(fb);
        return 0;
    }
    
    snd_frames_buffer_to_graph_buffer(graph_bits,
                                      fb,
                                      r,
                                      tr->frame_width,
                                      hres);
    mem_free(fb);
    return r;
}

/* Retrieves a scaled representation of the requested frames into the 
 * buffer given by gb. The buffer must be large enough to hold at least 
 * frame_count * sizeof(_graph_bits) elements. 
 */

AFframecount
track_graph_buffer_get(track *tr, 
                       _graph_bits gb,
                       AFframecount frame_offset,
                       AFframecount frame_count,
                       float hres) {
    AFframecount r = 0, frame_offset_old = frame_offset, frame_count_old = frame_count;
    size_t gbc_sz, soff = 0; 
    float q = hres;
    _graph_bits gbp = NULL, gbc = NULL;

    if(frame_count <= 0) {
        FAIL("frame_count <= 0\n");
        abort();
    }

    /* First try to get the data from the prescaled (graph) cache.
     * The graph cache stores data at a scaling factor of
     * tr->graph_hres, so there is no point in looking in the cache
     * when a scaling factor below tr->graph_hres is requested. The
     * cache may return a partial hit, in that case we scale directly
     * from raw data and combine that with the result from the
     * cache. FIXME: it is no longer true strictly that the cache can
     * return a partial hit, because now graph data is calculated the
     * moment data is put into the frame cache (it used to be lazy, i.e.
     * calculate the data when necessary, but that has problems). If
     * we get a partial hit that is a bug (has happened, in TODO). 
     */

    if(hres >= tr->graph_hres) {

        /* Need memory to scale up from cache. */

        gbc_sz = (size_t) ceil(frame_count / hres) * 
            sizeof(_graph_bits_t) * (hres / tr->graph_hres);
        gbc = mem_calloc(1, gbc_sz);
        if(!gbc) {
            FAIL("could not get memory for temporary graph cache buffer\n");
            return 0;
        }

        track_graph_cache_find(tr,
                               gbc,
                               &frame_offset,
                               &frame_count);
        /*
        DEBUG("%s: graph cache returned frame_offset: %ld, frame_count: %ld\n", 
              (frame_count == frame_count_old ? "MISS" : 
               (frame_count == 0 ? "HIT" : "PARTIAL")), frame_offset, frame_count);
        */

        /*
          if(frame_count != 0) {
            FAIL("partial hit, abort.\n");
            blocklist_dump(tr->bl);
            abort();
        }
        */

    }

    if(frame_count) {

        /* Get it either at the requested scaling factor or at a
           scaling factor suitable for the cache. */

        if(hres >= tr->graph_hres) {
            gbp = gbc;
            q = tr->graph_hres;
        } else {
            gbp = gb;
            q = hres;
        }
        
        soff = (frame_offset / q) - (frame_offset_old / q);
        
        r = track_graph_buffer_read(tr,
                                    &gbp[soff],
                                    frame_offset,
                                    frame_count, 
                                    q);

    }

    /* We now either have something fit for immediate consumption by
       the caller (i.e. a graph at the requested hres), or something
       fit for insertion into the cache (i.e. a graph at the cache
       hres). */

    if(hres >= tr->graph_hres) {

        /* OK, got something suitable for the cache. Put it in there and 
           scale from cache hres to requested hres. */
        
        if(r > 0)
            track_graph_cache_fill(tr,
                                   &gbc[soff],
                                   frame_offset,
                                   r);

        /* Scale the graph from the cache hres (probably 128) to 
           the requested hres. */

        snd_scale(gb,
                  gbc,
                  ((frame_count_old - frame_count) + r) / tr->graph_hres,
                  tr->frame_width,
                  hres / tr->graph_hres,
                  0,
                  0,
                  1);

        mem_free(gbc);
    }

    return (frame_count_old - frame_count) + r;
}

/* Removes frame_count frames at frame_offset and returns a list of
 * blocks containing the deleted frames.  
 *
 * Returns NULL (not enough memory to rearrange blocks or offset out
 * of range), or the start of the deleted block list. This function
 * may delete less frames than were requested if we run out of memory 
 * or frame_count exceeds the track frame count. FIXME: maybe better
 * to promise either full success or full failure.
 */

GList *
track_delete(track *tr,
             AFframecount frame_offset,
             AFframecount frame_count) {
    GList *del_blks;
    rwlock_wlock(&tr->rwl);
    del_blks = blocklist_delete(tr->bl, frame_offset, frame_count);
    rwlock_wunlock(&tr->rwl);
    return del_blks;
}

void
track_null_insert(track *tr,
                  AFframecount frame_offset,
                  AFframecount frame_count) {
    block *b;
    GList *l = NULL;
    DEBUG("creating NULL block, %ld frames\n", frame_count);
    b = block_new(CACHE_NULL, tr->frame_width, tr->graph_hres, frame_count);
    if(!b) {
        FAIL("could not create %ld frame NULL block\n", frame_count);
        return;
    }
    DEBUG("inserting NULL block\n");
    l = g_list_append(l, b);
    blocklist_blocks_insert(tr->bl, l, frame_offset);
    block_unref(b);
    l->data = NULL;
    g_list_free(l);
}


/* Inserts the given block list into the target track at the specified
 * offset. The given blocks are not actually copied, instead their
 * reference count is increased and they are added to the target block
 * list. The source block list should be destroyed by the caller.
 *
 * If the given offset equals the count for this track, then insert
 * becomes append and the source block list is appended. If the
 * given offset exceeds the count for this track, then a NULL block
 * is inserted to span the difference, followed by an append. 
 *
 * Returns 0 on success.
 */

int
track_insert(track *tr,
             GList *source,
             AFframecount frame_offset) {
    int r;
    
    rwlock_wlock(&tr->rwl);

    /* Append with zeroes to the required length first if
       necessary. */

    if(frame_offset > track_frame_count(tr)) 
        track_null_insert(tr, 
                          MIN(track_frame_count(tr), frame_offset), 
                          MAX(track_frame_count(tr), frame_offset) -
                          MIN(track_frame_count(tr), frame_offset));

    r = blocklist_blocks_insert(tr->bl, source, frame_offset);

    rwlock_wunlock(&tr->rwl);
    return r;
}

/* Converts an array of sample data into blocks and inserts them into
 * the track. If possible it tries to resize an existing block and fit
 * the data in there.
 *
 * The frame_offset can extend beyond the actual length of the
 * track. In this case the track is padded with (virtual) zero data up
 * to the required length.  
 *
 * Returns 0 on success, 1 if there was not enough memory.
 */

int
track_frame_buffer_put(track *tr,
                       _frame_bits frame_bits,
                       AFframecount frame_offset,
                       AFframecount frame_count) {
    int r;

    rwlock_wlock(&tr->rwl);

    /* Append with zeroes to the required length first if
       necessary. */

    if(frame_offset > track_frame_count(tr)) 
        track_null_insert(tr, 
                          MIN(track_frame_count(tr), frame_offset), 
                          MAX(track_frame_count(tr), frame_offset) -
                          MIN(track_frame_count(tr), frame_offset));

    r = blocklist_buffer_insert(tr->bl,
                                frame_bits,
                                tr->frame_width,
                                frame_offset,
                                frame_count);
    rwlock_wunlock(&tr->rwl);
    return r;
}

/* Same as track_frame_buffer_get but converts the data to 32 bits
 * signed integer before returning. 
 */

AFframecount
track_int32_frame_buffer_get(track *tr,
                             int32_t *int32_frame_bits,
                             AFframecount frame_offset,
                             AFframecount frame_count) {
    AFframecount r, i;
    _frame_bits fb;

    if(!(frame_count > 0 && frame_offset >= 0))
        return 0;

    if(tr->frame_width == 4) {
        fb = int32_frame_bits;
    } else {
        fb = mem_alloc(frame_count * tr->frame_width);
        if(!fb)
            return 0;
    }

    r = track_frame_buffer_get(tr, fb, frame_offset, frame_count);

    if(tr->frame_width == 4) 
        return r;

    for(i = 0; i < r; i++) {
        switch(tr->frame_width) {
        case 1:
            int32_frame_bits[i] = ((int8_t *)fb)[i] << 24;
            break;
        case 2:
            int32_frame_bits[i] = ((int16_t *)fb)[i] << 16;
            break;
        }
    }

    mem_free(fb);
    return r;
}

/* Same as track_frame_buffer_replace but expects sample data to 
 * be 32 bit unsigned format.
 */

int
track_int32_frame_buffer_replace(track *tr,
                                 int32_t *int32_frame_bits,
                                 AFframecount frame_offset,
                                 AFframecount frame_count) {
    GList *del_blks;
    int r;
    rwlock_wlock(&tr->rwl);
    del_blks = track_delete(tr,
                            frame_offset,
                            frame_count);
    if(!del_blks) {
        rwlock_wunlock(&tr->rwl);
        return 1;
    }
    r = track_int32_frame_buffer_put(tr,
                                     int32_frame_bits,
                                     frame_offset,
                                     frame_count);

    if(r) 
        track_insert(tr,
                     del_blks,
                     frame_offset);
    else 
        blocklist_blocks_destroy(del_blks);
    
    rwlock_wunlock(&tr->rwl);
    return r;
}

/* Same as track_frame_buffer_put but for 32 bit unsigned integers.
 */

int
track_int32_frame_buffer_put(track *tr,
                             int32_t *int32_frame_bits,
                             AFframecount frame_offset,
                             AFframecount frame_count) {
    AFframecount i;
    int r;
    _frame_bits fb;

    if(!(frame_count > 0 && frame_offset >= 0))
        return 1;

    if(tr->frame_width == 4) 
        return track_frame_buffer_put(tr, int32_frame_bits, frame_offset, frame_count);

    fb = mem_alloc(frame_count * tr->frame_width);
    if(!fb)
        return 1;

    for(i = 0; i < frame_count; i++) {
        switch(tr->frame_width) {
        case 1:
            ((int8_t *)fb)[i] = int32_frame_bits[i] >> 24;
            break;
        case 2:
            ((int16_t *)fb)[i] = int32_frame_bits[i] >> 16;
            break;
        }
    }

    r = track_frame_buffer_put(tr, fb, frame_offset, frame_count);
    mem_free(fb);
    return r;
}

/* Clones the track, depending on the method, either just the track
 * properties or the track data as well.
 */

track *
track_clone(track *tr,
            track_clone_method method) {
    track *tr_clone = track_new(tr->frame_width, 0);

    if(!tr_clone) {
        FAIL("could not create new track for clone\n");
        return NULL;
    }

    if(method & CLONE_DATA) {
        rwlock_rlock(&tr->rwl);
        if(tr->bl) {
            if(tr_clone->bl) 
                blocklist_destroy(tr_clone->bl);
            tr_clone->bl = blocklist_clone(tr->bl);
            if(!tr_clone->bl) {
                FAIL("could not clone block list for track clone\n");
                track_destroy(tr_clone);
                rwlock_runlock(&tr->rwl);
                return NULL;
            }
        }
        rwlock_runlock(&tr->rwl);
    }
    return tr_clone;
}

/* Destroys the track. */

void
track_destroy(track *tr) {
    if(tr->bl) {
        blocklist_destroy(tr->bl);
        tr->bl = NULL;
    }
    rwlock_destroy(&tr->rwl);
    mem_free(tr);
}

/* Creates a new track and reserves the specified amount of frame
 * storage for it. 
 */

track *
track_new(int frame_width,
          AFframecount frame_count) {
    AFframecount fc, fc_blk, fc_rq;
    int i, fail;
    block *b;
    GList *l = NULL;
    track *tr = mem_calloc(1, sizeof(track));    
    if(!tr)
        return NULL;

    fc_blk = BLOCK_SIZE;

    tr->frame_width = frame_width;
    tr->graph_hres = GRAPH_BITS_HRES;
    rwlock_init(&tr->rwl);

    fail = 0;
    for(fc = 0; fc < frame_count; fc += fc_blk, i++) {
        if(fc + fc_blk >= frame_count) 
            fc_rq = frame_count - fc;
        else
            fc_rq = fc_blk;

        b = block_new(CACHE_REAL, tr->frame_width, tr->graph_hres, fc_rq);
        if(!b) {
            fail = 1;
            break;
        }
        l = g_list_append(l, b);
    }

    tr->bl = blocklist_new(l);

    if(!tr->bl)
        fail = 1;

    if(fail) {
        track_destroy(tr);
        return NULL;
    }

    return tr;
}
