/*
 * Copyright (C) 2002 Pascal Haakmat.
 */

#include <errno.h>
#include <stdio.h>
#include <dlfcn.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <stdarg.h>
#include <glib.h>
#include <audiofile.h>
#include "mem.h"
#include "snd.h"
#include "track.h"
#include "gui.h"
#include "undo.h"
#include "action.h"
#include "play.h"
#include "file.h"
#include "shell.h"
#include "module.h"
#include "modutils.h"

GList *shells = NULL;
shell *clipboard_shell = NULL;
snd *clipboard = NULL;
snd *clipboard_buffers[MAX_TRACKS];
char *path = NULL;
extern int mem_fail_allocation_on_zero;
extern int module_count;
extern module modules[];
static action_result last_action_result;
static action_result false_action_result = { -1, 0, { 0 } };

action_group *
action_player_play(action *a) {
    shell *shl = a->shl;
    if(!shl->player.player_running) {
        /* Wrap. */
        if(!shl->player.record_mode &&
           shl->select_start == shl->select_end &&
           shl->select_start == snd_frame_count(shl->sr)) {
            shl->select_start = shl->select_end = 0;
            shl->hadjust->value = 0;
        }
        player_start(shl); 
    }
    return NULL;
}

action_group *
action_player_stop(action *a) {
    shell *shl = a->shl;
    if(shl->player.player_running) {
        player_stop(shl);
        if(shl->select_start == shl->select_end)
            shl->select_start = shl->select_end = shl->player.player_pos;
    }
    return NULL;
}

void 
gui_saveas_fileselection_ok_clicked(GtkWidget *w, 
                                    gpointer fs) {
    shell *shl;
    struct stat buf;
    int button;
    char *filename = strdup(gtk_file_selection_get_filename(GTK_FILE_SELECTION(fs)));
    if(path)
       mem_free(path);
    path = strdup(filename);
    shl = gtk_object_get_user_data(fs);
    
    gtk_widget_destroy(GTK_WIDGET(fs));
    
    if(stat(filename, &buf) == -1) {
        switch(errno) {
        case ENOENT:
            /* A component of the path file_name does not exist, or the path is an empty string. */
            break;
        default:
            gui_alert("The filename '%s' is invalid: %s", filename, strerror(errno));
           mem_free(filename);
            return;
        }
        /* Assuming file can be created. */

        snd_name_set(shl->sr, filename);

    } else {

        /* File exists. */

        button = gui_yes_no("Replace file?", "The file %s already exists. Replace?", 
                            filename);

        if(button == GUI_YES) {
            if(unlink(filename) == -1) {
                gui_alert("Cannot remove file %s: %s", filename, strerror(errno));
               mem_free(filename);
                return;
            } else {
                snd_name_set(shl->sr, filename);
            }
        } else {
           mem_free(filename);
            return;
        }

    }

    file_save(shl, FALSE);
   mem_free(filename);
}

action_group *
action_file_save_as(action *a) {
    GtkWidget *fs = gtk_file_selection_new("Save under what name?");
    
    if(path) {
        DEBUG("setting path: %s\n", path);
        gtk_file_selection_set_filename(GTK_FILE_SELECTION(fs), path);
    }

    gtk_object_set_user_data(GTK_OBJECT(fs), a->shl);
    gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(fs)->ok_button),
                       "clicked", 
                       GTK_SIGNAL_FUNC(gui_saveas_fileselection_ok_clicked), 
                       fs);
    
    gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(fs)->cancel_button),
                              "clicked", 
                              GTK_SIGNAL_FUNC(gtk_widget_destroy),
                              (gpointer) fs);
    
    gtk_widget_show(fs);
    return NULL;
}

action_group *
action_file_close(action *a) {
    int button, allow = 0;

    if(!a->shl->has_changed) {
        allow = 1;
    } else {
        if(a->shl == clipboard_shell) {
            allow = 1;
        } else if(a->shl->sr) {
            button = gui_yes_no("Close file?", "Close file %s without saving changes?", 
                                a->shl->sr->name);
            if(button == GUI_NO || button == GUI_CANCEL) 
                allow = 0;
            else
                allow = 1;
        } else {
            allow = 1;
        }
    }

    if(allow)
        gtk_object_destroy(GTK_OBJECT(a->shl->appwindow));

    ACTION_RESULT_RETURN_BOOL(a->id, !allow);
    return NULL;
}

action_group *
action_file_save(action *a) {

    if(!a->shl->has_name)
        action_file_save_as(a);

    file_save(a->shl, TRUE);

    return NULL;
}

void 
gui_open_fileselection_ok_clicked(GtkWidget *w, 
                                  gpointer fs) {
    char *filename = strdup(gtk_file_selection_get_filename(GTK_FILE_SELECTION(fs)));
    gtk_widget_destroy(GTK_WIDGET(fs));

    if(path)
       mem_free(path);
    path = strdup(filename);
    
    file_open(filename);
   mem_free(filename);    
}

action_group *
action_file_open(action *a) {
    GtkWidget *fs = gtk_file_selection_new("Select an audio file");
    
    if(path) {
        DEBUG("setting path: %s\n", path);
        gtk_file_selection_set_filename(GTK_FILE_SELECTION(fs), path);
    }

    gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(fs)->ok_button),
                       "clicked", 
                       GTK_SIGNAL_FUNC(gui_open_fileselection_ok_clicked), 
                       fs);
    
    gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(fs)->cancel_button),
                              "clicked", 
                              GTK_SIGNAL_FUNC(gtk_widget_destroy),
                              (gpointer) fs);
    
    gtk_widget_show(fs);
    return NULL;
}

action_group *
action_file_new(action *a) {

    file_new();
    return NULL;
}

action_group *
action_selection_to_loop(action *a) {
    if(a->shl->select_start == a->shl->select_end)
        return NULL;

    a->shl->loop = 1;
    a->shl->loop_start = a->shl->select_start;
    a->shl->loop_end = a->shl->select_end;
    return NULL;
}

action_group *
action_loop_to_selection(action *a) {
    if(a->shl->loop_start == a->shl->loop_end)
        return NULL;

    a->shl->select_start = a->shl->loop_start;
    a->shl->select_end = a->shl->loop_end;
    return NULL;
}

action_group *
action_selection_fit(action *a) {
    AFframecount delta = a->shl->select_end - a->shl->select_start;
    float hres = 1;

    for(hres = 1; 
        hres < 16384 && ((delta / hres) > GTK_WIDGET(a->shl->canvas)->allocation.width); 
        hres += hres);
    a->shl->hres = hres;
    shell_viewport_center(a->shl, a->shl->select_start, a->shl->select_end);
    
    DEBUG("hres: %f, delta: %ld, allocation.width: %d\n", hres, delta, 
          (GTK_WIDGET(a->shl->canvas)->allocation.width));
    return NULL;
}

action_group *
action_tracks_insert(action *a) {
    snd_tracks_insert(a->sr_target, 
                      a->sr_source,
                      a->channel_map);
    if(RESULT_IS_ERROR(a->sr_target)) 
        return NULL;

    mixer_configure(a->shl->mixer, 
                    OUTPUT_CHANNELS, //MIN(OUTPUT_CHANNELS, a->sr_target->channels), 
                    a->sr_target->channels);

    shell_status_default_set(a->shl);

    if(!gtk_main_iteration_do(FALSE))
        gtk_widget_queue_draw(GTK_WIDGET(a->shl->mixercanvas));

    if(!a->undo) 
        return NULL;
    
    a->shl->has_changed = 1;
    return action_group_new(1,
                            ACTION_TRACKS_DELETE_NEW(DONT_UNDO,
                                                     a->shl,
                                                     a->sr_target, 
                                                     a->channel_map));
    
}

action_group *
action_tracks_delete(action *a) {
    snd *del_sr;
    
    a->shl->select_channel_map = 0;
    del_sr = snd_tracks_delete(a->sr_target,
                               a->channel_map);
    
    if(RESULT_IS_ERROR(a->sr_target)) 
        return NULL;
    
    mixer_configure(a->shl->mixer, 
                    OUTPUT_CHANNELS, //MIN(OUTPUT_CHANNELS, a->sr_target->channels), 
                    a->sr_target->channels);
    
    shell_status_default_set(a->shl);

    if(!gtk_main_iteration_do(FALSE))
        gtk_widget_queue_draw(GTK_WIDGET(a->shl->mixercanvas));

    if(!a->undo) {
        snd_destroy(del_sr);
        return NULL;
    }
    
    a->shl->has_changed = 1;
    return action_group_new(1,
                            ACTION_TRACKS_INSERT_NEW(DONT_UNDO,
                                                     a->shl,
                                                     a->sr_target, 
                                                     del_sr, 
                                                     a->channel_map));

}

action_group *
action_copy(action *a) {
    action_group *ag;
    
    if(a->shl == clipboard_shell)
        return NULL;
    
    ag = action_group_new(3,
                          ACTION_CUT_NEW(DONT_UNDO,
                                         a->shl,
                                         a->sr_target,
                                         a->channel_map,
                                         a->offset,
                                         a->count),
                          ACTION_PASTE_NEW(DONT_UNDO,
                                           a->shl,
                                           a->sr_target,
                                           a->channel_map,
                                           a->offset,
                                           0),
                         ACTION_SELECT_NEW(DONT_UNDO,
                                           a->shl,
                                           a->sr_target,
                                           a->channel_map,
                                           a->offset,
                                           a->count));
    
    action_group_do(ag);
    return NULL;
}

action_group *
action_select(action *a) {
    AFframecount old_ss = a->shl->select_start,
        old_se = a->shl->select_end,
        old_sc = a->shl->select_channel_map;

    a->shl->select_channel_map = a->channel_map;
    
    if(a->shl->select_channel_map == 0) 
        a->shl->select_channel_map = 1;
    
    a->shl->select_start = a->offset;
    a->shl->select_end = a->offset + a->count;

    if(!a->undo) 
        return NULL;

    /* Don't save undo steps for simply moving the cursor (cursor
       appears when start == end). */

    if(old_ss == old_se)
        return NULL;

    return action_group_new(1,
                            ACTION_SELECT_NEW(DONT_UNDO,
                                              a->shl,
                                              a->shl->sr,
                                              old_sc,
                                              old_ss,
                                              old_se - old_ss));
        
}

action_group *
action_cut(action *a) {
    snd *del_sr;

    if(a->count < 1) {
        FAIL("cannot cut empty selection.\n");
        return NULL;
    }

    if(a->shl == clipboard_shell)
        return NULL;

    del_sr = snd_delete(a->sr_target,
                        a->channel_map,
                        a->offset, 
                        a->count);
    if(RESULT_IS_ERROR(a->sr_target)) {
        snd_insert(a->sr_target,
                   del_sr,
                   a->channel_map,
                   a->offset);
        snd_destroy(del_sr);
        return NULL;
    }

    snd_destroy(clipboard);

    clipboard = snd_clone(del_sr, 
                          CLONE_STRUCTURE |
                          CLONE_TRACK_STRUCTURE | 
                          CLONE_TRACK_DATA);    
    if(RESULT_IS_ERROR(del_sr)) {
        snd_error_set(a->sr_target,
                      "sorry, clipboard lost (%s)", 
                      snd_error_get(del_sr));
        snd_error_free(del_sr);
        snd_destroy(clipboard);
        clipboard = (snd *)snd_new();
        snd_name_set(clipboard, "clipboard");
        if(clipboard_shell) {
            clipboard_shell->sr = clipboard;
            mixer_configure(clipboard_shell->mixer, 
                            OUTPUT_CHANNELS, //MIN(OUTPUT_CHANNELS, clipboard_shell->sr->channels), 
                            clipboard_shell->sr->channels);
            shell_status_default_set(clipboard_shell);
            shell_redraw(clipboard_shell);
        }
        return NULL;
    }

    snd_name_set(clipboard, "clipboard");
    if(clipboard_shell) {
        clipboard_shell->sr = clipboard;
        mixer_configure(clipboard_shell->mixer, 
                        OUTPUT_CHANNELS, //MIN(OUTPUT_CHANNELS, clipboard_shell->sr->channels), 
                        clipboard_shell->sr->channels);
        shell_status_default_set(clipboard_shell);
        shell_redraw(clipboard_shell);
    }

    a->shl->select_start = a->offset;
    a->shl->select_end = a->offset;
    a->shl->select_channel_map = a->channel_map;

    shell_status_default_set(a->shl);

    if(!a->undo) {
        snd_destroy(del_sr);
        return NULL;
    }

    a->shl->has_changed = 1;
    return action_group_new(2,
                            ACTION_INSERT_NEW(DONT_UNDO,
                                              a->sr_target, 
                                              del_sr, 
                                              a->channel_map,
                                              a->offset),
                            ACTION_SELECT_NEW(DONT_UNDO,
                                              a->shl,
                                              a->sr_target,
                                              a->channel_map,
                                              a->offset,
                                              a->count));
    
}

action_group *
action_paste_mix(action *a) {
    int t, t2 = 0;
    snd *copy, *old_sr;
    action_group *undo = NULL;

    if(a->shl == clipboard_shell)
        return NULL;

    if(!clipboard_shell) {

        /* We need a visible shell window to get envelope values, see
           below. */

        gui_alert("Sorry, please show clipboard before doing paste mix...");
        return NULL;
    }

    copy = snd_clone(clipboard, 
                     CLONE_STRUCTURE |
                     CLONE_TRACK_STRUCTURE | 
                     CLONE_TRACK_DATA);
    if(RESULT_IS_ERROR(clipboard)) {
        snd_error_set(a->sr_target,
                      "cannot copy clipboard (%s)",
                      snd_error_get(clipboard));
        snd_error_free(clipboard);
        snd_destroy(copy);
        return NULL;
    }

    snd_name_set(copy, "clipboard_copy");

    snd_convert(copy,
                a->sr_target->frame_width,
                a->sr_target->rate);
    if(RESULT_IS_ERROR(copy)) {
        snd_error_set(a->sr_target, 
                      "could not convert clipboard to %d bits, %.2fhz (%s)",
                      8 * a->sr_target->frame_width,
                      a->sr_target->rate,
                      snd_error_get(copy));
        snd_destroy(copy);
        return NULL;
    }

    if(a->undo) 
        undo = action_group_undo_create(a->shl,
                                        a->channel_map, 
                                        a->offset,
                                        a->count,
                                        a->offset,
                                        a->count);

    /* FIXME: danger danger we swap the clipboard_shell->sr while
       processing the mix, but since the mix yields to the event loop
       other commands can come in and access or change the
       clipboard. The mix operation needs a shell because shells
       contain the envelopes that control the mix. However, shells
       have a visible window. So what we are really doing here is
       passing the visible window that we have as a temporary
       substitute. */

    old_sr = clipboard_shell->sr;
    clipboard_shell->sr = copy;
    rwlock_rlock(&a->shl->sr->rwl);
    rwlock_rlock(&copy->rwl);

    /* Find first non-empty track on clipboard. */

    for(t2 = 0; t2 < clipboard_shell->sr->channels; t2++)
        if(track_frame_count(clipboard_shell->sr->tracks[t2]))
            break;

    for(t = 0; t < a->shl->sr->channels && t2 < clipboard_shell->sr->channels; t++)
        if((1 << t) & a->channel_map)
            mix(a->shl,
                t,
                a->offset,
                clipboard_shell,
                t2++,
                0,
                MIN(snd_frame_count(copy), a->count));
    rwlock_runlock(&a->shl->sr->rwl);
    rwlock_runlock(&copy->rwl);
    clipboard_shell->sr = old_sr;
    
    snd_destroy(copy);

    shell_status_default_set(a->shl);
    a->shl->has_changed = 1;

    return undo;

}

action_group *
action_paste_fit(action *a) {
    int t;
    snd *copy, *del_sr;
    action_group *undo = NULL;
    int ins_map, tmp;

    if(a->shl == clipboard_shell)
        return NULL;

    copy = snd_clone(clipboard, 
                     CLONE_STRUCTURE |
                     CLONE_TRACK_STRUCTURE | 
                     CLONE_TRACK_DATA);
    if(RESULT_IS_ERROR(clipboard)) {
        snd_error_set(a->sr_target,
                      "cannot copy clipboard (%s)",
                      snd_error_get(clipboard));
        snd_error_free(clipboard);
        snd_destroy(copy);
        return NULL;
    }

    snd_name_set(copy, "clipboard_copy");

    snd_convert(copy,
                a->sr_target->frame_width,
                a->sr_target->rate);
    if(RESULT_IS_ERROR(copy)) {
        snd_error_set(a->sr_target, 
                      "could not convert clipboard to %d bits, %.2fhz (%s)",
                      8 * a->sr_target->frame_width,
                      a->sr_target->rate,
                      snd_error_get(copy));
        snd_destroy(copy);
        return NULL;
    }

    /* FIXME: error checking. Again this is ugly because we need to
     store the sample inside a shell before we can resample (same as
     paste_mix above). */

    if(a->undo) 
        undo = action_group_undo_create(a->shl,
                                        a->channel_map, 
                                        a->offset,
                                        a->count,
                                        a->offset,
                                        a->count);

    del_sr = snd_delete(a->shl->sr,
                        a->channel_map,
                        a->offset,
                        a->count);

    if(RESULT_IS_ERROR(a->shl->sr)) {
        if(del_sr) 
            snd_insert(a->shl->sr,
                       del_sr,
                       a->channel_map,
                       a->offset);
        snd_destroy(del_sr);
        return NULL;
    }

    ins_map = snd_insert(a->shl->sr,
                         copy,
                         a->channel_map,
                         a->offset);

    if(RESULT_IS_ERROR(a->shl->sr)) {
        if(del_sr) 
            snd_insert(a->shl->sr,
                       del_sr,
                       a->channel_map,
                       a->offset);
        snd_destroy(del_sr);
        return NULL;
    }

    rwlock_rlock(&a->shl->sr->rwl);
    for(t = 0; t < a->shl->sr->channels; t++) {
        if((1 << t) & ins_map) {            
            tmp = a->shl->markers[t]->marker_types_enabled;
            a->shl->markers[t]->marker_types_enabled = 0;
            resample(a->shl,
                     t,
                     a->offset,
                     a->count * ((double)snd_frame_count(copy) / a->count),
                     (double)snd_frame_count(copy) / a->count);
            a->shl->markers[t]->marker_types_enabled = tmp;
        }
    }
    rwlock_runlock(&a->shl->sr->rwl);

    snd_destroy(copy);
    snd_destroy(del_sr);

    shell_status_default_set(a->shl);
    a->shl->has_changed = 1;

    return undo;

}

action_group *
action_paste_over(action *a) {
    int i_map;
    snd *copy, *del_sr;
    AFframecount del_frame_count, pasted_frame_count;

    if(a->shl == clipboard_shell)
        return NULL;

    copy = snd_clone(clipboard, 
                     CLONE_STRUCTURE |
                     CLONE_TRACK_STRUCTURE | 
                     CLONE_TRACK_DATA);
    if(RESULT_IS_ERROR(clipboard)) {
        snd_error_set(a->sr_target,
                      "cannot copy clipboard (%s)",
                      snd_error_get(clipboard));
        snd_error_free(clipboard);
        snd_destroy(copy);
        return NULL;
    }

    snd_name_set(copy, "clipboard_copy");

    snd_convert(copy,
                a->sr_target->frame_width,
                a->sr_target->rate);
    if(RESULT_IS_ERROR(copy)) {
        snd_error_set(a->sr_target, 
                      "could not convert clipboard to %d bits, %.2fhz (%s)",
                      8 * a->sr_target->frame_width,
                      a->sr_target->rate,
                      snd_error_get(copy));
        snd_error_free(copy);
    }

    del_frame_count = MIN(snd_frame_count(copy), a->count);
    //    mem_fail_allocation_on_zero = 4;
    del_sr = snd_delete(a->sr_target,
                        a->channel_map,
                        a->offset,
                        del_frame_count);
    //    mem_fail_allocation_on_zero = 0;
    if(RESULT_IS_ERROR(a->sr_target)) {
        snd_destroy(copy);
        snd_destroy(del_sr);
        return NULL;

        /* FIXME: snd_delete may complete partially on error and
           return the frames deleted before the error was
           encountered. Should put deleted frames back here. */
    }

    snd_destroy(snd_delete(copy,
                           a->channel_map,
                           MIN(snd_frame_count(copy), del_frame_count),
                           snd_frame_count(copy) -
                           MIN(snd_frame_count(copy), del_frame_count)));
    i_map = snd_insert(a->sr_target,
                       copy,
                       a->channel_map,
                       a->offset);
    pasted_frame_count = snd_frame_count(copy);
    
    a->shl->select_start = a->offset;
    a->shl->select_end = a->offset + pasted_frame_count;
    a->shl->select_channel_map = a->channel_map;
    
    snd_destroy(copy);

    shell_status_default_set(a->shl);

    if(!a->undo) {
        snd_destroy(del_sr);
        return NULL;
    }

    a->shl->has_changed = 1;
    return action_group_new(3,
                            ACTION_DELETE_NEW(DONT_UNDO,
                                              a->shl,
                                              a->sr_target, 
                                              i_map, 
                                              a->offset, 
                                              del_frame_count),
                            ACTION_INSERT_NEW(DONT_UNDO,
                                              a->sr_target, 
                                              del_sr, 
                                              a->channel_map,
                                              a->offset),
                            ACTION_SELECT_NEW(DONT_UNDO,
                                              a->shl,
                                              a->sr_target, 
                                              a->channel_map, 
                                              a->offset,
                                              a->count));
    
}

action_group *
action_paste(action *a) {
    int i_map;
    snd *copy, *del_sr;

    if(a->shl == clipboard_shell)
        return NULL;

    copy = snd_clone(clipboard, 
                     CLONE_STRUCTURE |
                     CLONE_TRACK_STRUCTURE | 
                     CLONE_TRACK_DATA);
    if(RESULT_IS_ERROR(clipboard)) {
        snd_error_set(a->sr_target,
                      "cannot copy clipboard (%s)",
                      snd_error_get(clipboard));
        snd_error_free(clipboard);
        snd_destroy(copy);
        return NULL;
    }

    snd_name_set(copy, "clipboard_copy");

    snd_convert(copy,
                a->sr_target->frame_width,
                a->sr_target->rate);
    if(RESULT_IS_ERROR(copy)) {
        snd_error_set(a->sr_target, 
                      "could not convert clipboard to %d bits, %.2fhz (%s)",
                      8 * a->sr_target->frame_width,
                      a->sr_target->rate,
                      snd_error_get(copy));
        snd_error_free(copy);
    }

    del_sr = snd_delete(a->sr_target,
                        a->channel_map,
                        a->offset,
                        a->count);    
    if(RESULT_IS_ERROR(a->sr_target)) {
        snd_destroy(copy);
        snd_destroy(del_sr);
        return NULL;

        /* FIXME: snd_delete may complete partially on error and
           return the frames deleted before the error was
           encountered. Should put deleted frames back here. */
    }

    i_map = snd_insert(a->sr_target,
                       copy,
                       a->channel_map,
                       a->offset);
    
    a->shl->select_start = a->offset + snd_frame_count(copy);
    a->shl->select_end = a->shl->select_start;
    a->shl->select_channel_map = a->channel_map;
    
    snd_destroy(copy);
    
    shell_status_default_set(a->shl);
    
    if(!a->undo) {
        snd_destroy(del_sr);
        return NULL;
    }
    DEBUG("*** snd_frame_count(clipboard): %ld\n", snd_frame_count(clipboard));
    
    a->shl->has_changed = 1;
    return action_group_new(3,
                            ACTION_DELETE_NEW(DONT_UNDO,
                                              a->shl,
                                              a->sr_target, 
                                              i_map, 
                                              a->offset, 
                                              snd_frame_count(clipboard)),
                            ACTION_INSERT_NEW(DONT_UNDO,
                                              a->sr_target, 
                                              del_sr, 
                                              a->channel_map,
                                              a->offset),
                            ACTION_SELECT_NEW(DONT_UNDO,
                                              a->shl,
                                              a->sr_target, 
                                              a->channel_map, 
                                              a->offset,
                                              a->count));
    
}

action_group *
action_insert(action *a) {
    snd_insert(a->sr_target,
               a->sr_source,
               a->channel_map,
               a->offset);
    
    if(!a->undo) 
        return NULL;

    return action_group_new(1, 
                            ACTION_DELETE_NEW(DONT_UNDO,
                                              a->shl,
                                              a->sr_target,
                                              a->channel_map,
                                              a->offset,
                                              snd_frame_count(a->sr_source)));
}

action_group *
action_erase(action *a) {
    snd *del_sr;

    del_sr = snd_erase(a->sr_target,
                       a->channel_map,
                       a->offset,
                       a->count);
    
    if(!a->undo) {
        snd_destroy(del_sr);
        return NULL;
    }

    return action_group_new(2, 
                            ACTION_DELETE_NEW(DONT_UNDO,
                                              a->shl,
                                              a->sr_target,
                                              a->channel_map,
                                              a->offset,
                                              a->count),
                            ACTION_INSERT_NEW(DONT_UNDO,
                                              a->sr_target,
                                              del_sr,
                                              a->channel_map,
                                              a->offset));
}

action_group *
action_delete(action *a) {
    snd *del_sr;

    del_sr = snd_delete(a->sr_target,
                        a->channel_map,
                        a->offset,
                        a->count);
    if(RESULT_IS_ERROR(a->sr_target)) {
        snd_destroy(del_sr);
        return NULL;

        /* FIXME: snd_delete may complete partially on error and
           return the frames deleted before the error was
           encountered. Should put deleted frames back here. */
    }
    
    if(!a->undo) {
        snd_destroy(del_sr);
        return NULL;
    }

    a->shl->has_changed = 1;
    return action_group_new(1,
                            ACTION_INSERT_NEW(DONT_UNDO,
                                              a->sr_target, 
                                              del_sr, 
                                              a->channel_map, 
                                              a->offset));

}

/* FIXME: action_delete_track is a hack that is needed so that we can
 * construct undo's for actions that may change the size of each track
 * individually. For example, resampling can be controlled by using
 * markers. Because of this, when resampling 2 tracks of equal length,
 * you may end up with 2 tracks of different size. To undo this
 * action, we must delete a different number of frames from the two
 * tracks. Real solution is a selection model that can describe this
 * situation, i.e. something more powerful than just "select start",
 * "select end" and "select map".
 */
action_group *
action_delete_on_track(action *a) {
    snd *del_sr;

    del_sr = snd_delete(a->sr_target,
                        1 << a->channel_map,
                        a->offset,
                        a->count);
    if(RESULT_IS_ERROR(a->sr_target)) {
        snd_destroy(del_sr);
        return NULL;

        /* FIXME: snd_delete may complete partially on error and
           return the frames deleted before the error was
           encountered. Should put deleted frames back here. */
    }
    
    snd_destroy(del_sr);
    
    return NULL;
}

action_group * 
action_module_execute(action *a) {
    module_id id = a->channel_map;
    action_group *(*module_exec)(shell *, int);
    char *error;
    char canceltext[50];
    action_group *undo;

    if(a->shl->active_module_id > -1) {
        gui_alert("%s needs to complete first.",
                  modules[a->shl->active_module_id].name);
        return NULL;
    }

    a->shl->active_module_id = id;
    a->shl->module_cancel_requested = 0;

    DEBUG("executing module %s\n", modules[id].name);

    module_exec = dlsym(modules[id].handle, "module_execute");
    if((error = dlerror()) != NULL) {
        gui_alert("Could not execute %s: %s\n", modules[id].name, error);
        return NULL;
    }

    shell_status_push(a->shl, "%s ...", modules[id].name);
    snprintf(canceltext, 50, "Cancel '%s'", modules[id].name);
    gtk_label_set_text(GTK_LABEL(GTK_BIN(a->shl->cancelmodule)->child), 
                       canceltext);
    gtk_widget_set_sensitive(GTK_WIDGET(a->shl->cancelmodule), TRUE);
    a->shl->has_changed = 1;
    undo = (*module_exec)(a->shl, a->undo);
    a->shl->active_module_id = -1;
    DEBUG("module finished, got undo: %p\n", undo);

    /* We were quit, so don't update UI. */

    if(gtk_main_iteration_do(FALSE)) 
        return undo;

    shell_status_pop(a->shl);
    shell_status_default_set(a->shl);
    gtk_label_set_text(GTK_LABEL(GTK_BIN(a->shl->cancelmodule)->child), 
                       "Cancel");
    gtk_widget_set_sensitive(GTK_WIDGET(a->shl->cancelmodule), FALSE);

    return undo;
}

action_group *
action_module_open(action *a) {
    int id = a->channel_map;
    void *(*module_open)(module_id, shell *, int);
    char *error;

    if(a->shl->module_state[id].is_open) {
        DEBUG("module %s already open\n", modules[id].name);
        return NULL;
    }

    DEBUG("opening module %s\n", modules[id].name);

    module_open = dlsym(modules[id].handle, "module_open");
    if((error = dlerror()) != NULL) {
        DEBUG("module %s doesn't have module_open(), trying module_execute...\n",
              modules[id].name);
        return action_module_execute(a);
    } else {
        (*module_open)(id, a->shl, a->undo);
    }
    return NULL;
}

action_group *
action_exit(action *a) {
    GList *l;
    shell *shl;
    action_result *ar;

    DEBUG("shells: %p\n", shells);
    for(l = shells; l && l->data; ) {
        shl = l->data;
        l = l->next;
        ar = action_do(ACTION_FILE_CLOSE_NEW(shl));
        if(action_result_as_bool(ar)) {
            ACTION_RESULT_RETURN_BOOL(a->id, 0);
            return NULL;
        }
    }
    DEBUG("calling gtk_main_quit\n");
    gtk_main_quit();
    return NULL;
}

static action_desc action_descriptions[] = {
    { "new file", action_file_new, 0 },
    { "open file", action_file_open, 0 },
    { "save file", action_file_save, HAS_SHELL },
    { "save file as", action_file_save_as, HAS_SHELL },
    { "close file", action_file_close, HAS_SHELL },
    { "insert frames", action_insert, HAS_TARGET | HAS_SOURCE | HAS_MAP | HAS_OFFSET },
    { "erase frames", action_erase, HAS_SHELL | HAS_TARGET | HAS_SEL_SPEC },
    { "delete frames", action_delete, HAS_SHELL | HAS_TARGET | HAS_SEL_SPEC },
    { "delete frames on track", action_delete_on_track,
      HAS_SHELL | HAS_TARGET | HAS_TRACK_INDEX | HAS_OFFSET | HAS_COUNT },
    { "cut frames", action_cut, HAS_SHELL | HAS_TARGET | HAS_SEL_SPEC },
    { "paste frames", action_paste, HAS_SHELL | HAS_TARGET | HAS_SEL_SPEC },
    { "paste frames over", action_paste_over, HAS_SHELL | HAS_TARGET | HAS_SEL_SPEC },
    { "paste mix", action_paste_mix, HAS_SHELL | HAS_TARGET | HAS_SEL_SPEC },
    { "paste fit", action_paste_fit, HAS_SHELL | HAS_TARGET | HAS_SEL_SPEC },
    { "select frames", action_select, HAS_SHELL | HAS_TARGET | HAS_SEL_SPEC },
    { "copy frames", action_copy, HAS_SHELL | HAS_TARGET | HAS_SEL_SPEC },
    { "insert tracks", action_tracks_insert, HAS_SHELL | HAS_TARGET | HAS_SOURCE | NULL_SOURCE | HAS_MAP },
    { "delete tracks", action_tracks_delete, HAS_SHELL | HAS_TARGET | HAS_MAP },
    { "fit selection", action_selection_fit, HAS_SHELL },
    { "selection to loop", action_selection_to_loop, HAS_SHELL },
    { "loop to selection", action_loop_to_selection, HAS_SHELL },
    { "play", action_player_play, HAS_SHELL },
    { "stop", action_player_stop, HAS_SHELL },
    { "module_open", action_module_open, HAS_SHELL | HAS_MODULE_INDEX },
    { "module_execute", action_module_execute, HAS_SHELL | HAS_MODULE_INDEX },
    { "exit", action_exit, 0 },
    { "LAST", NULL }
};

/* It is legal to pass NULL as the action. In that case this function
 * returns a FALSE value. This way the result from action_new() can be
 * immediately fed to action_do(). The user is alerted of any errors
 * in action_new(). It is very unlikely for action_new() to fail
 * anyway since it mainly involves allocation a few bytes for the
 * action struct. 
 * Otherwise this function executes and destroys the action, passing 
 * any errors along to the user. 
 */

action_result *
action_do(action *a) {
    action_group *undo;

    /* Check consistency. */

    if(strcmp(action_descriptions[ACTION_LAST].name, "LAST") ||
       action_descriptions[ACTION_LAST].func != NULL) {
        FAIL("action_description list length does not match ACTION_LAST. abort.\n");
        abort();
    }

    if(!a) 
        return &false_action_result;

    if(a->id < 0 || a->id >= ACTION_LAST) {
        FAIL("invalid action id: %d\n", a->id);
        action_destroy(a);
        gui_alert("Internal error: attempt to execute unknown action %d.\n", a->id);
        return &false_action_result;
    }

    ACTION_RESULT_INIT(a->id);
    DEBUG("performing action %s\n", action_descriptions[a->id].name);
    undo = (*action_descriptions[a->id].func)(a);

    if(!a->suppress_redraw &&
       a->shl &&
       a->id != ACTION_FILE_CLOSE &&
       a->id != ACTION_EXIT &&
       !gtk_main_iteration_do(FALSE))
        shell_redraw(a->shl);

    //    DEBUG("undo: %p, a->in_group: %d\n",
    //          undo, a->in_group);
    if(undo) {
        if(!a->in_group)
            a->shl->undo_stack = 
                undo_push(a->shl->undo_stack, undo);
        else
            ACTION_RESULT_UNDO_SET(undo);
    }

    action_destroy(a);
    return ACTION_RESULT;
}

/**
 * This function makes of copy of the frames frame_offset to
 * frame_count with the given channel_map and then pushes a
 * "delete-insert-select" action sequence onto the undo stack. It's an
 * easy way to preserve sound region before it gets modified. FIXME:
 * error return, failure modes?
 */
action_group *
action_group_undo_create(shell *shl,
                         int channel_map,
                         AFframecount frame_offset,
                         AFframecount frame_count,
                         AFframecount delete_frame_offset,
                         AFframecount delete_frame_count) {
    action_group *ag;
    snd *del_sr, *ins_sr;

    del_sr = snd_delete(shl->sr,
                        channel_map,
                        frame_offset, 
                        frame_count);
    if(RESULT_IS_ERROR(shl->sr)) {
        snd_error_set(shl->sr,
                      "could not create undo for %ld-%ld (%s)",
                      frame_offset, frame_offset + frame_count, snd_error_get(shl->sr));
        if(del_sr) {
            snd_insert(shl->sr,
                       del_sr,
                       channel_map,
                       frame_offset);
            snd_destroy(del_sr);
        }
        return NULL;
    }

    ins_sr = snd_clone(del_sr, 
                       CLONE_STRUCTURE | 
                       CLONE_TRACK_DATA | 
                       CLONE_TRACK_STRUCTURE);
    if(RESULT_IS_ERROR(del_sr)) {
        snd_error_set(shl->sr, 
                      "could not create undo for %ld-%ld (%s)",
                      frame_offset, frame_offset + frame_count, snd_error_get(del_sr));
        snd_insert(shl->sr,
                   del_sr,
                   channel_map,
                   frame_offset);
        snd_destroy(del_sr);
        return NULL;
    }

    snd_insert(shl->sr,
               ins_sr,
               channel_map,
               frame_offset);
    snd_destroy(ins_sr);

    ag = action_group_new(3,
                          ACTION_DELETE_NEW(DONT_UNDO,
                                            shl,
                                            shl->sr,
                                            channel_map,
                                            delete_frame_offset,
                                            delete_frame_count),
                          ACTION_INSERT_NEW(DONT_UNDO,
                                            shl->sr, 
                                            del_sr, 
                                            channel_map,
                                            frame_offset),
                          ACTION_SELECT_NEW(DONT_UNDO,
                                            shl,
                                            shl->sr,
                                            channel_map,
                                            frame_offset,
                                            snd_frame_count(del_sr)));
    return ag;
}

void
action_group_dump(action_group *ag) {
    int i;
    DEBUG("action_group: %p, count: %d", 
          ag, ag->count);
    INFO(" (");
    for(i = 0; i < ag->count; i++) {
        INFO("%p", ag->a[i]);
        if(i < ag->count - 1)
            INFO(", ");
    }
    INFO(")\n");
    //    DEBUG("\n");
}

/* FIXME: failure here clutters up code everywhere, maybe promise
   that it never fails by eliminating mem allocation. */

action_group *
action_group_append(action_group *ag,
                    action *a) {
    action_group *ag_new;
    //    DEBUG("before append:\n");
    //    action_group_dump(ag);

    ag_new = mem_realloc(ag, sizeof(action_group) + 
                         (sizeof(action *) * (ag->count + 1)));
    if(!ag_new)
        return NULL;

    ag_new->a[ag_new->count] = a;
    ag_new->count++;
    //    DEBUG("after append:\n");
    //    action_group_dump(ag_new);
    return ag_new;
}

action_group *
action_group_prepend(action_group *ag,
                     action_group *ag_ins) {
    int i;
    action_group *ag_new;
    //    DEBUG("before prepend:\n");
    //    action_group_dump(ag);
    //    action_group_dump(ag_ins);

    ag_new = mem_alloc(sizeof(action_group) + 
                       (sizeof(action *) * (ag->count + ag_ins->count)));
    if(!ag_new)
        return NULL;
    ag_new->count = ag->count + ag_ins->count;
    for(i = 0; i < ag_ins->count; i++) 
        ag_new->a[i] = ag_ins->a[i];
    
    for(i = ag_ins->count; i < ag_new->count; i++)
        ag_new->a[i] = ag->a[i - ag_ins->count];

   mem_free(ag);
   mem_free(ag_ins);

   //    DEBUG("after prepend:\n");
   //    action_group_dump(ag_new);
   return ag_new;
}

void
action_group_do(action_group *ag) {
    int i;
    action *a;
    action_result *ar;
    action_group *undo = NULL, *ag_new;
    shell *shl = NULL;

    //    DEBUG("ag: %p\n", ag);
    for(i = 0; i < ag->count; i++) {
        a = ag->a[i];

        if(i < ag->count - 1)
            a->suppress_redraw = 1;
        a->in_group = 1;
        if(a->shl)
            shl = a->shl;
        
        //        DEBUG("a[%d]: %p...\n", i, a);
        ar = action_do(a);
        //        DEBUG("a[%d]: ...returns %p (undo: %p)\n", i, ar, ar->undo);
        if(ar->undo) {
            if(!undo) {
                undo = ar->undo;
                continue;
            }
            ag_new = action_group_prepend(undo, ar->undo);
            if(!ag_new) {
                FAIL("could not prepend action_groups for undo, " \
                     "undo may be lost or broken\n");
                continue;
            }
            undo = ag_new;
        }
    }
    ag->count = 0;
    action_group_destroy(ag);

    //    DEBUG("a->shl->undo_stack: %p, undo: %p\n", a->shl->undo_stack, undo);
    //    action_group_dump(undo);

    if(undo) {
        if(!shl) {
            FAIL("undo but no shell in any of the action in group");
            return;
        }
        shl->undo_stack = 
            undo_push(shl->undo_stack, undo);
    }

}

action_group *
action_group_new_empty(int count) {
    action_group *ag;

    ag = mem_alloc(sizeof(action_group) + (sizeof(action *) * count));
    if(!ag) {
        FAIL("cannot allocate memory for action group (%d items)\n", count);
        return NULL;
    }

    ag->count = count;
    return ag;
}

action_group *
action_group_new(int count, ...) {
    int i;
    va_list ap;
    action_group *ag;

    ag = action_group_new_empty(count);
    if(!ag) 
        return NULL;

    va_start(ap, count);

    for(i = 0; count--; i++) 
        ag->a[i] = va_arg(ap, action *);
    
    va_end(ap);

    return ag;
}

action *
action_new(action_id id,
           shell *shl,
           snd *sr_target,
           snd *sr_source,
           int channel_map,
           AFframecount offset,
           AFframecount count,
           int undo) {
    action *a;
    action_desc *ad;

    if(id < 0 && id > ACTION_LAST) {
        gui_alert("Internal error: attempt to create unknown action %d.\n", id);
        return NULL;
    }
    ad = &action_descriptions[id];

    if((ad->signature & HAS_SHELL) && !shl) {
        FAIL("attempt to create action '%s' but shell is undefined\n",
             ad->name);
        abort();
    }
    if((ad->signature & HAS_TARGET) && !sr_target) {
        FAIL("attempt to create action '%s' but target is undefined\n",
             ad->name);
        abort();
    }
    if((ad->signature & HAS_SOURCE) && !sr_source && !(ad->signature & NULL_SOURCE)) {
        FAIL("attempt to create action '%s' but source is undefined\n",
             ad->name);
        abort();
    }
    if((ad->signature & HAS_MAP) && channel_map == -1) {
        FAIL("attempt to create action '%s' but channel_map is undefined\n",
             ad->name);
        abort();
    }
    if((ad->signature & HAS_OFFSET) && offset == -1) {
        FAIL("attempt to create action '%s' but offset is undefined\n",
             ad->name);
        abort();
    }
    if((ad->signature & HAS_COUNT) && count == -1) {
        FAIL("attempt to create action '%s' but count is undefined\n",
             ad->name);
        abort();
    }
    if((ad->signature & HAS_MODULE_INDEX) && channel_map == -1) {
        FAIL("attempt to create action '%s' but module index is undefined\n",
             ad->name);
        abort();
    }
    if((ad->signature & HAS_TRACK_INDEX) && channel_map == -1) {
        FAIL("attempt to create action '%s' but track index is undefined\n",
             ad->name);
        abort();
    }
    a = mem_alloc(sizeof(action));
    if(!a) {
        FAIL("could not allocate new action (%s)\n", action_descriptions[id].name);
        gui_alert("Not enough memory to create action for %s.", action_descriptions[id].name);
        return NULL;
    }
    a->id = id;
    a->shl = shl;
    a->sr_target = sr_target;
    a->sr_source = sr_source;
    a->channel_map = channel_map;
    a->offset = offset;
    a->count = count;
    a->undo = undo;
    a->suppress_redraw = 0;
    a->in_group = 0;
    return a;
}

void
action_destroy(action *a) {
    if(a->sr_source) 
        snd_destroy(a->sr_source);
   mem_free(a);
}

void
action_group_destroy(action_group *ag) {
    int i;
    for(i = 0; i < ag->count; i++) 
        if(ag->a[i]) 
            action_destroy(ag->a[i]);
   mem_free(ag);
}

gboolean
action_result_as_bool(action_result *ar) {
    gboolean b = FALSE;

    if(!ar->defined)
        b = FALSE;
    if(!ar->v.str_val)
        b = FALSE;

    if(atoi(ar->v.str_val) == 1)
        b = TRUE;
    else
        b = FALSE;
    
    DEBUG("action: %s, return: %s\n", 
          action_descriptions[ar->id].name, b == TRUE ? "TRUE" : "FALSE");
    return b;
}

void
action_result_undo_set(action_result *ar,
                       action_group *undo) {
    ar->undo = undo;
}

void
action_result_bool_set(action_result *ar, 
                       action_id id,
                       gboolean b) {
    ar->id = id;
    ar->v.str_val = mem_alloc(2);

    if(!ar->v.str_val) {
        FAIL("cannot get two measly bytes for action result. returning 0 value, original value was %d", b);
        return;
    }

    ar->v.str_val[1] = '\0';
    ar->v.str_val[0] = b == FALSE ? '0' : '1';
    DEBUG("action: %s, new value: %s\n", 
          action_descriptions[ar->id].name, b == TRUE ? "TRUE" : "FALSE");
}

void
action_result_init(action_result *ar, 
                   action_id id) {
    ar->id = id;
    ar->defined = 0;
    ar->undo = NULL;
    if(ar->v.str_val) {
        mem_free(ar->v.str_val);
        ar->v.str_val = NULL;
    }
}
