/*
 * Copyright (C) 2002 Pascal Haakmat. 
 */

#include <stdlib.h>
#include <audiofile.h>
#include "config.h"
#include "mem.h"
#include "mixer.h"

float **
mixer_mixtable_new(int target_channels,
                   int source_channels) {
    int i;
    float **mt = NULL;
    
    mt = (float **) mem_alloc(target_channels * sizeof(float *));
    if(!mt) {
        FAIL("could not allocate mix table (%d bytes)\n", 
             target_channels * sizeof(float));
            return NULL;
    }
    DEBUG("allocated %d bytes for mixtable\n",
          target_channels * sizeof(float));
    for(i = 0; i < target_channels; i++) {
        mt[i] = mem_alloc(source_channels * sizeof(float));
        if(!mt[i]) {
            for(i--; i + 1 > 0; i--)
                mem_free(mt[i]);
            mem_free(mt);
            FAIL("could not allocate mix table rows\n");
            return NULL;
        }
    }
    return mt;
}

void
mixer_mixtable_destroy(float **mt,
                       int target_channels) {
    int i;
    for(i = 0; i < target_channels; i++) 
        mem_free(mt[i]);
    mem_free(mt);
    DEBUG("freed %d channel mixtable\n", target_channels);
}

void 
mixer_dump(mixer *m) {
    int i, j;
    DEBUG("target_channels: %d, source_channels: %d\n",
          m->target_channels, m->source_channels);
    for(i = 0; i < m->target_channels; i++) {
        INFO("%2i:", i);
        for(j = 0; j < m->source_channels; j++) {
            INFO(" %.3f", m->mixtable[i][j]);
            //            DEBUG("route dump: source: %d, target: %d: %f\n",
            //                  j, i, m->mixtable[i][j]);
        }
        INFO("\n");
    }
}

void
mixer_init(mixer *m) {
    int i, j;
    if(!m->target_channels)
        return;
    for(i = 0; i < m->target_channels; i++) {
        for(j = 0; j < m->source_channels; j++) 
            m->mixtable[i][j] = 0; //(j == i) ? 1 : 0;
    }
    mixer_dump(m);
}

void
mixer_configure(mixer *m,
                int target_channels,
                int source_channels) {
    int i, j;

    if(target_channels > MAX_TRACKS ||
       source_channels > MAX_TRACKS) {
        FAIL("illegal configure request, target_channels: %d, source_channels: %d\n",
             target_channels, source_channels);
        abort();
    }

    DEBUG("m->target_channels: %d, target_channels: %d, m->source_channels: %d, source_channels: %d\n",
          m->target_channels, target_channels, m->source_channels, source_channels);
    for(i = 0; i < target_channels; i++) {
        for(j = m->source_channels; j < source_channels; j++) {
            DEBUG("[%d][%d] = 0\n", i, j);
            m->mixtable[i][j] = i == j ? 1 : 0;
        }
    }
    m->target_channels = target_channels;
    m->source_channels = source_channels;
    mixer_dump(m);

    DEBUG("target_channels: %d, source_channels: %d\n", 
          m->target_channels, m->source_channels);
}

void
mixer_destroy(mixer *m) {
    DEBUG("destroying mixer\n");
    mixer_mixtable_destroy(m->mixtable, m->target_channels);
    mem_free(m);
}

mixer *
mixer_new(int target_channels,
          int source_channels) {
    mixer *m = calloc(sizeof(mixer), 1);
    if(!m) {
        FAIL("could not allocate mixer object.\n");
        return NULL;
    }

    m->mixtable = mixer_mixtable_new(MAX_TRACKS, MAX_TRACKS);

    if(!m->mixtable) {
        FAIL("could not allocate mixtable.\n");
        free(m);
        return NULL;
    }

    mixer_configure(m, target_channels, source_channels);
    mixer_init(m);
    
    return m;
}

void
mixer_mux(mixer *m,
          _frame_bits fb_target,
          _frame_bits *fb_sources,
          int frame_width,
          AFframecount frame_count) {
    int i, j, k;

    for(i = 0; i < frame_count; i++) {
        for(j = 0; j < m->target_channels; j++) {
            switch(frame_width) {
            case 1:
                for(k = 0; k < m->source_channels; k++) 
                    ((int8_t *)fb_target)[(i * m->target_channels) + j] +=
                        (int8_t) (((float) ((int8_t **)fb_sources)[k][i]) *
                                    m->mixtable[j][k]);
                break;
            case 2:
                for(k = 0; k < m->source_channels; k++) 
                    ((int16_t *)fb_target)[(i * m->target_channels) + j] +=
                        (int16_t) (((float) ((int16_t **)fb_sources)[k][i]) *
                                     m->mixtable[j][k]);
                break;
            case 4:
                for(k = 0; k < m->source_channels; k++) 
                    ((int32_t *)fb_target)[(i * m->target_channels) + j] +=
                        (int32_t) (((float) ((int32_t **)fb_sources)[k][i]) *
                                     m->mixtable[j][k]);
                break;
            }
        }
    }
}

void
mixer_demux(_frame_bits fb_target,
            _frame_bits fb_source,
            int channel,
            int channel_count,
            int frame_width,
            AFframecount frame_count) {
    for(frame_count--; frame_count + 1 > 0; frame_count--) {
        switch(frame_width) {
        case 1:
            *((int8_t *)FB1(fb_target, frame_width, frame_count)) =
                *((int8_t *)FB(fb_source, frame_width, channel_count, frame_count, channel));
            break;
        case 2:
            *((int16_t *)FB1(fb_target, frame_width, frame_count)) =
                *((int16_t *)FB(fb_source, frame_width, channel_count, frame_count, channel));
            break;
        case 4:
            *((int32_t *)FB1(fb_target, frame_width, frame_count)) =
                *((int32_t *)FB(fb_source, frame_width, channel_count, frame_count, channel));
            break;
        }
    }
}

void 
mixer_buffers_free(int tracks,
                   _frame_bits fb_downmix,
                   _frame_bits *fb_sources) {
    int i;
    for(i = 0; i < tracks; i++) {
        if(fb_sources[i])
            mem_free(fb_sources[i]);
        fb_sources[i] = NULL;
    }
    DEBUG("freed mixer buffers %p\n", fb_downmix);
    if(fb_downmix)
        mem_free(fb_downmix);
}

_frame_bits
mixer_buffers_alloc(int frame_width,
                    int tracks,
                    _frame_bits *fb_downmix,
                    _frame_bits *fb_sources,
                    AFframecount frame_count) {
    int i;
    for(i = 0; i < tracks; i++)
        fb_sources[i] = NULL;
    *fb_downmix = mem_alloc(frame_width * tracks * frame_count);
    if(!*fb_downmix) {
        FAIL("could not get memory for downmix buffer (%d tracks).\n", tracks);
        return NULL;
    }
    for(i = 0; i < tracks; i++) {
        fb_sources[i] = mem_alloc(frame_width * frame_count);
        if(fb_sources[i] == NULL) {
            FAIL("could not get memory for track buffer %d\n", i);
            for(i--; i + 1 > 0; i--) {
                mem_free(fb_sources[i]);
                fb_sources[i] = NULL;
            }
            mem_free(*fb_downmix);
            *fb_downmix = NULL;
            return NULL;
        }
    }
    DEBUG("mixer buffers allocated: muxbuf: %ld bytes@%p, srcbufs: %ld bytes (* %d)\n",
          frame_width * tracks * frame_count, *fb_downmix, frame_width * frame_count, tracks);

    return *fb_downmix;
}
