/*
 * Copyright (C) 2002 Pascal Haakmat.
 */

#include <string.h>
#include <stdlib.h>
#include <errno.h>
#include <sys/time.h>
#include "config.h"
#include "mem.h"
#include "snd.h"
#include "gui.h"

extern int errno;
static char *nomem_error_str = "out of memory!";

void
snd_convert(snd *sr,
            int sample_width,
            float sample_rate) {
    int i, r;
    int32_t *int32_frame_buffer;
    snd *converted_sr;
    AFframecount frame_offset, frame_count, c;
    track *old_tracks[MAX_TRACKS];

    rwlock_wlock(&sr->rwl);

    if(sample_width != sr->frame_width) {
        int32_frame_buffer = mem_alloc(BLOCK_SIZE * sizeof(int32_t));
        if(!int32_frame_buffer) {
            snd_error_set(sr, "not enough memory to convert to %d bits",
                          sample_width * 8);
            rwlock_wunlock(&sr->rwl);
            return;
        }

        converted_sr = snd_clone(sr,
                                 CLONE_STRUCTURE |
                                 CLONE_TRACK_STRUCTURE);
        if(RESULT_IS_ERROR(sr)) {
            snd_error_set(sr, "not enough memory to convert to %d bits (%s)",
                          sample_width * 8, snd_error_get(sr));
            mem_free(int32_frame_buffer);
            rwlock_wunlock(&sr->rwl);
            return;
        }
        converted_sr->frame_width = sample_width;
        
        for(i = 0; i < sr->channels; i++) {
            converted_sr->tracks[i]->frame_width = sample_width;
            frame_count = track_frame_count(sr->tracks[i]);
            c = MIN(frame_count, BLOCK_SIZE);
            for(frame_offset = 0; frame_count; c = MIN(frame_count, 
                                                       BLOCK_SIZE)) {
                c = track_int32_frame_buffer_get(sr->tracks[i],
                                                 int32_frame_buffer,
                                                 frame_offset,
                                                 c);
                r = track_int32_frame_buffer_put(converted_sr->tracks[i],
                                                 int32_frame_buffer,
                                                 frame_offset,
                                                 c);
                if(r) {
                    mem_free(int32_frame_buffer);
                    snd_destroy(converted_sr);
                    snd_error_set(sr, 
                                  "not enough memory to convert to %d bits",
                                  sample_width * 8);
                    rwlock_wunlock(&sr->rwl);
                    return;
                }

                frame_offset += c;
                frame_count -= c;
                if(c < 1) 
                    frame_count = 0;
            }
            old_tracks[i] = sr->tracks[i];
        }

        /* Steal the converted data from the converted_sr and put the 
           old data in the converted_sr so we can destroy it. */

        sr->frame_width = sample_width;
        for(i = 0; i < converted_sr->channels; i++) {
            sr->tracks[i] = converted_sr->tracks[i];
            converted_sr->tracks[i] = old_tracks[i];
        }

        snd_destroy(converted_sr);
        mem_free(int32_frame_buffer);
    }
    
    if(sample_rate != sr->rate) 
        sr->rate = sample_rate;

    rwlock_wunlock(&sr->rwl);
 
   return;
}

void
snd_tracks_dump(snd *sr) {
    int i;
    //    DEBUG("'%s' claims to have %d tracks, dumping all %d:\n", 
    //          sr->name, sr->channels, MAX_TRACKS);
    
    for(i = 0; i < sr->channels; i++) 
        INFO("%2d: %p, bl->l: %p, frame_count: %ld\n", 
             i, sr->tracks[i], sr->tracks[i]->bl->l, track_frame_count(sr->tracks[i]));    
}

/* Deletes the tracks specified by channel_map from the specified snd and 
 * returns the deleted tracks. 
 */

snd *
snd_tracks_delete(snd *sr,
                  int channel_map) {
    int i, j, nt = 0, ch = sr->channels;
    snd *del_sr;

    rwlock_wlock(&sr->rwl);

    for(i = 0; i < ch; i++)
        if((1 << i) & channel_map) 
            nt++;

    if(nt == 0) {
        snd_error_set(sr,
                      "scared to resize to 0 tracks");
        rwlock_wunlock(&sr->rwl);
        return NULL;
    }

    del_sr = snd_clone(sr, CLONE_STRUCTURE);
    
    if(RESULT_IS_ERROR(sr)) {
        snd_error_set(sr,
                      "unable to clone to store deleted tracks (%s)",
                      snd_error_get(sr));
        rwlock_wunlock(&sr->rwl);
        return NULL;
    }

    snd_name_set(del_sr, "deleted tracks");

    nt = 0;
    for(i = 0; i < ch; i++) {
        if((1 << i) & channel_map) {
            DEBUG("reparenting track %d\n", i);
            del_sr->tracks[nt++] = sr->tracks[i];
            sr->tracks[i] = NULL;
        }
    }

    del_sr->channels = nt;

    sr->channels = 0;
    for(i = 0; i < MAX_TRACKS; i++) {
        if(sr->tracks[i]) {
            sr->channels++;
            continue;
        }

        for(j = i; j < ch; j++) {
            if(sr->tracks[j]) {
                sr->channels++;
                sr->tracks[i] = sr->tracks[j];
                sr->tracks[j] = NULL;
                break;
            }
        }
    }

    rwlock_wunlock(&sr->rwl);
    return del_sr;
}

/* Inserts the tracks specified by ins_sr into sr according to the
 * given channel_map. ins_sr may be NULL in which case empty tracks
 * will be inserted. When the channel_map specifies more channels than
 * exist in the given sr, then this function appends tracks.
 *
 * ins_sr can be NULL.
 *
 * On error, the error value for sr is set.
 * FIXME: doesn't return anything useful.
 */

void
snd_tracks_insert(snd *sr,
                  snd *ins_sr,
                  int channel_map) {
    int i, j = 0, k, discount = 0;
    track *tr = NULL;

    rwlock_wlock(&sr->rwl);

    if(sr->channels >= MAX_TRACKS) {
        snd_error_set(sr,
                      "cannot add new tracks, too many tracks.");
        rwlock_wunlock(&sr->rwl);
        return;
    }

    DEBUG("channel_map: %d\n", channel_map);

    for(i = 0; i < MAX_TRACKS; i++) {
        if((1 << i) & channel_map) { 
            DEBUG("inserting at %d because channel_map says so\n", i);
            if(ins_sr) {
                if(j < ins_sr->channels) {
                    tr = ins_sr->tracks[j];
                    ins_sr->tracks[j] = NULL;
                    j++;
                }
            } else {
                tr = track_new(sr->frame_width, 0);
                if(!tr) 
                    FAIL("could not create new empty track, don't know what to do.\n");
            }
            
            DEBUG("reparenting track from source pos %d to destination pos %d\n", j, i);
            for(k = MAX_TRACKS - 1; k > i; k--) 
                sr->tracks[k] = sr->tracks[k-1];

            sr->channels += 1 - discount;
            discount = 0;
            sr->tracks[i] = tr;
        }

        /* Insert turns into append. Bit strange. */

        if((1 << (i + 1)) & channel_map) {
            discount++;
            sr->channels++;
        }
    }

    if(ins_sr)
        ins_sr->channels = 0;

    rwlock_wunlock(&sr->rwl);
}

/* Returns a copy of the given sr. The method variable specifies
 * whether to clone just the basic sound properties, or whether to
 * also clone the tracks, or whether to also clone the track data. 
 *
 * On error, the error value for sr is set and NULL is returned.
 */

snd *
snd_clone(snd *sr,
          snd_clone_method method) {
    int t;
    snd *copy;

    if(RESULT_IS_ERROR(sr))
        return 0;

    rwlock_rlock(&sr->rwl);

    /* Create a new snd. */

    copy = (snd *)snd_new();
    
    if(!copy || RESULT_IS_ERROR(copy)) {
        snd_error_set(sr,
                      "unable to clone sound object (%s)",
                      snd_error_get(copy));
        snd_destroy(copy);
        rwlock_runlock(&sr->rwl);
        return NULL;
    }

    copy->frame_width = sr->frame_width;
    copy->channels = sr->channels;
    copy->rate = sr->rate;

    /* Create some empty tracks to go with that snd. */

    if(method & CLONE_TRACK_STRUCTURE) {
        for(t = 0; t < sr->channels; t++) {
            copy->tracks[t] = track_clone(sr->tracks[t], method);
            if(copy->tracks[t] == NULL) {
                snd_error_set(sr,
                              "unable to clone track %d", t);
                snd_destroy(copy);
                rwlock_runlock(&sr->rwl);
                return NULL;
            }
        }
    }

    rwlock_runlock(&sr->rwl);
    return copy;
}

/* Inserts one snd into another.
 * channel_map is a mask specifying how to map the channels in 
 * ins_sr to the channels in sr.
 * e.g. 1101  means: 
 *      |||`- put first available track in src onto track 0 in sr.
 *      ||`-- do not put anything on track 1 in sr.
 *      |`--- put second available track in src onto track 2 in sr.
 *      `---- put third available track in src onto track 3 in sr.
 * The number of bits looked at is always smaller than sr->channels.
 * An 'available track' is a track that has a non-zero frame count.
 * This function returns a channel_map specifying which channels
 * were actually mapped. 
 */

int
snd_insert(snd *sr,
           snd *ins_sr,
           int channel_map,
           AFframecount frame_offset) {
    int t, u, i_map = 0, err = 0;

    if(RESULT_IS_ERROR(sr))
        return 0;

    rwlock_rlock(&sr->rwl);

    for(t = 0, u = 0; t < sr->channels && u < ins_sr->channels; t++) {
        if((1 << t) & channel_map) {
            while(u < ins_sr->channels &&
                  !track_frame_count(ins_sr->tracks[u])) u++;
            if(u == ins_sr->channels) {
                DEBUG("no more tracks available for insert (u: %d)\n", u);
                break;
            }
            err = track_insert(sr->tracks[t],
                               ins_sr->tracks[u++]->bl->l,
                               frame_offset);

            /* FIXME: If insert fails, we are out of memory. We log
               the error and return, but without reversing the effects
               of what we did up until now. */

            if(err) {
                snd_error_set(sr,
                              "track %d insert at %ld failed, cannot recover",
                              t, frame_offset);
                rwlock_runlock(&sr->rwl);
                return i_map;
            }
            i_map |= (1 << t);
        }
    }

    rwlock_runlock(&sr->rwl);

    return i_map;
}

/* Deletes frames from the given sr and returns them.  On error, the
 * error value is set for sr. The return value is then either NULL in
 * case no frames were deleted, or a snd struct specifying the deleted
 * frames before the error occurred. 
 */

snd *
snd_delete(snd *sr,
           int channel_map,
           AFframecount frame_offset,
           AFframecount frame_count) {
    int t;
    GList *del_blks;
    snd *del_sr;

    if(RESULT_IS_ERROR(sr))
        return NULL;

    rwlock_rlock(&sr->rwl);

    del_sr = snd_clone(sr, CLONE_STRUCTURE | CLONE_TRACK_STRUCTURE);

    if(RESULT_IS_ERROR(sr)) {
        snd_error_set(sr,
                      "unable to clone to store deleted frames (%s)",
                      snd_error_get(del_sr));
        snd_destroy(del_sr);
        rwlock_wunlock(&sr->rwl);
        return NULL;
    }

    snd_name_set(del_sr, "deleted frames");

    for(t = 0; t < sr->channels; t++) {
        del_blks = NULL;
        if((1 << t) & channel_map) 
            del_blks = track_delete(sr->tracks[t],
                                    frame_offset,
                                    frame_count);

        blocklist_blocks_set(del_sr->tracks[t]->bl, del_blks);
    }
    
    rwlock_runlock(&sr->rwl);

    return del_sr;
}

void
snd_pad(snd *sr,
        int channel_map,
        AFframecount frame_count) {
    int t;
    GList *l = NULL;
    block *b;

    if(RESULT_IS_ERROR(sr))
        return;

    rwlock_rlock(&sr->rwl);

    DEBUG("padding to frame_count: %ld\n", frame_count);
    for(t = 0; t < sr->channels; t++) {
        if((1 << t) & channel_map) {
            if(track_frame_count(sr->tracks[t]) < frame_count) {
                DEBUG("creating NULL block, %ld frames\n", frame_count);
                b = block_new(CACHE_NULL, 
                              sr->tracks[t]->frame_width, 
                              sr->tracks[t]->graph_hres, 
                              frame_count - track_frame_count(sr->tracks[t]));
                if(!b) {
                    FAIL("could not create %ld frame NULL block\n", frame_count);
                    return;
                }
                DEBUG("inserting NULL block\n");
                l = g_list_append(l, b);
                track_insert(sr->tracks[t], l, track_frame_count(sr->tracks[t]));
                block_unref(b);
                l->data = NULL;
                g_list_free(l);
                l = NULL;

            }
        }
    }

    rwlock_runlock(&sr->rwl);
    return;
}

snd *
snd_erase(snd *sr,
          int channel_map,
          AFframecount frame_offset,
          AFframecount frame_count) {
    int t;
    snd *del_sr;
    GList *l = NULL;
    block *b;

    if(RESULT_IS_ERROR(sr))
        return NULL;

    rwlock_rlock(&sr->rwl);

    snd_pad(sr, channel_map, frame_offset + frame_count);

    del_sr = snd_delete(sr, channel_map, frame_offset, frame_count);

    DEBUG("erased frames: \n");
    snd_dump(del_sr);
    for(t = 0; t < sr->channels; t++) {
        if((1 << t) & channel_map) {
            DEBUG("creating NULL block, %ld frames\n", frame_count);
            b = block_new(CACHE_NULL, 
                          sr->tracks[t]->frame_width, 
                          sr->tracks[t]->graph_hres, 
                          frame_count);
            if(!b) {
                FAIL("could not create %ld frame NULL block\n", frame_count);
                return del_sr;
            }
            DEBUG("inserting NULL block\n");
            l = g_list_append(l, b);
            track_insert(sr->tracks[t], l, frame_offset);
            block_unref(b);
            l->data = NULL;
            g_list_free(l);
            l = NULL;
        }
    }

    rwlock_runlock(&sr->rwl);
    return del_sr;
}

int
snd_track_count(snd *sr) {
    return sr->channels;
}

AFframecount 
snd_frame_count(snd *sr) {
    int t;
    AFframecount s, p = 0;
    rwlock_rlock(&sr->rwl);
    for(t = 0; t < sr->channels; t++) {        
        s = track_frame_count(sr->tracks[t]);
        if(s > p) 
            p = s;
    }
    rwlock_runlock(&sr->rwl);
    return p;
}

double 
snd_frames_to_time(const snd *sr, AFframecount c) {
    return ((double) c / sr->rate);
}

inline graph_bits_unit
snd_frame_to_graph(int frame_width,
                   _frame_bits frame_bits) {
    switch(frame_width) {
    case 1:
        return -(*((int8_t *)frame_bits));
    case 2:
        return -(*((int16_t *)frame_bits) >> 8);
    case 4:
        return -(*((int32_t *)frame_bits) >> 24);
    default:
        return 0;
    }
}

void 
snd_scale(_graph_bits graph_bits,
          void *source_bits,
          AFframecount frame_count,
          int frame_width,
          float skip_amount,
          AFframecount src_off,
          AFframecount dst_off,
          int source_is_graph_bits) {
    AFframecount c;
    _graph_bits_t p = { 0, 0 }, s = { 0, 0 };
    _graph_bits gbp = (_graph_bits)source_bits;
    _frame_bits fbp = (_frame_bits)source_bits;
    int k;
    float i;

    for(i = 0, c = 0; 
        i < frame_count; 
        i += skip_amount, c++) {
        for(k = 0; k < skip_amount && i+k < frame_count; k++) {
            if(source_is_graph_bits)
                s = gbp[(AFframecount) (src_off+i+k)];
            else
                s.l = s.h = snd_frame_to_graph(frame_width, 
                                               fbp + (frame_width * 
                                                      (size_t)(src_off+i+k)));
            /* Low/high algorithm. */

            if(!k || (s.l < p.l))
                p.l = s.l;
            if(!k || (s.h >= p.h))
                p.h = s.h;
        }
        graph_bits[dst_off+c] = p;
    }
}

void 
snd_frames_buffer_to_graph_buffer(_graph_bits graph_bits,
                                  _frame_bits frame_bits,
                                  AFframecount frame_count,
                                  int frame_width,
                                  float hres) {
    snd_scale(graph_bits,
              frame_bits,
              frame_count,
              frame_width,
              hres,
              0,
              0,
              0);

}

void
snd_info_dump(snd *sr) {
    snd_dump(sr);
}

void
snd_name_set(snd *sr,
             const char *name) {
    char *p;

    if(!name) {
        snd_error_set(sr, 
                      "name == NULL\n");
        return;
    }

    p = strdup(name);

    if(p == NULL) {
        snd_error_set(sr, 
                      RESULT_ERROR_NOMEM);
        return;
    }

    if(sr->name)
        mem_free(sr->name);

    sr->name = p;
}

/* Demultiplexes frame_count frames from the interleaved samples
 * stored in fb_source into the supplied (temporary) buffer fb_target,
 * (which must be big enough to contain frame_count samples), then
 * inserts them into the given snd at the position specified by
 * frame_offset. channel_count specifies how many channels are
 * interleaved in fb_source buffer. channel_map is a bitfield
 * specifying on which tracks in sr the demuxed data should appear. It
 * should never specify more channels than given in
 * channel_count. Used to insert data from file or recording. FIXME:
 * return code?
 */

int
snd_demux(snd *sr,
          _frame_bits fb_target,
          _frame_bits fb_source,
          int channel_map,
          int channel_count,
          AFframecount frame_offset,
          AFframecount frame_count) {
    int i, t;

    if(RESULT_IS_ERROR(sr))
        return 1;
    
    rwlock_rlock(&sr->rwl);

    for(i = 0, t = 0; i < MAX_TRACKS; i++) {
        if((1 << i) & channel_map) {
            mixer_demux(fb_target,
                        fb_source,
                        t++,
                        channel_count,
                        sr->frame_width,
                        frame_count);

            if(track_frame_buffer_put(sr->tracks[i],
                                      fb_target,
                                      frame_offset,
                                      frame_count)) {
                FAIL("error trying to put %ld frames into track %d at %ld\n",
                     frame_count, i, frame_offset);
            }

            /* This will recalculate the possibly
               changed frame count. (also messy) */
            
            snd_frame_count(sr);
        }
    }
    rwlock_runlock(&sr->rwl);

    return 0;
}

/* Multiplexes and mixes the channels specified by channel_map and the
 * array fb_sources into the buffer given by fb_target. Returns the
 * number of frames actually mixed. This may be shorter than the
 * requested number of frames, because tracks may have differing
 * lengths. Used to write to file or device. 
 */

AFframecount
snd_mux(snd *sr,
        _frame_bits fb_target,
        _frame_bits *fb_sources,
        mixer *mixer,
        int channel_map,
        AFframecount frame_offset,
        AFframecount frame_count) {
    AFframecount high = 0, r;
    int i; //, source_channels = 0;

    if(RESULT_IS_ERROR(sr))
        return 0;

    rwlock_rlock(&sr->rwl);

    for(i = 0; i < sr->channels; i++) {
        memset(fb_sources[i], '\0', frame_count * sr->frame_width);
        if((1 << i) & channel_map) {

            /* FIXME: have to decide whether channel_map determines
               which channels to mux (i.e. all other channels become
               zero), or whether channel_map also determines which
               target tracks they appear on (i.e. all other channels
               disappear). This makes a difference for output to file
               or to an audio device. */

            r = track_frame_buffer_get(sr->tracks[i],
                                       fb_sources[i], //fb_sources[source_channels++],
                                       frame_offset,
                                       frame_count);
            if(r > high)
                high = r;
        }
    }
    /*
    mixer_configure(mixer,
                    mixer->target_channels,
                    source_channels);
    */
    mixer_mux(mixer,
              fb_target,
              fb_sources,
              sr->frame_width,
              high);
    rwlock_runlock(&sr->rwl);
    
    return high;
}

void 
snd_error_free(snd *s) {
    
    if(s && s->error_str && s->error_str != nomem_error_str)
        mem_free(s->error_str);
    if(s) s->error_str = NULL;

}

const char *
snd_error_get(snd *s) {
    if(!s)
        return nomem_error_str;

    if(!s->error_str)
        return NULL;

    return s->error_str;
}

void
snd_error_set(snd *s, 
              const char *format,
              ...) {
    va_list ap;

    if(!s) {
        FAIL("s == NULL\n");
        return;
    }

    snd_error_free(s);

    if(format == RESULT_ERROR_NOMEM) {
        s->error_str = nomem_error_str;
        return;
    }

    s->error_str = (char *) mem_alloc(strlen(format) + 4097);

    if(!s->error_str) {
        s->error_str = nomem_error_str;
        return;
    }
    
    va_start(ap, format);
    vsnprintf(s->error_str, strlen(format) + 4096, format, ap);
    va_end(ap);

}

void
snd_dump(snd *sr) {
    INFO("  file %s\n", sr->name);
    INFO("  format  : %d\n", sr->fmt);
    INFO("  endian  : %d\n", sr->endian);
    INFO("  channels: %d\n", sr->channels);
    INFO("  rate    : %f\n", sr->rate);
    INFO("  frame_w : %d\n", sr->frame_width);
    INFO("  frame_ct: %ld\n", snd_frame_count(sr));
    INFO("  time    : %f\n", sr->time);
    snd_tracks_dump(sr);
    //    INFO("tracks:\n");
    //    for(i = 0; i < sr->channels; i++) 
    //        INFO("%d: %p\n", i, sr->tracks[i]);
}

/* Deletes this sound object and releases all resources associated
 * with it. It is harmless to supply a NULL argument.
 */

void 
snd_destroy(snd *sr) {
    int i;

    if(!sr) {

        /* NULL is allowed. */

        DEBUG("sr == NULL\n");
        return;
    }

    DEBUG("destroying '%s'\n", sr->name);

    for(i = 0; i < sr->channels; i++) 
        if(sr->tracks[i])
            track_destroy(sr->tracks[i]);
    
    snd_error_free(sr);

    rwlock_destroy(&sr->rwl);

    if(sr->name)
        mem_free(sr->name);
    mem_free(sr);
}

snd *
snd_new() {
    int i;
    snd *sr = (snd *)mem_calloc(sizeof(snd), 1);

    if(!sr)
        return NULL;
 
    rwlock_init(&sr->rwl);
    sr->fmt = AF_SAMPFMT_TWOSCOMP;
    sr->endian = AF_BYTEORDER_LITTLEENDIAN;
    sr->channels = 0;
    sr->rate = DEFAULT_SAMPLE_RATE;
    sr->frame_width = DEFAULT_SAMPLE_WIDTH;
    sr->time = 0;
    sr->name = NULL;

    for(i = 0; i < MAX_TRACKS; i++)
        sr->tracks[i] = NULL;

    return sr;
}

