/*
 * Copyright (C) 2002 Pascal Haakmat.
 */

#include <math.h>
#include <gnome.h>
#include "shell.h"
#include "gui.h"

GdkColor point_rec_col       = { 0, 0xFFFF, 0x0000, 0x0000 };
GdkColor point_play_col      = { 0, 0x0000, 0xFFFF, 0x0000 };
GdkColor wave_col            = { 0, 0x0000, 0xCCCC, 0x4444 };
GdkColor zero_col            = { 0, 0x0000, 0xCCCC, 0x4444 };
GdkColor wsel_col            = { 0, 0x0000, 0x8888, 0x3333 };
GdkColor sel_col             = { 0, 0x0000, 0x5555, 0x2222 };
GdkColor sel_bg_col          = { 0, 0xE5E5, 0xE5E5, 0xE5E5 };
GdkColor block_col           = { 0, 0x0000, 0x5555, 0x2222 };
GdkColor mark_col            = { 0, 0xFFFF, 0xFFFF, 0x0000 };
GdkColor grid_col            = { 0, 0x5A5A, 0x0000, 0xAFAF };
GdkColor grid_font_col       = { 0, 0xE5E5, 0xE5E5, 0xE5E5 };
GdkColor info_font_col       = { 0, 0xE5E5, 0xE5E5, 0xE5E5 };
GdkColor info_font_rec_col   = { 0, 0xFFFF, 0x0000, 0x0000 };
GdkColor info_font_play_col  = { 0, 0x0000, 0xFFFF, 0x0000 };
GdkColor slope_marker_cols[3] = { { 0, 0x6B6B, 0xF0F0, 0xF9F9 },
                                  { 0, 0x4848, 0x9F9F, 0xA5A5 },
                                  { 0, 0xFFFF, 0xFFFF, 0xFFFF } };
GdkPixmap *stipple = NULL;
GdkPixmap *mixerlevel = NULL;
extern GdkFont *grid_font;
extern GdkFont *info_font;

void
draw_pointer(GdkDrawable *drawable,
             shell *shl,
             GdkGC *gc) {
    int t, x, b;
    AFframecount pos = shl->player.player_pos;
    GdkGCValues gc_vals;

    if(!shl->player.player_running)
        return;

    if(pos < shl->hadjust->value ||
       pos > shl->hadjust->value + 
       (GTK_WIDGET(shl->canvas)->allocation.width * shl->hres))
        return;

    gdk_gc_get_values(gc, &gc_vals);
    if(shl->player.record_mode)
        gdk_gc_set_foreground(gc, &point_rec_col);
    else
        gdk_gc_set_foreground(gc, &point_play_col);

    x = (pos - shl->hadjust->value) / shl->hres;
    for(b = 0, t = shl->vadjust->value; t < shl->sr->channels; t++, b++)
        gdk_draw_line(drawable,
                      gc,
                      x, b * shl->vres,
                      x, (b * shl->vres) + shl->vres);

    gdk_gc_set_foreground(gc, &gc_vals.foreground);
}

void
draw_cursor(GdkDrawable *drawable,
            shell *shl) {
    int t, b, cursor_pixel_pos;
    GdkGC *gc;
    cursor_pixel_pos = (shl->select_start - ceil(shl->hadjust->value)) / shl->hres;
    
    if(!shl->sr)
        return;
    /*
    if(shl->player.player_running)
        return;
    */

    gc = &GTK_WIDGET(shl->canvas)->style->white_gc[GTK_WIDGET_STATE(GTK_WIDGET(shl->canvas))];

    if(shl->select_start != shl->select_end) 
        return;
    
    if(shl->select_start < shl->hadjust->value ||
       shl->select_start > shl->hadjust->value + 
       (GTK_WIDGET(shl->canvas)->allocation.width * shl->hres))
        return;
    
    //    gdk_gc_set_function(gc, GDK_INVERT);
    for(b = 0, t = shl->vadjust->value; t < shl->sr->channels; t++, b++)
        if((1 << t) & shl->select_channel_map)
            gdk_draw_line(drawable,
                          gc,
                          cursor_pixel_pos, b * shl->vres,
                          cursor_pixel_pos, ((b * shl->vres) + shl->vres) - 1);
    //gdk_gc_set_function(gc, GDK_COPY);
}

/*
int 
draw_periodic(shell *shl) {
    if(!shl->sr)
        return TRUE;
    
    if(shl->player.player_running)
        return TRUE;
    
    if(shl->player.player_stopped == 1) {
        shell_redraw(shl);
        shl->player.player_stopped = 0;
    }
    
    draw_cursor(shl);

    return TRUE;
}
*/

void 
draw_loop(GdkDrawable *drawable,
          GdkGC *gc,
          shell *shl,
          AFframecount frame_offset,
          AFframecount frame_count) {
    AFframecount x1, x2, i, t;
    GdkGCValues gc_vals;

    if(shl->loop_start == shl->loop_end)
        return;

    if(shl->loop_start < frame_offset && shl->loop_end < frame_offset)
        return;

    if(shl->loop_start > frame_offset + frame_count && 
       shl->loop_end > frame_offset + frame_count)
        return;

    /* At least one visible. */

    gdk_gc_get_values(gc, &gc_vals);

    x1 = shl->loop_start - frame_offset;
    x2 = shl->loop_end - frame_offset;

    x1 /= shl->hres;
    x2 /= shl->hres;

    if(x1 > frame_count / shl->hres)
        x1 = -1;
    if(x2 > frame_count / shl->hres)
        x2 = -1;
    
    gdk_gc_set_foreground(gc, &mark_col);
    if(!shl->loop)
        gdk_gc_set_line_attributes(gc, 
                                   gc_vals.line_width, 
                                   GDK_LINE_ON_OFF_DASH, 
                                   gc_vals.cap_style, 
                                   gc_vals.join_style);
    for(t = 0, i = shl->vadjust->value; i < shl->sr->channels; i++, t++) { 
        if(x1 >= 0)
            gdk_draw_line(drawable,
                          gc,
                          x1, t * shl->vres,
                          x1, (t + 1) * shl->vres);
        if(x2 >= 0) 
            gdk_draw_line(drawable,
                          gc,
                          (x2 - 1), t * shl->vres,
                          (x2 - 1), (t + 1) * shl->vres);
    }
    
    if(!shl->loop)
        gdk_gc_set_line_attributes(gc, 
                                   gc_vals.line_width, 
                                   gc_vals.line_style,
                                   gc_vals.cap_style, 
                                   gc_vals.join_style);

    gdk_gc_set_foreground(gc, &gc_vals.foreground);
}

void 
draw_selection(GdkDrawable *drawable,
               GdkGC *gc,
               shell *shl,
               AFframecount frame_offset,
               AFframecount frame_count) {
    
    AFframecount x1, x2, i, t;
    //    int u;
    GdkGCValues gc_vals;
    char mask[] = { 0x9 };

    if(shl->select_start == shl->select_end)
        return;

    if(shl->select_start < frame_offset && shl->select_end < frame_offset)
        return;

    if(shl->select_start > frame_offset + frame_count && 
       shl->select_end > frame_offset + frame_count)
        return;

    /* At least one visible. */

    x1 = shl->select_start - frame_offset;
    x2 = shl->select_end - frame_offset;

    x1 = ceil(x1 / shl->hres);
    x2 = ceil(x2 / shl->hres);

    if(x1 < 0)
        x1 = 0;
    if(x2 > frame_count / shl->hres)
        x2 = frame_count / shl->hres;

    if(x2 < 0)
        return;

    gdk_gc_get_values(gc, &gc_vals);

    //    DEBUG("x1: %ld, x2: %ld\n", x1, x2);
    gdk_gc_set_foreground(gc, &sel_col);
    
    if(shl->envelope_enabled) {
        if(!stipple) {
            stipple = gdk_bitmap_create_from_data(NULL,
                                                  mask,
                                                  2,
                                                  2);
        }
        gdk_gc_set_background(gc, &sel_bg_col);
        gdk_gc_set_stipple(gc, stipple);
        gdk_gc_set_fill(gc, GDK_STIPPLED);
    }

    for(t = 0, i = shl->vadjust->value; i < shl->sr->channels; i++, t++)
        if((1 << i) & shl->select_channel_map) 
            gdk_draw_rectangle(drawable,
                               gc,
                               TRUE,
                               x1, 
                               t * shl->vres,
                               MAX(x2 - x1, 1),
                               shl->vres);

    /*4 beat selection
    for(u = 0; u < 4; u++) 
        for(t = 0, i = shl->vadjust->value; i < shl->sr->channels; i++, t++)
            if((1 << i) & shl->select_channel_map) 
                gdk_draw_rectangle(drawable,
                                   gc,
                                   FALSE,
                                   x1 + (u * (MAX(x2 - x1, 1) / 4)),  
                                   t * shl->vres,
                                   MAX(x2 - x1, 1) / 4,
                                   shl->vres);
    */
    gdk_gc_set_fill(gc, gc_vals.fill);
    gdk_gc_set_foreground(gc, &gc_vals.foreground);
    gdk_gc_set_background(gc, &gc_vals.background);

}

void 
draw_zero(GdkDrawable *drawable,
          GdkGC *gc,
          shell *shl,
          unsigned int width) {
    unsigned int c, d;
    GdkGCValues gc_vals;

    gdk_gc_get_values(gc, &gc_vals);
    
    /* Draw zero line. */
    
    gdk_gc_set_foreground(gc, &zero_col);
    
    for(d = 0, c = shl->vadjust->value; c <= shl->sr->channels; c++, d++) 
        gdk_draw_line(drawable,
                      gc,
                      0, (shl->vres * d) - (shl->vres / 2),
                      width - 1, (shl->vres * d) - (shl->vres / 2));
    
    gdk_gc_set_foreground(gc, &gc_vals.foreground);
}

void
draw_sample(GdkDrawable *drawable,
            GdkGC *gc,
            shell *shl, 
            AFframecount frame_offset,
            AFframecount frame_count) {
    unsigned int yo, t, i, x, mx, fact;
    int sel_first_visible = 0, sel_last_visible = 0;
    AFframecount r;
    GdkGCValues gc_vals;

    if(!shl->graph_bits_buffer)
        return;

    gdk_gc_get_values(gc, &gc_vals);

    if(frame_count <= 1) {
        FAIL("frame_count: %ld, frame_offset: %ld\n", frame_count, frame_offset);
        return;
    }

    if(shl->select_start != shl->select_end) {
        sel_first_visible = MAX(0, (shl->select_start - frame_offset) / shl->hres);
        sel_last_visible = ((shl->select_start - frame_offset) / shl->hres) + 
            ((shl->select_end - shl->select_start) / shl->hres);
    }

    for(t = 0, i = shl->vadjust->value; i < shl->sr->channels; i++, t++) {
        r = track_graph_buffer_get(shl->sr->tracks[i], 
                                   shl->graph_bits_buffer,
                                   frame_offset, 
                                   frame_count, 
                                   shl->hres);
        if(RESULT_IS_ERROR(shl->sr)) {
            FAIL("track_graph_buffer_get returned error: %s\n", 
                 shl->sr->error_str);
            snd_error_free(shl->sr);
            return;
        }

        gdk_gc_set_foreground(gc, &wave_col);

        /* Draw track. */
        
        yo = t * shl->vres + (shl->vres / 2);
        mx = r / shl->hres;
        fact = GRAPH_BITS_VRES / shl->vres;
        for(x = 0; x < mx; x++) {
            if(((1 << i) & shl->select_channel_map) && 
               sel_first_visible != sel_last_visible &&
               x >= sel_first_visible && x < sel_last_visible && 
               sel_last_visible > 0) 
                gdk_gc_set_foreground(gc, &wsel_col);
            gdk_draw_line(drawable, 
                          gc, 
                          x, (shl->graph_bits_buffer[x].l / fact) + yo,
                          x, (shl->graph_bits_buffer[x].h / fact) + yo);
            gdk_gc_set_foreground(gc, &wave_col);
        }
    }
    gdk_gc_set_foreground(gc, &gc_vals.foreground);
}

void
draw_blocks(GdkDrawable *drawable,
            GdkGC *gc,
            shell *shl, 
            AFframecount frame_offset,
            AFframecount frame_count) {
    unsigned int yo, i, t = 0;
    AFframecount cur;
    GdkGCValues gc_vals;
    GList *l;
    
    gdk_gc_get_values(gc, &gc_vals);
    
    if(frame_count <= 1) {
        FAIL("frame_count: %ld, frame_offset: %ld\n", 
             frame_count, frame_offset);
        return;
    }
    
    /* Draw the internal block representation of the sample. */
    
    gdk_gc_set_foreground(gc, &block_col);
    for(i = shl->vadjust->value; i < shl->sr->channels; i++) {

        l = shl->sr->tracks[i]->bl->l;

        if(!l)
            continue;

        cur = ((block *)l->data)->frame_count;
        yo = t++ * shl->vres;

        while(l && cur < frame_offset + frame_count) {
            if(cur >= frame_offset) 
                gdk_draw_line(drawable,
                              gc,
                              (cur - frame_offset) / shl->hres, 
                              yo,
                              (cur - frame_offset) / shl->hres, 
                              yo + shl->vres);

            l = l->next;
            if(l)
                cur += ((block *)l->data)->frame_count;
        }
    }
    gdk_gc_set_foreground(gc, &gc_vals.foreground);
}

void
draw_waveform(GdkDrawable *drawable,
              GtkWidget *widget,
              GdkRectangle *area,
              shell *shl) {    
    int o = shl->hres > 1 ? ((int) shl->hadjust->value % (int) shl->hres) : 0;
    draw_sample(drawable,
                widget->style->fg_gc[widget->state],
                shl,
                (int) shl->hadjust->value - o,
                (widget->allocation.width * shl->hres));
    //    shl->prev_hadjust_value = shl->hadjust->value;
}

void
draw_envelope(GdkDrawable *drawable,
              GdkGC *gc,
              shell *shl,
              enum marker_type type,
              AFframecount frame_offset,
              AFframecount frame_count) {
    int i;
    struct marker *m;
    int x, y, px = -1, py = -1, has_prev, lstyle;
    AFframecount frame_offset_old = 0;
    GdkGCValues gc_vals;
    gdk_gc_get_values(gc, &gc_vals);
    lstyle = gc_vals.line_style;

    if(!shl->envelope_enabled)
        lstyle = GDK_LINE_ON_OFF_DASH;
    
    gdk_gc_set_foreground(gc, &slope_marker_cols[type - 1]);
    gdk_gc_set_line_attributes(gc, 
                               2,
                               lstyle,
                               gc_vals.cap_style, 
                               gc_vals.join_style);
    gdk_gc_set_function(gc, GDK_OR);

    for(i = shl->vadjust->value; i < shl->sr->channels; i++) {
        has_prev = 0;
        frame_offset = frame_offset_old;
        m = marker_list_previous(shl->markers[i], 
                                 frame_offset,
                                 type);

        if(m) {
            has_prev = 1;
            px = ((m->frame_offset / shl->hres)) -
                (shl->hadjust->value / shl->hres);
            py = ((shl->vres * i) + (shl->vres / 2) - 
                  m->multiplier * (shl->vres / 2));
            gdk_draw_rectangle(drawable,
                               gc,
                               FALSE,
                               px - (MARKER_HANDLE_SIZE / 2), 
                               py - (MARKER_HANDLE_SIZE / 2),
                               MARKER_HANDLE_SIZE,
                               MARKER_HANDLE_SIZE);
            frame_offset = m->frame_offset + 1;
        }

        m = marker_list_next(shl->markers[i], 
                             frame_offset,
                             type);
        
        for(; m; m = marker_list_next(shl->markers[i], 
                                      frame_offset,
                                      type)) {
            x = ((m->frame_offset / shl->hres)) - 
                (shl->hadjust->value / shl->hres);
            y = ((shl->vres * (i - shl->vadjust->value)) + (shl->vres / 2) - 
                 m->multiplier * (shl->vres / 2));
            gdk_draw_rectangle(drawable,
                               gc,
                               FALSE,
                               x - (MARKER_HANDLE_SIZE / 2), 
                               y - (MARKER_HANDLE_SIZE / 2),
                               MARKER_HANDLE_SIZE, 
                               MARKER_HANDLE_SIZE);
            if(has_prev)
                gdk_draw_line(drawable,
                              gc,
                              x, 
                              y,
                              px, 
                              py);
            has_prev = 1;
            px = x;
            py = y;

            frame_offset = m->frame_offset + 1;
            if(m->frame_offset > frame_offset + frame_count)
                break;
        }
    }
    gdk_gc_set_function(gc, gc_vals.function);
    gdk_gc_set_line_attributes(gc, 
                               gc_vals.line_width,
                               gc_vals.line_style,
                               gc_vals.cap_style, 
                               gc_vals.join_style);
    gdk_gc_set_foreground(gc, &gc_vals.foreground);
}

void
draw_grid(GdkDrawable *drawable,
          GdkGC *gc,
          shell *shl,
          AFframecount frame_offset,
          AFframecount frame_count) {
    unsigned int xbase, yo, i, t = 0, ascent, descent, width, dummy;
    char s[20];
    AFframecount cur;
    GdkGCValues gc_vals;
    GdkPoint handle[4];

    gdk_gc_get_values(gc, &gc_vals);
    
    if(frame_count <= 1) {
        FAIL("frame_count: %ld, frame_offset: %ld\n", 
             frame_count, frame_offset);
        return;
    }
    
    /* FIXME: reverse loop order for efficiency. */

    gdk_gc_set_foreground(gc, &grid_col);
    for(i = shl->vadjust->value; i < shl->sr->channels; i++) {

        yo = t++ * shl->vres;
        
        for(cur = frame_offset - (frame_offset % shl->grid.gap);
            cur < frame_offset + frame_count && cur < snd_frame_count(shl->sr); 
            cur += MAX(shl->hres, shl->grid.gap)) {
            xbase = ((cur - frame_offset) / shl->hres);
            gdk_gc_set_foreground(gc, &grid_col);
            gdk_draw_line(drawable,
                          gc,
                          xbase, 
                          yo,
                          xbase, 
                          yo + shl->vres);
            if(i != shl->vadjust->value) 
                continue;
            
            grid_format(&shl->grid,
                        cur,
                        s,
                        20);
            gdk_string_extents(grid_font,
                               s,
                               &dummy, // lbearing
                               &dummy, // rbearing
                               &width,
                               &ascent,
                               &descent);
            handle[0].x = xbase;
            handle[0].y = yo;
            handle[1].x = xbase + width + ascent + descent;
            handle[1].y = yo;
            handle[2].x = xbase + width;
            handle[2].y = yo + ascent + descent;
            handle[3].x = xbase;
            handle[3].y = yo + ascent + descent;
            gdk_draw_polygon(drawable,
                             gc,
                             1,
                             handle,
                             4);
            gdk_gc_set_foreground(gc, &grid_font_col);
            gdk_draw_string(drawable,
                            grid_font,
                            gc,
                            xbase + 2,
                            yo + ascent,
                            s);
        }
    }
    gdk_gc_set_foreground(gc, &gc_vals.foreground);
}

void 
draw_obscured(GdkDrawable *drawable,
              GtkWidget *widget,
              GdkRectangle *area,
              shell *shl) {
    gdk_gc_set_clip_rectangle(widget->style->fg_gc[widget->state],
                              area);

    gdk_draw_rectangle(drawable,
                       widget->style->black_gc,
                       TRUE,
                       0, 0,
                       widget->allocation.width,
                       shl->vres * (shl->sr->channels - 
                                    (int) shl->vadjust->value));
    draw_selection(drawable,
                   widget->style->fg_gc[widget->state],
                   shl,
                   shl->hadjust->value,
                   widget->allocation.width * shl->hres);
                       
    gdk_draw_line(drawable,
                  widget->style->black_gc,
                  0, shl->vres * (shl->sr->channels - 
                                  shl->vadjust->value) + 1,
                  widget->allocation.width, shl->vres * 
                  (shl->sr->channels - shl->vadjust->value) + 1);

    if(shl->draw_zero)
        draw_zero(drawable,
                  widget->style->mid_gc[widget->state],
                  shl,
                  (widget->allocation.width < 
                   snd_frame_count(shl->sr) / shl->hres) ? 
                  widget->allocation.width : 
                  snd_frame_count(shl->sr) / shl->hres);
        
    if(shl->show_grid)
        draw_grid(drawable,
                  widget->style->fg_gc[widget->state],
                  shl,
                  (int) shl->hadjust->value,
                  widget->allocation.width * shl->hres);
    
    gdk_gc_set_clip_rectangle(widget->style->fg_gc[widget->state],
                              NULL);
    
}

void 
draw_exposed(GdkDrawable *drawable,
             GtkWidget *widget,
             GdkRectangle *area,
             shell *shl) {
    int i;

    draw_loop(drawable,
              widget->style->fg_gc[widget->state],
              shl,
              shl->hadjust->value,
              widget->allocation.width * shl->hres);
    
    if(shl->debug_flags & DEBUG_FLAG_DRAW_BLOCKS)
        draw_blocks(drawable,
                    widget->style->fg_gc[widget->state],
                    shl,
                    /* FIXME: broken for shl->hres < 0. */
                    (int) shl->hadjust->value - 
                    ((int) shl->hadjust->value % (int) shl->hres),
                    widget->allocation.width * shl->hres);
    
    if(shl->draw_envelope || shl->envelope_enabled) {
        
        for(i = 0; i < MAX_TRACKS; i++)
            shl->markers[i]->marker_types_enabled = 
                MARKER_SLOPE | MARKER_SLOPE_AUX;

        draw_envelope(drawable,
                      widget->style->fg_gc[widget->state],
                      shl,
                      MARKER_SLOPE,
                      shl->hadjust->value,
                      widget->allocation.width * shl->hres);
        draw_envelope(drawable,
                      widget->style->fg_gc[widget->state],
                      shl,
                      MARKER_SLOPE_AUX,
                      shl->hadjust->value,
                      widget->allocation.width * shl->hres);

        if(!shl->envelope_enabled)
            for(i = 0; i < MAX_TRACKS; i++)
                shl->markers[i]->marker_types_enabled = 0;
    }
    
    draw_pointer(drawable,
                 shl,
                 widget->style->fg_gc[widget->state]);
}

void
draw_mixer_canvas(GtkWidget *widget,
                  GdkRectangle *area,
                  shell *shl) {
    int i, j, y = 0, y2 = 0;
    char s[8];
    GdkGC *gc = widget->style->fg_gc[widget->state];
    GdkGCValues gc_vals;
    GdkDrawable *drawable = shl->mixerpixmap;

    if(!shl->sr)
        return;

    gdk_gc_get_values(gc, &gc_vals);
    
    gdk_draw_rectangle(drawable,
                       widget->style->bg_gc[widget->state],
                       TRUE,
                       0, 0,
                       widget->allocation.width,
                       widget->allocation.height);
    gdk_draw_rectangle(drawable,
                       gc,
                       TRUE,
                       0, 0,
                       widget->allocation.width,
                       (shl->sr->channels - shl->vadjust->value) * shl->vres);
    
    gdk_gc_set_foreground(gc, &info_font_col);
    y += 2;
    for(i = shl->vadjust->value; i < shl->sr->channels; i++) {
        for(j = 0; j < shl->mixer->target_channels; j++) {
            y2 = y + j * MIXER_LEVEL_HEIGHT + j;
            snprintf(s, 128, "%d", j + 1);
            gdk_draw_string(drawable,
                            info_font,
                            gc,
                            1,
                            y2 + MIXER_LEVEL_HEIGHT,
                            s);
            if(!mixerlevel) 
                mixerlevel = gdk_pixmap_create_from_xpm(widget->window,
                                                        NULL,
                                                        NULL,
                                                        MIXER_LEVEL_FILE);
            gdk_draw_pixmap(drawable,
                            gc,
                            mixerlevel,
                            0, 0,
                            8, y2,
                            MIXER_LEVEL_WIDTH * shl->mixer->mixtable[j][i],
                            MIXER_LEVEL_HEIGHT);
        }
        y += shl->vres;
    }

    gdk_gc_set_foreground(gc, &gc_vals.foreground);

    gdk_draw_pixmap(widget->window,
                    widget->style->fg_gc[GTK_WIDGET_STATE(widget)],
                    shl->mixerpixmap,
                    area->x, area->y,
                    area->x, area->y,
                    area->width, area->height);
}

int 
draw_info_line(GtkWidget *widget,
               GdkGC *gc,
               char *s, 
               int x,
               int y,
               int align) {
    int dummy, width, ascent, descent;
    gdk_string_extents(info_font,
                       s,
                       &dummy, // lbearing
                       &dummy, // rbearing
                       &width,
                       &ascent,
                       &descent);
    gdk_draw_string(widget->window,
                    info_font,
                    gc,
                    align ? ((widget->allocation.width - width) - x) : x,
                    y + ascent + descent,
                    s);
    return y + ascent + descent;
}

void
draw_info_canvas(GtkWidget *widget,
                 GdkRectangle *area,
                 shell *shl) {
    int ycenter = 0, y = 0, width, dummy, ascent, descent;
    char s[40], f[20];
    GdkGC *gc = widget->style->fg_gc[widget->state];
    GdkGCValues gc_vals;

    if(!shl->sr)
        return;

    gdk_gc_get_values(gc, &gc_vals);

    snprintf(s, 40, "%s, %d bits", shl->sr->name, shl->sr->frame_width * 8);
    gdk_string_extents(info_font,
                       s,
                       &dummy, // lbearing
                       &dummy, // rbearing
                       &width,
                       &ascent,
                       &descent);
    ycenter = (widget->allocation.height / 2) - ((ascent + descent) + 2);
    gdk_draw_rectangle(widget->window,
                       gc,
                       TRUE,
                       0, ycenter,
                       widget->allocation.width,
                       ycenter + ((ascent + descent) * 2) + 2);

    gdk_gc_set_foreground(gc, &info_font_col);
    if(shl->player.player_running) {
        if(shl->player.record_mode) 
            gdk_gc_set_foreground(gc, &info_font_rec_col);
        else
            gdk_gc_set_foreground(gc, &info_font_play_col);
    }
    snprintf(s, 40, "%d bits", shl->sr->frame_width * 8);
    y = draw_info_line(widget, gc, s, 1, ycenter + 1, 0);
    snprintf(s, 40, "%.2fhz", shl->sr->rate);
    y = draw_info_line(widget, gc, s, 1, ycenter + y, 0);
    
    grid_format(&shl->grid, shl->select_start, f, 20);
    snprintf(s, 40, "SS:%10s", f);
    y = draw_info_line(widget, gc, s, 1, ycenter + 0, 1);
    grid_format(&shl->grid, shl->select_end, f, 20);
    snprintf(s, 40, "SE:%10s", f);
    y = draw_info_line(widget, gc, s, 1, ycenter + y, 1);

    gdk_string_extents(info_font,
                       s,
                       &dummy, // lbearing
                       &dummy, // rbearing
                       &width,
                       &ascent,
                       &descent);

    grid_format(&shl->grid, shl->loop_start, f, 20);
    snprintf(s, 40, "LS:%10s  ", f);
    y = draw_info_line(widget, gc, s, width, ycenter + 0, 1);
    grid_format(&shl->grid, shl->loop_end, f, 20);
    snprintf(s, 40, "LE:%10s  ", f);
    y = draw_info_line(widget, gc, s, width, ycenter + y, 1);

    gdk_string_extents(info_font,
                       s,
                       &dummy, // lbearing
                       &dummy, // rbearing
                       &width,
                       &ascent,
                       &descent);

    grid_format(&shl->grid, (shl->last_mouse_x * shl->hres) + shl->hadjust->value, f, 20);
    snprintf(s, 40, "MX:%10s  ", f);
    y = draw_info_line(widget, gc, s, width * 2, ycenter + 0, 1);
    if(!shl->player.player_running) {
        snprintf(s, 40, "MY:% 10.5f  ", shl->last_mouse_y);
        y = draw_info_line(widget, gc, s, width * 2, ycenter + y, 1);
    }
    if(shl->player.player_running) {
        grid_format(&shl->grid, shl->player.player_pos, f, 20);
        snprintf(s, 40, "%s:%10s  ", (shl->player.record_mode ? "R" : "P"), f);
        y = draw_info_line(widget, gc, s, width * 2, ycenter + y, 1);
    }

    gdk_gc_set_foreground(gc, &gc_vals.foreground);
}

void
draw_canvas(GtkWidget *widget,
            GdkRectangle *area,
            shell *shl) {
    
    /* All error handling goes through here. Errors are flagged on the
       sound object, polled and displayed, then freed. */
    
    if(shl->sr) {
        if(RESULT_IS_ERROR(shl->sr)) {
            gui_alert("Error: %s", snd_error_get(shl->sr));
            snd_error_free(shl->sr);
        }
    }

    /* Draw unto pixmap. */

    if(shl->pixmap) {
        gdk_draw_rectangle(shl->pixmap,
                           widget->style->bg_gc[widget->state],
                           TRUE,
                           0, 0,
                           widget->allocation.width,
                           widget->allocation.height);
        if(shl->sr) {
            rwlock_rlock(&shl->sr->rwl);
            draw_obscured(shl->pixmap,
                          widget,
                          area,
                          shl);
            draw_waveform(shl->pixmap, 
                          widget,
                          area,
                          shl);
            draw_exposed(shl->pixmap,
                         widget,
                         area,
                         shl);
            draw_cursor(shl->pixmap,
                        shl);
            rwlock_runlock(&shl->sr->rwl);
        }
    }
    
    /* Show the pixmap. */
    
    if(shl->pixmap) 
        gdk_draw_pixmap(widget->window,
                        widget->style->fg_gc[GTK_WIDGET_STATE(widget)],
                        shl->pixmap,
                        area->x, area->y,
                        area->x, area->y,
                        area->width, area->height);

    gtk_widget_queue_draw(GTK_WIDGET(shl->infocanvas));
}
