/*
 * Copyright (C) 2002 Pascal Haakmat.
 */

#include <sys/time.h>
#include <errno.h>
#include <audiofile.h>
#include "mem.h"
#include "shell.h"
#include "gui.h"
#include "file.h"

extern int errno;

static char untitled_name[] = "Untitled\0        ";
static int untitled_counter = 1;

int
file_load(shell *shl,
          const char *filename) {
    int fail, i, input_map;
    size_t fb_sz, fb_dmx_sz;
    _frame_bits fb = NULL, fb_dmx = NULL;
    AFfilehandle fh = AF_NULL_FILEHANDLE;
    AFframecount frame_offset = 0, 
        frame_count = 0,
        total = 0, 
        r;
    long start;
    float perc;
    struct timeval tv;
    snd *sr;
 
    sr = (snd *)snd_new();
    if(RESULT_IS_ERROR(sr)) {
        gui_alert("Out of memory trying to create sound object: %s.", 
                  snd_error_get(sr));
        snd_destroy(sr);
        return -1;
    }

    snd_name_set(sr, filename);

    /* Open file. */

    fh = afOpenFile(sr->name, "r", AF_NULL_FILESETUP);
    if(fh == AF_NULL_FILEHANDLE) {
        gui_alert("Error loading %s: %s. \n" \
                  "Perhaps this is not an audio file or the \n" \
                  "format is not supported.", 
                  sr->name, strerror(errno));
        goto failed;
    }
    
    afGetSampleFormat(fh, 
                      AF_DEFAULT_TRACK, 
                      &sr->fmt, 
                      &sr->frame_width);

    sr->frame_width = sr->frame_width + (sr->frame_width % 8);
    if(sr->frame_width == 24)
        sr->frame_width = 32;
    sr->frame_width /= 8;
    sr->endian = afGetByteOrder(fh, AF_DEFAULT_TRACK);
    sr->channels = afGetChannels(fh, AF_DEFAULT_TRACK);
    sr->rate = afGetRate(fh, AF_DEFAULT_TRACK);
    grid_rate_set(&shl->grid, sr->rate);
    //    sr->frame_count = afGetFrameCount(fh, AF_DEFAULT_TRACK);
    //    sr->time = snd_frames_to_time(sr, );

    if(sr->channels < 1 || sr->channels > MAX_TRACKS) {
        gui_alert("Sorry, %d channel sound format not supported (%s).",
                  sr->channels, sr->name);
        goto failed;
    }
    
    if(sr->frame_width < 1 || sr->frame_width > 4) {
        gui_alert("Sorry, %d bits sample format not supported (%s).",
                  sr->frame_width * 8, sr->name);
        goto failed;
    }

    input_map = 0;
    for(i = 0; i < sr->channels; i++)
        input_map += (1 << i);

    shl->select_channel_map = input_map;
    shl->sr = sr;

    mixer_configure(shl->mixer, 
                    OUTPUT_CHANNELS, //MIN(OUTPUT_CHANNELS, shl->sr->channels), 
                    shl->sr->channels);

    /* Reserve & setup memory. */

    fail = 0;
    for(i = 0; i < sr->channels; i++)
        sr->tracks[i] = NULL;
    for(i = 0; i < sr->channels; i++) {
        sr->tracks[i] = track_new(sr->frame_width, 0);
        if(sr->tracks[i] == NULL) {
            fail = 1;
            for(i--; i >= 0; i--)
                track_destroy(sr->tracks[i]);
            break;
        }
    }

    if(fail) {
        gui_alert("Out of memory allocating %d tracks.", 
                  sr->channels);
        goto failed;
    }

    fb_sz = BLOCK_SIZE * sr->frame_width * sr->channels;
    fb_dmx_sz = BLOCK_SIZE * sr->frame_width;

    fb = mem_alloc(fb_sz);
    if(!fb) {
        gui_alert("Out of memory (need %d bytes).", fb_sz);
        goto failed;
    }
    fb_dmx = mem_alloc(fb_dmx_sz);
    if(!fb_dmx) {
        gui_alert("Out of memory (need %d bytes).", fb_dmx_sz);
        goto failed;
    }

    /* Read audio file.  

       afSetVirtualSampleFormat() causes libaudiofile to return signed
       twos-complement samples regardless of the file's real sample
       format. */

    afSetVirtualSampleFormat(fh,
                             AF_DEFAULT_TRACK,
                             AF_SAMPFMT_TWOSCOMP,
                             sr->frame_width * 8);

    r = afSeekFrame(fh, AF_DEFAULT_TRACK, frame_offset);
    if(r != frame_offset) {
        gui_alert("Seek error in %s: requested: %ld, returned: %ld.", 
                  sr->name, frame_offset, r);
        goto failed;
    }

    gettimeofday(&tv, NULL);
    start = tv.tv_sec;

    frame_count = afGetFrameCount(fh, AF_DEFAULT_TRACK);//sr->frame_count;

    rwlock_rlock(&sr->rwl);

    for(frame_offset = 0; 
        frame_offset < frame_count; 
        frame_offset += LOAD_BUF_SIZE) {

        r = afReadFrames(fh, 
                         AF_DEFAULT_TRACK,
                         fb,
                         MIN(frame_count - total, LOAD_BUF_SIZE));
        if(r <= 0)
            break;

        snd_demux(sr,
                  fb_dmx,
                  fb,
                  input_map,
                  sr->channels,
                  frame_offset,
                  r);
        
        if(RESULT_IS_ERROR(sr)) {
            gui_alert("Error loading %s: %s", 
                      sr->name,
                      snd_error_get(sr));
            break;
        }
        
        /* Progress meter behaves strange for values around
           0.01. */
        
        perc = (float) ((float) total / 
                        (float) (frame_count));
        if(perc > 0.02)
            gtk_progress_set_percentage(shl->progress, perc);
        
        shell_redraw(shl);

        /* Process GUI events, exit if that kills us. */
        
        while(gtk_events_pending()) 
            if(gtk_main_iteration_do(FALSE)) 
                goto failed;

        total += r;
    }

    rwlock_runlock(&sr->rwl);

    DEBUG("loaded %ld frames, expected %ld frames, calculated %ld frames\n",
          total, frame_count, snd_frame_count(sr));

    if(total != frame_count) 
        gui_alert("File might be damaged, expected %ld frames but read only %ld.\n",
                  frame_count, total);

    //    sr->frame_count = total;

    gettimeofday(&tv, NULL);
    INFO("read %ld frames (%ld bytes) in %ld secs (%ld bytes/sec)\n", 
         snd_frame_count(sr), 
         snd_frame_count(sr) * sr->frame_width * sr->channels, 
         tv.tv_sec - start,
         (snd_frame_count(sr) * sr->frame_width * sr->channels / 
          ((tv.tv_sec - start < 1 ? 1 : tv.tv_sec - start))));

    mem_free(fb);
    mem_free(fb_dmx);
    afCloseFile(fh);
    return 0;
    
 failed:
    shl->sr = sr;
    if(fb)
        mem_free(fb);
    if(fb_dmx)
        mem_free(fb_dmx);
    if(fh != AF_NULL_FILEHANDLE)
        afCloseFile(fh);
    return -1;
}

/* FIXME: clean this up. */

int
file_store(shell *shl) {
    AFfilesetup file_setup;
    AFfilehandle fh = AF_NULL_FILEHANDLE;
    AFframecount off, ct, r, total_written = 0;
    int written = 0, i;
    _frame_bits fb_muxbuf, fb_srcbufs[MAX_TRACKS];
    mixer *output_mixer;
    snd *sr = shl->sr;
    int tracks = sr->channels, channel_map = 0;
    struct timeval tv;
    long start;
    float perc;

    if(!sr->name) 
        abort();

    /* Setup local stuff. */

    file_setup = afNewFileSetup();

    if(!file_setup) {
        gui_alert("Not enough memory to create AFfilesetup. " \
                  "Very low memory.");
        return -1;
    }
    
    fb_muxbuf = mixer_buffers_alloc(sr->frame_width,
                                    tracks,
                                    &fb_muxbuf,
                                    fb_srcbufs,
                                    BLOCK_SIZE);

    if(!fb_muxbuf) {
        gui_alert("Cannot get temporary memory for output buffer.");
        afFreeFileSetup(file_setup);
        return -1;
    }

    output_mixer = mixer_new(0, 0);

    if(!output_mixer) {
        gui_alert("Cannot create output mixer for %d tracks. " \
                  "Very low memory.", tracks);
        goto recover;
    }

    mixer_configure(output_mixer, tracks, tracks);

    DEBUG("whitespace...\n\n\n\n");
    mixer_dump(output_mixer);

    /* Setup the output file. */

    afInitFileFormat(file_setup,
                     AF_FILE_WAVE);
    afInitSampleFormat(file_setup,
                       AF_DEFAULT_TRACK,
                       sr->fmt,
                       sr->frame_width * 8);
    afInitByteOrder(file_setup,
                    AF_DEFAULT_TRACK,
                    sr->endian);
    afInitChannels(file_setup,
                   AF_DEFAULT_TRACK,
                   tracks);
    afInitRate(file_setup,
               AF_DEFAULT_TRACK,
               sr->rate);
    fh = afOpenFile(sr->name, "w", file_setup);

    if(fh == AF_NULL_FILEHANDLE) {
        gui_alert("Could not open %s for writing. Check filename " \
                  "and permissions.", sr->name);
        goto recover;
    }

    afSetVirtualSampleFormat(fh,
                             AF_DEFAULT_TRACK,
                             AF_SAMPFMT_TWOSCOMP,
                             sr->frame_width * 8);
    
    for(i = 0; i < sr->channels; i++) 
        if(track_frame_count(sr->tracks[i])) 
            channel_map |= (1 << i);

    gettimeofday(&tv, NULL);
    start = tv.tv_sec;

    DEBUG("starting mux-to-disk: name: %s, tracks: %d, channel_map: %d, frame_count: %ld\n",
          sr->name, tracks, channel_map, snd_frame_count(sr));

    gtk_progress_set_percentage(shl->progress, 0);
    ct = snd_frame_count(shl->sr);
    off = 0;
    r = 1;
    while(ct && r) {
        while(gtk_events_pending())
            gtk_main_iteration();

        memset(fb_muxbuf, 
               '\0', 
               SAVE_BUF_SIZE * sr->frame_width * output_mixer->target_channels);
        r = snd_mux(sr,
                    fb_muxbuf,
                    fb_srcbufs, 
                    output_mixer, 
                    channel_map, 
                    off, 
                    SAVE_BUF_SIZE);
        written = afWriteFrames(fh, 
                                AF_DEFAULT_TRACK,
                                fb_muxbuf, 
                                r);

        if(written == -1) 
            break;
        
        total_written += written;
        ct -= r;
        off += r;

        /* Progress meter behaves strange for values around
           0.01. */
        
        perc = (float) ((float) off / 
                        (float) (snd_frame_count(shl->sr)));
        if(perc > 0.02)
            gtk_progress_set_percentage(shl->progress, perc);
    }
    gtk_progress_set_percentage(shl->progress, 0);
    
    if(ct || total_written != snd_frame_count(shl->sr)) 
        gui_alert("Save %s: not all frames were written (%ld of %ld). " \
                  "Some data is not on disk. Check disk space (%s).",
                  shl->sr->name, 
                  total_written,
                  snd_frame_count(shl->sr),
                  written == -1 ? strerror(errno) : "Unknown error");

    if(afCloseFile(fh)) 
        gui_alert("An error occurred trying to close %s. Some data " \
                  "may not have been written to disk. Check disk space (%s).",
                  shl->sr->name,
                  strerror(errno));

    gettimeofday(&tv, NULL);
    INFO("wrote %ld frames (%ld bytes) in %ld secs (%ld bytes/sec)\n",
         total_written,
         total_written * sr->frame_width * tracks,
         tv.tv_sec - start,
         (total_written * sr->frame_width * tracks /
          ((tv.tv_sec - start < 1 ? 1 : tv.tv_sec - start))));

    if(output_mixer)
        mixer_destroy(output_mixer);

    mixer_buffers_free(tracks, fb_muxbuf, fb_srcbufs);
    afFreeFileSetup(file_setup);
    return 0;

 recover:
    if(output_mixer)
        mixer_destroy(output_mixer);

    mixer_buffers_free(tracks, fb_muxbuf, fb_srcbufs);
    afFreeFileSetup(file_setup);
    return -1;
}

shell *
file_open(const char *filename) {
    shell *shl;

    /* Something killed us so bail. */

    if(gtk_main_iteration_do(FALSE))
        return NULL;

    shl = shell_new();
    if(!shl) {
        gui_alert("Could not create interface, you are probably out of memory.");
        return NULL;
    }
    gtk_widget_show(GTK_WIDGET(shl->appwindow));

    shell_status_push(shl, "Loading %s ...", filename);
    shell_cursor_set(shl, GDK_WATCH);
    gtk_progress_set_percentage(shl->progress, 0);
    gtk_widget_set_sensitive(GTK_WIDGET(shell_bindings_find(shl, "close")), 0);

    if(file_load(shl, filename)) {

        while(gtk_events_pending()) 
            if(gtk_main_iteration_do(FALSE)) 
                return NULL;

        /* Failed to load file for some reason so initialize to new
           file. */

        snprintf(untitled_name, 12, "Untitled%d", untitled_counter++);
        snd_name_set(shl->sr, untitled_name);
        shl->sr->fmt = AF_SAMPFMT_TWOSCOMP;
        shl->sr->endian = AF_BYTEORDER_LITTLEENDIAN;
        shl->sr->channels = 0;
        shl->sr->rate = DEFAULT_SAMPLE_RATE;
        grid_rate_set(&shl->grid, shl->sr->rate);
        shl->sr->frame_width = DEFAULT_SAMPLE_WIDTH;
        //        shl->sr->frame_count = 0;
        //        shl->sr->time = 0;
        snd_tracks_insert(shl->sr, NULL, 1);
        
    } else {

        shl->has_name = 1;

    }

    /* Configure mixer for the number of tracks that we have. */

    mixer_configure(shl->mixer, 
                    OUTPUT_CHANNELS, //MIN(OUTPUT_CHANNELS, shl->sr->channels), 
                    shl->sr->channels);

    gtk_progress_set_percentage(shl->progress, 0);
    shell_cursor_set(shl, GDK_LEFT_PTR);
    gtk_widget_set_sensitive(GTK_WIDGET(shell_bindings_find(shl, "close")), 1);
    shell_status_default_set(shl);
    shell_redraw(shl);

    /* Something killed us while loading the file so bail. */

    if(gtk_main_iteration_do(FALSE))
        return NULL;

    return shl;
}

void
file_save(shell *shl, 
          gboolean do_rename) {
    char *newname;
    if(!shl->sr->name) {
        FAIL("file has no name\n");
        return;
    }

    if(do_rename) {
        newname = mem_alloc(strlen(shl->sr->name) + 5);
        if(!newname) {
            FAIL("cannot get memory for rename\n");
            return;
        }

        strcpy(newname, shl->sr->name);
        strcat(newname, ".bak");

        if(rename(shl->sr->name, newname)) {
            FAIL("could not rename file for saving as %s\n", newname);
            mem_free(newname);
            return;
        }

        DEBUG("renamed '%s' to '%s'\n", shl->sr->name, newname);
        mem_free(newname);
    }

    shell_status_push(shl, "Saving %s ...", shl->sr->name);
    shell_cursor_set(shl, GDK_WATCH);
    if(file_store(shl)) {
        shl->has_changed = 1;
    } else {
        shl->has_changed = 0;
    }
    shell_cursor_set(shl, GDK_LEFT_PTR);
    shell_status_default_set(shl);
    shell_redraw(shl);
    shl->has_name = 1;

}

int
file_new() {
    snd *sr;
    shell *shl;

    shl = shell_new();
    if(!shl) {
        FAIL("cannot create new shell\n");
        return -1;
    }
    gtk_widget_show(GTK_WIDGET(shl->appwindow));

    sr = (snd *)snd_new();    
    if(RESULT_IS_ERROR(sr)) {
        gui_alert("Error creating new sound object, you are probably out of memory: %s.", 
                  snd_error_get(sr));
        snd_destroy(sr);
        return -1;
    }

    snprintf(untitled_name, 12, "Untitled%d", untitled_counter++);
    snd_name_set(sr, untitled_name);
    snd_tracks_insert(sr, NULL, 1);
    mixer_configure(shl->mixer, OUTPUT_CHANNELS, sr->channels);
    shl->has_name = 0;
    shl->sr = sr;
    grid_rate_set(&shl->grid, sr->rate);
    shell_status_default_set(shl);
    shell_redraw(shl);
    return 0;
}

