/*
 * Copyright (C) 2002 Pascal Haakmat.
 */

#include <stdlib.h>
#include <glib.h>
#include <audiofile.h>
#include "config.h"
#include "marker.h"

#define USE_MCACHE 0

void
marker_dump(struct marker *m) {
    DEBUG("marker: frame_offset: %ld, type: %d, multiplier: %f, label: %s\n",
          m->frame_offset, m->type, m->multiplier, m->label);
}

struct marker *
marker_new(AFframecount frame_offset,
           enum marker_type type, 
           float multiplier,
           char *label) { 
    struct marker *m = malloc(sizeof(struct marker));
    if(!m) {
        FAIL("could not allocate memory for new marker\n");
        return NULL;
    }

    m->frame_offset = frame_offset;
    m->type = type;
    m->label = label;
    m->multiplier = multiplier;
    m->flags = 0;
    return m;
}

void
marker_destroy(struct marker *m) {
    if(m->label)
        free(m->label);
    free(m);
}

double
marker_list_slope_value(struct marker_list *ml,
                        AFframecount frame_offset,
                        enum marker_type type) {
    struct marker *mn = NULL, *mp = NULL;
    double a, r, b, dn, dp, max, min;
    AFframecount x;

    if((ml->marker_types_enabled & type) == 0) 
        return 0;

#if USE_MCACHE
    if(ml->left_cache[type] != ml->right_cache[type] &&
       frame_offset >= ml->left_cache[type] &&
       frame_offset < ml->right_cache[type]) {
        if(ml->last_request[type] + 1 == frame_offset) {
            ml->last_return[type] = ml->last_return[type] + 
                ml->slope[type];
            ml->last_request[type] = frame_offset;
            return ml->last_return[type];
        } else if(ml->last_request[type] - 1 == frame_offset) {
            ml->last_return[type] = ml->last_return[type] - 
                ml->slope[type];
            ml->last_request[type] = frame_offset;
            return ml->last_return[type];
        }
    }
#endif

    mn = marker_list_next(ml,
                          frame_offset,
                          type);
    mp = marker_list_previous(ml,
                              frame_offset,
                              type);
    if(!(mn && mp))
        return 0;

    if(mn->frame_offset - mp->frame_offset <= 0) {
        FAIL("mn->frame_offset <= mp->frame_offset\n");
        return 0;
    }
    
    if(mp->multiplier == mn->multiplier) {
        return mp->multiplier;
    } else if(mp->multiplier > mn->multiplier) {
        max = mp->multiplier;
        min = mn->multiplier;
    } else {
        max = mn->multiplier;
        min = mp->multiplier;
    }
    a = (max - min) / (mn->frame_offset - mp->frame_offset);
    if(mp->multiplier > mn->multiplier) 
        a = -a;
    x = frame_offset - mp->frame_offset;
    b = mp->multiplier;
    
    r = (a * x) + b;

    dp = ABS(mp->multiplier) - ABS(r);
    dn = ABS(mn->multiplier) - ABS(r); 

    if(dp < -0.001 && dn < -0.001) {
        FAIL("rounding error: r: %f, mp->multiplier: %f, mn->multiplier: %f, a: %f, b: %f\n",
             r, mp->multiplier, mn->multiplier, a, b);
        r = CLAMP(r, MIN(mp->multiplier, mn->multiplier), MAX(mp->multiplier, mn->multiplier));
        FAIL("clamped to %f\n", r);
    }

    ml->left_cache[type] = mp->frame_offset;
    ml->right_cache[type] = mn->frame_offset;
    ml->last_request[type] = frame_offset;
    ml->slope[type] = a;

    return r;
}

void
marker_list_marker_delete(struct marker_list *ml,
                          struct marker *m) {
    GList *l;
    ml->left_cache[m->type] = ml->right_cache[m->type] = 0;
    for(l = ml->markers; l; l = l->next) {
        if((struct marker *)(l->data) == m) {
            ml->markers = g_list_remove_link(ml->markers, l);
            marker_destroy(m);
            g_list_free(l);
            break;
        }
    }
}

void
marker_list_marker_position_set(struct marker_list *ml,
                                struct marker *m,
                                AFframecount frame_offset) {
    struct marker *mn = NULL, *mp = NULL;
    mn = marker_list_next(ml,
                          m->frame_offset + 1,
                          m->type);
    mp = marker_list_previous(ml,
                              m->frame_offset,
                              m->type);
    if(mn && frame_offset >= mn->frame_offset)
        frame_offset = mn->frame_offset - 1;
    if(mp && frame_offset <= mp->frame_offset)
        frame_offset = mp->frame_offset + 1;
    m->frame_offset = frame_offset;
}

void
marker_list_dump(struct marker_list *ml) {
    GList *l;
    for(l = ml->markers; l; l = l->next) 
        marker_dump((struct marker *)(l->data));
}

/* FIXME: slow */

struct marker *
marker_list_next(struct marker_list *ml,
                 AFframecount frame_offset,
                 enum marker_type type) {
    GList *l;
    struct marker *m;
    for(l = ml->markers; l; l = l->next) {
        m = (struct marker *)(l->data); 
        if(m->frame_offset >= frame_offset && 
           m->type == type &&
           !(m->flags & MARKER_IS_DISABLED)) 
            return m;
    }
    return NULL;
}

/* FIXME: slow */

struct marker *
marker_list_previous(struct marker_list *ml,
                     AFframecount frame_offset,
                     enum marker_type type) {
    GList *l;
    struct marker *m, *prev = NULL;
    for(l = ml->markers; l; l = l->next) {
        m = (struct marker *)(l->data); 
        if(m->frame_offset < frame_offset &&
           m->type == type &&
           (m->flags & MARKER_IS_DISABLED) == 0)
            prev = m;
        if(m->frame_offset >= frame_offset && 
           m->type == type) 
            break;
    }
    return prev;
}


void
marker_list_delete(struct marker_list *ml,
                   AFframecount frame_offset,
                   AFframecount frame_count) {
    GList *l, *l2;
    struct marker *m;
    ml->left_cache[MARKER_SLOPE] = ml->right_cache[MARKER_SLOPE] = 0;
    ml->left_cache[MARKER_SLOPE_AUX] = ml->right_cache[MARKER_SLOPE_AUX] = 0;
    for(l = ml->markers; l; ) {
        m = (struct marker *)(l->data);
        if(m->frame_offset > frame_offset + frame_count)
            break;
        if(m->frame_offset >= frame_offset &&
           m->frame_offset <= frame_offset + frame_count) {
            marker_destroy(m);
            l2 = l->next;
            ml->markers = g_list_remove_link(ml->markers, l);
            g_list_free(l);
            l = l2;
            continue;
        }
        l = l->next;
    }

    /* Subtract. */
    /*
    for(; l; l = l->next) {
        m = (struct marker *)(l->data);
        m->frame_offset = MAX(0, m->frame_offset - frame_count);
        }*/
}

gint
marker_list_do_insert(gconstpointer a,
                      gconstpointer b) {
    struct marker *m1, *m2;
    m1 = (struct marker *)a;
    m2 = (struct marker *)b;
    if(m1->frame_offset > m2->frame_offset)
        return 1;
    return 0;
}

/* 
 * Returns the marker that was inserted or NULL. 
 */

struct marker *
marker_list_insert(struct marker_list *ml,
                   AFframecount frame_offset,
                   enum marker_type type,
                   float multiplier,
                   char *label) {
    struct marker *m, *m2;

    ml->left_cache[type] = ml->right_cache[type] = 0;
    
    m2 = marker_list_next(ml, frame_offset, type);
    if(m2 && m2->frame_offset == frame_offset)
        return NULL;

    m = marker_new(frame_offset, type, multiplier, label);

    if(!m) 
        return NULL;

    ml->markers = g_list_insert_sorted(ml->markers, 
                                       m, 
                                       marker_list_do_insert);
    return m;
}

struct marker_list *
marker_list_new() {
    struct marker_list *ml = calloc(1, sizeof(struct marker_list));
    if(!ml)
        return NULL;
    ml->marker_types_enabled = 0;
    ml->markers = NULL;
    return ml;
}

void
marker_list_do_destroy(gpointer l,
                       gpointer user_data) {
    DEBUG("destroying 1 marker\n");
    marker_destroy((struct marker *)(user_data));
    ((GList *)l)->data = NULL;
}

void
marker_list_destroy(struct marker_list *ml) {
    /* FIXME */
    //    g_list_foreach(ml->markers, (GFunc)marker_list_do_destroy, NULL);
    //    g_list_free(ml->markers);
    //    free(ml);
}
