/*
 * Copyright (C) 2002 Pascal Haakmat.
 */

#include <unistd.h>
#include <sys/types.h>
#include <dirent.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>
#include <dlfcn.h>
#include <wordexp.h>
#include "config.h"
#include "module.h"

module modules[MAX_MODULES];
int module_count = 0;

void
module_exit() {
    int i;
    for(i = 0; i < module_count; i++) {
        DEBUG("destroying module '%s'\n", modules[i].name);
        dlclose(modules[i].handle);
        free(modules[i].name);
        free(modules[i].fname);
    }
}

void
module_load(const char *path) {
    void *handle;
    char *error;
    module *mod;
    module *(*module_ctor)();

    if(module_count + 1 >= MAX_MODULES) {
        FAIL("won't load %s because maximum number of modules exceeded.\n",
             path);
        return;
    }

    handle = dlopen(path, RTLD_NOW);
    if (!handle) {
        FAIL("could not open %s: %s\n", path, dlerror());
        return;
    }

    module_ctor = dlsym(handle, "module_new");
    if((error = dlerror()) != NULL) {
        FAIL("could not open %s: %s\n", path, error);
        dlclose(handle);
        return;
    }

    mod = (*module_ctor)();

    if(!mod || mod->error) {
        FAIL("failed to create module. %s.\n", 
             mod ? mod->error : "unspecified error");
        dlclose(handle);
        return;
    }

    modules[module_count].name = strdup(mod->name);
    modules[module_count].fname = strdup(path);
    modules[module_count].handle = handle;
    module_count++;
    DEBUG("created module %d, '%s'\n", module_count, mod->name);
    
}

int
module_compare(const void *a, 
               const void *b) {
    return strcmp(((module *)a)->name, ((module *)b)->name);
}

int
module_collect(char *path) {
    char tmp[4096];
    struct dirent **namelist;
    int i, n;

    n = scandir(path, &namelist, 0, alphasort);
    if (n < 0) {
        FAIL("unable to load modules in %s: %s\n", path, strerror(errno));
        return 0;
    }

    for(i = 0; i < n; i++) {
        snprintf(tmp, 4096, "%s/%s", path, namelist[i]->d_name);
        free(namelist[i]);
        module_load(tmp);
    }

    free(namelist);

    qsort(modules, module_count, sizeof(module), module_compare);

    return 0;
}

int
module_init() {
    int i, r;
    char *sp = strdup(MODULE_PATH), *t;
    wordexp_t exp;
    if(!sp) {
        FAIL("unable to initialize modules, not enough memory to copy MODULE_PATH, bye\n");
        return 1;
    }
    while((t = strsep(&sp, ":"))) {
        r = wordexp(t, &exp, 0);
        if(r) {
            FAIL("unable to initalize modules in path %s, invalid path?\n",
                 t);
            continue;
        }
        DEBUG("collecting modules in path %s\n", t);
        for(i = 0; i < exp.we_wordc; i++) {
            DEBUG("expansion %d: %s\n", i, exp.we_wordv[i]);
            module_collect(exp.we_wordv[i]);
        }
        wordfree(&exp);
    }

    return 0;
}

