/*
 * Copyright (C) 2002 Pascal Haakmat.
 */

#include <math.h>
#include <gtk/gtk.h>
#include <glade/glade.h>
#include "../modutils.h"

module modinfo;

enum generator_types {
    SIN_GENERATOR         = (1 << 0),
    TRIANGLE_GENERATOR    = (1 << 1),
    SAW_GENERATOR         = (1 << 2),
    SQUARE_GENERATOR      = (1 << 3)
};

int32_t 
sine_generator(AFframecount offset,
               double rate,
               double frequency) {
    double rps = 2 * M_PI / rate; 
    return (int32_t)(sin(offset * rps * frequency) * INT32_MAX);
}

int32_t
triangle_generator(AFframecount offset,
                   double rate,
                   double frequency) {
    double rps = 2 * M_PI / rate; 
    double y = fmod(offset * rps * frequency, 2 * M_PI);
    if(y < M_PI / 2)
        return (int32_t)((y / (M_PI / 2)) * INT32_MAX);
    else if(y >= M_PI / 2 && y < M_PI) {
        y -= M_PI;
        return (int32_t)(-(y / (M_PI / 2)) * INT32_MAX);
    } else if(y >= M_PI && y < M_PI + (M_PI / 2)) {
        y -= M_PI;
        return (int32_t)(-(y / (M_PI / 2)) * INT32_MAX);
    } else {
        y -= M_PI + (M_PI / 2);
        return (int32_t)((y / (M_PI / 2)) * INT32_MAX) + INT32_MAX + 1;
    }
}

int32_t
saw_generator(AFframecount offset,
              double rate,
              double frequency) {
    double rps = 2 * M_PI / rate; 
    double y = fmod(offset * rps * frequency, 2 * M_PI);
    if(y < M_PI)
        return (int32_t)(((((y / (2 * M_PI)) - .5) * 4) + 1) * INT32_MAX) + (INT32_MAX);
    else
        return (int32_t)(((((y / (2 * M_PI)) - .5) * 4) - 1) * INT32_MAX) + (INT32_MAX);
}

int32_t
square_generator(AFframecount offset,
                 double rate,
                 double frequency) {
    double rps = 2 * M_PI / rate; 
    double y = fmod(offset * rps * frequency, 2 * M_PI);
    if(y < M_PI)
        return INT32_MAX;
    else
        return INT32_MIN;
}

typedef struct {
    char *control_name;
    int32_t (*generator_func)(AFframecount offset,
                              double rate,
                              double frequency);
} generator;

generator generator_info[] = {
    { "sine", sine_generator },
    { "triangle", triangle_generator },
    { "saw", saw_generator },
    { "square", square_generator },
    { NULL, NULL }
};

#define GENERATOR_COUNT 4

typedef struct {
    module_id id;
    shell *shl;
    AFframecount duration;
    float frequency;
    int generators;
    GtkSpinButton *duration_control;
    GtkSpinButton *frequency_control;
    GtkCheckButton *generator_controls[GENERATOR_COUNT];
    GtkWidget *dialog;
} tonegen_params;

#define TONEGEN_GLADE_FILE "tonegen.glade"

module *
module_new() {
    MODULE_INIT(&modinfo, "Tone Generator");
    return &modinfo;
}

void
on_apply_clicked(GtkWidget *w,
                 gpointer user_data) {
    int i;
    struct _module_state *module_state = 
        gtk_object_get_user_data(GTK_OBJECT(gtk_widget_get_toplevel(w)));
    tonegen_params *p = (tonegen_params *)module_state->data;
    p->duration = gtk_spin_button_get_value_as_float(p->duration_control) *
        p->shl->grid.gap;
    p->frequency = gtk_spin_button_get_value_as_float(p->frequency_control);
    p->generators = 0;
    for(i = 0; i < GENERATOR_COUNT; i++)
        if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(p->generator_controls[i])))
            p->generators |= (1 << i);
    action_do(ACTION_MODULE_EXECUTE_NEW(WITH_UNDO, 
                                        p->shl,
                                        p->id));
}

void
on_ok_clicked(GtkWidget *w,
              gpointer user_data) {
    struct _module_state *module_state = 
        gtk_object_get_user_data(GTK_OBJECT(gtk_widget_get_toplevel(w)));
    tonegen_params *p = (tonegen_params *)module_state->data;
    on_apply_clicked(w, NULL);
    gtk_object_destroy(GTK_OBJECT(p->dialog));
}

void
on_close_clicked(GtkWidget *w,
                  gpointer user_data) {
    struct _module_state *module_state = 
        gtk_object_get_user_data(GTK_OBJECT(gtk_widget_get_toplevel(w)));
    tonegen_params *p = (tonegen_params *)module_state->data;
    gtk_object_destroy(GTK_OBJECT(p->dialog));
}

void
on_dialog_destroy(GtkWidget *w,
                  gpointer user_data) {
    struct _module_state *module_state = 
        gtk_object_get_user_data(GTK_OBJECT(gtk_widget_get_toplevel(w)));
    module_state->is_open = 0;
    free(module_state->data);
}

void
module_open(module_id id,
            shell *shl, 
            int undo) {
    GladeXML *xml;
    GtkWidget *w;
    int i;
    char path[4096];
    tonegen_params *p = mem_calloc(1, sizeof(tonegen_params));
    
    if(!p) {
        gui_alert("Not enough memory.");
        return;
    }

    snprintf(path, 4096, "%s/%s", dirname(modules[id].fname), TONEGEN_GLADE_FILE);
    DEBUG("loading interface %s\n", path);
    xml = glade_xml_new(path, "dialog");
    
    if(!xml) {
        gui_alert("Tone Generator: could not load interface %s.", path);
        free(p);
        return;
    }

    shl->module_state[id].is_open = 1;
    shl->module_state[id].data = p;
    p->id = id;
    p->shl = shl;
    p->duration_control = GTK_SPIN_BUTTON(glade_xml_get_widget(xml, "duration"));
    p->frequency_control = GTK_SPIN_BUTTON(glade_xml_get_widget(xml, "frequency"));
    
    for(i = 0; i < GENERATOR_COUNT; i++)
        p->generator_controls[i] = 
            GTK_CHECK_BUTTON(glade_xml_get_widget(xml, generator_info[i].control_name));

    p->dialog = GTK_WIDGET(glade_xml_get_widget(xml, "dialog"));
    gtk_signal_connect(GTK_OBJECT(p->dialog), "destroy", on_dialog_destroy, shl);
    w = GTK_WIDGET(glade_xml_get_widget(xml, "ok"));
    gtk_signal_connect(GTK_OBJECT(w), "clicked", on_ok_clicked, shl);
    w = GTK_WIDGET(glade_xml_get_widget(xml, "close"));
    gtk_signal_connect(GTK_OBJECT(w), "clicked", on_close_clicked, shl);
    w = GTK_WIDGET(glade_xml_get_widget(xml, "apply"));
    gtk_signal_connect(GTK_OBJECT(w), "clicked", on_apply_clicked, shl);

    gtk_object_set_user_data(GTK_OBJECT(p->dialog), 
                             GINT_TO_POINTER(&shl->module_state[id]));
    gtk_object_unref(GTK_OBJECT(xml));
}

void
module_close(mod_state *module_state) {
    gtk_object_destroy(GTK_OBJECT(((tonegen_params *)module_state->data)->dialog));
}

AFframecount
tonegen(shell *shl,
        int track,
        AFframecount start_offset,
        AFframecount end_offset,
        float frequency,
        int generators) {
    int i, j, gens_active = 0;
    double ampl_adjust, freq_adjust, atten = 0;
    ITERATOR_INIT(start_offset, end_offset - start_offset);
    for(i = 0; i < GENERATOR_COUNT; i++)
        if((1 << i) & generators)
            gens_active++;
    DEBUG("gens_active: %d\n", gens_active);
    if(gens_active)
        atten = (double)1 / (double)gens_active;
    ITERATOR_SKEL(shl, shl->sr->tracks[track], ;, 
                  for(i = 0; i < MIN(EFFECT_BUF_SIZE, iter_frame_count); i++) {
                      int32_frame_bits[i] = 0;
                      for(j = 0; j < GENERATOR_COUNT; j++) {
                          if((1 << j) & generators) {
                              ampl_adjust = marker_list_slope_value(shl->markers[track],
                                                                    iter_frame_offset + i,
                                                                    MARKER_SLOPE) + 1;
                              freq_adjust = marker_list_slope_value(shl->markers[track],
                                                                    iter_frame_offset + i,
                                                                    MARKER_SLOPE_AUX);
                              freq_adjust = (frequency * freq_adjust) + frequency;
                              int32_frame_bits[i] += 
                                  generator_info[j].generator_func(iter_frames_processed + i,
                                                                   shl->sr->rate,
                                                                   freq_adjust) * ampl_adjust * atten;
                          }
                      }
                  }
                  track_int32_frame_buffer_put(shl->sr->tracks[track],
                                               int32_frame_bits,
                                               iter_frame_offset,
                                               MIN(EFFECT_BUF_SIZE, iter_frame_count));
                  iter_read = iter_written = MIN(EFFECT_BUF_SIZE, iter_frame_count));
                  
    ITERATOR_EXIT();
    return iter_frames_processed;
}


action_group *
module_execute(shell *shl, 
               int undo) {
    AFframecount start = shl->select_start,
        end;
    int map = shl->select_channel_map;
    int t, id = shl->active_module_id, generators;
    float frequency;
    AFframecount highest = 0, frames_generated;
    tonegen_params *p = NULL;
    action_group *undo_ag = NULL, *tmp_ag;
    action *undo_a = NULL;
    p = shl->module_state[id].data;

    if(!shl->module_state[id].is_open) { 
        gui_alert("Tone Generator does not have defaults.");
        return NULL;
    }

    DEBUG("tonegen, duration: %ld, frequency: %f, generator mask: %d\n",
          p->duration, p->frequency, p->generators);

    end = shl->select_start + p->duration;
    frequency = p->frequency;
    generators = p->generators;

    if(undo) {
        undo_ag = action_group_new(0);
        if(!undo_ag) {
            gui_alert("Could not create undo for Tone Generator, out of memory.\n");
            return NULL;
        }
    }

    rwlock_rlock(&shl->sr->rwl);
    for(t = 0; t < snd_track_count(shl->sr); t++) {
        if((1 << t) & map && t < shl->sr->channels) { 

            /* In principle tonegen always generates the amount 
               of frames we specify but because it may be cancelled
               midway we must create a seperate delete action for
               every track we touch. */

            frames_generated = tonegen(shl,
                                       t,
                                       start,
                                       end,
                                       frequency,
                                       generators);
            if(undo) {
                undo_a = ACTION_DELETE_ON_TRACK_NEW(DONT_UNDO,
                                                    shl,
                                                    shl->sr,
                                                    t,
                                                    start,
                                                    frames_generated);
                if(!undo_a) {
                    FAIL("help, could not create delete action for undo\n");
                    break;
                }
                tmp_ag = action_group_append(undo_ag, undo_a);
                if(!tmp_ag) {
                    FAIL("help, could not append delete action to group\n");
                    break;
                }
                undo_ag = tmp_ag;
            }
            if(frames_generated > highest)
                highest = frames_generated;
        }
    }

    if(undo) {
        undo_a = ACTION_SELECT_NEW(DONT_UNDO,
                                   shl,
                                   shl->sr,
                                   map,
                                   shl->select_start,
                                   shl->select_end - shl->select_start);
        if(!undo_a) {
            FAIL("help, could not create select action for undo\n");
        } else {
            tmp_ag = action_group_append(undo_ag, undo_a);
            if(!tmp_ag) 
                FAIL("help, could not append select action to group\n");
            else
                undo_ag = tmp_ag;
        }
    }

    shl->select_end = shl->select_start + highest;
    rwlock_runlock(&shl->sr->rwl);
    return undo_ag;
}


