/*
 * Copyright (C) 2002 Pascal Haakmat.
 */

#include <math.h>
#include <audiofile.h>
#include "mem.h"
#include "shell.h"
#include "module.h"
#include "action.h"

char *curdir = ".";
char *pathpart = NULL;

const char *
dirname(const char *s) {
    char *s2;
    s2 = rindex(s, '/');
    if(!s2)
        return curdir;
    if(pathpart) 
        mem_free(pathpart);
    pathpart = mem_alloc(s2 - s + 1);
    if(!pathpart) {
        FAIL("cannot create dirname\n");
        return curdir;
    }
    memcpy(pathpart, s, s2 - s);
    pathpart[s2 -s] = '\0';
    return pathpart;
}

void
mix(shell *target_shell,
    int target_channel,
    AFframecount target_offset,
    shell *source_shell,
    int source_channel,
    AFframecount source_offset,
    AFframecount frame_count) {
    int i;
    int32_t *out_buffer;
    double marker_value = 0;
    AFframecount source_read = 0;
    ITERATOR_INIT(target_offset, frame_count);

    out_buffer = calloc(1, EFFECT_BUF_SIZE * sizeof(int32_t));
    if(!out_buffer) {
        FAIL("not enough memory for mix buffer (%d bytes)\n",
             EFFECT_BUF_SIZE * sizeof(int32_t));
        ITERATOR_EXIT();
        return;
    }

    ITERATOR(target_shell, target_shell->sr->tracks[target_channel], 
             for(i = 0; i < iter_read; i++) {
                 marker_value = 
                     marker_list_slope_value(target_shell->markers[target_channel],
                                             iter_frame_offset + i,
                                             MARKER_SLOPE); 
                 out_buffer[i] = int32_frame_bits[i] + 
                     (int32_frame_bits[i] * marker_value);
             }
             source_read = track_int32_frame_buffer_get(source_shell->sr->tracks[source_channel],
                                                        int32_frame_bits,
                                                        source_offset,
                                                        iter_read);
             for(i = 0; i < source_read; i++) {
                 marker_value = 
                     marker_list_slope_value(source_shell->markers[source_channel],
                                             source_offset + i,
                                             MARKER_SLOPE); 
                 out_buffer[i] += int32_frame_bits[i] + 
                     (int32_frame_bits[i] * marker_value);
             }
             DEBUG("iter_frame_offset: %ld\n", iter_frame_offset);
             blocklist_blocks_destroy(track_delete(target_shell->sr->tracks[target_channel],
                                               iter_frame_offset,
                                               iter_read));
             track_int32_frame_buffer_put(target_shell->sr->tracks[target_channel],
                                          out_buffer,
                                          iter_frame_offset,
                                          iter_read);
             source_offset += iter_read;
             memset(out_buffer, '\0', iter_read * sizeof(int32_t)));
    
    free(out_buffer);
    ITERATOR_EXIT();
}

AFframecount
resample(shell *shl,
         int track,
         AFframecount start_offset,
         AFframecount end_offset,
         double factor) {
    int32_t *out_buffer;
    double j = 0, adjusted_factor = factor, marker_value;
    AFframecount out_buf_count = ceil((1.0f / (factor / 2)) * EFFECT_BUF_SIZE);
    AFframecount out_buf_count_iteration, total_frames_written = 0;

    ITERATOR_INIT(start_offset, end_offset - start_offset);

    out_buffer = calloc(1, out_buf_count * sizeof(int32_t));
    if(!out_buffer) {
        FAIL("not enough memory for resample buffer (%ld bytes)\n",
             out_buf_count * sizeof(int32_t));
        ITERATOR_EXIT();
        return 0;
    }

    ITERATOR(shl, shl->sr->tracks[track], 
             for(out_buf_count_iteration = 0, j = 0; floor(j) < iter_read;
                 out_buf_count_iteration++, j += adjusted_factor) {
                 out_buffer[out_buf_count_iteration] = int32_frame_bits[(int)floor(j)];
                 marker_value = marker_list_slope_value(shl->markers[track],
                                                        iter_frame_offset + out_buf_count_iteration,
                                                        MARKER_SLOPE); 
                 if(marker_value == 0)
                     continue;
                 if(marker_value < 0) 
                     adjusted_factor = factor - ((factor * -marker_value) / 2);
                 else
                     adjusted_factor = factor + (factor * marker_value);
                 if(adjusted_factor < factor / 2) {
                     FAIL("factor: %f, adjusted_factor: %f, marker_value: %f\n", 
                          factor, adjusted_factor, marker_value);
                     abort();
                 }

             }
             blocklist_blocks_destroy(track_delete(shl->sr->tracks[track],
                                               iter_frame_offset,
                                               iter_read));
             track_int32_frame_buffer_put(shl->sr->tracks[track],
                                          out_buffer,
                                          iter_frame_offset,
                                          out_buf_count_iteration);
             iter_written = out_buf_count_iteration;
             total_frames_written += iter_written);
    
    free(out_buffer);
    ITERATOR_EXIT();
    return total_frames_written;
}

void
reverse(shell *shl,
        int track,
        AFframecount start_offset,
        AFframecount end_offset) {
    int i;
    int32_t f;
    AFframecount insert_offset = start_offset;
    ITERATOR_INIT(start_offset, end_offset - start_offset);
    ITERATOR(shl, shl->sr->tracks[track], 
             for(i = 0; i < (iter_read / 2); i++) {
                 f = int32_frame_bits[iter_read - (i + 1)];
                 int32_frame_bits[iter_read - (i + 1)] = int32_frame_bits[i];
                 int32_frame_bits[i] = f;
             }
             blocklist_blocks_destroy(track_delete(shl->sr->tracks[track],
                                                   iter_frame_offset,
                                                   iter_read));
             
             track_int32_frame_buffer_put(shl->sr->tracks[track],
                                          int32_frame_bits,
                                          insert_offset,
                                          iter_read));
    ITERATOR_EXIT();
}

void
amplify(shell *shl,
        int track,
        AFframecount start_offset,
        AFframecount end_offset,
        double factor,
        double slope) {
    int i;
    ITERATOR_INIT(start_offset, end_offset - start_offset);
    ITERATOR(shl, shl->sr->tracks[track],
             for(i = 0; i < iter_read; i++) {
                 int32_frame_bits[i] = int32_frame_bits[i] * 
                     (factor + 
                      (factor * marker_list_slope_value(shl->markers[track],
                                                        iter_frame_offset,
                                                        MARKER_SLOPE)) - 
                      (slope * (iter_frames_processed + i)));
             }
             track_int32_frame_buffer_replace(shl->sr->tracks[track],
                                              int32_frame_bits,
                                              iter_frame_offset,
                                              iter_read));
    ITERATOR_EXIT();
}

int32_t
find_peak(shell *shl,
          int track,
          AFframecount start_offset,
          AFframecount end_offset) {
    int i;
    int32_t peak = 1, val = 0;
    ITERATOR_INIT(start_offset, end_offset - start_offset);
    ITERATOR(shl, shl->sr->tracks[track],
             for(i = 0; i < iter_read; i++) {
                 val = ABS(int32_frame_bits[i]);
                 if(val > peak)
                     peak = val;
             });
    ITERATOR_EXIT();
    return peak;
}

action_group *
fade(shell *shl, 
     int undo,
     double factor,
     double slope) {
    int t, map = shl->select_channel_map;
    AFframecount start = shl->select_start, end = shl->select_end;
    action_group *undo_ag = NULL;
    
    rwlock_rlock(&shl->sr->rwl);
    
    if(undo) 
        undo_ag = action_group_undo_create(shl,
                                           map, 
                                           start,
                                           end - start,
                                           start,
                                           end - start);
    
    DEBUG("slope: %f\n", slope);
    
    for(t = 0; t < snd_track_count(shl->sr); t++) {
        if((1 << t) & map) {
            DEBUG("fading track %d\n", t);
            amplify(shl,
                    t,
                    start,
                    end,
                    factor,
                    slope);
        }
    }
    rwlock_runlock(&shl->sr->rwl);

    return undo_ag;
}

AFframecount
find_zero(shell *shl,
          int track,
          AFframecount start_offset,
          AFframecount end_offset) {
    AFframecount r = 0, abs_offset = start_offset, frame_count;
    int i, cur_sign = 0, prev_sign = -1, delta = 1, count_down = 0;
    _frame_bits frame_bits = mem_alloc(EFFECT_BUF_SIZE * 4);

    if(!frame_bits)
        return start_offset;

    if(start_offset > end_offset)
        count_down = 1;

    frame_count = ABS(end_offset - start_offset);

    if(count_down) {
        delta = -1;
        start_offset = MAX(0, start_offset - MIN(EFFECT_BUF_SIZE, frame_count));
    }

    do {
        r = track_int32_frame_buffer_get(shl->sr->tracks[track],
                                         frame_bits,
                                         start_offset,
                                         MIN(EFFECT_BUF_SIZE, frame_count));
        if(r <= 0)
            break;
        
        for(i = count_down ? r - 1 : 0; 
            count_down ? i >= 0 : i < r; 
            i += delta, abs_offset += delta) {

            cur_sign = (((int32_t *)frame_bits)[i] > 0 ? 1 : 0);
            
            /* First iteration, fix start point. */
            
            if(prev_sign == -1)
                prev_sign = cur_sign;

            /* Sign change. */
            
            if(cur_sign != prev_sign && prev_sign != -1) {
                DEBUG("found nul: abs_offset: %ld\n", abs_offset);
                mem_free(frame_bits);
                return abs_offset;
            }
            prev_sign = cur_sign;
        }

        start_offset += (delta * r);
        frame_count -= r;
        
    } while(r > 0);

    mem_free(frame_bits);
    return start_offset;
}

