/*
 * Copyright (C) 2002 Pascal Haakmat.
 */

#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <pthread.h>
#include <sys/time.h>
#include <sys/ioctl.h>
#include <sys/soundcard.h>
#include <sys/types.h>
#include <audiofile.h>
#include "config.h"
#include "shell.h"
#include "snd.h"
#include "mixer.h"
#include "play.h"
#include "action.h"
#include "undo.h"

extern int errno;

int
player_device_init(int audio_fd, 
                   int frame_width,
                   int channels, 
                   int speed,
                   int *format) { 
    int reqformat, reqchannels, reqspeed, frags;

    /* "This call switches the device to full duplex mode and makes the
       driver prepared for full duplex access. This must be done
       before checking the DSP_CAP_DUPLEX bit, since otherwise the
       driver may report that the device doesn't support full
       duplex." (oss.pdf p. 106) */
    /*
    if(record_mode)
        if(ioctl(audio_fd, SNDCTL_DSP_SETDUPLEX, 1) == -1) 
            FAIL("Could not set device to full duplex: %s.\n", 
                 strerror(errno));
    
    if(ioctl(audio_fd, SNDCTL_DSP_GETCAPS, &caps) == -1) 
        FAIL("Could not get %s device capabilities: %s\n", 
             AUDIO_DEVICE, strerror(errno));

    if(record_mode && (!(caps & DSP_CAP_DUPLEX) ||
                       !(caps & DSP_CAP_TRIGGER))) {
        FAIL("Soundcard has insufficient capabilities for play and record, " \
             "require DSP_CAP_DUPLEX DSP_CAP_TRIGGER, got %s%s\n",
             (caps & DSP_CAP_DUPLEX) ? "DSP_CAP_DUPLEX " : "",
             (caps & DSP_CAP_TRIGGER) ? "DSP_CAP_TRIGGER " : "");
        return 0;
    }
    */
    /* "This ioctl call must be used as early as possible. The optimum
       location is immediately after opening the device." (oss.pdf,
       p. 98) */
    
    frags = DSP_FRAGS; 
    if (ioctl(audio_fd, SNDCTL_DSP_SETFRAGMENT, &frags)) 
        perror("could not set fragments");
    
    /* FIXME: We ask audiofile to always return signed data, so we do
       not need to consider signed/unsigned issues. Endianness issues
       are not considered however, that needs work. */

    switch(frame_width) {
    case 1:
#ifdef SIGNED_INT8_TO_UNSIGNED_INT8_FLAG
        /* Arghhh, AFMT_S8 doesn't work for some cards and the values
           returned by SNDCTL_DSP_GETFMTS are totally useless (they
           won't tell you whether AFMT_U8 _is_ supported for
           example). */
        *format = AFMT_U8;
#else
        *format = AFMT_S8;
#endif
        break;
    case 2:
        *format = AFMT_S16_NE;
        break;
    case 4:
        *format = AFMT_S32_NE;
        break;
    }

    /* Set format (sign, bits, byteorder). */

    reqformat = *format;    
    if(ioctl(audio_fd, SNDCTL_DSP_SETFMT, &reqformat) == -1) {
        perror("SNDCTL_DSP_SETFMT");
        return 0;
    }
    if(*format != reqformat) {
        FAIL("Device doesn't support format\n");
        return 0;
    }

    /* Set channels. */

    reqchannels = channels;
    if (ioctl(audio_fd, SNDCTL_DSP_CHANNELS, &reqchannels) == -1) {
        perror("SNDCTL_DSP_CHANNELS"); 
        return 0;
    }    
    if(channels != reqchannels) {
        FAIL("device doesn't support %d channels\n", channels);
        return 0;
    }

    /* Set speed. */

    reqspeed = speed;    
    if (ioctl(audio_fd, SNDCTL_DSP_SPEED, &reqspeed)==-1) {
        perror("SNDCTL_DSP_SPEED"); 
        return 0;
    }

    if(reqspeed < (speed - 10) ||
       reqspeed > (speed + 10))
        FAIL("Warning, device requires %d samplerate (requested %d)\n",
             reqspeed, speed);

    DEBUG("playback speed: %d\n", reqspeed);
    return 1;
}

/* Open record and play device. We don't use full-duplex, because on
   Linux record and play device can be the same device, and this is
   easier. */
int
player_dsp_init(int *play_audio_fd, 
                int *rec_audio_fd,
                int *format, 
                int frame_width,
                int channels, 
                int input_channels,
                int speed,
                int record_mode) {
    int enable_bits;
    *play_audio_fd = open(PLAY_DEVICE, O_WRONLY);
    if(*play_audio_fd == -1) {
        FAIL("Could not open %s for write.\n", PLAY_DEVICE);
        return 0;
    }
    if(!player_device_init(*play_audio_fd,
                           frame_width,
                           channels,
                           speed,
                           format)) {
        FAIL("Could not initialize %s.\n", PLAY_DEVICE);
        return 0;
    }

    if(record_mode) {
        *rec_audio_fd = open(RECORD_DEVICE, O_RDONLY);
        if(*rec_audio_fd == -1) {
            FAIL("Could not open %s for read.\n", RECORD_DEVICE);
            close(*play_audio_fd);
            return 0;
        }
        if(!player_device_init(*rec_audio_fd,
                               frame_width,
                               input_channels,
                               speed,
                               format)) {
            FAIL("Could not initialize %s.\n", RECORD_DEVICE);
            return 0;
        }
    }
    /* Disable playback while we fill the play buffer. */
    
    if(record_mode) {
        enable_bits = ~PCM_ENABLE_OUTPUT;
        if(ioctl(*play_audio_fd, SNDCTL_DSP_SETTRIGGER, &enable_bits) == -1) 
            FAIL("SNDCTL_DSP_SETTRIGGER for play device %s, play/record synchro suboptimal: %s",
                 strerror(errno), PLAY_DEVICE);
        enable_bits = ~PCM_ENABLE_INPUT;
        if(ioctl(*rec_audio_fd, SNDCTL_DSP_SETTRIGGER, &enable_bits) == -1) 
            FAIL("SNDCTL_DSP_SETTRIGGER for record device %s, play/record synchro suboptimal: %s",
                 strerror(errno), RECORD_DEVICE);
    }
    
    return 1;
}

/*void
player_gdk_unlock(void *ptr) {
    gdk_threads_leave();
}
*/

void
player_graph_update(shell *shl) {
    AFframecount pp = shl->player.player_pos, 
        ls = shl->loop_start,
        le = shl->loop_end,
        w = (GTK_WIDGET(shl->canvas)->allocation.width * shl->hres),
        fvis = shl->hadjust->value,
        lvis = fvis + w;

    /* FIXME: this logic is a bit crummy. */

    if(shl->follow_playback) {
        if(pp > fvis && pp < lvis) {
            if(pp > ls && pp < le &&
               !(ls > fvis && le < lvis)) {
                shl->hadjust->value = pp - (w / 2);
                gtk_adjustment_changed(shl->hadjust);
            } else {
                if(!(ls > fvis && le < lvis)) {
                    if(pp > (w / 2) +
                       fvis &&
                       !(pp > shl->select_start && pp < shl->select_end)) {
                        shl->hadjust->value = pp - (w / 2);
                        gtk_adjustment_changed(shl->hadjust);
                    }
                }
            }
        } else if(shl->player.record_mode) {
            shl->hadjust->value = pp - (w / 2);
            gtk_adjustment_changed(shl->hadjust);
        }            
    }

    shell_redraw(shl);
}

void
player_buffer_convert(_frame_bits fb,
                      int frame_width,
                      AFframecount frame_count,
                      int format) {
    AFframecount i;

    /* This function should convert the frames in fb to the format
       specified in format. That means byte order, sign and frame
       width translation. The bits in fb are always two's complement.
       Byte order is defined by libaudiofile. FIXME: what kind of byte
       order does libaudiofile return? (think native byte order) */
    
    /* Some cards cannot playback signed 8 bit samples. So we convert
       them to unsigned 8 bit. */

    if(format == AFMT_U8) {
        if(frame_width == 1) {
            for(i = 0; i < frame_count; i++) {
                ((int8_t *)fb)[i] = ((int8_t *)fb)[i] + 128;
            }
        } else {
            FAIL("cannot convert from frame_width %d to 1\n",
                 frame_width);
        }
    }    
}

void
player_dsp_trigger(shell *shl) {
    int enable_bits = PCM_ENABLE_OUTPUT;
    if(ioctl(shl->player.play_audio_fd, SNDCTL_DSP_SETTRIGGER, &enable_bits) == -1) 
        FAIL("SNDCTL_DSP_SETTRIGGER: failed to trigger playback on %s\n", 
             PLAY_DEVICE);
    enable_bits = PCM_ENABLE_INPUT;
    if(ioctl(shl->player.rec_audio_fd, SNDCTL_DSP_SETTRIGGER, &enable_bits) == -1) 
        FAIL("SNDCTL_DSP_SETTRIGGER: failed to trigger playback on %s\n",
             RECORD_DEVICE);
}

int
player_play(shell *shl,
            int format,
            int mix_map,
            int target_channels,
            int frame_width,
            AFframecount frame_count) {
    player *p = &shl->player;
    int played_frames;

    memset(p->muxbuf, '\0', PLAY_BUF_SIZE * frame_width * target_channels);
    played_frames = snd_mux(shl->sr,
                            p->muxbuf, 
                            p->srcbufs,
                            shl->mixer,
                            mix_map,
                            p->player_pos,
                            MIN(frame_count, PLAY_BUF_SIZE));
    player_buffer_convert(p->muxbuf, 
                          frame_width, 
                          played_frames * target_channels, 
                          format);
    if(write(p->play_audio_fd, 
             p->muxbuf, 
             played_frames * frame_width * target_channels) == -1) {
        
        /* write() returns EAGAIN when the buffer is full and
           non-blocking operations are selected (that is,
           SND_DSP_SETTRIGGER is active). */
        
        if(errno != EAGAIN) {
            FAIL("write failed on %s: %s\n", PLAY_DEVICE, strerror(errno));
            return -1;
        }
    }
    
    return played_frames;
}

int
player_record(shell *shl,
              int loop_count,
              int input_channels,
              int frame_width,
              AFframecount frame_count) {
    snd *del_sr;
    player *p = &shl->player;
    int recorded_frames;

    if(!p->record_mode)
        return 0;

    recorded_frames = read(p->rec_audio_fd,
                           p->muxbuf,
                           MIN(frame_count * frame_width * input_channels,
                               PLAY_BUF_SIZE * frame_width * input_channels)) /
        (frame_width * input_channels);
    
    if(recorded_frames == -1)
        return -1;
    
    if(recorded_frames == 0) 
        return 0;
    
    /* In replace mode we need to delete the number of frames that we
       just recorded before we insert the recorded data. */
    
    if(p->replace_mode && p->player_pos < snd_frame_count(shl->sr)) {
        
        /* FIXME: error checking. (added but it creaks) */
        
        del_sr = snd_delete(shl->sr,
                            p->undo_channel_map,
                            p->player_pos,
                            recorded_frames);
        if(RESULT_IS_ERROR(shl->sr)) {
            FAIL("Could not overwrite frames in record: %s\n",
                 snd_error_get(shl->sr));
            shl->loop = 0;
            snd_destroy(del_sr);
            return -1;
        }
        
        /* In replace mode we need to keep an undo buffer for the
           frames that we deleted. But in loop mode, we can stop
           making undos after the first loop has completed. */
        
        if(loop_count == 0) {            
            if(p->undo_sr) 
                snd_insert(p->undo_sr,
                           del_sr,
                           p->undo_channel_map,
                           snd_frame_count(p->undo_sr));
            else {
                p->undo_sr = 
                    snd_clone(del_sr,
                              CLONE_STRUCTURE | 
                              CLONE_TRACK_STRUCTURE | 
                              CLONE_TRACK_DATA);
                if(RESULT_IS_ERROR(del_sr)) {
                    FAIL("cannot clone deleted frames for undo buffer: %s\n",
                         snd_error_get(del_sr));
                    shl->loop = 0;
                    snd_destroy(p->undo_sr);
                    p->undo_sr = NULL;
                    snd_destroy(del_sr);                            
                    return -1;
                }
            }
        }
        snd_destroy(del_sr);
    }
    snd_demux(shl->sr,
              p->srcbufs[0],
              p->muxbuf,
              p->undo_channel_map,
              input_channels,
              p->player_pos,
              recorded_frames);
    
    return recorded_frames;
}

void
player_cleanup(void *ptr) {
    shell *shl = ptr;
    player *p = &shl->player;

    DEBUG("player_cancel_requested: %d, shl->loop: %d, shl->loop_start: %ld, shl->loop_end: %ld\n", 
          shl->player.player_cancel_requested, shl->loop, shl->loop_start, shl->loop_end);

    if(p->play_audio_fd != -1)
        close(p->play_audio_fd);
    p->play_audio_fd = -1;
    if(p->rec_audio_fd != -1)
        close(p->rec_audio_fd);
    p->rec_audio_fd = -1;

    if(p->player_running && shl->select_start == shl->select_end) {
        DEBUG("new cursor pos: %ld\n", p->player_pos);
        shl->select_start = shl->select_end = p->player_pos;
    }

    mixer_buffers_free(MAX_TRACKS,
                       p->muxbuf, 
                       p->srcbufs);
    DEBUG("player undo info: undo_start_offset: %ld, undo_end_offset: %ld, undo_sr: %p\n",
          p->undo_start_offset, p->undo_end_offset, p->undo_sr);

    if(p->undo_sr) {
        DEBUG("dumping undo_sr info:\n");
        snd_info_dump(p->undo_sr);
    }
    DEBUG("record_mode: %d\n", p->record_mode);

    if(p->record_mode) {
        if(p->replace_mode && p->undo_sr) {
            DEBUG("pushing undo for replacement record\n");
            shl->undo_stack = 
                undo_push(shl->undo_stack,
                          action_group_new(3,
                                           ACTION_DELETE_NEW(DONT_UNDO,
                                                             shl,
                                                             shl->sr, 
                                                             p->undo_channel_map, 
                                                             p->undo_start_offset, 
                                                             p->undo_end_offset - p->undo_start_offset),
                                           ACTION_INSERT_NEW(DONT_UNDO,
                                                             shl->sr, 
                                                             p->undo_sr, 
                                                             p->undo_channel_map,
                                                             p->undo_start_offset),
                                           ACTION_SELECT_NEW(DONT_UNDO,
                                                             shl,
                                                             shl->sr, 
                                                             p->undo_channel_map, 
                                                             p->undo_start_offset,
                                                             p->undo_end_offset - p->undo_start_offset)));
        } else {
            
            DEBUG("pushing undo for insertion record\n");
            shl->undo_stack = 
                undo_push(shl->undo_stack,
                          action_group_new(2,
                                           ACTION_DELETE_NEW(DONT_UNDO,
                                                             shl,
                                                             shl->sr, 
                                                             p->undo_channel_map, 
                                                             p->undo_start_offset, 
                                                             p->undo_end_offset - p->undo_start_offset),
                                           ACTION_SELECT_NEW(DONT_UNDO,
                                                             shl,
                                                             shl->sr, 
                                                             p->undo_channel_map, 
                                                             p->undo_start_offset,
                                                             0)));
        }
    }

    pthread_mutex_lock(&p->player_running_lock);
    pthread_cond_signal(&p->player_running_cond);
    p->player_running = 0;
    pthread_mutex_unlock(&p->player_running_lock);

    /* Final redraw to restore "normal" display and get rid of player
       specific display stuff (e.g. pointer). */

    gdk_threads_enter();
    player_graph_update(shl);
    gdk_threads_leave();
    p->player_cancel_honored = 1;

    DEBUG("player cleanup complete\n");
}

/* The player thread entry point. The player thread handles
 * playback/record. 
 */

void *
player_thread(void *ptr) {
    int i,
        format, 
        frame_width, 
        target_channels,
        iteration = 0, 
        mix_map,
        input_channels = 0, 
        loop_count = 0,
        selection_is_empty = 0,
        trigger_activate = 0;
    AFframecount prev_player_pos, 
        played_frames = 0,
        recorded_frames = 0, 
        frame_count, 
        s, 
        e,
        first_visible,
        last_visible,
        pos_adjust = 0;
    shell *shl = (shell *)ptr;
    player *p = &shl->player;

    frame_width = shl->sr->frame_width;
    target_channels = shl->mixer->target_channels;
    loop_count = 0;
    p->muxbuf = NULL;
    for(i = 0; i < MAX_TRACKS; i++)
        p->srcbufs[i] = NULL;

    p->undo_channel_map = shl->select_channel_map;
    p->record_mode = shl->record_mode;
    p->replace_mode = shl->replace_mode;
    p->undo_sr = NULL;

    /* Setup record mode stuff. The mix_map is used to mute tracks 
       that are being recorded on. */

    mix_map = 0xff;
    if(p->record_mode) {
        trigger_activate = 1;
        shl->has_changed = 1;
        if(!p->undo_channel_map)
            p->undo_channel_map = 1;
        
        /* Subtract selected tracks from the mix_map. */

        for(i = 0; i < MAX_TRACKS; i++) {
            if((1 << i) & p->undo_channel_map) {
                mix_map -= (1 << i);
                input_channels++;
            }
        }
        if(input_channels > MAX_TRACKS)
            input_channels = MAX_TRACKS;
        DEBUG("input_channels: %d, mix_map: %d, p->undo_channel_map: %d\n",
              input_channels, mix_map, p->undo_channel_map);
    }

    pthread_cleanup_push(player_cleanup, shl);

    /* Signal that we are up and running now. It's important that this
       happens before initialization may fail or player_start()
       deadlocks on failure. */

    pthread_mutex_lock(&p->player_running_lock);
    p->player_running = 1;
    pthread_cond_signal(&p->player_running_cond);
    pthread_mutex_unlock(&p->player_running_lock);

    if(!player_dsp_init(&p->play_audio_fd, 
                        &p->rec_audio_fd,
                        &format, 
                        frame_width, 
                        target_channels, 
                        input_channels,
                        shl->sr->rate,
                        p->record_mode))
        goto cleanup;    

    if(!mixer_buffers_alloc(frame_width,
                            MAX_TRACKS,
                            &p->muxbuf, 
                            p->srcbufs,
                            PLAY_BUF_SIZE)) {
        FAIL("could not allocate mixer buffers\n");
        goto cleanup;
    }

    DEBUG("player running, record_mode: %d, replace_mode: %d\n",
          p->record_mode, p->replace_mode);

    /* Repeat while loop is active and not cancel_requested. */

    do {

        s = LOOP_IS_ACTIVE(shl) ? shl->loop_start : shl->select_start;
        e = LOOP_IS_ACTIVE(shl) ? shl->loop_end : shl->select_end;

        if(s == e) {
            selection_is_empty = 1;
            e = snd_frame_count(shl->sr);
        }

        p->player_pos = s;
        prev_player_pos = p->player_pos;

        if(loop_count == 0) 
            p->undo_start_offset = s;

        /* In record mode, if there is no selection, just keep
           recording in PLAY_BUF_SIZE chunks. Otherwise, the number of
           frames to play is end - start. */

        if(p->record_mode && selection_is_empty)
            frame_count = PLAY_BUF_SIZE;
        else
            frame_count = MAX(0, e - s);
        
        while(frame_count && !shl->player.player_cancel_requested) {            

            played_frames = player_play(shl,
                                        format,
                                        mix_map,
                                        target_channels,
                                        frame_width,
                                        frame_count);

            if(played_frames == -1) {
                FAIL("player error, played_frames: -1\n");
                goto cleanup;
            }

            if(trigger_activate) {
                player_dsp_trigger(shl);
                trigger_activate = 0;
            }

            recorded_frames = player_record(shl, 
                                            loop_count,
                                            input_channels, 
                                            frame_width, 
                                            frame_count);

            if(recorded_frames == -1) {
                FAIL("player error, recorded_frames: -1\n");
                goto cleanup;
            }

            if(played_frames == 0 && recorded_frames == 0) {
                FAIL("nothing played, nothing recorded, bailing out\n");
                goto cleanup;
            }

            /* Now determine how many bytes were actually
               played/recorded and adjust the player_pos and
               frame_count (bytes-to-go) accordingly. */

            pos_adjust += ABS(prev_player_pos - p->player_pos);
            prev_player_pos = p->player_pos;
            
            if(p->record_mode) {

                /* Frame count (bytes-to-record) never diminishes if
                   there is no selection. */
            
                p->player_pos += recorded_frames;
                if(!selection_is_empty) 
                    frame_count = MAX(0, e - p->player_pos);
                if(loop_count == 0)
                    p->undo_end_offset = p->player_pos;
            } else {
                p->player_pos += played_frames;
                frame_count = MAX(0, e - p->player_pos);
            }

            /* Draw. */

            if(pos_adjust > shl->hres && (iteration++ % DRAW_DIV == 0)) {
                pos_adjust = 0;
                gdk_threads_enter();
                player_graph_update(shl);
                if(shl->player.record_mode)
                    shell_status_default_set(shl);
                gdk_threads_leave();
            }
            if(shl->debug_flags & DEBUG_FLAG_STEP_MODE)
                sleep(2);
        }

        if(!LOOP_IS_ACTIVE(shl) || shl->player.player_cancel_requested)
            break;

        /* FIXME: move to beginning of outer loop. */

        p->player_pos = shl->loop_start;
        frame_count = shl->loop_end - shl->loop_start;
        first_visible = shl->hadjust->value;
        last_visible = first_visible + 
            (GTK_WIDGET(shl->canvas)->allocation.width * shl->hres);

        /* Do something sensible when looping wraps. Only adjust the
           viewport when the loop start is not visible and the loop
           end is visible. Otherwise the user is probably just
           scanning through the file and wouldn't want to be suddenly
           sent someplace else. */

        if(shl->follow_playback && LOOP_IS_ACTIVE(shl) &&
           (shl->loop_start < first_visible || 
            shl->loop_start > last_visible) &&
           (shl->loop_end > first_visible &&
            shl->loop_end < last_visible)) {
            gdk_threads_enter();
            shell_viewport_center(shl, shl->loop_start, shl->loop_start);
            gdk_threads_leave();
        }
        loop_count++;

    } while(LOOP_IS_ACTIVE(shl) && !shl->player.player_cancel_requested);

 cleanup:
    pthread_cleanup_pop(1);
    return NULL;
}

/* Stops the running player thread by setting player_cancel_requested
 * to true (which is polled from the player thread), then waiting for
 * the player thread cleanup handler to signal completion.
 */

void
player_stop(shell *shl) {
    struct timeval now;
    struct timespec timeout;

    if(!shl->player.player_running)
        return;

    /* Avoid recursion. FIXME: this is necessary only because the
       interface can recursively invoke player_stop(), which it
       shouldn't allow (see below as well). */

    if(shl->player.player_cancel_requested)
        return;
    
    DEBUG("requesting cancellation for player thread\n");
    shl->player.player_cancel_requested = 1;
    shl->player.player_cancel_honored = 0;

    /* Careful now. We need to wait for the player thread to execute
       its cleanup handler and terminate. But we are being called from
       the GTK event handler and thus blocking event handling. This
       may cause the player thread to block, since it needs to issue
       redraws, which in turn issues events that never get handled as
       long as we are blocking the GTK event handler. When the player
       thread blocks waiting for the redraw events to be processed,
       then we block waiting for the player thread to cleanup and
       terminate, and deadlock occurs. Therefore, we cannot wait for
       the player thread indefinitely. Instead we use a polling
       approach and invoke the GTK event handler manually on each
       iteration to ensure that the player thread does not get
       stuck. */
       
    pthread_mutex_lock(&shl->player.player_running_lock);
    while(shl->player.player_running) {
        gettimeofday(&now, NULL);
        timeout.tv_sec = now.tv_sec;
        timeout.tv_nsec = now.tv_usec * 1000;
        pthread_cond_timedwait(&shl->player.player_running_cond, 
                               &shl->player.player_running_lock,
                               &timeout);

        /* Process pending GTK events. FIXME: this can cause
           player_stop() to be invoked again, which we catch above. */

        while(gtk_events_pending())
            gtk_main_iteration_do(FALSE);
    }
    pthread_mutex_unlock(&shl->player.player_running_lock);

    /* Wait for cleanup handler issue final redraw. */

    while(!shl->player.player_cancel_honored) {
        while(gtk_events_pending())
            gtk_main_iteration();
        usleep(1000);
    }

    /* Cleanup handler exits now and thread should terminate. */

    DEBUG("joining player thread...\n");
    pthread_join(shl->player.player_thread, NULL);
    DEBUG("player thread joined\n");

    shl->player.player_cancel_requested = 0;
}

void
player_start(shell *shl) {
    int retcode;
    struct timeval now;
    struct timespec timeout;
    if(shl->player.player_cancel_requested)
        return;
    pthread_mutex_lock(&shl->player.player_running_lock);
    shl->player.player_cancel_requested = 0;
    if(pthread_create(&shl->player.player_thread, 
                      NULL,
                      player_thread,
                      shl)) {
        FAIL("failed to create player thread.\n");
        pthread_mutex_unlock(&shl->player.player_running_lock);
        return;
    }
    DEBUG("waiting for player thread to initialize...\n");
    while(!shl->player.player_running) {

        /* This needs cond_timedwait() and periodic
           gtk_main_iteration() invocation, otherwise deadlocks can
           occur against gdk_threads_enter() in player_cleanup(). */

        gettimeofday(&now, NULL);
        timeout.tv_sec = now.tv_sec;
        timeout.tv_nsec = now.tv_usec + 500000;
        retcode = pthread_cond_timedwait(&shl->player.player_running_cond, 
                                         &shl->player.player_running_lock,
                                         &timeout);

        if(retcode != ETIMEDOUT) {
            DEBUG("pthread_cond_wait failed: %s\n", strerror(errno));
            pthread_mutex_unlock(&shl->player.player_running_lock);
            return;
        }

        while(gtk_events_pending())
            gtk_main_iteration_do(FALSE);
    }
    DEBUG("ok, player thread initialized.\n");
    pthread_mutex_unlock(&shl->player.player_running_lock);
}
