/*
 * Copyright (C) 2002 Pascal Haakmat.
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <dlfcn.h>
#include <math.h>
#include <gnome.h>
#include <glade/glade.h>
#include "config.h"
#include "grid.h"
#include "module.h"
#include "action.h"
#include "draw.h"
#include "undo.h"
#include "gui.h"
#include "play.h"
#include "modutils.h"
#include "shell.h"

extern shell *clipboard_shell;
extern int module_count;
extern module modules[];
extern int mem_fail_allocation_on_zero;
GList *shells;

int
shell_width_get(shell *shl) {
    return GTK_WIDGET(shl->canvas)->allocation.width;
}

int
shell_height_get(shell *shl) {
    return GTK_WIDGET(shl->canvas)->allocation.height;
}

void 
shell_vadjust_set(shell *shl, 
                  double value) {
    
    if(!shl->sr)
        return;
    shl->vadjust->lower = 0;
    shl->vadjust->upper = shl->sr->channels;
    shl->vadjust->page_size = 1;
    shl->vadjust->step_increment = shl->vadjust->page_increment = 1;
    gtk_adjustment_changed(shl->vadjust);
    gtk_adjustment_value_changed(shl->vadjust);

    /* Show/hide the vertical scrollbar. */

    if(shl->vadjust->value != 0 ||
       shl->sr->channels * shl->vres > shell_height_get(shl)) 
        gtk_widget_show(GTK_WIDGET(shl->vscrollbar));
    else 
        gtk_widget_hide(GTK_WIDGET(shl->vscrollbar));
}

void
shell_hadjust_set(shell *shl, 
                  double value) {
    if(!shl->sr)
        return;

    shl->hadjust->upper = snd_frame_count(shl->sr);
    shl->hadjust->lower = 0;
    shl->hadjust->step_increment = shl->hres < 1 ? 1 : shl->hres * 10;
    shl->hadjust->page_size = 
        floor(shell_width_get(shl) * shl->hres);
    shl->hadjust->page_increment = shl->hadjust->page_size - shl->hres;
    shl->hadjust->value = 
        rint(value) + shl->hadjust->page_size > shl->hadjust->upper ? 
        shl->hadjust->upper - shl->hadjust->page_increment : rint(value);
    gtk_adjustment_changed(shl->hadjust);
    gtk_adjustment_value_changed(shl->hadjust);
}

void
shell_viewport_center(shell *shl, 
                      AFframecount start,
                      AFframecount end) {
    AFframecount delta = end - start;
    shl->hadjust->value = start;

    if(delta < shell_width_get(shl) * shl->hres)
        shl->hadjust->value -= 
            ((shell_width_get(shl) * shl->hres) - delta) / 2;

    shell_hadjust_set(shl, shl->hadjust->value);
}

void 
shell_hruler_set(shell *shl) {
    if(shl->sr) 
        gtk_ruler_set_range(GTK_RULER(shl->hruler),
                            snd_frames_to_time(shl->sr, shl->hadjust->value),
                            snd_frames_to_time(shl->sr, shl->hadjust->value + 
                                               (GTK_WIDGET(shl->hruler)->allocation.width) * 
                                               shl->hres),
                            snd_frames_to_time(shl->sr, shl->hadjust->value),
                            shl->sr->time * (shl->hres < 1 ? 
                                             1 / shl->hres : shl->hres));
}

void
shell_hres_set(shell *shl,
               float zoom) {
    shl->hres = zoom;
    shell_redraw(shl);
}

void
shell_vres_set(shell *shl,
               int zoom) {
    shl->vres = zoom;
    gtk_widget_queue_draw(GTK_WIDGET(shl->mixercanvas));
    shell_redraw(shl);
}

void
window_cursor_set(GdkWindow *w,
                  GdkCursorType type) {
    GdkCursor *cursor = gdk_cursor_new(type);
    gdk_window_set_cursor(w, cursor);
}

void 
shell_cursor_set(shell *shl, 
                 GdkCursorType type) {
    window_cursor_set(GTK_WIDGET(shl->canvas)->window, type);
}

void
shell_redraw(shell *shl) {
    if(!shl->sr) 
        return;

    shell_hadjust_set(shl, shl->hadjust->value);
    shell_vadjust_set(shl, shl->vadjust->value);
    gtk_window_set_title(GTK_WINDOW(shl->appwindow), shell_path_strip(shl->sr->name));
    shell_hruler_set(shl);
    gtk_widget_queue_draw(GTK_WIDGET(shl->canvas));
}

char *
shell_path_strip(const char *const_p) {
    char *p = (char *)const_p;
    char *name_ptr;

    if(!p) 
        return NULL;

    for(name_ptr = p + (strlen(p) - 1); 
        name_ptr > p && *name_ptr != '/'; 
        name_ptr--);
    
    if(*name_ptr == '/')
        name_ptr++;
    else
        name_ptr = p;

    return name_ptr;
}

/* Signal/event handlers. */

void
shell_size_allocate_signal(GtkWidget *widget,
                           GtkAllocation *allocation,
                           shell *shl) {

    /* This function attaches to the window's "size_allocate"
       signal. The assumption is that we get here after the drawing
       area's "configure" handler has run (see below). We simply force
       another resize of the drawing area here after processing any
       outstanding events. Otherwise it may happen in some cases that
       the drawing area gets the wrong size; in particular, when I use
       my window manager's maximize function, and if this causes the
       vertical scrollbar to be hidden (because there is now enough
       vertical screen space to show all the tracks), then without
       this code, the drawing area will be sized as if the scrollbar
       was not there. The strange thing is that this problem does not
       happen when resizing a window by its corner. */

    while(gtk_events_pending())
        if(gtk_main_iteration())
            return;
    gtk_widget_queue_resize(GTK_WIDGET(shl->canvas));
}

gboolean 
shell_focus_in_event(GtkWidget *widget,
                     GdkEventFocus *event,
                     shell *shl) {
    return FALSE;
}

gboolean 
shell_focus_out_event(GtkWidget *widget,
                      GdkEventFocus *event,
                      shell *shl) {
    return FALSE;
}

void 
shell_vscrollbar_value_changed(GtkAdjustment *adj, 
                               shell *shl) {
    adj->value = (int) adj->value;
    gtk_widget_queue_draw(GTK_WIDGET(shl->mixercanvas));
    gtk_widget_queue_draw(GTK_WIDGET(shl->canvas));
}

void 
shell_hscrollbar_value_changed(GtkAdjustment *adj, 
                               shell *shl) {
    adj->value = floor(adj->value);
    shell_hruler_set(shl);
    gtk_widget_queue_draw(GTK_WIDGET(shl->canvas));
}

void
shell_draw_grid_toggle_activate(GtkMenuItem *menuitem,
                                shell *shl) {
    shl->show_grid = shl->show_grid ? 0 : 1;
    shell_redraw(shl);
}

void
shell_snap_to_grid_toggle_activate(GtkMenuItem *menuitem,
                                   shell *shl) {
    shl->snap_to_grid = shl->snap_to_grid ? 0 : 1;
    shell_redraw(shl);
}

void
shell_grid_configure(shell *shl) {

    /* Reset invalid unit entry. */

    if(shl->grid.units != gtk_spin_button_get_value_as_float(shl->gridunits))
        gtk_spin_button_set_value(shl->gridunits,
                                  shl->grid.units);
    shell_redraw(shl);
}

void
shell_grid_bpm_changed(GtkSpinButton *spinbutton,
                       shell *shl) {
    grid_bpm_set(&shl->grid, 
                 gtk_spin_button_get_value_as_float(spinbutton));
    shell_grid_configure(shl);
}

void
shell_grid_units_changed(GtkSpinButton *spinbutton,
                         shell *shl) {
    grid_units_set(&shl->grid, 
                   gtk_spin_button_get_value_as_float(spinbutton));
    shell_grid_configure(shl);
}

void
shell_grid_measurement_changed(GtkOptionMenu *optionmenu,
                               shell *shl) {
    enum grid_measurement m = option_menu_get_active(shl->gridmeasurement);
    grid_measurement_set(&shl->grid, m);
    if(m == GRID_BEATS) 
        gtk_widget_show(GTK_WIDGET(shl->gridbpm));
    else
        gtk_widget_hide(GTK_WIDGET(shl->gridbpm));
    shell_grid_configure(shl);
}

gint 
shell_mixercanvas_configure_event(GtkWidget *widget, 
                                  GdkEventConfigure *event,
                                  shell *shl) {
    size_t gb_sz;

    if(shl->mixerpixmap)
        gdk_pixmap_unref(shl->mixerpixmap);

    shl->mixerpixmap = gdk_pixmap_new(widget->window,
                                      widget->allocation.width,
                                      widget->allocation.height,
                                      -1);

    gb_sz = (size_t) widget->allocation.width * sizeof(_graph_bits_t);
    shl->graph_bits_buffer = calloc(1, gb_sz);
    
    return TRUE;

}

gint 
shell_canvas_configure_event(GtkWidget *widget, 
                             GdkEventConfigure *event,
                             shell *shl) {
    size_t gb_sz;

    if(shl->pixmap)
        gdk_pixmap_unref(shl->pixmap);

    if(shl->graph_bits_buffer)
        free(shl->graph_bits_buffer);

    shl->pixmap = gdk_pixmap_new(widget->window,
                                 widget->allocation.width,
                                 widget->allocation.height,
                                 -1);

    gb_sz = (size_t) widget->allocation.width * sizeof(_graph_bits_t);
    DEBUG("gb_sz: %d\n", gb_sz);
    shl->graph_bits_buffer = calloc(1, gb_sz);

    if(!shl->graph_bits_buffer) 
        FAIL("not enough memory to allocate peak buffer (%d bytes)\n",
             gb_sz);
    
    /* for caching waveform bitmap?
       if(shl->graph_pixmap)
       gdk_pixmap_unref(shl->graph_pixmap);

       shl->graph_pixmap = gdk_pixmap_new(widget->window,
       widget->allocation.width,
       snd_frame_count(shl->sr) * shl->vres,
       -1);
    */

    shell_vadjust_set(shl, shl->vadjust->value);
    shell_hadjust_set(shl, shl->hadjust->value);
    shell_hruler_set(shl);

    return TRUE;

}

gboolean 
shell_infocanvas_expose_event(GtkWidget *widget, 
                              GdkEventExpose *event, 
                              shell *shl) {
    draw_info_canvas(widget, &event->area, shl);
    return TRUE;
}

gboolean 
shell_mixercanvas_expose_event(GtkWidget *widget, 
                                   GdkEventExpose *event, 
                                   shell *shl) {
    draw_mixer_canvas(widget, &event->area, shl);
    return TRUE;
}

gboolean 
shell_mixercanvas_button_press_event(GtkWidget *widget,
                                     GdkEventButton *event,
                                     shell *shl) {
    int index_rel = event->y / shl->vres;
    int index_abs = index_rel + shl->vadjust->value;
    int dst_track;
    int rel_x = 0;
    int y = event->y;
    float mixervalue = 0;

    y -= (index_rel * shl->vres) + 1;

    DEBUG("press on track %d, remainder: %d\n", index_abs, y);
    shl->source_channel_being_dragged = -1;
    shl->target_channel_being_dragged = -1;

    if(index_abs < 0 || index_abs >= shl->sr->channels)
        return TRUE;

    dst_track = y / (MIXER_LEVEL_HEIGHT + 1);
    DEBUG("think target channel %d?\n", dst_track);
    
    if(dst_track < 0 || dst_track >= shl->mixer->target_channels)
        return TRUE;

    shl->source_channel_being_dragged = index_abs;
    shl->target_channel_being_dragged = dst_track;

    if(event->x > 7 && event->x < 7 + MIXER_LEVEL_WIDTH) {
        rel_x = event->x - 8;
        mixervalue = (double)rel_x / MIXER_LEVEL_WIDTH;
        DEBUG("think target mix value %f\n", mixervalue);
    } else if(event->x < 8) {
        mixervalue = 0;
    } else {
        mixervalue = 1;
    }
    DEBUG("mixervalue: %f\n", mixervalue);
    shl->mixer->mixtable[dst_track][index_abs] = mixervalue;

    gtk_widget_queue_draw(GTK_WIDGET(shl->mixercanvas));
    return TRUE;
}

gboolean
shell_mixercanvas_motion_notify_event(GtkWidget *widget,
                                      GdkEventMotion *event,
                                      shell *shl) {
    int rel_x;
    float mixervalue = 0;

    if(shl->source_channel_being_dragged == -1 ||
       shl->target_channel_being_dragged == -1)
        return TRUE;

    if(event->x > 7 && event->x < 7 + MIXER_LEVEL_WIDTH) {
        rel_x = event->x - 8;
        mixervalue = (double)rel_x / MIXER_LEVEL_WIDTH;
        DEBUG("think target mix value %f\n", mixervalue);
    } else if(event->x < 8) {
        mixervalue = 0;
    } else {
        mixervalue = 1;
    }
    DEBUG("mixervalue: %f\n", mixervalue);

    shl->mixer->mixtable[shl->target_channel_being_dragged]
        [shl->source_channel_being_dragged] = mixervalue;

    gtk_widget_queue_draw(GTK_WIDGET(shl->mixercanvas));
    return TRUE;
}

gboolean 
shell_mixercanvas_button_release_event(GtkWidget *widget,
                                       GdkEventButton *event,
                                       shell *shl) {
    DEBUG("release\n");
    shl->source_channel_being_dragged = -1;
    shl->target_channel_being_dragged = -1;
    return TRUE;
}

gboolean 
shell_canvas_expose_event(GtkWidget *widget, 
                          GdkEventExpose *event, 
                          shell *shl) {
    draw_canvas(widget, &event->area, shl);
    return TRUE;
}

gboolean 
shell_canvas_button_press_event(GtkWidget *widget,
                                GdkEventButton *event,
                                shell *shl) {
    AFframecount ss, se, mxtol;
    int track_bit, sc = shl->select_channel_map;
    double y, mytol;
    struct marker *mn, *mp, *m = NULL;
    enum marker_type type = MARKER_SLOPE;

    if(!shl->sr)
        return FALSE;

    track_bit = (event->y / shl->vres) + shl->vadjust->value;

    if(track_bit > shl->sr->channels - 1)
        return FALSE;

    /* Check for pending marker drag on previous button
       press/motion. */

    if(shl->marker_being_dragged) {
        DEBUG("releasing marker\n");
        if((shl->marker_being_dragged->flags & MARKER_IS_DISABLED) == 
           MARKER_IS_DISABLED) {
            DEBUG("marker disabled, deleting marker\n");
            marker_list_marker_delete(shl->markers[shl->marker_dragged_on_track],
                                      shl->marker_being_dragged);
        }
        shl->marker_being_dragged = NULL;
        shl->marker_dragged_on_track = -1;
    }

    shl->source_channel_being_dragged = -1;
    shl->target_channel_being_dragged = -1;

    /* Alt key selects MARKER_SLOPE_AUX. */

    if((event->state & GDK_MOD1_MASK) == GDK_MOD1_MASK) 
        type = MARKER_SLOPE_AUX;

    if(shl->envelope_enabled) {

        /* Find nearest marker. */

        mn = marker_list_next(shl->markers[track_bit],
                              MAX(0, PIXEL_TO_FRAME(event->x) - 1),
                              type);
        mp = marker_list_previous(shl->markers[track_bit],
                                  MAX(0, PIXEL_TO_FRAME(event->x) - 1),
                                  type);
        if(mn && !mp)
            m = mn;
        else if(mp && !mn)
            m = mp;
        else if(mp && mn)
            m = DISTANCE(PIXEL_TO_FRAME(event->x), mp->frame_offset) > 
                DISTANCE(PIXEL_TO_FRAME(event->x), mn->frame_offset) ?
                mn : mp;

        if(m) {
            DEBUG("nearest marker:\n");
            marker_dump(m);
            y = (-((event->y - ((track_bit - shl->vadjust->value) * shl->vres)) -
                   (shl->vres / 2)) / (shl->vres / 2));
            mxtol = shl->hres * (MARKER_HANDLE_SIZE / 2);
            if(PIXEL_TO_FRAME(event->x) > m->frame_offset - mxtol &&
               PIXEL_TO_FRAME(event->x) < m->frame_offset + mxtol) {
                DEBUG("is in marker handle x range\n");
                mytol = ((double)1 / (shl->vres / 2)) * (MARKER_HANDLE_SIZE / 2);
                if(y > m->multiplier - mytol &&
                   y < m->multiplier + mytol) {
                    DEBUG("is in marker handle y range, dragging marker on track %d\n",
                          track_bit);
                    shl->marker_being_dragged = m;
                    shl->marker_dragged_on_track = track_bit;
                    return TRUE;
                }
            }
        }
        DEBUG("no marker in the neighbourhood or not in range\n");
        y = (-((event->y - ((track_bit - shl->vadjust->value) * shl->vres)) -
               (shl->vres / 2)) / (shl->vres / 2));
        shl->marker_being_dragged = marker_list_insert(shl->markers[track_bit],
                                                       PIXEL_TO_FRAME(event->x),
                                                       type,
                                                       y,
                                                       NULL);
        DEBUG("new marker: %p, x: %ld, y: %f, track: %d\n", 
              shl->marker_being_dragged, (AFframecount)PIXEL_TO_FRAME(event->x), y, track_bit);
        if(shl->marker_being_dragged)
            shl->marker_dragged_on_track = track_bit;
        marker_list_dump(shl->markers[track_bit]);
        gtk_widget_queue_draw(GTK_WIDGET(shl->canvas));
        return TRUE;
    }

    /* Click + control key modifies which channels are selected. */

    if(event->state & GDK_CONTROL_MASK) {
        shl->select_channel_map = (shl->select_channel_map & (1 << track_bit)) ? 
            shl->select_channel_map & ~(1 << track_bit) : 
            shl->select_channel_map | (1 << track_bit);
        gtk_widget_queue_draw(GTK_WIDGET(shl->canvas));
        return TRUE;
    }
    
    shl->select_flex = PIXEL_TO_FRAME(event->x);
    
    if(shl->select_flex < 0 || shl->select_flex > snd_frame_count(shl->sr))
        return FALSE;
    
    shl->select_pivot = (DISTANCE(shl->select_flex, shl->select_start) >
                         DISTANCE(shl->select_flex, shl->select_end) ? 
                         shl->select_start : shl->select_end);
    
    if(!(event->state & GDK_SHIFT_MASK)) {
        
        sc = (1 << track_bit);
        shl->select_pivot = shl->select_flex = PIXEL_TO_FRAME(event->x);
        
    }

    /* Using an action here gives us undo. */

    ss = MIN(shl->select_pivot, shl->select_flex);
    se = MAX(shl->select_pivot, shl->select_flex);
    
    action_do(ACTION_SELECT_NEW(WITH_UNDO,
                                shl,
                                shl->sr,
                                sc,
                                ss,
                                se - ss));
    return TRUE;
}

gboolean
shell_canvas_motion_notify_event(GtkWidget *widget,
                                 GdkEventMotion *event,
                                 shell *shl) {
    int track_bit = (event->y / shl->vres) + shl->vadjust->value;
    double y = (-((event->y - ((track_bit - shl->vadjust->value) * shl->vres)) -
                  (shl->vres / 2)) / (shl->vres / 2));
    enum marker_type type = MARKER_SLOPE;
    AFframecount mpos;

    /* User came from mixer canvas, is meaningless here. */

    if(shl->source_channel_being_dragged != -1)
        return FALSE;

    shl->last_mouse_x = event->x;
    shl->last_mouse_y = y;

    if(shl->last_mouse_x < 0) {
        event->x = 0;
        shl->last_mouse_x = 0;
    }

    gtk_widget_queue_draw(GTK_WIDGET(shl->infocanvas));

    if((event->state & GDK_MOD1_MASK) == GDK_MOD1_MASK) 
        type = MARKER_SLOPE_AUX;

    if(!shl->sr)
        return FALSE;

    //    if((event->state & GDK_MOD1_MASK) == GDK_MOD1_MASK && 
    //       SLOPE_ENABLED(track_bit, MARKER_SLOPE)) 
    //        return FALSE;
    
    if(!(event->state & GDK_BUTTON1_MASK))
        return FALSE;

    /* Dragging a marker. */

    if(shl->marker_being_dragged) {
        mpos = PIXEL_TO_FRAME(event->x);

        if(shl->snap_to_grid) 
            mpos -= mpos % shl->grid.gap;

        shl->marker_being_dragged->multiplier = y;
        marker_list_marker_position_set(shl->markers[shl->marker_dragged_on_track],
                                        shl->marker_being_dragged,
                                        mpos);
        if(track_bit != shl->marker_dragged_on_track) 
            shl->marker_being_dragged->flags |= MARKER_IS_DISABLED;
        else
            shl->marker_being_dragged->flags &= ~MARKER_IS_DISABLED;
        gtk_widget_queue_draw(GTK_WIDGET(shl->canvas));
        return TRUE;
    }

    if(shl->envelope_enabled)
        return FALSE;

    shl->select_flex = PIXEL_TO_FRAME(event->x);

    if(shl->select_flex < 0)
        shl->select_flex = 0;
    if(shl->select_flex > snd_frame_count(shl->sr))
        shl->select_flex = snd_frame_count(shl->sr);

    shl->select_start = MIN(shl->select_pivot, shl->select_flex);
    shl->select_end = MAX(shl->select_pivot, shl->select_flex);
    gtk_widget_queue_draw(GTK_WIDGET(shl->canvas));

    return TRUE;
}

gboolean 
shell_canvas_button_release_event(GtkWidget *widget,
                                  GdkEventButton *event,
                                  shell *shl) {
    if(!shl->sr)
        return FALSE;

    /* User came from mixer canvas, is meaningless here. */

    if(shl->source_channel_being_dragged != -1)
        return FALSE;

    if(shl->envelope_enabled)
        return FALSE;

    if(event->state & GDK_CONTROL_MASK)
        return FALSE;

    if(event->y > (shl->sr->channels - shl->vadjust->value) * shl->vres)
        return FALSE;

    shl->select_flex = PIXEL_TO_FRAME(event->x);

    if(shl->select_flex < 0)
        shl->select_flex = 0;
    if(shl->select_flex > snd_frame_count(shl->sr))
        shl->select_flex = snd_frame_count(shl->sr);

    shl->select_start = MIN(shl->select_pivot, shl->select_flex);
    shl->select_end = MAX(shl->select_pivot, shl->select_flex);
    gtk_widget_queue_draw(GTK_WIDGET(shl->canvas));

    return TRUE;
}

void
shell_draw_zero_line_toggle_activate(GtkMenuItem *menuitem,
                                     shell *shl) {
    shl->draw_zero = shl->draw_zero ? 0 : 1;
}

void
shell_draw_envelope_toggle_activate(GtkMenuItem *menuitem,
                                    shell *shl) {
    shl->draw_envelope = shl->draw_envelope ? 0 : 1;
    DEBUG("draw_envelope: %d\n", shl->draw_envelope);
    gtk_widget_queue_draw(GTK_WIDGET(shl->canvas));    
}

void
shell_envelope_enable_toggle_activate(GtkMenuItem *menuitem,
                                      shell *shl) {
    int i;
    shl->envelope_enabled = shl->envelope_enabled ? 0 : 1;
    DEBUG("envelope_enabled: %d\n", shl->envelope_enabled);
    if(shl->envelope_enabled)
        for(i = 0; i < MAX_TRACKS; i++)
            shl->markers[i]->marker_types_enabled = 
                MARKER_SLOPE | MARKER_SLOPE_AUX;
    else
        for(i = 0; i < MAX_TRACKS; i++)
            shl->markers[i]->marker_types_enabled = 0;
    
    gtk_widget_queue_draw(GTK_WIDGET(shl->canvas));    
}

void
shell_play_activate(GtkMenuItem *menuitem,
                    shell *shl) {
    if(shl->player.player_running) 
        action_do(ACTION_PLAYER_STOP_NEW(shl));
    else { 
        shl->record_mode = 0;
        action_do(ACTION_PLAYER_PLAY_NEW(shl));
    }
}

void
shell_stop_activate(GtkMenuItem *menuitem,
                    shell *shl) {
    action_do(ACTION_PLAYER_STOP_NEW(shl));
}

void
shell_cue_play_activate(GtkMenuItem *menuitem,
                        shell *shl) {
    AFframecount 
        s = LOOP_IS_ACTIVE(shl) ? shl->loop_start : shl->select_start;
    shl->player.player_pos = s;
    shell_viewport_center(shl, shl->player.player_pos, 
                          shl->player.player_pos);
    
    if(!shl->player.player_running)
        shell_play_activate(menuitem, shl);
}

void
shell_scrub_left_activate(GtkMenuItem *menuitem,
                          shell *shl) {
    shl->player.player_pos = 
        MAX(0, shl->player.player_pos - SCRUB_AMOUNT);
    shell_viewport_center(shl, shl->player.player_pos, shl->player.player_pos);
}

void
shell_scrub_right_activate(GtkMenuItem *menuitem,
                           shell *shl) {
    shl->player.player_pos = MIN(snd_frame_count(shl->sr),
                                 shl->player.player_pos + SCRUB_AMOUNT);
    shell_viewport_center(shl, shl->player.player_pos, 
                          shl->player.player_pos);
}

void
shell_loop_toggle_activate(GtkMenuItem *menuitem,
                           shell *shl) {
    shl->loop = shl->loop ? 0 : 1;
    shell_redraw(shl);
}

void
shell_record_replace_toggle_activate(GtkMenuItem *menuitem,
                                     shell *shl) {
    shl->replace_mode = shl->replace_mode ? 0 : 1;
    shell_redraw(shl);
}

void
shell_record_activate(GtkMenuItem *menuitem,
                      shell *shl) {
    shl->record_mode = 1;
    if(shl->player.player_running) 
        action_do(ACTION_PLAYER_STOP_NEW(shl));
    else 
        action_do(ACTION_PLAYER_PLAY_NEW(shl));
}

void
shell_follow_playback_toggle_activate(GtkMenuItem *menuitem,
                                      shell *shl) {
    shl->follow_playback = shl->follow_playback ? 0 : 1;
    shell_redraw(shl);
}

void
shell_loop_activate(GtkMenuItem *menuitem,
                    shell *shl) {
    shl->loop = shl->loop ? 0 : 1;
}

void 
shell_undo_activate(GtkMenuItem *menuitem,
                    shell *shl) {
    if(!shl->sr)
        return;

    shl->undo_stack = undo_pop(shl->undo_stack);
    shell_status_default_set(shl);
    shell_redraw(shl);
}

void
shell_saveas_activate(GtkWidget *w,
                      shell *shl) {
    action_do(ACTION_FILE_SAVE_AS_NEW(shl));
}

void
shell_save_activate(GtkWidget *w,
                    shell *shl) {
    action_do(ACTION_FILE_SAVE_NEW(shl));
}

void 
shell_copy_activate(GtkMenuItem *menuitem,
                    shell *shl) {
    action_do(ACTION_COPY_NEW(DONT_UNDO,
                              shl,
                              shl->sr,
                              shl->select_channel_map,
                              shl->select_start,
                              shl->select_end - shl->select_start));
}

void 
shell_paste_activate(GtkMenuItem *menuitem,
                     shell *shl) {
    action_do(ACTION_PASTE_NEW(WITH_UNDO,
                               shl,
                               shl->sr,
                               shl->select_channel_map,
                               shl->select_start,
                               shl->select_end - shl->select_start));
}

void 
shell_paste_fit_activate(GtkMenuItem *menuitem,
                         shell *shl) {
    action_do(ACTION_PASTE_FIT_NEW(WITH_UNDO,
                                   shl,
                                   shl->sr,
                                   shl->select_channel_map,
                                   shl->select_start,
                                   shl->select_end - shl->select_start));
}

void 
shell_paste_mix_activate(GtkMenuItem *menuitem,
                         shell *shl) {
    action_do(ACTION_PASTE_MIX_NEW(WITH_UNDO,
                                   shl,
                                   shl->sr,
                                   shl->select_channel_map,
                                   shl->select_start,
                                   shl->select_end - shl->select_start));
}

void 
shell_paste_over_activate(GtkMenuItem *menuitem,
                          shell *shl) {
    action_do(ACTION_PASTE_OVER_NEW(WITH_UNDO,
                                    shl,
                                    shl->sr,
                                    shl->select_channel_map,
                                    shl->select_start,
                                    shl->select_end - shl->select_start));
}

void
shell_insert_tracks_activate(GtkMenuItem *menuitem,
                             shell *shl) {
    int map = 0, i, j = shl->sr->channels;

    for(i = 0; i < shl->sr->channels; i++) 
        if((1 << i) & shl->select_channel_map)
            map |= (1 << j);

    action_group_do(action_group_new(2,
                                     ACTION_TRACKS_INSERT_NEW(WITH_UNDO,
                                                              shl,
                                                              shl->sr,
                                                              NULL,
                                                              map),
                                     ACTION_SELECT_NEW(WITH_UNDO,
                                                       shl,
                                                       shl->sr,
                                                       map,
                                                       shl->select_start,
                                                       shl->select_end - shl->select_start)));
}

void 
shell_delete_tracks_activate(GtkMenuItem *menuitem,
                             shell *shl) {
    action_do(ACTION_TRACKS_DELETE_NEW(WITH_UNDO,
                                       shl,
                                       shl->sr,
                                       shl->select_channel_map));
}

void
shell_cut_activate(GtkMenuItem *menuitem,
                   shell *shl) {
    action_do(ACTION_CUT_NEW(WITH_UNDO,
                             shl,
                             shl->sr,
                             shl->select_channel_map,
                             shl->select_start,
                             shl->select_end - shl->select_start));
}

void 
shell_clear_activate(GtkMenuItem *menuitem,
                     shell *shl) {
    action_do(ACTION_DELETE_NEW(WITH_UNDO,
                                shl,
                                shl->sr,
                                shl->select_channel_map,
                                shl->select_start,
                                shl->select_end - shl->select_start));
}

void 
shell_erase_activate(GtkMenuItem *menuitem,
                     shell *shl) {
    action_do(ACTION_ERASE_NEW(WITH_UNDO,
                               shl,
                               shl->sr,
                               shl->select_channel_map,
                               shl->select_start,
                               shl->select_end - shl->select_start));

}

void 
shell_crop_activate(GtkMenuItem *menuitem,
                    shell *shl) {
    action_group *ag;
    action *a;
    ag = action_group_new(0);
    a = ACTION_DELETE_NEW(WITH_UNDO, 
                          shl,
                          shl->sr,
                          shl->select_channel_map,
                          shl->select_end,
                          snd_frame_count(shl->sr) - shl->select_end);
    ag = action_group_append(ag, a);
    a = ACTION_DELETE_NEW(WITH_UNDO,
                          shl,
                          shl->sr,
                          shl->select_channel_map,
                          0,
                          shl->select_start);
    ag = action_group_append(ag, a);
    a = ACTION_SELECT_NEW(WITH_UNDO,
                          shl,
                          shl->sr,
                          shl->select_channel_map,
                          0,
                          shl->select_end - shl->select_start);
    ag = action_group_append(ag, a);
    action_group_do(ag);
}

void 
shell_insert_markers_activate(GtkMenuItem *menuitem,
                              shell *shl) {
    int t;
    for(t = 0; t < shl->sr->channels; t++) {
        if((1 << t) & shl->select_channel_map) {
            marker_list_insert(shl->markers[t],
                               shl->select_start,
                               MARKER_SLOPE,
                               0,
                               NULL);
            if(shl->select_start != shl->select_end) 
                marker_list_insert(shl->markers[t],
                                   shl->select_end,
                                   MARKER_SLOPE,
                                   0,
                                   NULL);
        }
    }
    shell_redraw(shl);
}

void 
shell_delete_markers_activate(GtkMenuItem *menuitem,
                              shell *shl) {
    int t;
    for(t = 0; t < shl->sr->channels; t++) 
        if((1 << t) & shl->select_channel_map)
            marker_list_delete(shl->markers[t],
                               shl->select_start,
                               shl->select_end - shl->select_start);
    shell_redraw(shl);
}

void 
shell_selection_to_loop_activate(GtkMenuItem *menuitem,
                                 shell *shl) {
    action_do(ACTION_SELECTION_TO_LOOP_NEW(shl));
}

void 
shell_loop_to_selection_activate(GtkMenuItem *menuitem,
                                 shell *shl) {
    action_do(ACTION_LOOP_TO_SELECTION_NEW(shl));
}

void 
shell_select_all_activate(GtkMenuItem *menuitem,
                          shell *shl) {
    action_do(ACTION_SELECT_NEW(WITH_UNDO,
                                shl,
                                shl->sr,
                                shl->select_channel_map,
                                0,
                                snd_frame_count(shl->sr)));
}

void
shell_properties_cancel_clicked(GtkWidget *w,
                                shell *shl) {
    DEBUG("cancel\n");
}

void
shell_properties_ok_clicked(GtkWidget *w,
                            shell *shl) {
    struct properties *props = &shl->props;
    int i, tmp;
    float old_sample_rate = shl->sr->rate;

    shell_cursor_set(shl, GDK_WATCH);
    window_cursor_set(GTK_WIDGET(shl->props.dialog)->window, 
                      GDK_WATCH);
    while(gtk_events_pending())
        gtk_main_iteration_do(FALSE);
    snd_convert(shl->sr,
                props->sample_width,
                props->sample_rate);
    if(RESULT_IS_ERROR(shl->sr)) {
        gui_alert("Could not convert %s: %s", 
                  shl->sr->name, shl->sr->error_str);
        snd_error_free(shl->sr);
        window_cursor_set(GTK_WIDGET(shl->props.dialog)->window, 
                          GDK_LEFT_PTR);
        shell_cursor_set(shl, GDK_LEFT_PTR);
        return;
    }
    grid_rate_set(&shl->grid, shl->sr->rate);

    shell_redraw(shl);
    window_cursor_set(GTK_WIDGET(shl->props.dialog)->window, 
                      GDK_LEFT_PTR);
    shell_cursor_set(shl, GDK_LEFT_PTR);

    if(!gtk_toggle_button_get_active(props->resample))
        return;

    if(props->sample_rate == old_sample_rate)
        return;

    shell_cursor_set(shl, GDK_WATCH);
    window_cursor_set(GTK_WIDGET(shl->props.dialog)->window, 
                      GDK_WATCH);
    shell_status_push(shl, "Please wait, resampling to %.2fhz...",
                      props->sample_rate);
    for(i = 0; i < shl->sr->channels; i++) {
        tmp = shl->markers[i]->marker_types_enabled;
        shl->markers[i]->marker_types_enabled = 0;
        resample(shl,
                 i,
                 0,
                 snd_frame_count(shl->sr),
                 (double)shl->sr->rate / old_sample_rate);
        shl->markers[i]->marker_types_enabled = tmp;
    }
    shell_status_pop(shl);
    gtk_toggle_button_set_active(props->resample, FALSE);    
    window_cursor_set(GTK_WIDGET(shl->props.dialog)->window, 
                      GDK_LEFT_PTR);
    shell_cursor_set(shl, GDK_LEFT_PTR);
    shell_redraw(shl);
}

void
shell_properties_size_labels_set(shell *shl) {
    char s[50];
    size_t current_size, new_size;
    struct properties *props = &shl->props;
    int resample_toggle = gtk_toggle_button_get_active(props->resample);
    current_size = shl->sr->channels * 
        snd_frame_count(shl->sr) *
        shl->sr->frame_width;
    new_size = ceil(shl->sr->channels * 
                    snd_frame_count(shl->sr) *
                    props->sample_width *
                    (resample_toggle ?
                     (props->sample_rate / shl->sr->rate) :
                     1));
    snprintf(s, 50, "%d", current_size);
    gtk_label_set_text(props->cur_label, s);
    snprintf(s, 50, "%d", new_size);
    gtk_label_set_text(props->new_label, s);
}

void
shell_properties_sample_rate_changed(GtkSpinButton *w,
                                     shell *shl) {
    DEBUG("sample rate\n");
    shl->props.sample_rate = 
        gtk_spin_button_get_value_as_float(w);
    shell_properties_size_labels_set(shl);
}

void
shell_properties_sample_width_8_clicked(GtkToggleButton *w,
                                        shell *shl) {
    DEBUG("sample width 8\n");
    if(gtk_toggle_button_get_active(w)) 
        shl->props.sample_width = 1;        
    shell_properties_size_labels_set(shl);
}

void
shell_properties_sample_width_16_clicked(GtkToggleButton *w,
                                         shell *shl) {
    DEBUG("sample width 16\n");
    if(gtk_toggle_button_get_active(w)) 
        shl->props.sample_width = 2;
    shell_properties_size_labels_set(shl);
}

void
shell_properties_sample_width_32_clicked(GtkToggleButton *w,
                                         shell *shl) {
    DEBUG("sample width 32\n");
    if(gtk_toggle_button_get_active(w)) 
        shl->props.sample_width = 4;
    shell_properties_size_labels_set(shl);
}

void
shell_properties_resample_toggled(GtkWidget *w,
                                  shell *shl) {
    DEBUG("resample\n");
    shell_properties_size_labels_set(shl);
}

void
shell_properties_activate(GtkMenuItem *menuitem,
                          shell *shl) {
    shl->props.sample_rate = shl->sr->rate;
    shl->props.sample_width = shl->sr->frame_width;
    shell_properties_size_labels_set(shl);
    gtk_widget_show(GTK_WIDGET(shl->props.dialog));
}

void
shell_fit_selection_to_window_activate(GtkMenuItem *menuitem,
                                       shell *shl) {
    action_do(ACTION_SELECTION_FIT_NEW(shl));
}

void
shell_selection_to_4_beats_activate(GtkWidget *w,
                                    shell *shl) {

    DEBUG("blah blah\n");
    if(shl->select_end == shl->select_start)
        return;

    gtk_option_menu_set_history(shl->gridmeasurement, GRID_BEATS);
    shell_grid_measurement_changed(shl->gridmeasurement, shl);
    grid_bpm_from_frames_set(&shl->grid, (shl->select_end - shl->select_start) / 4);
    gtk_spin_button_set_value(shl->gridbpm, shl->grid.bpm);
    shell_grid_configure(shl);
    shell_redraw(shl);
}

void
shell_zoom_16_1_activate(GtkWidget *w,
                         shell *shl) {
    shell_hres_set(shl, 0.0625);
}

void
shell_zoom_8_1_activate(GtkWidget *w,
                        shell *shl) {
    shell_hres_set(shl, 0.125);
}

void
shell_zoom_4_1_activate(GtkWidget *w,
                        shell *shl) {
    shell_hres_set(shl, 0.25);
}

void
shell_zoom_2_1_activate(GtkWidget *w,
                        shell *shl) {
    shell_hres_set(shl, 0.5);
}

void
shell_zoom_1_1_activate(GtkWidget *w,
                        shell *shl) {
    shell_hres_set(shl, 1);
}

void
shell_zoom_1_8_activate(GtkWidget *w,
                        shell *shl) {
    shell_hres_set(shl, 8);
}

void
shell_zoom_1_32_activate(GtkWidget *w,
                         shell *shl) {
    shell_hres_set(shl, 32);
}

void
shell_zoom_1_64_activate(GtkWidget *w,
                         shell *shl) {
    shell_hres_set(shl, 64);
}

void
shell_zoom_1_128_activate(GtkWidget *w,
                          shell *shl) {
    shell_hres_set(shl, 128);
}

void
shell_zoom_1_256_activate(GtkWidget *w,
                          shell *shl) {
    shell_hres_set(shl, 256);
}

void
shell_zoom_1_512_activate(GtkWidget *w,
                          shell *shl) {
    shell_hres_set(shl, 512);
}

void
shell_zoom_1_1024_activate(GtkWidget *w,
                           shell *shl) {
    shell_hres_set(shl, 1024);
}

void
shell_zoom_1_2048_activate(GtkWidget *w,
                           shell *shl) {
    shell_hres_set(shl, 2048);
}

void
shell_zoom_1_4096_activate(GtkWidget *w,
                           shell *shl) {
    shell_hres_set(shl, 4096);
}

void
shell_zoom_in_activate(GtkWidget *w,
                       shell *shl) {
    shl->hres -= (shl->hres / 2);
    if(shl->hres < HRES_MIN)
        shl->hres = HRES_MIN;
    shell_hres_set(shl, shl->hres);
}

void
shell_zoom_out_activate(GtkWidget *w,
                        shell *shl) {
    shl->hres += shl->hres;
    if(shl->hres > HRES_MAX)
        shl->hres = HRES_MAX;
    shell_hres_set(shl, shl->hres);
}

void
shell_small_activate(GtkWidget *w,
                     shell *shl) {
    shell_vres_set(shl, 64);
}

void
shell_medium_activate(GtkWidget *w,
                      shell *shl) {
    shell_vres_set(shl, 128);
}

void
shell_large_activate(GtkWidget *w,
                     shell *shl) {
    shell_vres_set(shl, 256);
}

void
shell_left_activate(GtkMenuItem *menuitem,
                    shell *shl) {
    AFframecount 
        ss = (shl->select_start == shl->select_end ? 
              MAX(0, shl->select_start - MAX(1, shl->hres)) :
              shl->select_start);
    
    if(shl->snap_to_grid) 
        ss -= ss % shl->grid.gap;

    action_do(ACTION_SELECT_NEW(WITH_UNDO,
                                shl,
                                shl->sr,
                                shl->select_channel_map,
                                ss,
                                0));
}

void
shell_right_activate(GtkMenuItem *menuitem,
                     shell *shl) {
    AFframecount ss;

    ss = (shl->select_start == shl->select_end ?                                
          MIN(snd_frame_count(shl->sr), shl->select_end + 
              MAX(1, shl->hres)) : 
          shl->select_end);

    if(shl->snap_to_grid) {
        ss -= ss % shl->grid.gap;
        ss += shl->grid.gap;
        ss = MIN(snd_frame_count(shl->sr), ss);
    }

    action_do(ACTION_SELECT_NEW(WITH_UNDO,
                                shl,
                                shl->sr,
                                shl->select_channel_map,
                                ss,
                                0));
}

void
shell_track_up_activate(GtkMenuItem *menuitem,
                        shell *shl) {
    int first_sel_track = 1, i;
    for(i = 0; i < shl->sr->channels; i++) {
        if((1 << i) & shl->select_channel_map) {
            first_sel_track = i;
            break;
        }
    }
    
    action_do(ACTION_SELECT_NEW(DONT_UNDO,
                                shl,
                                shl->sr,
                                1 << MAX(first_sel_track - 1, 0),
                                shl->select_start,
                                0));
}

void
shell_track_down_activate(GtkMenuItem *menuitem,
                          shell *shl) {
    int last_sel_track = 0, i;
    for(i = 0; i < shl->sr->channels; i++)
        if((1 << i) & shl->select_channel_map)
            last_sel_track = i;
    
    action_do(ACTION_SELECT_NEW(DONT_UNDO,
                                shl,
                                shl->sr,
                                1 << MIN(last_sel_track + 1, 
                                         shl->sr->channels - 1),
                                shl->select_start,
                                0));
}

void
shell_left_select_activate(GtkMenuItem *menuitem,
                           shell *shl) {
    AFframecount ss, se;

    ss = MAX(0, shl->select_start - MAX(1, shl->hres));
    se = shl->select_end;
    
    if(shl->snap_to_grid) 
        ss = MAX(0, ((ss + shl->grid.gap) - 
                     (ss % shl->grid.gap)) - shl->grid.gap);

    action_do(ACTION_SELECT_NEW(DONT_UNDO,
                                shl,
                                shl->sr,
                                shl->select_channel_map,
                                ss,
                                se - ss));
}

void
shell_right_select_activate(GtkMenuItem *menuitem,
                            shell *shl) {
    AFframecount ss, se;

    ss = shl->select_start;
    se = MIN(snd_frame_count(shl->sr), 
             shl->select_end + MAX(1, shl->hres));

    if(shl->snap_to_grid)
        se = MIN(snd_frame_count(shl->sr), 
                 (se - (se % shl->grid.gap)) + shl->grid.gap);

    action_do(ACTION_SELECT_NEW(DONT_UNDO,
                                shl,
                                shl->sr,
                                shl->select_channel_map,
                                ss,
                                se - ss));
}

void
shell_track_up_select_activate(GtkMenuItem *menuitem,
                               shell *shl) {
    int next_sel_track = shl->sr->channels - 1, i;
    for(i = 0; i < shl->sr->channels; i++) {
        if((1 << i) & shl->select_channel_map) {
            next_sel_track = MAX(0, i - 1);
            break;
        }
    }
    
    action_do(ACTION_SELECT_NEW(DONT_UNDO,
                                shl,
                                shl->sr,
                                (1 << next_sel_track) | shl->select_channel_map,
                                shl->select_start,
                                shl->select_end - shl->select_start));
}

void
shell_track_down_select_activate(GtkMenuItem *menuitem,
                                 shell *shl) {
    int next_sel_track = 0, i;
    for(i = 0; i < shl->sr->channels; i++)
        if((1 << i) & shl->select_channel_map)
            next_sel_track = i + 1;
    
    if(next_sel_track >= shl->sr->channels)
        return;

    action_do(ACTION_SELECT_NEW(DONT_UNDO,
                                shl,
                                shl->sr,
                                (1 << next_sel_track) | shl->select_channel_map,
                                shl->select_start,
                                shl->select_end - shl->select_start));
}

void
shell_left_transfer_activate(GtkMenuItem *menuitem,
                             shell *shl) {
    action_do(ACTION_SELECT_NEW(DONT_UNDO,
                                shl,
                                shl->sr,
                                shl->select_channel_map,
                                MAX(0, shl->select_start - MAX(1, shl->hres)),
                                shl->select_end - shl->select_start));
}

void
shell_right_transfer_activate(GtkMenuItem *menuitem,
                              shell *shl) {
    action_do(ACTION_SELECT_NEW(DONT_UNDO,
                                shl,
                                shl->sr,
                                shl->select_channel_map,
                                shl->select_start + MAX(1, shl->hres),
                                shl->select_end - shl->select_start));
}

void
shell_left_jump_activate(GtkMenuItem *menuitem,
                         shell *shl) {
    action_do(ACTION_SELECT_NEW(DONT_UNDO,
                                shl,
                                shl->sr,
                                shl->select_channel_map,
                                MAX(0, shl->select_start - 
                                    (shl->select_end - shl->select_start)),
                                shl->select_end - shl->select_start));
}

void
shell_right_jump_activate(GtkMenuItem *menuitem,
                          shell *shl) {
    action_do(ACTION_SELECT_NEW(DONT_UNDO,
                                shl,
                                shl->sr,
                                shl->select_channel_map,
                                shl->select_start +
                                (shl->select_end - shl->select_start),
                                shl->select_end - shl->select_start));
}

void
shell_module_cancel_activate(GtkMenuItem *menuitem,
                             shell *shl) {
    shl->module_cancel_requested = 1;
}

void 
shell_module_activate(GtkMenuItem *menuitem,
                      shell *shl) {
    int module = GPOINTER_TO_INT(gtk_object_get_user_data(GTK_OBJECT(menuitem)));
    action_do(ACTION_MODULE_OPEN_NEW(WITH_UNDO,
                                     shl,
                                     module));
    
}

void
shell_mixer_dump_activate(GtkWidget *w,
                          shell *shl) {
    mixer_dump(shl->mixer);
}

void
shell_blocks_dump_activate(GtkWidget *w,
                           shell *shl) {
    int i;
    if(!shl->sr)
        return;

    for(i = 0; i < shl->sr->channels; i++) {
        DEBUG("blocks for track %d\n", i);
        blocklist_dump(shl->sr->tracks[i]->bl);
    }
}

void
shell_sound_info_dump_activate(GtkWidget *w,
                               shell *shl) {
    if(!shl->sr)
        return;

    snd_info_dump(shl->sr);
}

void
shell_join_blocks_activate(GtkWidget *w,
                           shell *shl) {
    //    int i;
    if(!shl->sr)
        return;
    /*
    rwlock_rlock(&shl->sr->rwl);
    for(i = 0; i < shl->sr->channels; i++)
        track_compact(shl->sr->tracks[i]);
    rwlock_runlock(&shl->sr->rwl);
    */
}

void
shell_fail_next_allocation_toggle_activate(GtkWidget *w,
                                           shell *shl) {
    mem_fail_allocation_on_zero = mem_fail_allocation_on_zero > 0 ? 0 : 1;
    DEBUG("mem_fail_allocation_on_zero: %d\n",
          mem_fail_allocation_on_zero);
}

void
shell_step_mode_toggle_activate(GtkWidget *w,
                                  shell *shl) {
    if(shl->debug_flags & DEBUG_FLAG_STEP_MODE)
        shl->debug_flags &= ~DEBUG_FLAG_STEP_MODE;
    else
        shl->debug_flags |= DEBUG_FLAG_STEP_MODE;
    shell_redraw(shl);
}

void
shell_draw_blocks_toggle_activate(GtkWidget *w,
                                  shell *shl) {
    if(shl->debug_flags & DEBUG_FLAG_DRAW_BLOCKS)
        shl->debug_flags &= ~DEBUG_FLAG_DRAW_BLOCKS;
    else
        shl->debug_flags |= DEBUG_FLAG_DRAW_BLOCKS;
    shell_redraw(shl);
}

void
shell_about_activate(GtkWidget *w,
                     shell *shl) {
    const gchar *authors[] = {
        "Pascal Haakmat",
        NULL
    };
    gtk_widget_show(gnome_about_new("GNUsound", VERSION,
                                    "Copyright 2002 Pascal Haakmat",
                                    (const gchar **) authors,
                                    _("GNOME U sound editor"),
                                    NULL));

}

void
shell_destroy_undo_activate(GtkWidget *w,
                            shell *shl) {
    undo_destroy(shl->undo_stack);
    shl->undo_stack = NULL;
}

gboolean
shell_close(shell *shl) {
    action_result *ar = action_do(ACTION_FILE_CLOSE_NEW(shl));
    return action_result_as_bool(ar);
}

gboolean
shell_delete_event(GtkWidget *w,
                   GdkEvent *ev,
                   shell *shl) {
    return shell_close(shl);
}

gboolean
shell_delete_signal(GtkWidget *w,
                    shell *shl) {
    return shell_close(shl);
}

void
shell_bindings_free(shell *shl) {
    if(!shl->bindings)
        return;

    free(shl->bindings);
}

void
shell_modules_close(shell *shl) {
    int i;
    void (*module_close_func)(struct _module_state *module_state);
    char *error;
    for(i = 0; i < module_count; i++) {
        DEBUG("module %d: is_open: %d\n", i, shl->module_state[i].is_open);
        if(shl->module_state[i].is_open) {
            module_close_func = dlsym(modules[i].handle, "module_close");
            if((error = dlerror()) != NULL) {
                FAIL("module %s is open, " \
                     "but an error occurred trying to close it: %s\n",
                     modules[i].name, error);
            } else {
                if(module_close_func) {
                    DEBUG("closing module %s\n", modules[i].name);
                    (*module_close_func)(&shl->module_state[i]);
                }
            }
        }
    }
}

void
shell_destroy(shell *shl) {
    int i;
    player_stop(shl);
    pthread_mutex_destroy(&shl->player.player_running_lock);
    pthread_cond_destroy(&shl->player.player_running_cond);
    shell_modules_close(shl);
    undo_destroy(shl->undo_stack);
    mixer_destroy(shl->mixer);
    for(i = 0; i < MAX_TRACKS; i++)
        marker_list_destroy(shl->markers[i]);
    if(shl == clipboard_shell)
        clipboard_shell = NULL;
    else 
        snd_destroy(shl->sr);
    shells = g_list_remove(shells, shl);
    if(shl->pixmap)
        gdk_pixmap_unref(shl->pixmap);
    if(shl->graph_bits_buffer)
        free(shl->graph_bits_buffer);
    shell_bindings_free(shl);
    if(!g_list_length(shells))
        gtk_main_quit();
}

void
shell_destroy_signal(void *obj,
                     shell *shl) {
    shell_destroy(shl);
}

void
shell_status_pop(shell *shl) {
    gnome_appbar_pop(shl->appbar);
}

void
shell_status_push(shell *shl, 
                  const char *format, 
                  ...) {
    char status[4096];
    va_list ap;

    va_start(ap, format);
    vsnprintf(status, 4096, format, ap);
    va_end(ap);

    gnome_appbar_push(shl->appbar, status);
}

void
shell_status_default_set(shell *shl) {
    if(!shl->sr)
        return;

    shell_status_pop(shl);
    shell_status_push(shl,
                      "%s  %.2f sec.  %ld frames", 
                      shell_path_strip(shl->sr->name),
                      snd_frames_to_time(shl->sr, snd_frame_count(shl->sr)), 
                      snd_frame_count(shl->sr));
    shell_properties_size_labels_set(shl);
}

signal_bindings bindings[] = {

    /* Basic window and canvas events. */

    { "canvas", "expose_event", (GtkSignalFunc)shell_canvas_expose_event },
    { "canvas", "configure_event", (GtkSignalFunc)shell_canvas_configure_event },
    { "canvas", "button_press_event", (GtkSignalFunc)shell_canvas_button_press_event },
    { "canvas", "button_release_event", (GtkSignalFunc)shell_canvas_button_release_event },
    { "canvas", "motion_notify_event", (GtkSignalFunc)shell_canvas_motion_notify_event },
    { "infocanvas", "expose_event", (GtkSignalFunc)shell_infocanvas_expose_event },
    { "mixercanvas", "expose_event", (GtkSignalFunc)shell_mixercanvas_expose_event },
    { "mixercanvas", "configure_event", (GtkSignalFunc)shell_mixercanvas_configure_event },
    { "mixercanvas", "button_press_event", (GtkSignalFunc)shell_mixercanvas_button_press_event },
    { "mixercanvas", "button_release_event", (GtkSignalFunc)shell_mixercanvas_button_release_event },
    { "mixercanvas", "motion_notify_event", (GtkSignalFunc)shell_mixercanvas_motion_notify_event },
    { "appwindow", "size_allocate", (GtkSignalFunc)shell_size_allocate_signal },
    { "appwindow", "delete_event", (GtkSignalFunc)shell_delete_event },
    { "appwindow", "focus_in_event", (GtkSignalFunc)shell_focus_in_event },
    { "appwindow", "focus_out_event", (GtkSignalFunc)shell_focus_out_event },
    { "appwindow", "destroy", (GtkSignalFunc)shell_destroy_signal },

    /* Toolbar buttons. */
    
    { "b_newfile", "clicked", (GtkSignalFunc)gui_new_activate },
    { "b_open", "clicked", (GtkSignalFunc)gui_open_activate },
    { "b_save", "clicked", (GtkSignalFunc)shell_save_activate },
    { "b_cut", "clicked", (GtkSignalFunc)shell_cut_activate },
    { "b_copy", "clicked", (GtkSignalFunc)shell_copy_activate },
    { "b_paste", "clicked", (GtkSignalFunc)shell_paste_activate },
    { "b_clear", "clicked", (GtkSignalFunc)shell_clear_activate },
    { "b_play", "clicked", (GtkSignalFunc)shell_play_activate },
    { "b_cue_play", "clicked", (GtkSignalFunc)shell_cue_play_activate },
    { "b_stop", "clicked", (GtkSignalFunc)shell_stop_activate },
    { "b_scrub_left", "clicked", (GtkSignalFunc)shell_scrub_left_activate },
    { "b_scrub_right", "clicked", (GtkSignalFunc)shell_scrub_right_activate },
    { "b_record", "clicked", (GtkSignalFunc)shell_record_activate },
    
    /* Grid toolbar. */

    { "grid_units", "changed", (GtkSignalFunc)shell_grid_units_changed },
    { "grid_bpm", "changed", (GtkSignalFunc)shell_grid_bpm_changed },

    /* File menu. */

    { "newfile", "activate", (GtkSignalFunc)gui_new_activate },
    { "open", "activate", (GtkSignalFunc)gui_open_activate },
    { "save", "activate", (GtkSignalFunc)shell_save_activate },
    { "saveas", "activate", (GtkSignalFunc)shell_saveas_activate },
    { "close", "activate", (GtkSignalFunc)shell_delete_signal },
    { "exit", "activate", (GtkSignalFunc)gui_exit_activate },

    /* Edit menu. */

    { "undo", "activate", (GtkSignalFunc)shell_undo_activate },
    { "cut", "activate", (GtkSignalFunc)shell_cut_activate },
    { "paste", "activate", (GtkSignalFunc)shell_paste_activate },
    { "paste_fit", "activate", (GtkSignalFunc)shell_paste_fit_activate },
    { "paste_over", "activate", (GtkSignalFunc)shell_paste_over_activate },
    { "paste_mix", "activate", (GtkSignalFunc)shell_paste_mix_activate },
    { "copy", "activate", (GtkSignalFunc)shell_copy_activate },
    { "clear", "activate", (GtkSignalFunc)shell_clear_activate },
    { "erase", "activate", (GtkSignalFunc)shell_erase_activate },
    { "crop", "activate", (GtkSignalFunc)shell_crop_activate },
    { "insert_tracks", "activate", (GtkSignalFunc)shell_insert_tracks_activate },
    { "delete_tracks", "activate", (GtkSignalFunc)shell_delete_tracks_activate },
    { "insert_markers", "activate", (GtkSignalFunc)shell_insert_markers_activate },
    { "delete_markers", "activate", (GtkSignalFunc)shell_delete_markers_activate },
    { "select_all", "activate", (GtkSignalFunc)shell_select_all_activate },
    { "selection_to_loop", "activate", (GtkSignalFunc)shell_selection_to_loop_activate },
    { "loop_to_selection", "activate", (GtkSignalFunc)shell_loop_to_selection_activate },
    { "fit_selection_to_window", "activate", (GtkSignalFunc)shell_fit_selection_to_window_activate },
    { "show_clipboard", "activate", (GtkSignalFunc)gui_show_clipboard_activate },
    { "properties", "activate", (GtkSignalFunc)shell_properties_activate },

    /* Properties dialog. */

    { "resample", "toggled", (GtkSignalFunc)shell_properties_resample_toggled },
    { "sample_rate", "changed", (GtkSignalFunc)shell_properties_sample_rate_changed },
    { "sample_width_8", "clicked", (GtkSignalFunc)shell_properties_sample_width_8_clicked },
    { "sample_width_16", "clicked", (GtkSignalFunc)shell_properties_sample_width_16_clicked },
    { "sample_width_32", "clicked", (GtkSignalFunc)shell_properties_sample_width_32_clicked },
    { "ok", "clicked", (GtkSignalFunc)shell_properties_ok_clicked },
    { "cancel", "clicked", (GtkSignalFunc)shell_properties_cancel_clicked },

    /* View menu. */

    { "16_1", "activate", (GtkSignalFunc)shell_zoom_16_1_activate },
    { "8_1", "activate", (GtkSignalFunc)shell_zoom_8_1_activate },
    { "4_1", "activate", (GtkSignalFunc)shell_zoom_4_1_activate },
    { "2_1", "activate", (GtkSignalFunc)shell_zoom_2_1_activate },
    { "1_1", "activate", (GtkSignalFunc)shell_zoom_1_1_activate },
    { "1_8", "activate", (GtkSignalFunc)shell_zoom_1_8_activate },
    { "1_32", "activate", (GtkSignalFunc)shell_zoom_1_32_activate },
    { "1_64", "activate", (GtkSignalFunc)shell_zoom_1_64_activate },
    { "1_128", "activate", (GtkSignalFunc)shell_zoom_1_128_activate },
    { "1_256", "activate", (GtkSignalFunc)shell_zoom_1_256_activate },
    { "1_512", "activate", (GtkSignalFunc)shell_zoom_1_512_activate },
    { "1_1024", "activate", (GtkSignalFunc)shell_zoom_1_1024_activate },
    { "1_2048", "activate", (GtkSignalFunc)shell_zoom_1_2048_activate },
    { "1_4096", "activate", (GtkSignalFunc)shell_zoom_1_4096_activate },
    { "zoom_in", "activate", (GtkSignalFunc)shell_zoom_in_activate },
    { "zoom_out", "activate", (GtkSignalFunc)shell_zoom_out_activate },
    { "small", "activate", (GtkSignalFunc)shell_small_activate },
    { "medium", "activate", (GtkSignalFunc)shell_medium_activate },
    { "large", "activate", (GtkSignalFunc)shell_large_activate },
    { "draw_zero_line_toggle", "activate", (GtkSignalFunc)shell_draw_zero_line_toggle_activate },
    { "draw_envelope_toggle", "activate", (GtkSignalFunc)shell_draw_envelope_toggle_activate },

    /* Select menu. */
    
    { "left", "activate", (GtkSignalFunc)shell_left_activate },
    { "right", "activate", (GtkSignalFunc)shell_right_activate },
    { "track_up", "activate", (GtkSignalFunc)shell_track_up_activate },
    { "track_down", "activate", (GtkSignalFunc)shell_track_down_activate },
    { "left_select", "activate", (GtkSignalFunc)shell_left_select_activate },
    { "right_select", "activate", (GtkSignalFunc)shell_right_select_activate },
    { "track_up_select", "activate", (GtkSignalFunc)shell_track_up_select_activate },
    { "track_down_select", "activate", (GtkSignalFunc)shell_track_down_select_activate },
    { "left_transfer", "activate", (GtkSignalFunc)shell_left_transfer_activate },
    { "right_transfer", "activate", (GtkSignalFunc)shell_right_transfer_activate },
    { "left_jump", "activate", (GtkSignalFunc)shell_left_jump_activate },
    { "right_jump", "activate", (GtkSignalFunc)shell_right_jump_activate },
    { "draw_grid_toggle", "activate", (GtkSignalFunc)shell_draw_grid_toggle_activate },
    { "snap_to_grid_toggle", "activate", (GtkSignalFunc)shell_snap_to_grid_toggle_activate },
    { "envelope_enable_toggle", "activate", (GtkSignalFunc)shell_envelope_enable_toggle_activate },
    { "selection_to_4_beats", "activate", (GtkSignalFunc)shell_selection_to_4_beats_activate },

    /* Playback menu. */

    { "play", "activate", (GtkSignalFunc)shell_play_activate },
    { "cue_play", "activate", (GtkSignalFunc)shell_cue_play_activate },
    { "record", "activate", (GtkSignalFunc)shell_record_activate },
    { "scrub_left", "activate", (GtkSignalFunc)shell_scrub_left_activate },
    { "scrub_right", "activate", (GtkSignalFunc)shell_scrub_right_activate },
    { "loop_toggle", "activate", (GtkSignalFunc)shell_loop_toggle_activate },
    { "follow_playback_toggle", "activate", (GtkSignalFunc)shell_follow_playback_toggle_activate },
    { "record_replace_toggle", "activate", (GtkSignalFunc)shell_record_replace_toggle_activate },

    /* Debug menu. */

    { "dump_blocks", "activate", (GtkSignalFunc)shell_blocks_dump_activate },
    { "dump_sound_info", "activate", (GtkSignalFunc)shell_sound_info_dump_activate },
    { "dump_mixer", "activate", (GtkSignalFunc)shell_mixer_dump_activate },
    { "destroy_undo", "activate", (GtkSignalFunc)shell_destroy_undo_activate },
    { "draw_blocks_toggle", "activate", (GtkSignalFunc)shell_draw_blocks_toggle_activate },
    { "step_mode_toggle", "activate", (GtkSignalFunc)shell_step_mode_toggle_activate },
    { "join_blocks", "activate", (GtkSignalFunc)shell_join_blocks_activate },
    { "fail_next_allocation_toggle", "activate", (GtkSignalFunc)shell_fail_next_allocation_toggle_activate },

    /* Help menu. */

    { "about", "activate", (GtkSignalFunc)shell_about_activate },
    { NULL, NULL, NULL }
};

int
shell_bindings_alloc(shell *shl) {
    int i;
    for(i = 0; bindings[i].widget_name; i++);

    shl->bindings = malloc(i * sizeof(action_bindings));

    if(!shl->bindings) {
        FAIL("could not allocate %d bindings\n", i);
        return -1;
    }

    for(i = 0; bindings[i].widget_name; i++) {
        shl->bindings[i].object = NULL;
        shl->bindings[i].ref = 0;
    }

    return 0;
}

GtkObject *
shell_bindings_find(shell *shl,
                    const char *name) {
    int i;
    for(i = 0; bindings[i].widget_name; i++) 
        if(!strcmp(name, bindings[i].widget_name))
            return shl->bindings[i].object;
    FAIL("cannot find binding for %s\n", name);
    abort();
}

void
shell_bindings_disable(shell *shl,
                       char *names[]) {
    int i;
    GtkWidget *w;
    
    for(i = 0; names[i]; i++) {
        w = GTK_WIDGET(shell_bindings_find(shl, names[i]));
        gtk_widget_hide(w);
    }
}

void
shell_signals_connect(shell *shl,
                      GladeXML *xml) {
    int i;
    GtkObject *o;

    for(i = 0; bindings[i].widget_name; i++) {
        o = GTK_OBJECT(glade_xml_get_widget(xml, bindings[i].widget_name));
        if(!o) {
            FAIL("cannot bind %s\n", bindings[i].widget_name);
            abort();
        }
        shl->bindings[i].object = o;
        gtk_signal_connect(o, 
                           bindings[i].signal_name,
                           bindings[i].signal_func, 
                           shl);
    }
}

shell *
shell_new() {
    GladeXML *xml;
    GdkEventConfigure cfg; 
    GtkMenuItem *tools_trigger, *debug_trigger, *separator;
    GtkMenu *tools;
    GtkMenuItem *item;
    GtkToolbar *toolbar;
    GList *iterator;
    shell *shl;
    struct stat statbuf;
    int i, err;

    shl = calloc(sizeof(shell), 1);

    if(!shl) {
        FAIL("could not allocate memory for new shell object\n");
        return NULL;
    }
    if(shell_bindings_alloc(shl)) {
        FAIL("could not alloc shell bindings\n");
        shell_bindings_free(shl);
        return NULL;
    }

    shells = g_list_append(shells, shl);

    shl->undo_stack = NULL;
    shl->mixer = mixer_new(0, 0);
    shl->hres = 128;
    shl->vres = 128;
    shl->sr = NULL;
    shl->select_start = shl->select_end = shl->select_pivot = shl->select_flex = 0;
    shl->select_channel_map = 1;
    shl->player.player_pos = 0;
    shl->record_mode = 0;
    shl->replace_mode = 0;
    pthread_mutex_init(&shl->player.player_running_lock, NULL);
    pthread_cond_init(&shl->player.player_running_cond, NULL);
    shl->draw_zero = 1;
    shl->follow_playback = 1;
    shl->loop = 1;
    shl->show_grid = 1;
    shl->draw_envelope = 1;
    shl->envelope_enabled = 0;
    shl->snap_to_grid = 0;
    shl->pixmap = NULL;
    shl->graph_bits_buffer = NULL;
    shl->active_module_id = -1;
    shl->module_cancel_requested = 0;
    grid_bpm_set(&shl->grid, 120);
    grid_units_set(&shl->grid, 1);
    grid_rate_set(&shl->grid, DEFAULT_SAMPLE_RATE);
    grid_measurement_set(&shl->grid, GRID_SECONDS);

    for(i = 0; i < MAX_TRACKS; i++)
        shl->markers[i] = marker_list_new();
    shl->marker_being_dragged = NULL;
    shl->marker_dragged_on_track = -1;
    shl->source_channel_being_dragged = -1;
    shl->target_channel_being_dragged = -1;
    shl->last_mouse_y = 0;

    err = stat(PRIMARY_GLADE_FILE, &statbuf);
    if(!err) 
        xml = glade_xml_new(PRIMARY_GLADE_FILE, NULL);
    else
        xml = glade_xml_new(SECONDARY_GLADE_FILE, NULL);

    if(!xml) {
        FAIL("could not find interface definition, " "looked at %s:%s.", 
             PRIMARY_GLADE_FILE, SECONDARY_GLADE_FILE);
        return NULL;
    }

    shl->appwindow = GNOME_APP(glade_xml_get_widget(xml, "appwindow"));
    shl->props.dialog = GNOME_DIALOG(glade_xml_get_widget(xml, "propertiesdialog"));
    shl->props.cur_label = GTK_LABEL(glade_xml_get_widget(xml, "current_size"));
    shl->props.new_label = GTK_LABEL(glade_xml_get_widget(xml, "new_size"));
    shl->props.resample = GTK_TOGGLE_BUTTON(glade_xml_get_widget(xml, "resample"));
    shl->appbar = GNOME_APPBAR(glade_xml_get_widget(xml, "appbar"));
    shl->canvas = GTK_DRAWING_AREA(glade_xml_get_widget(xml, "canvas")); 
    shl->infocanvas = GTK_DRAWING_AREA(glade_xml_get_widget(xml, "infocanvas")); 
    shl->mixercanvas = GTK_DRAWING_AREA(glade_xml_get_widget(xml, "mixercanvas")); 
    shl->hruler = GTK_HRULER(glade_xml_get_widget(xml, "hruler"));
    shl->hscrollbar = GTK_RANGE(glade_xml_get_widget(xml, "hscrollbar"));
    shl->vscrollbar = GTK_RANGE(glade_xml_get_widget(xml, "vscrollbar"));
    shl->hadjust = gtk_range_get_adjustment(shl->hscrollbar);
    shl->vadjust = gtk_range_get_adjustment(shl->vscrollbar);
    shl->progress = gnome_appbar_get_progress(shl->appbar);
    shl->gridunits = GTK_SPIN_BUTTON(glade_xml_get_widget(xml, "grid_units"));
    shl->gridbpm = GTK_SPIN_BUTTON(glade_xml_get_widget(xml, "grid_bpm"));
    shl->gridmeasurement = GTK_OPTION_MENU(glade_xml_get_widget(xml, "grid_measurement"));
    debug_trigger = GTK_MENU_ITEM(glade_xml_get_widget(xml, "debug"));
    tools_trigger = GTK_MENU_ITEM(glade_xml_get_widget(xml, "tools"));
    toolbar = GTK_TOOLBAR(glade_xml_get_widget(xml, "toolbar_file"));
    gtk_toolbar_set_style(toolbar, GTK_TOOLBAR_ICONS);
    toolbar = GTK_TOOLBAR(glade_xml_get_widget(xml, "toolbar_player"));
    gtk_toolbar_set_style(toolbar, GTK_TOOLBAR_ICONS);
    gtk_signal_connect(GTK_OBJECT(shl->hadjust), "value_changed",
                       GTK_SIGNAL_FUNC(shell_hscrollbar_value_changed), shl);
    gtk_signal_connect(GTK_OBJECT(shl->vadjust), "value_changed",
                       GTK_SIGNAL_FUNC(shell_vscrollbar_value_changed), shl);

    for(iterator = GTK_MENU_SHELL(GTK_MENU(gtk_option_menu_get_menu(shl->gridmeasurement)))->children; iterator; iterator = iterator->next)         
        gtk_signal_connect(GTK_OBJECT(iterator->data), "activate",
                           GTK_SIGNAL_FUNC(shell_grid_measurement_changed),
                           shl);

    shell_signals_connect(shl, xml);

    gtk_object_unref(GTK_OBJECT(xml));

    /* Setup the tools menu. The modules structure is initialized in
       module.c. */

    tools = GTK_MENU(gtk_menu_new());
    gtk_menu_set_title(tools, "Tools");

    item = GTK_MENU_ITEM(gtk_tearoff_menu_item_new());
    gtk_widget_show(GTK_WIDGET(item));
    gtk_menu_append(GTK_MENU(tools), GTK_WIDGET(item));
    shl->cancelmodule = GTK_MENU_ITEM(gtk_menu_item_new_with_label("Cancel"));
    gtk_menu_append(GTK_MENU(tools), GTK_WIDGET(shl->cancelmodule));
    gtk_widget_show(GTK_WIDGET(shl->cancelmodule));
    gtk_widget_set_sensitive(GTK_WIDGET(shl->cancelmodule), FALSE);
    gtk_widget_add_accelerator(GTK_WIDGET(shl->cancelmodule),
                               "activate",
                               gtk_accel_group_get_default(),
                               GDK_period,
                               GDK_CONTROL_MASK,
                               GTK_ACCEL_VISIBLE);
    gtk_signal_connect(GTK_OBJECT(shl->cancelmodule),
                       "activate",
                       shell_module_cancel_activate,
                       shl);
    separator = GTK_MENU_ITEM(gtk_menu_item_new());
    gtk_menu_append(GTK_MENU(tools), GTK_WIDGET(separator));
    gtk_widget_show(GTK_WIDGET(separator));
    for(i = 0; i < module_count; i++) {
        item = GTK_MENU_ITEM(gtk_menu_item_new_with_label(modules[i].name));
        gtk_object_set_user_data(GTK_OBJECT(item), GINT_TO_POINTER(i));
        gtk_menu_append(GTK_MENU(tools), GTK_WIDGET(item));
        gtk_widget_show(GTK_WIDGET(item));
        gtk_signal_connect(GTK_OBJECT(item), 
                           "activate",
                           shell_module_activate, 
                           shl);
    }
    gtk_menu_item_set_submenu(GTK_MENU_ITEM(tools_trigger),
                              GTK_WIDGET(tools));

#ifndef DEBUG_FLAG
    gtk_widget_hide(GTK_WIDGET(debug_trigger));
#endif
    gtk_widget_realize(GTK_WIDGET(shl->appwindow));

    /* Explicitly send another configure event because we might miss
       the first, I think. */

    cfg.type = GDK_CONFIGURE;
    cfg.window = GTK_WIDGET(shl->canvas)->window;
    cfg.send_event = 0;
    cfg.x = 0;
    cfg.y = 0;
    cfg.width = 500;
    cfg.height = 500;
    gtk_main_do_event((GdkEvent *)&cfg);

    return shl;
}
