/* Copyright (C) 2002 Pascal Haakmat */

#ifndef SND_H
#define SND_H

#include <audiofile.h>
#include "config.h"
#include "track.h"
#include "mixer.h"
#include "rwlock.h"

#define RESULT_ERROR_NOMEM NULL
#define RESULT_IS_ERROR(s) (!(s) || ((s) && ((snd *)s)->error_str))

typedef enum {
    CLONE_STRUCTURE,
    CLONE_TRACK_STRUCTURE,
    CLONE_TRACK_DATA
} snd_clone_method;

typedef struct _snd {

    /* Error condition on this snd (see RESULT_IS_ERROR macro). */

    char *error_str;

    /* Name/identifier (may be NULL). */

    char *name;

    /* The format as returned by libaudiofile. */

    int fmt;

    /* Endianness. */

    int endian;

    /* Number of channels, i.e. number of valid tracks in tracks array
       below. */

    int channels;

    /* Rate in frames per second. */

    double rate;

    /* Frame width in bytes. */

    int frame_width;

    /* Duration in seconds. */

    double time;

    /* Read/write lock. Read lock must be acquired before traversing
       the tracks array below. */

    rwlock rwl;

    /* Array for the tracks. The channels field above
       specifies how many tracks there are. */

    struct _track *tracks[MAX_TRACKS];
} snd;

snd *
snd_new();

snd *
snd_clone(snd *sr,
          snd_clone_method method);

void 
snd_destroy(snd *sr);

int
snd_insert(snd *sr,
           snd *src,
           int channel_map,
           AFframecount frame_offset);

snd *
snd_erase(snd *sr,
          int channel_map,
          AFframecount frame_offset,
          AFframecount frame_count);

snd *
snd_delete(snd *sr,
           int channel_map,
           AFframecount frame_offset,
           AFframecount frame_count);

void
snd_tracks_insert(snd *sr,
                  snd *src,
                  int channel_map);

snd *
snd_tracks_delete(snd *sr,
                  int channel_map);

int
snd_demux(snd *sr,
          _frame_bits fb_target,
          _frame_bits fb_source,
          int channel_map,
          int channel_count,
          AFframecount frame_offset,
          AFframecount frame_count);

AFframecount
snd_mux(snd *sr,
        _frame_bits fb_target,
        _frame_bits *fb_sources,
        mixer *mixer,
        int channel_map,
        AFframecount frame_offset,
        AFframecount frame_count);

void
snd_name_set(snd *sr,
             const char *filename);

int
snd_track_count(snd *sr);

AFframecount
snd_frame_count(snd *sr);

double 
snd_frames_to_time(const snd *sr, 
                   AFframecount c);

void
snd_convert(snd *sr,
            int sample_width,
            float sample_rate);

void
snd_info_dump(snd *sr);

void
snd_dump(snd *sr);

void 
snd_frames_buffer_to_graph_buffer(_graph_bits graph_bits,
                                  _frame_bits frame_bits,
                                  AFframecount frame_count,
                                  int frame_width,
                                  float hres);

void 
snd_scale(_graph_bits graph_bits,
          void *source_bits,
          AFframecount frame_count,
          int frame_width,
          float skip_amount,
          AFframecount src_off,
          AFframecount dst_off,
          int source_is_graph_bits);

void 
snd_error_free(snd *s);

void
snd_error_set(snd *s, 
              const char *format,
              ...);

const char *
snd_error_get(snd *s);

#endif /* ! SND_H */
