/*
    Copyright (C) 2001 CodeFactory AB
    Copyright (C) 2001 Thomas Nyberg <thomas@codefactory.se>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
#include <stdlib.h>
#include "alsa.h"

#include <sys/soundcard.h>

#define MK_FORMAT(n, d) {SND_PCM_FMT_##n, AFMT_##n, d, #n}

static struct {
	gint mask;
	gint format;
	gint depth;
	gchar *name;
} alsa_formats[] = {
	MK_FORMAT(S8, 8),
	MK_FORMAT(U8, 8),
	MK_FORMAT(S16_LE, 16),
	MK_FORMAT(U16_LE, 16),
	MK_FORMAT(S16_BE, 16),
	MK_FORMAT(U16_BE, 16),
};
#undef MK_FORMAT
#define NUM_FORMATS (sizeof(alsa_formats) / sizeof(alsa_formats[0]))

GType
_gst_alsa_formats_get_type(void)
{
	static GType alsa_formats_type = 0;
	static GEnumValue alsa_formats[] = {
		{8, "8", "8 bits"},
		{16, "16", "16 bits"},
		{0, NULL, NULL},
	};

	if (!alsa_formats_type)
		alsa_formats_type = g_enum_register_static("GstALSAFormats", alsa_formats);

	return alsa_formats_type;
}

GType
_gst_alsa_channels_get_type(void)
{
	static GType alsa_channels_type = 0;
	static GEnumValue alsa_channels[] = {
		{1, "1", "Mono"},
		{2, "2", "Stereo"},
		{0, NULL, NULL},
	};

	if (!alsa_channels_type)
		alsa_channels_type = g_enum_register_static("GstALSAChannels", alsa_channels);

	return alsa_channels_type;
}



gint
_gst_oss_format_to_alsa_format(gint format)
{
	switch (format) {
	case AFMT_U8:
		return SND_PCM_SFMT_U8;
	case AFMT_S16_LE:
		return SND_PCM_SFMT_S16_LE;
	case AFMT_S16_BE:
		return SND_PCM_SFMT_S16_BE;
	case AFMT_S8:
		return SND_PCM_SFMT_S8;
	case AFMT_U16_LE:
		return SND_PCM_SFMT_U16_LE;
	case AFMT_U16_BE:
		return SND_PCM_SFMT_U16_BE;
	default:
		ALSA_DEBUG_PRINT("Format: %d\n", format);
		g_assert_not_reached();
		break;
	}

	return -1;
}

gint
_gst_alsa_format_to_oss_format(gint format)
{
	switch (format) {
	case SND_PCM_SFMT_U8:
		return AFMT_U8;
	case SND_PCM_SFMT_S16_LE:
		return AFMT_S16_LE;
	case SND_PCM_SFMT_S16_BE:
		return AFMT_S16_BE;
	case SND_PCM_SFMT_S8:
		return AFMT_S8;
	case SND_PCM_SFMT_U16_LE:
		return AFMT_U16_LE;
	case SND_PCM_SFMT_U16_BE:
		return AFMT_U16_BE;
	default:
		ALSA_DEBUG_PRINT("Format: %d\n", format);
		g_assert_not_reached();
		break;
	}
	return -1;
}

GstCaps *
_gst_alsa_create_caps(GstALSA *this)
{
	GstCaps *caps = NULL;
	GstProps *props;
	GstCaps *caps_list = NULL;
	gint i;

	if (this->cap) {
		for (i = 0;i < NUM_FORMATS;i++)
			if (this->cap->formats & alsa_formats[i].mask) {
				props = gst_props_new("format", GST_PROPS_STRING("int"),
						      "law", GST_PROPS_INT(0),
						      "depth", GST_PROPS_INT(alsa_formats[i].depth),
						      "width", GST_PROPS_INT(alsa_formats[i].depth),
						      "rate", GST_PROPS_INT_RANGE(this->cap->min_rate, 
										  this->cap->max_rate),
						      "channels", GST_PROPS_INT_RANGE(this->cap->min_channels, 
										      this->cap->max_channels),
						      NULL);

				caps = gst_caps_new (g_strdup(alsa_formats[i].name), "audio/raw", props);

				caps_list = gst_caps_prepend (caps_list, caps);
			}
	} else {
		/* this is bad, we have no idea of knowing anything */

		ALSA_ERROR("No capabilities availible");

		g_assert_not_reached();
	}

	return caps_list;
}

/* 
 * Parses the caps-struct, "borrowed" from gstosssink.c
 */
static gboolean
gst_alsa_parse_caps (GstALSA *this, GstCaps *caps)
{
	gint law, endianness, width, depth;
	gboolean sign;
	gint format = -1;
  
	width = gst_caps_get_int (caps, "width");
	depth = gst_caps_get_int (caps, "depth");

	if (width != depth) return FALSE;

	law = gst_caps_get_int (caps, "law");
	endianness = gst_caps_get_int (caps, "endianness");
	sign = gst_caps_get_boolean (caps, "signed");

	if (law == 0) {
		if (width == 16) {
			if (sign == TRUE) {
				if (endianness == G_LITTLE_ENDIAN)
					format = AFMT_S16_LE;
				else if (endianness == G_BIG_ENDIAN)
					format = AFMT_S16_BE;
			}
			else {
				if (endianness == G_LITTLE_ENDIAN)
					format = AFMT_U16_LE;
				else if (endianness == G_BIG_ENDIAN)
					format = AFMT_U16_BE;
			}
		}
		else if (width == 8) {
			if (sign == TRUE) {
				format = AFMT_S8;
			}
			else {
				format = AFMT_U8;
			}
		}
	}
	
	if (format == -1) 
		return FALSE;

	this->format = format;
	this->rate = gst_caps_get_int(caps, "rate");
	this->channels = gst_caps_get_int(caps, "channels");
	
	return TRUE;
}

/*
 * Negotiates the caps, "borrowed" from gstosssink.c
 */
GstPadNegotiateReturn
_gst_alsa_negotiate(GstPad *pad, GstCaps **caps, gpointer *user_data)
{
	GstALSA *this;

	g_return_val_if_fail (pad != NULL, GST_PAD_NEGOTIATE_FAIL);
	g_return_val_if_fail (GST_IS_PAD (pad), GST_PAD_NEGOTIATE_FAIL);

	/* we have no cast, since we have different types */
	this = (GstALSA *)gst_pad_get_parent(pad);

	ALSA_DEBUG_MSG("negotiate");

	// we decide
	if (user_data == NULL) {
		*caps = NULL;
		return GST_PAD_NEGOTIATE_TRY;
	}
	// have we got caps?
	else if (*caps) {
		if (this->handle == NULL)
			return GST_PAD_NEGOTIATE_FAIL;

		if (gst_alsa_parse_caps(this, *caps)) {
			/* sync the params */
			snd_pcm_channel_flush(this->handle, this->card->channel);

			_gst_alsa_channel_prepare(this);

			snd_pcm_channel_prepare(this->handle, this->card->channel);

			return GST_PAD_NEGOTIATE_AGREE;
		}

		// FIXME check if the sound card was really set to these caps,
		// else send out another caps..

		return GST_PAD_NEGOTIATE_FAIL;
	}
  
	return GST_PAD_NEGOTIATE_FAIL;
}

gboolean
_gst_alsa_channel_prepare(GstALSA *this)
{
	snd_pcm_channel_params_t param;
	gint ret;

	g_return_val_if_fail(this != NULL, FALSE);
	g_return_val_if_fail(this->card != NULL, FALSE);

	ALSA_DEBUG_PRINT("Preparing channel: %d %dHz, %d channels\n", this->format, this->rate, this->channels);

	param.format.interleave = 1; /* We assume the data will be interleaved */

	param.format.format = _gst_oss_format_to_alsa_format(this->format);
	g_return_val_if_fail(param.format.format != -1, FALSE);

	param.format.rate = this->rate;
	param.format.voices = this->channels;
	
	param.channel = this->card->channel;
	param.start_mode = SND_PCM_START_DATA; /* SND_PCM_START_FULL; */
	param.stop_mode = SND_PCM_STOP_STOP;

	if (this->cap->stream_support) {

		param.mode = SND_PCM_MODE_STREAM;
		/*
		 * We do not want the buffer to be to small nor to large
		 */
#if defined(ALSA_QUEUE_SIZE) && ALSA_QUEUE_SIZE > 0		
		param.buf.stream.queue_size = MIN(ALSA_QUEUE_SIZE, this->cap->buffer_size);

		if (param.buf.stream.queue_size == 0) /* the value of buffer_size might be zero */
			param.buf.stream.queue_size = ALSA_QUEUE_SIZE;
#else
		param.buf.stream.queue_size = this->cap->buffer_size; /* maximum size */

		if (param.buf.stream.queue_size == 0) /* if this happens, we will have to make a value up */
			param.buf.stream.queue_size = 8192; /* sounds good on my card */
#endif

		param.buf.stream.fill = SND_PCM_FILL_SILENCE_WHOLE;

	} else if (this->cap->block_support) {

		param.mode = SND_PCM_MODE_BLOCK;
		param.buf.block.frag_size = ALSA_BUFFER_SIZE;
		param.buf.block.frags_min = 1;
		param.buf.block.frags_max = -1;

		if (this->buffer == NULL) /* we will need a buffer here */
			this->buffer = g_malloc(sizeof(gchar) * ALSA_BUFFER_SIZE);

	} else {
		ALSA_ERROR("neither stream nor block is supported by hardware\n");
		return FALSE;
	}

	if ((ret = snd_pcm_channel_params(this->handle, &param))) {
		ALSA_ERROR(snd_strerror(ret));
		return FALSE;
	}

	return TRUE;
}

void
_gst_alsa_flush(GstALSA *this)
{
	gint ret;
	g_assert(this != NULL);

	if (GST_FLAG_IS_SET(this, GST_ALSA_OPEN)) {
		if ((ret = snd_pcm_channel_flush(this->handle, this->card->channel))) {
			ALSA_ERROR(snd_strerror(ret));
			return;
		} 

	}

}

void
_gst_alsa_close_audio(GstALSA *this)
{
	g_assert(this != NULL);

	if (this->handle) {
		_gst_alsa_flush(this);
		snd_pcm_close(this->handle);
	}
	this->handle = NULL;

	GST_FLAG_UNSET(this, GST_ALSA_OPEN);
}

gboolean
_gst_alsa_open_audio(GstALSA *this)
{
	gint ret;
	ALSA_DEBUG_MSG(__FUNCTION__);
	g_assert(this != NULL);

	if (this->handle)
		_gst_alsa_close_audio(this);

	if ((ret = snd_pcm_open(&this->handle, this->card->card, this->card->device, this->card->mode))) {
		ALSA_ERROR(snd_strerror(ret));
		return FALSE;
	}

	if (_gst_alsa_channel_prepare(this) == FALSE) {
		_gst_alsa_close_audio(this);
		return FALSE;
	}

	if ((ret = snd_pcm_channel_prepare(this->handle, this->card->channel))) {
		ALSA_ERROR(snd_strerror(ret));
		_gst_alsa_close_audio(this);
		return FALSE;
	}


	GST_FLAG_SET(this, GST_ALSA_OPEN);
	return TRUE;
}


ALSACardCap *
_gst_alsa_probe_capabilities(const ALSACard *card)
{
	snd_pcm_t *handle;
	snd_pcm_channel_info_t info;
	gint ret;
	ALSACardCap *cap;

	g_assert(card != NULL);

	if ((ret = snd_pcm_open(&handle, card->card, card->device, card->mode))) {
		ALSA_ERROR(snd_strerror(ret));
		return NULL;
	}
	info.channel = card->channel;
	if ((ret = snd_pcm_channel_info(handle, &info))) {
		ALSA_ERROR(snd_strerror(ret));

		snd_pcm_close(handle);
		return NULL;
	}
	snd_pcm_close(handle);

	cap = g_new0(ALSACardCap, 1);
	cap->min_rate = info.min_rate;
	cap->max_rate = info.max_rate;
	cap->min_channels = info.min_voices;
	cap->max_channels = info.max_voices;
	cap->buffer_size = info.buffer_size;

	cap->formats = info.formats;

	cap->mmap_support = (info.flags & SND_PCM_CHNINFO_MMAP) ? 1 : 0;
	cap->stream_support = (info.flags & SND_PCM_CHNINFO_STREAM) ? 1 : 0;
	cap->block_support = (info.flags & SND_PCM_CHNINFO_BLOCK) ? 1 : 0;


#ifdef ALSA_DEBUG
	printf("Subdevice(%d): %s\n", info.subdevice, info.subname);
	printf("Rate: %d-%dHz\n", info.min_rate, info.max_rate);
	printf("Channels: %d-%d\n", info.min_voices, info.max_voices);
	printf("Buffersize: %dbytes\n", info.buffer_size);

	printf("Supports:");
	if (info.flags & SND_PCM_CHNINFO_MMAP)
		printf(" mmap");
	if (info.flags & SND_PCM_CHNINFO_STREAM)
		printf(" stream-mode");
	if (info.flags & SND_PCM_CHNINFO_BLOCK)
		printf(" blockmode");
	printf("\n");
#endif
	return cap;
}

void
_gst_alsa_probe_cards(ALSACardFound playback_func, ALSACardFound capture_func)
{
	int card;
	int pcm_dev;
	gchar *name;
	snd_ctl_t *handle;
	struct snd_ctl_hw_info hw_info;
	snd_pcm_info_t pcm_info;

	ALSA_DEBUG_MSG("Probing soundcards...");
	for (card = 0;card < snd_cards();card++) {
		snd_card_get_longname(card, &name);
		ALSA_DEBUG_PRINT("Soundcard: %s found\n", name);

		snd_ctl_open(&handle, card);

		if (snd_ctl_hw_info(handle, &hw_info) == 0) {
			ALSA_DEBUG_PRINT("PCM-devs: %d\n", hw_info.pcmdevs);

			for (pcm_dev = 0;pcm_dev < hw_info.pcmdevs;pcm_dev++) {

				if (snd_ctl_pcm_info(handle, pcm_dev, &pcm_info) == 0) {
					ALSA_DEBUG_PRINT("Name: %s\n", pcm_info.name);

					if ((pcm_info.flags & SND_PCM_INFO_PLAYBACK) == SND_PCM_INFO_PLAYBACK && playback_func != NULL)
							(*playback_func)(card, pcm_dev, pcm_info.name);

					if ((pcm_info.flags & SND_PCM_INFO_CAPTURE) == SND_PCM_INFO_CAPTURE && capture_func != NULL)
							(*capture_func)(card, pcm_dev, pcm_info.name);

#if 0
					if ((pcm_info.flags & SND_PCM_INFO_DUPLEX) == SND_PCM_INFO_DUPLEX)
						g_print("DUPLEX-support\n");
					if ((pcm_info.flags & SND_PCM_INFO_DUPLEX_RATE) == SND_PCM_INFO_DUPLEX_RATE)
						g_print("DUPLEX-support, but rate must be the same\n");
					if ((pcm_info.flags & SND_PCM_INFO_DUPLEX_MONO) == SND_PCM_INFO_DUPLEX_MONO)
						g_print("DUPLEX-support, but only one channel for each direction\n");
#endif

				} else
					ALSA_ERROR("unable to obtain info about pcm-device");
			}
				
		}
		

		snd_ctl_close(handle);


		free(name);
	}
}













