/* Gnome-Streamer
 * Copyright (C) <1999> Erik Walthinsen <omega@cse.ogi.edu>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <dirent.h>
#include <errno.h>
#include <linux/cdrom.h>

#include <dvdsrc.h>


GstElementDetails dvdsrc_details = {
  "DVD Source",
  "Source/File/DVD",
  "Asynchronous read from encrypted DVD disk",
  VERSION,
  "Erik Walthinsen <omega@cse.ogi.edu>",
  "(C) 1999",
};


/* DVDSrc signals and args */
enum {
  /* FILL ME */
  LAST_SIGNAL
};

enum {
  ARG_0,
  ARG_LOCATION,
  ARG_DEVICE,
  ARG_TITLE_KEY,
  ARG_BYTESPERREAD,
  ARG_OFFSET,
  ARG_SIZE,
};


static void 			dvdsrc_class_init	(DVDSrcClass *klass);
static void 			dvdsrc_init		(DVDSrc *dvdsrc);

static void 			dvdsrc_set_property		(GObject *object, guint prop_id, const GValue *value, GParamSpec *pspec);
static void 			dvdsrc_get_property		(GObject *object, guint prop_id, GValue *value, GParamSpec *pspec);

static GstBuffer *		dvdsrc_get		(GstPad *pad);
//static GstBuffer *		dvdsrc_get_region	(GstPad *pad,gulong offset,gulong size);

static GstElementStateReturn 	dvdsrc_change_state 	(GstElement *element);


static GstElementClass *parent_class = NULL;
//static guint dvdsrc_signals[LAST_SIGNAL] = { 0 };

GType
dvdsrc_get_type (void) 
{
  static GType dvdsrc_type = 0;

  if (!dvdsrc_type) {
    static const GTypeInfo dvdsrc_info = {
      sizeof(DVDSrcClass),      NULL,
      NULL,
      (GClassInitFunc)dvdsrc_class_init,
      NULL,
      NULL,
      sizeof(DVDSrc),
      0,
      (GInstanceInitFunc)dvdsrc_init,
    };
    dvdsrc_type = g_type_register_static (GST_TYPE_ELEMENT, "DVDSrc", &dvdsrc_info, 0);
  }
  return dvdsrc_type;
}

static void
dvdsrc_class_init (DVDSrcClass *klass) 
{
  GObjectClass *gobject_class;
  GstElementClass *gstelement_class;

  gobject_class = (GObjectClass*)klass;
  gstelement_class = (GstElementClass*)klass;

  parent_class = g_type_class_ref (GST_TYPE_ELEMENT);

  g_object_class_install_property(G_OBJECT_CLASS(klass), ARG_LOCATION,
    g_param_spec_string("location","location","location",
                        NULL, G_PARAM_READWRITE)); // CHECKME
  g_object_class_install_property(G_OBJECT_CLASS(klass), ARG_DEVICE,
    g_param_spec_string("device","device","device",
                        NULL, G_PARAM_READWRITE)); // CHECKME
  g_object_class_install_property(G_OBJECT_CLASS(klass), ARG_TITLE_KEY,
    g_param_spec_string("title_key","title_key","title_key",
                        NULL, G_PARAM_READWRITE)); // CHECKME
  g_object_class_install_property(G_OBJECT_CLASS(klass), ARG_BYTESPERREAD,
    g_param_spec_int("bytesperread","bytesperread","bytesperread",
                     G_MININT,G_MAXINT,0,G_PARAM_READABLE)); // CHECKME
  g_object_class_install_property(G_OBJECT_CLASS(klass), ARG_OFFSET,
    g_param_spec_int("offset","offset","offset",
                     G_MININT,G_MAXINT,0,G_PARAM_READWRITE)); // CHECKME
  g_object_class_install_property(G_OBJECT_CLASS(klass), ARG_SIZE,
    g_param_spec_int("size","size","size",
                     G_MININT,G_MAXINT,0,G_PARAM_READABLE)); // CHECKME

  gobject_class->set_property = dvdsrc_set_property;
  gobject_class->get_property = dvdsrc_get_property;

  gstelement_class->change_state = dvdsrc_change_state;
}

static void 
dvdsrc_init (DVDSrc *dvdsrc) 
{
  dvdsrc->srcpad = gst_pad_new ("src", GST_PAD_SRC);
  gst_pad_set_get_function (dvdsrc->srcpad, dvdsrc_get);
  gst_element_add_pad (GST_ELEMENT (dvdsrc), dvdsrc->srcpad);

  dvdsrc->filename = "/cdrom/VIDEO_TS/VTS_02_1.VOB";
  dvdsrc->device = "/dev/dvd";
//  dvdsrc->device = NULL;
  dvdsrc->title_key = NULL;
  dvdsrc->fd = 0;
  dvdsrc->curoffset = 0;
  dvdsrc->bytes_per_read = 2048;
  dvdsrc->seq = 0;
  dvdsrc->new_seek = FALSE;
}


static void 
super_seek (int fd, guint64 offset) 
{
  int i;

  lseek (fd, offset%0x7FFFFFFF, SEEK_SET);
  for (i=0; i < offset/0x7FFFFFFF; i++) {
    lseek (fd, 0x7FFFFFFF, SEEK_CUR);
  }
}

static void 
dvdsrc_set_property (GObject *object, guint prop_id, const GValue *value, GParamSpec *pspec) 
{
  DVDSrc *src;

  /* it's not null if we got it, but it might not be ours */
  g_return_if_fail (GST_IS_DVDSRC (object));
  
  src = DVDSRC (object);

  switch (prop_id) {
    case ARG_LOCATION:
      /* the element must be stopped in order to do this */
//      g_return_if_fail(!GST_FLAG_IS_SET(src,GST_STATE_RUNNING));

      if (src->filename) g_free (src->filename);
      /* clear the filename if we get a NULL (is that possible?) */
      if (g_value_get_string (value) == NULL)
        src->filename = NULL;
      /* otherwise set the new filename */
      else
        src->filename = g_strdup (g_value_get_string (value));  
      break;
    case ARG_DEVICE:
      /* the element must be stopped in order to do this */
//      g_return_if_fail(!GST_FLAG_IS_SET(src,GST_STATE_RUNNING));

      if (src->device) g_free (src->device);
      /* reset the device if we get a NULL (is that possible?) */
      if (g_value_get_string (value) == NULL)
        src->device = "/dev/dvd";
      /* otherwise set the new filename */
      else
        src->device = g_strdup (g_value_get_string (value));  
      break;
    case ARG_TITLE_KEY:
      /* the element must be stopped in order to do this */
//      g_return_if_fail(!GST_FLAG_IS_SET(src,GST_STATE_RUNNING));
      src->title_key = g_value_get_pointer (value);
      break;
    case ARG_BYTESPERREAD:
      src->bytes_per_read = g_value_get_int (value);
      break;
    case ARG_OFFSET:
      src->curoffset = g_value_get_int (value) & ~0x000007ff;
      if (GST_FLAG_IS_SET (src, DVDSRC_OPEN))
        super_seek (src->fd, (DVDSRC_BASEOFFSET*2048) + src->curoffset);
      src->new_seek = TRUE;
      break;
    default:
      break;
  }

}

static void 
dvdsrc_get_property (GObject *object, guint prop_id, GValue *value, GParamSpec *pspec) 
{
  DVDSrc *src;

  /* it's not null if we got it, but it might not be ours */
  g_return_if_fail (GST_IS_DVDSRC (object));
  
  src = DVDSRC (object);

  switch (prop_id) {
    case ARG_LOCATION:
      g_value_set_string (value, src->filename);
      break;
    case ARG_DEVICE:
      g_value_set_string (value, src->device);
      break;
    case ARG_TITLE_KEY:
      g_value_set_pointer (value, src->title_key);
      break;
    case ARG_BYTESPERREAD:
      g_value_set_int (value, src->bytes_per_read);
      break;
    case ARG_OFFSET:
      g_value_set_int (value, src->curoffset);
      break;
    case ARG_SIZE:
      g_value_set_int (value, src->size);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static GstBuffer *
dvdsrc_get (GstPad *pad) 
{
  DVDSrc *dvdsrc;
  GstBuffer *buf;
  glong readbytes;

  g_return_val_if_fail (pad != NULL, NULL);
  g_return_val_if_fail (GST_IS_PAD (pad), NULL);

  dvdsrc = DVDSRC (gst_pad_get_parent (pad));
  g_return_val_if_fail (GST_FLAG_IS_SET (dvdsrc, DVDSRC_OPEN),NULL);

  /* create the buffer */
  // FIXME: should eventually use a bufferpool for this
  buf = gst_buffer_new ();
  g_return_val_if_fail (buf, NULL);

  /* allocate the space for the buffer data */
  GST_BUFFER_DATA (buf) = g_malloc (dvdsrc->bytes_per_read);
  g_return_val_if_fail (GST_BUFFER_DATA (buf) != NULL, NULL);

  /* read it in from the file */
  readbytes = read (dvdsrc->fd, GST_BUFFER_DATA (buf), dvdsrc->bytes_per_read);
  if (readbytes == 0) {
    gst_element_signal_eos (GST_ELEMENT (dvdsrc));
    return NULL;
  }

  // we can only decrypt full sectors, but if at end, signal EOS anyway
  if (readbytes < dvdsrc->bytes_per_read)
    GST_BUFFER_FLAG_SET(buf,GST_BUFFER_EOS);
  else {
    if (*(GST_BUFFER_DATA (buf)+20) & 0x30) {
      css_descramble (GST_BUFFER_DATA (buf), dvdsrc->title_key);
    }
  }

  if (dvdsrc->new_seek) {
    GST_BUFFER_FLAG_SET (buf, GST_BUFFER_FLUSH);
    dvdsrc->new_seek = FALSE;
  }

  GST_BUFFER_OFFSET (buf) = dvdsrc->curoffset;
  GST_BUFFER_SIZE (buf) = readbytes;
  dvdsrc->curoffset += readbytes;

  return buf;
}

/* open the file, necessary to go to RUNNING state */
static gboolean 
dvdsrc_open_file (DVDSrc *src) 
{
  struct stat fileinfo, devinfo;

  g_return_val_if_fail (!GST_FLAG_IS_SET (src, DVDSRC_OPEN), FALSE);

  // if we don't have the title-key yet, we have to go 'find' it
  if (src->title_key == NULL) {
    // if we have the device path, our life is easier, else find it
    if (src->device == NULL) {
      DIR *devdir;
      struct dirent *dirent;
      gchar *path;

      // stat the file to get the device major:minor
      if (stat (src->filename, &fileinfo) < 0) {
	g_print ("%s", src->filename);
	perror (" ");
	return FALSE;
      }

      // open up /dev to find the right device
      devdir = opendir ("/dev");
      // look through all the devices for a match
      while ((dirent = readdir (devdir))) {
        path = g_strconcat ("/dev/", dirent->d_name, NULL);
        stat (path, &devinfo);
        if ((devinfo.st_rdev == fileinfo.st_dev) && S_ISBLK(devinfo.st_mode)) {
          src->device = path;
          break;
        } else
          g_free (path);
      }
      closedir (devdir);
    }

    /* reset the device */
    if (src->device != NULL) {
       int dvdfd;
       dvd_authinfo ai;
       int i;
       guchar disk_key[2048];
       gint lba;

       // open the device
       dvdfd = open (src->device, O_RDONLY | O_NONBLOCK);

       // reset the drive's auth codes
       for (i=0; i<4; i++) {
         memset (&ai, 0, sizeof (ai));
         ai.type = DVD_INVALIDATE_AGID;
         ai.lsa.agid = i;
         ioctl (dvdfd, DVD_AUTH, &ai);
       }

       // get the LBA (what's that?, logical block address?  looks so...)
       lba = path_to_lba (src->filename);
       // special case if we aren't root...
       if (lba == -1) {
         g_print ("lba failed\n");
       }
       else {
         g_print ("lba is %d\n",lba);
       }

       // allocate the title_key
       src->title_key = g_malloc (5);

       // get the disk key
       authenticate (dvdfd, lba, FALSE, disk_key, src->title_key);
       // get the title key
       authenticate (dvdfd, lba, TRUE, disk_key, src->title_key);

       css_decrypttitlekey (src->title_key, disk_key);

       // close the device
       close (dvdfd);
    } else {
      fprintf (stderr,"sorry, couldn't find DVD device for '%s'\n", src->filename);
      exit (2);
    }
  }

  stat (src->filename, &fileinfo);

  src->size = fileinfo.st_size;
  g_print("size %ld\n", src->size);

  /* open the file */
  src->fd = open (src->filename,O_RDONLY);
  if (src->fd < 0) {
    perror ("open()");
    gst_element_error (GST_ELEMENT (src), "opening file");
    return FALSE;
  }

  // seek to the correct position
  super_seek (src->fd, DVDSRC_BASEOFFSET*2048);
  src->curoffset = 0;

  GST_FLAG_SET (src, DVDSRC_OPEN);
  
  return TRUE;
}

/* close the file */
static void 
dvdsrc_close_file (DVDSrc *src) 
{
  g_return_if_fail (GST_FLAG_IS_SET (src, DVDSRC_OPEN));

  /* close the file */
  close (src->fd);

  /* zero out a lot of our state */
  src->fd = 0;
  src->curoffset = 0;
  src->seq = 0;

  GST_FLAG_UNSET (src, DVDSRC_OPEN);
}

static GstElementStateReturn
dvdsrc_change_state (GstElement *element)
{
  g_return_val_if_fail (GST_IS_DVDSRC (element), GST_STATE_FAILURE);

  GST_DEBUG (0,"gstdisksrc: state pending %d\n", GST_STATE_PENDING (element));

  /* if going down into NULL state, close the file if it's open */
  if (GST_STATE_PENDING (element) == GST_STATE_NULL) {
    if (GST_FLAG_IS_SET (element, DVDSRC_OPEN))
      dvdsrc_close_file (DVDSRC (element));
  /* otherwise (READY or higher) we need to open the file */
  } else {
    if (!GST_FLAG_IS_SET (element, DVDSRC_OPEN)) {
      if (!dvdsrc_open_file (DVDSRC (element)))
        return GST_STATE_FAILURE;
    }
  }

  /* if we haven't failed already, give the parent class a chance to ;-) */
  if (GST_ELEMENT_CLASS (parent_class)->change_state)
    return GST_ELEMENT_CLASS (parent_class)->change_state (element);

  return GST_STATE_SUCCESS;
}

static gboolean
plugin_init (GModule *module, GstPlugin *plugin)
{
  GstElementFactory *factory;

  /* create an elementfactory for the dvdsrc element */
  factory = gst_elementfactory_new ("dvdsrc", GST_TYPE_DVDSRC,
                                    &dvdsrc_details);
  g_return_val_if_fail (factory != NULL, NULL);
  
  gst_plugin_add_factory (plugin, factory);
  
  return TRUE;
}

GstPluginDesc plugin_desc = {
  GST_VERSION_MAJOR,
  GST_VERSION_MINOR,
  "dvdsrc",
  plugin_init
};

