#include <ctype.h>
#include <stdlib.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#include "mpg123.h"

/* max = 1728 */


//#define debug(format,args...) g_print(format,##args)
#define debug(format,args...)

int mpg123_tabsel[2][3][16] =
{
  {
    {0, 32, 64, 96, 128, 160, 192, 224, 256, 288, 320, 352, 384, 416, 448,},
    {0, 32, 48, 56, 64, 80, 96, 112, 128, 160, 192, 224, 256, 320, 384,},
    {0, 32, 40, 48, 56, 64, 80, 96, 112, 128, 160, 192, 224, 256, 320,}},

  {
    {0, 32, 48, 56, 64, 80, 96, 112, 128, 144, 160, 176, 192, 224, 256,},
    {0, 8, 16, 24, 32, 40, 48, 56, 64, 80, 96, 112, 128, 144, 160,},
    {0, 8, 16, 24, 32, 40, 48, 56, 64, 80, 96, 112, 128, 144, 160,}},
};

long mpg123_freqs[9] =
{44100, 48000, 32000, 22050, 24000, 16000, 11025, 12000, 8000};

static void mpg123_get_II_stuff(struct mpg123_frame *fr)
{
  static int translate[3][2][16] =
  {
    {
      {0, 2, 2, 2, 2, 2, 2, 0, 0, 0, 1, 1, 1, 1, 1, 0},
      {0, 2, 2, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0}},
    {
      {0, 2, 2, 2, 2, 2, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      {0, 2, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}},
    {
      {0, 3, 3, 3, 3, 3, 3, 0, 0, 0, 1, 1, 1, 1, 1, 0},
      {0, 3, 3, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0}}};

  int table, sblim;
  static int sblims[5] =
  {27, 30, 8, 12, 30};

  if (fr->layer != MPG_LAYER_1)
    table = 4;
  else
    table = translate[fr->samplerate_index][2 - fr->stereo][fr->bitrate_index];
  sblim = sblims[table];

  fr->II_sblimit = sblim;
}

#define HDRCMPMASK 0xfffffd00

int mpg123_head_check(unsigned long head) {
  printf("checking 0x%08x: ",(unsigned int)head);
  /* if it's not a valid sync */
  if ((head & 0xffe00000) != 0xffe00000) {
    printf("invalid sync\n");return 0; }
  /* if it's an invalid MPEG version */
  if (((head >> 19) & 3) == 0x1) {
    printf("invalid MPEG version\n");return 0; }
  /* if it's an invalid layer */
  if (!((head >> 17) & 3)) {
    printf("invalid layer\n");return 0; }
  /* if it's an invalid bitrate */
  if (((head >> 12) & 0xf) == 0x0) {
    printf("invalid bitrate\n");return 0; }
  if (((head >> 12) & 0xf) == 0xf) {
    printf("invalid bitrate\n");return 0; }
  /* if it's an invalid samplerate */
  if (((head >> 10) & 0x3) == 0x3) {
    printf("invalid samplerate\n");return 0; }
  if (((head >> 19) & 1) == 1 &&
      ((head >> 17) & 3) == 3 &&
      ((head >> 16) & 1) == 1) {
    printf("bad for unknown reason\n");return 0; }
  printf("good\n");
  return 1;
}

/*
 * decode a header and write the information
 * into the frame structure
 */
int mpg123_decode_header(struct mpg123_frame *fr,unsigned long header) {
//  unsigned long header;

//  header = (inbuf[0] << 24) | (inbuf[1] << 16) |
//           (inbuf[2] << 8) | inbuf[3];
//  header = getbits(&fr->gb,32);

  debug("header is %08x\n",header);

  fr->mpegver = (header >> 19) & 0x3;
  fr->layer_index = (header >> 17) & 0x3;
  fr->layer = 4 - fr->layer_index;
  fr->bitrate_index = (header >> 12) & 0xf;
  fr->samplerate_index = (header >> 10) & 0x3;

  if (header & (1 << 20)) {
    fr->lsf = (header & (1 << 19)) ? 0 : 1;
    fr->mpeg25 = 0;
    fr->samplerate_index += 3 * fr->lsf;
  } else {
    fr->lsf = 1;
    fr->mpeg25 = 1;
    fr->samplerate_index += 6;
  }

  fr->bitrate = mpg123_tabsel[fr->lsf][fr->layer - 1][fr->bitrate_index];
  fr->samplerate = mpg123_freqs[fr->samplerate_index];

  debug("MPEG %d.0 layer %d stream, %d Kbps, freq %d\n",
          4-fr->mpegver,fr->layer,fr->bitrate,fr->samplerate);

  fr->protection = ((header >> 16) & 0x1) ^ 0x1;
  fr->padding = (header >> 9) & 0x1;
  fr->extension = (header >> 8) & 0x1;
  fr->mode = (header >> 6) & 0x3;
  fr->mode_ext = (header >> 4) & 0x3;
  fr->copyright = (header >> 3) & 0x1;
  fr->original = (header >> 2) & 0x1;
  fr->emphasis = header & 0x3;

  fr->stereo = (fr->mode == MPG_MD_MONO) ? 1 : 2;
  fr->single = -1;

  fr->bpf = mpg123_compute_bpf(fr);
  fr->tpf = mpg123_compute_tpf(fr);

  switch (fr->layer) {
  case MPG_LAYER_1:
    fr->jsbound = (fr->mode == MPG_MD_JOINT_STEREO) ?
	         (fr->mode_ext << 2) + 4 : 32;
	 fr->framesize  = (long) mpg123_tabsel[fr->lsf][0][fr->bitrate_index] * 12000;
    fr->framesize /= mpg123_freqs[fr->samplerate_index];
    fr->framesize  = ((fr->framesize+fr->padding)<<2)-4;
    break;
  case MPG_LAYER_2:
//    mpg123_get_II_stuff(fr);
    fr->jsbound = (fr->mode == MPG_MD_JOINT_STEREO) ?
	         (fr->mode_ext << 2) + 4 : fr->II_sblimit;
    fr->framesize = (long) mpg123_tabsel[fr->lsf][1][fr->bitrate_index] * 144000;
    fr->framesize /= mpg123_freqs[fr->samplerate_index];
    fr->framesize += fr->padding - 4;
    break;
  case MPG_LAYER_3:
    if (fr->lsf)
      fr->ssize = (fr->stereo == 1) ? 9 : 17;
    else
      fr->ssize = (fr->stereo == 1) ? 17 : 32;
    //fprintf(stderr,"ssize is %d\n",fr->ssize);
    if (fr->protection)
      fr->ssize += 2;
    fr->framesize  = (long) mpg123_tabsel[fr->lsf][2][fr->bitrate_index] * 144000;
    fr->framesize /= mpg123_freqs[fr->samplerate_index]<<(fr->lsf);
    fr->framesize = fr->framesize + fr->padding - 4;
    break;
  default:
    return 0;
  }
  return 1;
}

long mpg123_compute_bpf(struct mpg123_frame *fr)
{
  long bpf;

  if (fr->layer == MPG_LAYER_1) {
    bpf = fr->bitrate * 12000;
    bpf /= mpg123_freqs[fr->samplerate_index];
    bpf = ((bpf + fr->padding) << 2);
  } else {
    bpf = fr->bitrate * 144000;
    bpf /= mpg123_freqs[fr->samplerate_index];
    bpf += fr->padding;
  }

  return bpf;
}

/*
long mpg123_bpf_from_header(unsigned long header) {
  int layer,lsf,bitrate,samplerate_index,padding;
  long bpf;

  layer = 4 - ((header >> 19) & 0x3);
  lsf = (header & (1 << 20)) ? ((header & (1 << 19)) ? 0 : 1) : 1;
  bitrate = mpg123_tabsel[lsf][layer - 1][((header >> 12) & 0xf)];
  samplerate_index = (header >> 10) & 0x3;
  padding = (header >> 9) & 0x1;

  if (layer == MPG_LAYER_1) {
    bpf = bitrate * 12000;
    bpf /= mpg123_freqs[samplerate_index];
    bpf = ((bpf + padding) << 2);
  } else {
    bpf = bitrate * 144000;
    bpf /= mpg123_freqs[samplerate_index];
    bpf += padding;
  }

  return bpf;
}
*/

double mpg123_compute_tpf(struct mpg123_frame *fr)
{
  return (((fr->layer == MPG_LAYER_1) ? 384.0 : 1152.0) /
	  mpg123_freqs[fr->samplerate_index]);
}

void mpg123_set_pointer(struct mpg123_frame *fr, long bytes)
{
  gst_getbits_t *gb = &fr->gb;

  if (bytes) {
    memcpy(fr->bsbuf + fr->ssize - bytes,fr->bsbufold+fr->fsizeold-bytes,bytes);
  }
  gst_getbits_newbuf(gb, (fr->bsbuf + fr->ssize - bytes), 512);
}

void mpg123_set_synth_functions(struct mpg123_decoder *de) {
  typedef int (*func) (struct mpg123_frame *fr, double *, int, unsigned char *, int *);
  typedef int (*func_mono) (double *, unsigned char *, int *);
  int ds = de->down_sample;
  int p8 = 0;

  static func funcs[2][4] =
  { { mpg123_synth_1to1,/*
      mpg123_synth_2to1,
      mpg123_synth_4to1,
      mpg123_synth_ntom */ },
    { /*mpg123_synth_1to1_8bit,
      mpg123_synth_2to1_8bit,
      mpg123_synth_4to1_8bit,
      mpg123_synth_ntom_8bit */ }
  };

/*
  static func_mono funcs_mono[2][2][4] =
  { { { mpg123_synth_1to1_mono2stereo,
        mpg123_synth_2to1_mono2stereo,
        mpg123_synth_4to1_mono2stereo,
        mpg123_synth_ntom_mono2stereo },
      { mpg123_synth_1to1_8bit_mono2stereo,
        mpg123_synth_2to1_8bit_mono2stereo,
        mpg123_synth_4to1_8bit_mono2stereo,
        mpg123_synth_ntom_8bit_mono2stereo } },
    { { mpg123_synth_1to1_mono,
        mpg123_synth_2to1_mono,
        mpg123_synth_4to1_mono,
        mpg123_synth_ntom_mono },
      { mpg123_synth_1to1_8bit_mono,
        mpg123_synth_2to1_8bit_mono,
        mpg123_synth_4to1_8bit_mono,
        mpg123_synth_ntom_8bit_mono } }
  };*/

  if (de->mono == 1)
    p8 = 1;
  de->fr.synth = funcs[p8][ds];
//  fr->synth_mono = funcs_mono[1][p8][ds];

//  if (p8) {
//    mpg123_make_conv16to8_table();
//  }
}
