/* Gnome-Streamer
 * Copyright (C) <1999> Erik Walthinsen <omega@cse.ogi.edu>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */


//#define DEBUG_ENABLED
#include <gst/gstclock.h>
#include "gstmpeg_play.h"

/* elementfactory information */
static GstElementDetails gst_mpeg_play_details = {
  "mpeg_play mpeg1 video decoder",
  "Filter/Decoder/Video",
  "Uses modified mpeg_play code to decode MPEG1 video streams",
  VERSION,
  "Wim Taymans <wim.taymans@tvd.be>",
  "(C) 2000",
};

/* Mpeg_play signals and args */
enum {
  FRAME_DECODED,
  LAST_SIGNAL
};

enum {
  ARG_0,
  /* FILL ME */
};

static GstPadTemplate*
src_factory (void)
{
  return
    gst_padtemplate_new (
  	"src",
  	GST_PAD_SRC,
  	GST_PAD_ALWAYS,
  	gst_caps_new (
  	  "mpeg_play_src",
    	  "video/raw",
	  gst_props_new (
    	    "format",    GST_PROPS_FOURCC (GST_MAKE_FOURCC ('I','4','2','0')),
    	    "width",     GST_PROPS_INT_RANGE (16, 4096),
    	    "height",    GST_PROPS_INT_RANGE (16, 4096),
	    NULL)),
	NULL);
}

static GstPadTemplate*
sink_factory (void)
{
  return
    gst_padtemplate_new (
  	"sink",
  	GST_PAD_SINK,
  	GST_PAD_ALWAYS,
  	gst_caps_new (
  	  "mpeg_play_sink",
    	  "video/mpeg",
	  gst_props_new (
    	    "mpegversion",  GST_PROPS_INT (1),
    	    "systemstream", GST_PROPS_BOOLEAN (FALSE),
    	    "sliced",       GST_PROPS_BOOLEAN (TRUE),
	    NULL)),
	NULL);
}

static void	gst_mpeg_play_class_init	(GstMpeg_playClass *klass);
static void	gst_mpeg_play_init		(GstMpeg_play *mpeg_play);

static void	gst_mpeg_play_chain		(GstPad *pad, GstBuffer *buf);
static void	gst_mpeg_play_qos		(GstPad *pad, glong qos_message);

static GstElementClass *parent_class = NULL;
static guint gst_mpeg_play_signals[LAST_SIGNAL] = { 0 };

static GstPadTemplate *src_template, *sink_template;


//static GstElementStateReturn gst_mpeg_play_change_state(GstElement *element);

GType
gst_mpeg_play_get_type (void)
{
  static GType mpeg_play_type = 0;

  if (!mpeg_play_type) {
    static const GTypeInfo mpeg_play_info = {
      sizeof(GstMpeg_playClass),      NULL,
      NULL,
      (GClassInitFunc)gst_mpeg_play_class_init,
      NULL,
      NULL,
      sizeof(GstMpeg_play),
      0,
      (GInstanceInitFunc)gst_mpeg_play_init,
    };
    mpeg_play_type = g_type_register_static(GST_TYPE_ELEMENT, "GstMpeg_play", &mpeg_play_info, 0);
  }
  return mpeg_play_type;
}

static void
gst_mpeg_play_class_init (GstMpeg_playClass *klass)
{
  GObjectClass *gobject_class;
  GstElementClass *gstelement_class;

  gobject_class = (GObjectClass*)klass;
  gstelement_class = (GstElementClass*)klass;

  parent_class = g_type_class_ref(GST_TYPE_ELEMENT);

  gst_mpeg_play_signals[FRAME_DECODED] =
    g_signal_newc("frame_decoded", G_TYPE_FROM_CLASS(klass), G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET(GstMpeg_playClass,frame_decoded), NULL, NULL,
                  g_cclosure_marshal_VOID__POINTER,G_TYPE_NONE,1,
                  GST_TYPE_MPEG_PLAY);
}

static void
gst_mpeg_play_init (GstMpeg_play *mpeg_play)
{
  /* create the sink and src pads */
  mpeg_play->sinkpad = gst_pad_new_from_template(sink_template, "sink");
  gst_pad_set_chain_function(mpeg_play->sinkpad,gst_mpeg_play_chain);
  gst_pad_set_qos_function(mpeg_play->sinkpad,gst_mpeg_play_qos);
  gst_pad_set_caps(mpeg_play->sinkpad, gst_pad_get_padtemplate_caps (mpeg_play->sinkpad));
  gst_element_add_pad(GST_ELEMENT(mpeg_play),mpeg_play->sinkpad);

  mpeg_play->srcpad = gst_pad_new_from_template(src_template, "src");
  gst_element_add_pad(GST_ELEMENT(mpeg_play),mpeg_play->srcpad);

  /* initialize the mpeg_play decoder state */
  mpeg_play->decoder = NewVidStream();
  mpeg_play->parse_state = FIRST_BLOCK;
  mpeg_play->next_time = 0;
  mpeg_play->in_flush = FALSE;

	// reset the initial video state
  mpeg_play->width = -1;
  mpeg_play->height = -1;
}

static void
gst_mpeg_play_qos (GstPad *pad, glong qos_message)
{
  GstMpeg_play *mpeg_play;

  mpeg_play = GST_MPEG_PLAY (GST_OBJECT_PARENT (pad));

  GST_DEBUG (0,"mpeg_play: received QoS message %ld %llu\n", qos_message, mpeg_play->time_interval);
  return;

  // our buffer arrived too late
  if (qos_message < 0)
  {
    qos_message = -qos_message;
    // less than a frame, must be ok
    if (qos_message < mpeg_play->time_interval) {
      //mpeg_play->next_time += qos_message/2;
      SetBFlag(FALSE);
      GST_DEBUG (0,"mpeg_play: message acceptable %ld %llu\n", qos_message, mpeg_play->time_interval);
    }
    // we need to skip a frame
    else {
      GST_DEBUG (0,"mpeg_play: message unacceptable %ld %llu\n", qos_message, mpeg_play->time_interval);
      SetBFlag(TRUE);
    }
  }
  // our buffers arrived too soon
  else {
    SetBFlag(FALSE);
    if (qos_message > mpeg_play->time_interval) {
      //mpeg_play->next_time -= qos_message;
    }
  }
}

// This must match decoder and encoder tables
static double video_rate[16] =
{
  0.0,
  24000.0/1001.,
  24.0,
  25.0,
  30000.0/1001.,
  30.0,
  50.0,
  60000.0/1001.,
  60.0,
  1,
  5,
  10,
  12,
  15,
  0,
  0
};

static void
gst_mpeg_play_chain (GstPad *pad, GstBuffer *buf)
{
  GstMpeg_play *mpeg_play;
  guchar *data;
  gulong size;
  GstBuffer *outbuf;

  g_return_if_fail(pad != NULL);
  g_return_if_fail(GST_IS_PAD(pad));
  g_return_if_fail(buf != NULL);
  //g_return_if_fail(GST_IS_BUFFER(buf));

  mpeg_play = GST_MPEG_PLAY (GST_OBJECT_PARENT (pad));

  data = (guchar *)GST_BUFFER_DATA(buf);
  size = GST_BUFFER_SIZE(buf);
  GST_DEBUG (0,"gst_mpeg_play_chain: got buffer of %lu bytes in '%s' time %llu\n",size,
          GST_OBJECT_NAME (mpeg_play), GST_BUFFER_TIMESTAMP(buf));

  //mpeg_play->next_time = MAX(GST_BUFFER_TIMESTAMP(buf), mpeg_play->next_time);

  if (GST_BUFFER_FLAG_IS_SET(buf, GST_BUFFER_FLUSH)) {
    mpeg_flush(mpeg_play->decoder);
    mpeg_play->in_flush = TRUE;
    SetBFlag(FALSE);
    mpeg_play->next_time = GST_BUFFER_TIMESTAMP(buf);
    GST_DEBUG (0,"mpeg_play: flush %lld\n", mpeg_play->next_time);
  }

  // we have a new block
  mpeg_play->parse_state |= START_BLOCK;
  mpeg_play->parse_state &= ~FINISHED_BLOCK;

  // while we have not finished this block
  while (!(mpeg_play->parse_state & FINISHED_BLOCK)) {
    GST_DEBUG (0,"gst_mpeg_play_chain: about to decode a frame\n");

    // decode this block
    mpeg_play->parse_state = mpegVidRsrc(GST_BUFFER_TIMESTAMP(buf), mpeg_play->decoder, data, size, mpeg_play->parse_state);

    GST_DEBUG (0,"gst_mpeg_play_chain: decoded  frame\n");

    // the decoding gave a new picture
    if (mpeg_play->parse_state & NEW_PICTURE) {

      g_signal_emit (G_OBJECT(mpeg_play),gst_mpeg_play_signals[FRAME_DECODED], 0,
		                         mpeg_play);

      if (GST_PAD_CONNECTED(mpeg_play->srcpad)) {
        //outbuf = gst_buffer_new();
        outbuf = mpeg_play->decoder->current->buffer;
	gst_buffer_ref(outbuf);

        // see if we have a new size
        if ((mpeg_play->decoder->h_size != mpeg_play->width) ||
          (mpeg_play->decoder->v_size != mpeg_play->height)) {


	  gst_pad_set_caps (mpeg_play->srcpad, 
			  gst_caps_new (
				  "mpeg_play_caps",
				  "video/raw",
				  gst_props_new (
					  "format",  GST_PROPS_FOURCC (GST_MAKE_FOURCC ('I','4','2','0')),
					  "width",   GST_PROPS_INT (mpeg_play->decoder->h_size),
					  "height",  GST_PROPS_INT (mpeg_play->decoder->v_size),
					  NULL)));

	  mpeg_play->width = mpeg_play->decoder->h_size;
	  mpeg_play->height = mpeg_play->decoder->v_size;

          mpeg_play->time_interval = 1000000/video_rate[mpeg_play->decoder->picture_rate];

          GST_INFO (GST_CAT_PLUGIN_INFO, "MPEG1 %dx%d, %g fps", mpeg_play->width, mpeg_play->height,
		video_rate[mpeg_play->decoder->picture_rate]);

          mpeg_play->outsize = mpeg_play->width * mpeg_play->height+
			          mpeg_play->width * mpeg_play->height / 2;

        }

	if (mpeg_play->in_flush) {
          GST_BUFFER_FLAG_SET(outbuf, GST_BUFFER_FLUSH);
	  mpeg_play->in_flush = FALSE;
	}
	else {
          GST_BUFFER_FLAG_UNSET(outbuf, GST_BUFFER_FLUSH);
	}

	if (mpeg_play->next_time == 0) {
          GST_BUFFER_TIMESTAMP(outbuf) = mpeg_play->decoder->current->show_time;
          mpeg_play->next_time = mpeg_play->decoder->current->show_time;
	}
	else {
          GST_BUFFER_TIMESTAMP(outbuf) = mpeg_play->next_time;
	}
        GST_DEBUG (0,"gst_mpeg_play_chain: making buffer with timestamp %llu\n", GST_BUFFER_TIMESTAMP(outbuf));
        GST_BUFFER_SIZE(outbuf) = mpeg_play->outsize;
        //GST_BUFFER_DATA(outbuf) = g_malloc(mpeg_play->outsize);
        //memcpy(GST_BUFFER_DATA(outbuf), mpeg_play->decoder->current->luminance, mpeg_play->outsize);

        gst_pad_push(mpeg_play->srcpad,outbuf);
        GST_DEBUG (0,"gst_mpeg_play_chain: pushed buffer\n");
      }
      else {
        g_warning ("gst_mpeg_play_chain: pad not connected !\n");
      }

      mpeg_play->next_time += mpeg_play->time_interval;
    }
  }
  if (mpeg_play->parse_state & SKIPPED_PICTURE && !mpeg_play->in_flush) {
    GST_DEBUG (0,"skipped %d\n", mpeg_play->decoder->skipped);
    mpeg_play->next_time += 1000000/(video_rate[mpeg_play->decoder->picture_rate]
		    - mpeg_play->decoder->skipped ) * (mpeg_play->decoder->skipped);
  }

  gst_buffer_unref(buf);
}

static gboolean
plugin_init (GModule *module, GstPlugin *plugin)
{
  GstElementFactory *factory;

  /* this filter needs the getbits package */
  if (!gst_library_load("gstgetbits")) {
    gst_info("mpeg_play:: could not load support library: 'gstgetbits'\n");
    return FALSE;
  }
  if (!gst_library_load("gstidct")) {
    gst_info("mpeg_play:: could not load support library: 'gstidct'\n");
    return FALSE;
  }

  /* create an elementfactory for the mpeg_play element */
  factory = gst_elementfactory_new("mpeg_play",GST_TYPE_MPEG_PLAY,
                                   &gst_mpeg_play_details);
  g_return_val_if_fail(factory != NULL, FALSE);

  src_template = src_factory ();
  gst_elementfactory_add_padtemplate (factory, src_template);

  sink_template = sink_factory ();
  gst_elementfactory_add_padtemplate (factory, sink_template);

  gst_plugin_add_factory(plugin,factory);

  return TRUE;
}

GstPluginDesc plugin_desc = {
  GST_VERSION_MAJOR,
  GST_VERSION_MINOR,
  "mpeg_play",
  plugin_init
};
