/* 
 * video.c --
 *
 *      This file contains C code that implements the video decoder model.
 *
 */

/*
 * Copyright (c) 1995 The Regents of the University of California.
 * All rights reserved.  
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose, without fee, and without written agreement is
 * hereby granted, provided that the above copyright notice and the following
 * two paragraphs appear in all copies of this software.
 *
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY FOR
 * DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
 * OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF THE UNIVERSITY OF
 * CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND THE UNIVERSITY OF CALIFORNIA HAS NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */

/*
 * Portions of this software Copyright (c) 1995 Brown University.
 * All rights reserved.
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose, without fee, and without written agreement
 * is hereby granted, provided that the above copyright notice and the
 * following two paragraphs appear in all copies of this software.
 * 
 * IN NO EVENT SHALL BROWN UNIVERSITY BE LIABLE TO ANY PARTY FOR
 * DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
 * OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF BROWN
 * UNIVERSITY HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * BROWN UNIVERSITY SPECIFICALLY DISCLAIMS ANY WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 * PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS ON AN "AS IS"
 * BASIS, AND BROWN UNIVERSITY HAS NO OBLIGATION TO PROVIDE MAINTENANCE,
 * SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */

#include <stdio.h>
#include <stdlib.h>

#ifndef MIPS
#include <sys/time.h>
#else
#include <sys/types.h>
#include <sys/system.h>
#endif

//#define DEBUG_ENABLED
#include <libs/getbits/gstgetbits.h>

#include "config.h"
#include "decoders.h"
#include "video.h"
//#include "util.h"
#include "proto.h"

//#define MMX_TRACE
#ifdef HAVE_LIBMMX
#include "mmx.h"
#endif



/* Declarations of functions. */
static void ReconIMBlock();
static void ReconPMBlock();
static void ReconBMBlock();
static void ReconBiMBlock();
static void DoPictureDisplay();
static int ParseSeqHead();
static int ParseGOP();
static int ParsePicture();
static int ParseSlice();
static int ParseMacroBlock();
static void ProcessSkippedPFrameMBlocks();
static void ProcessSkippedBFrameMBlocks();

static mpeg_play_PictImage *NewPictImage(mpeg_play_VidStream *vid_stream);
static void DestroyPictImage(mpeg_play_PictImage *apictimage);

/*
   Changes to make the code reentrant:
     de-globalized: totNumFrames, realTimeStart
       curBits, ReconPMBlock statics, first, [lc]max[xy]
       vid_stream, Parse_done, seekValue, ReadPack static, sys_layer,
       bitOffset, bitLength, bitBuffer, curVidStream,
     X globals to xinfo (window, et al)
     use vid_stream->film_has_ended instead of FilmState
     lookup tables only initialized once, global as possible
        (default_intra_matrix, zigzag, zigzag_direct, scan)
     get rid of setjmp, long jmp
   -lsh@cs.brown.edu (Loring Holden)
 */

extern void form_component_prediction_C (unsigned char *src, unsigned char *dst, short int *blockvals,
			  int lx, int lx2, int w, int h, int x, int y, int dx, int dy, int zflag);
extern void form_component_prediction_mmx (unsigned char *src, unsigned char *dst, short int *blockvals,
			  int lx, int lx2, int w, int h, int x, int y, int dx, int dy, int zflag);
extern void form_component_prediction_mmxe (unsigned char *src, unsigned char *dst, short int *blockvals,
			  int lx, int lx2, int w, int h, int x, int y, int dx, int dy, int zflag);

extern void form_component_prediction_bidir_C(unsigned char *srcb, unsigned char *srcf, 
			unsigned char *dst, short int *blockvals,
			int lx, int lx2, int w, int h, int x, int y, 
			int dbx, int dby, int dfx, int dfy, int zflag);
extern void form_component_prediction_bidir_mmx(unsigned char *srcb, unsigned char *srcf, 
			unsigned char *dst, short int *blockvals,
			int lx, int lx2, int w, int h, int x, int y, 
			int dbx, int dby, int dfx, int dfy, int zflag);
extern void form_component_prediction_bidir_mmxe(unsigned char *srcb, unsigned char *srcf, 
			unsigned char *dst, short int *blockvals,
			int lx, int lx2, int w, int h, int x, int y, 
			int dbx, int dby, int dfx, int dfy, int zflag);

static void (*form_component_prediction) (unsigned char *src, unsigned char *dst, short int *blockvals,
			  int lx, int lx2, int w, int h, int x, int y, int dx, int dy, int zflag);
static void (*form_component_prediction_bidir) (unsigned char *srcb, unsigned char *srcf, 
			unsigned char *dst, short int *blockvals,
			int lx, int lx2, int w, int h, int x, int y, 
			int dbx, int dby, int dfx, int dfy, int zflag);


static const int zigzag[64][2] = {
	{0, 0}, {1, 0}, {0, 1}, {0, 2}, {1, 1}, {2, 0}, {3, 0}, {2, 1}, 
	{1, 2}, {0, 3}, {0, 4}, {1, 3}, {2, 2}, {3, 1}, {4, 0}, {5, 0}, 
	{4, 1}, {3, 2}, {2, 3}, {1, 4}, {0, 5}, {0, 6}, {1, 5}, {2, 4},
        {3, 3}, {4, 2}, {5, 1}, {6, 0}, {7, 0}, {6, 1}, {5, 2}, {4, 3}, 
	{3, 4}, {2, 5}, {1, 6}, {0, 7}, {1, 7}, {2, 6}, {3, 5}, {4, 4}, 
	{5, 3}, {6, 2}, {7, 1}, {7, 2}, {6, 3}, {5, 4}, {4, 5}, {3, 6},
        {2, 7}, {3, 7}, {4, 6}, {5, 5}, {6, 4}, {7, 3}, {7, 4}, {6, 5}, 
	{5, 6}, {4, 7}, {5, 7}, {6, 6}, {7, 5}, {7, 6}, {6, 7}, {7, 7}};

static int zigzag_direct[64] = {
  0, 1, 8, 16, 9, 2, 3, 10, 17, 24, 32, 25, 18, 11, 4, 5, 12,
  19, 26, 33, 40, 48, 41, 34, 27, 20, 13, 6, 7, 14, 21, 28, 35,
  42, 49, 56, 57, 50, 43, 36, 29, 22, 15, 23, 30, 37, 44, 51,
  58, 59, 52, 45, 38, 31, 39, 46, 53, 60, 61, 54, 47, 55, 62, 63
};

static int zigzag_direct_transpose[64] = {
  0*8+0/* 0*/, 1*8+0/* 1*/, 0*8+1/* 8*/, 0*8+2/*16*/, 1*8+1/* 9*/, 2*8+0/* 2*/, 3*8+0/* 3*/, 2*8+1/*10*/,
  1*8+2/*17*/, 0*8+3/*24*/, 0*8+4/*32*/, 1*8+3/*25*/, 2*8+2/*18*/, 3*8+1/*11*/, 4*8+0/* 4*/, 5*8+0/* 5*/,
  4*8+1/*12*/, 3*8+2/*19*/, 2*8+3/*26*/, 1*8+4/*33*/, 0*8+5/*40*/, 0*8+6/*48*/, 1*8+5/*41*/, 2*8+4/*34*/,
  3*8+3/*27*/, 4*8+2/*20*/, 5*8+1/*13*/, 6*8+0/* 6*/, 7*8+0/* 7*/, 6*8+1/*14*/, 5*8+2/*21*/, 4*8+3/*28*/,
  3*8+4/*35*/, 2*8+5/*42*/, 1*8+6/*49*/, 0*8+7/*56*/, 1*8+7/*57*/, 2*8+6/*50*/, 3*8+5/*43*/, 4*8+4/*36*/,
  5*8+3/*29*/, 6*8+2/*22*/, 7*8+1/*15*/, 7*8+2/*23*/, 6*8+3/*30*/, 5*8+4/*37*/, 4*8+5/*44*/, 3*8+6/*51*/,
  2*8+7/*58*/, 3*8+7/*59*/, 4*8+6/*52*/, 5*8+5/*45*/, 6*8+4/*38*/, 7*8+3/*31*/, 7*8+4/*39*/, 6*8+5/*46*/,
  5*8+6/*53*/, 4*8+7/*60*/, 5*8+7/*61*/, 6*8+6/*54*/, 7*8+5/*47*/, 7*8+6/*55*/, 6*8+7/*62*/, 7*8+7/*63*/
};


/* Set up array for fast conversion from row/column coordinates to
   zig zag order.
*/
static const int scan[8][8] = {
  {0, 1, 5, 6, 14, 15, 27, 28},
  {2, 4, 7, 13, 16, 26, 29, 42},
  {3, 8, 12, 17, 25, 30, 41, 43},
  {9, 11, 18, 24, 31, 40, 44, 53},
  {10, 19, 23, 32, 39, 45, 52, 54},
  {20, 22, 33, 38, 46, 51, 55, 60},
  {21, 34, 37, 47, 50, 56, 59, 61},
  {35, 36, 48, 49, 57, 58, 62, 63}};

/* non-linear quantization coefficient table */
static unsigned char Non_Linear_quantizer_scale[32] = 
{
  0, 1, 2, 3, 4, 5, 6, 7,
  8,10,12,14,16,18,20,22,
  24,28,32,36,40,44,48,52,
  56,64,72,80,88,96,104,112
};


/* Initialize P and B skip flags. */
static int No_P_Flag = FALSE;
static int No_B_Flag = FALSE;

/* Max lum, chrom indices for illegal block checking. */


/*
 * We use a lookup table to make sure values stay in the 0..255 range.
 * Since this is cropping (ie, x = (x < 0)?0:(x>255)?255:x; ), wee call this
 * table the "crop table".
 * MAX_NEG_CROP is the maximum neg/pos value we can handle.
 */

#define MAX_NEG_CROP 2048
#define NUM_CROP_ENTRIES (2048+2*MAX_NEG_CROP)
unsigned char cropTbl[NUM_CROP_ENTRIES];

/*
 *--------------------------------------------------------------
 *
 * InitCrop --
 *
 *      Initializes cropTbl - this was taken from newVidStream so
 *      that it wasn't done for each new video stream
 *
 * Results:
 *	None
 *
 * Side effects:
 *      cropTbl will be initialized
 *
 *--------------------------------------------------------------
 */
static void
InitCrop()
{
  int i;

  /* Initialize crop table. */
  for (i = (-MAX_NEG_CROP); i < NUM_CROP_ENTRIES - MAX_NEG_CROP; i++) {
    if (i <= 0) {
      cropTbl[i + MAX_NEG_CROP] = 0;
    } else if (i >= 255) {
      cropTbl[i + MAX_NEG_CROP] = 255;
    } else {
      cropTbl[i + MAX_NEG_CROP] = i;
    }
  }

}


/*
 *--------------------------------------------------------------
 *
 * NewVidStream --
 *
 *	Allocates and initializes a VidStream structure. Takes
 *      as parameter requested size for buffer length.
 *
 * Results:
 *	A pointer to the new VidStream structure.
 *
 * Side effects:
 *      None.
 *
 *--------------------------------------------------------------
 */

mpeg_play_VidStream *
NewVidStream()
{
  int i, j;
  mpeg_play_VidStream *new;
  static const unsigned char default_intra_matrix[64] = {
    8, 16, 19, 22, 26, 27, 29, 34,
    16, 16, 22, 24, 27, 29, 34, 37,
    19, 22, 26, 27, 29, 34, 34, 38,
    22, 22, 26, 27, 29, 34, 37, 40,
    22, 26, 27, 29, 32, 35, 40, 48,
    26, 27, 29, 32, 35, 40, 48, 58,
    26, 27, 29, 34, 38, 46, 56, 69,
    27, 29, 35, 38, 46, 56, 69, 83};

  /* Allocate memory for new structure. */
  new = (mpeg_play_VidStream *) malloc(sizeof(mpeg_play_VidStream));

  /* Initialize pointers to extension and user data. */
  new->group.user_data =
    new->picture.extra_info = new->picture.user_data =
    new->slice.extra_info =
    new->user_data = NULL;

  /* Copy default intra matrix. */
  for (i = 0; i < 8; i++) {
    for (j = 0; j < 8; j++) {
      new->intra_quant_matrix[i][j] = default_intra_matrix[i * 8 + j];
    }
  }

  /* Initialize non intra quantization matrix. */
  for (i = 0; i < 8; i++) {
    for (j = 0; j < 8; j++) {
      new->non_intra_quant_matrix[i][j] = 16;
    }
  }

  /* Initialize pointers to image spaces. */
  new->current = new->past = new->future = NULL;
  for (i = 0; i < RING_BUF_SIZE; i++) {
    new->ring[i] = NULL;
  }


  /* Initialize fields that used to be global */
  new->film_has_ended = FALSE;

  InitCrop();
  init_tables();
  /* setup default MPEG1 flags */
  new->MPEG2 = 0;
  new->pool = NULL;
  new->chunk = NULL;
  new->progressive_sequence = 1;
  new->progressive_frame = 1;
  new->picture.picture_structure = FRAME_PICTURE;
  new->picture.frame_pred_frame_dct = 1;
  new->chroma_format = CHROMA420;
  new->idct = gst_idct_new(GST_IDCT_DEFAULT);

  if (GST_IDCT_TRANSPOSE(new->idct)) {
    new->zigzag_scan = zigzag_direct_transpose;
  }
  else {
    new->zigzag_scan = zigzag_direct;
  }
  gst_getbits_init(&new->gb, NULL, NULL);

#ifdef HAVE_CPU_I386
  if (gst_cpu_get_flags() & GST_CPU_FLAG_MMXEXT) 
  {
    form_component_prediction = form_component_prediction_mmxe;
    form_component_prediction_bidir = form_component_prediction_bidir_mmxe;
    GST_INFO (GST_CAT_PLUGIN_INFO, "using MMXEXT optimised motion compensation");
  }
  else 
#ifdef HAVE_LIBMMX
  if (gst_cpu_get_flags() & GST_CPU_FLAG_MMX) 
  {
    form_component_prediction = form_component_prediction_mmx;
    form_component_prediction_bidir = form_component_prediction_bidir_mmx;
    GST_INFO (GST_CAT_PLUGIN_INFO, "using MMX optimised motion compensation");
  }
  else 
#endif
#endif
  {
    form_component_prediction = form_component_prediction_C;
    form_component_prediction_bidir = form_component_prediction_bidir_C;
    GST_INFO (GST_CAT_PLUGIN_INFO, "using normal motion compensation");
  }

  /* Return structure. */
  return new;
}


/*
 *--------------------------------------------------------------
 *
 * ResetVidStream --
 *
 *	Re-initializes a VidStream structure. Takes
 *      as parameter a pointer to the VidStream to reset.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *      None.
 *
 *--------------------------------------------------------------
 */

void
ResetVidStream(vid)
  mpeg_play_VidStream *vid;
{
  int i;

  /* Initialize pointers to image spaces. */
  vid->current = vid->past = vid->future = NULL;

  /* Initialize rings */
  for (i = 0; i < RING_BUF_SIZE; i++)
    vid->ring[i]->locked = 0;  /* Unlock */

  gst_getbits_init(&vid->gb, NULL, NULL);

  /* We are at the beginning of the film, so film has not ended */
  vid->film_has_ended = FALSE;

}

/*
 *--------------------------------------------------------------
 *
 * DestroyVidStream --
 *
 *	Deallocates a VidStream structure.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */
void
DestroyVidStream(astream)
  mpeg_play_VidStream *astream;
{
  int i;

  if (astream->ext_data != NULL)
    free(astream->ext_data);

  if (astream->user_data != NULL)
    free(astream->user_data);

  if (astream->group.user_data != NULL)
    free(astream->group.user_data);

  if (astream->picture.extra_info != NULL)
    free(astream->picture.extra_info);

  if (astream->picture.user_data != NULL)
    free(astream->picture.user_data);

  if (astream->slice.extra_info != NULL)
    free(astream->slice.extra_info);

  if (astream->buf_start != NULL)
    free(astream->buf_start);

  for (i = 0; i < RING_BUF_SIZE; i++) {
    if (astream->ring[i] != NULL) {
      DestroyPictImage(astream->ring[i]);
      astream->ring[i] = NULL;
    }
  }

  free((char *) astream);
}



/*
 *--------------------------------------------------------------
 *
 * NewPictImage --
 *
 *	Allocates and initializes a PictImage structure.
 *      The width and height of the image space are passed in
 *      as parameters.
 *
 * Results:
 *	A pointer to the new PictImage structure.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static mpeg_play_PictImage *
NewPictImage(vid_stream)
  mpeg_play_VidStream *vid_stream;
{
  mpeg_play_PictImage *new;

  /* Allocate memory space for new structure. */
  new = (mpeg_play_PictImage *) malloc(sizeof(mpeg_play_PictImage));

  new->buffer = NULL;
  
  /* Reset locked flag. */
  new->locked = 0;

  /* Return pointer to new structure. */
  return new;
}


/*
 *--------------------------------------------------------------
 *
 * DestroyPictImage --
 *
 *	Deallocates a PictImage structure.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */
static void
DestroyPictImage(apictimage)
  mpeg_play_PictImage *apictimage;
{
  free(apictimage);
}

void mpeg_flush(vid_stream) 
  mpeg_play_VidStream *vid_stream;
{
  int i;
  GST_DEBUG (0,"flush\n");
  vid_stream->future = NULL;
  vid_stream->past = NULL;
  for (i = 0; i< RING_BUF_SIZE; i++) {
    if (vid_stream->ring[i]) 
      vid_stream->ring[i]->locked = 0;
  }
  vid_stream->state = REGULAR_PARSE;
}

/*
 *--------------------------------------------------------------
 *
 * mpegVidRsrc --
 *
 *      Parses bit stream until MB_QUANTUM number of
 *      macroblocks have been decoded or current slice or
 *      picture ends, whichever comes first. If the start
 *      of a frame is encountered, the frame is time stamped
 *      with the value passed in time_stamp. If the value
 *      passed in buffer is not null, the video stream buffer
 *      is set to buffer and the length of the buffer is
 *      expected in value passed in through length. The current
 *      video stream is set to vid_stream. If vid_stream
 *      is passed as NULL, a new VidStream structure is created
 *      and initialized and used as the current video stream.
 *
 * Results:
 *      a integer indicating the number of new frames.
 *
 * Side effects:
 *      Bit stream is irreversibly parsed. If a picture is completed,
 *      a function is called to display the frame at the correct time.
 *
 *--------------------------------------------------------------
 */

int 
mpegVidRsrc(time_stamp, vid_stream, inbuf, inlen, parse_state)
  TimeStamp  time_stamp;
  mpeg_play_VidStream *vid_stream;
  char *inbuf;
  long inlen;
  int  parse_state;
{
  unsigned long data;
  gboolean found = FALSE;
  gst_getbits_t *gb = &vid_stream->gb;
	  
#define DATA_IN_BUFFER ((inlen) - ((((unsigned long *)gst_getbits_bufferpos(gb))-vid_stream->swapbuf)<<2)) 

  GST_DEBUG (0,"mpegVidRsrc %p\n", inbuf);

  parse_state &= ~NEW_PICTURE;
  parse_state &= ~SKIPPED_PICTURE;
  vid_stream->skipped = 0;


  // if we have a new block to parse, setup the buffers
  if (parse_state & START_BLOCK) {
					
    vid_stream->swapbuf = (unsigned long *)inbuf;

    /* Initialize bitstream i/o fields. */
    gst_getbits_newbuf(gb, inbuf, inlen);

    parse_state &= ~START_BLOCK;
  }

  /*
   * If called for the first time, find start code, make sure it is a
   * sequence start code.
   */
  if (parse_state & FIRST_BLOCK) {
    parse_state &= ~FIRST_BLOCK;
    next_start_code(vid_stream, DATA_IN_BUFFER);  /* sets curBits */
    data = gst_showbits32(gb);
    if (data != SEQ_START_CODE) {
      fprintf(stderr, "mpeg_play: This is not an MPEG video stream. (%08lx)\n",data);
      DestroyVidStream(vid_stream);
      parse_state |= FINISHED_BLOCK;
      return parse_state;
    }
  } 

  next_start_code(vid_stream, DATA_IN_BUFFER);  

  //while ((vid_stream->buffer-vid_stream->swapbuf)<<2 < inlen-4) {
  while (DATA_IN_BUFFER > 4) {

    if (vid_stream->MPEG2) {
      parse_state |= FINISHED_BLOCK;
      return parse_state;
    }
	  
    if (vid_stream->status == SKIP_PICTURE) {
      GST_DEBUG (0,"skipping \n");
      if (!next_bits(32, PICTURE_START_CODE, vid_stream)) {
        if (next_bits(32, GOP_START_CODE, vid_stream))  {
          found = TRUE;
	}
        else if (next_bits(32, SEQ_END_CODE, vid_stream)) {
          found = TRUE;
	}
      }
      else found = TRUE;
      if (!found) {
	gst_flushbitsn(gb, 24);
	goto done;
      }
    }
    
    data = gst_showbits32(gb);
    /* Get next 32 bits (size of start codes). */
    GST_DEBUG (0,"showing 32 (%08lx) %ld %ld %d\n", data, DATA_IN_BUFFER, inlen, found);

    // if we have found something else than e slice and we were parsing it...
    if ((vid_stream->state == IN_SLICE_PARSE) && 
	   ((data < SLICE_MIN_START_CODE) || (data > SLICE_MAX_START_CODE)) &&
	    (data != SEQUENCE_ERROR_CODE)) {

      vid_stream->state = REGULAR_PARSE;
      DoPictureDisplay(vid_stream, 0);
      parse_state |= NEW_PICTURE | CONTINUE_BLOCK;
      return parse_state;
    }

    /*
     * Process according to start code (or parse macroblock if not a start code
     * at all).
     */

    switch (data) {

      case SEQ_END_CODE:
      case 0x000001b9:   /*  handle ISO_11172_END_CODE too */

        /* Display last frame. */
        if (vid_stream->future != NULL) {
          vid_stream->current = vid_stream->future;
	  DoPictureDisplay(vid_stream,0);
	  parse_state |= NEW_PICTURE;
        }
        vid_stream->film_has_ended=TRUE;
	parse_state |= FINISHED_BLOCK;
	return parse_state;
        break;

      case SEQ_START_CODE:

        /* Sequence start code. Parse sequence header. */
        if (ParseSeqHead(vid_stream) != PARSE_OK) {
          fprintf(stderr, "mpeg_play: error parsing seqhead !!\n");
          goto error;
        }
        vid_stream->state = IN_SEQ_PARSE;
        goto done;

      case GOP_START_CODE:

        /* Group of Pictures start code. Parse gop header. */
        if (ParseGOP(vid_stream) != PARSE_OK) {
          fprintf(stderr, "mpeg_play: error parsing gophead !!\n");
          goto error;
        }
        vid_stream->state = IN_GOP_PARSE;
        goto done;

      case PICTURE_START_CODE:

        /* Picture start code. Parse picture header and first slice header. */
        vid_stream->status = ParsePicture(vid_stream, time_stamp);

        vid_stream->state = IN_PICTURE_PARSE;

        if (vid_stream->status == SKIP_PICTURE) {
          parse_state |= SKIPPED_PICTURE;
	  vid_stream->skipped++;
	  found = FALSE;
          gst_flushbitsn(gb, 24);
          GST_DEBUG (0,"mpeg_play: status == SKIP_PICTURE  \n");
          goto done;
        } else if (vid_stream->status != PARSE_OK) {
          fprintf(stderr, "mpeg_play: status  parse picture != PARSE_OK %d\n", vid_stream->status);
          goto error;
        }
        goto done;

      case USER_START_CODE:
        gst_flushbits32(gb);
	switch (vid_stream->state) {
          case IN_SEQ_PARSE:
            //fprintf(stderr, "user code found\n");
            if (vid_stream->user_data != NULL) {
              free(vid_stream->user_data);
              vid_stream->user_data = NULL;
            }
            vid_stream->user_data = get_user_data(vid_stream);
   	    break;
          case IN_GOP_PARSE:
            if (vid_stream->group.user_data != NULL) {
              free(vid_stream->group.user_data);
              vid_stream->group.user_data = NULL;
            }
            vid_stream->group.user_data = get_user_data(vid_stream);
	    break;
          case IN_PICTURE_PARSE:
            if (vid_stream->picture.user_data != NULL) {
              free(vid_stream->picture.user_data);
              vid_stream->picture.user_data = NULL;
            }
            vid_stream->picture.user_data = get_user_data(vid_stream);
	    break;
	}
        goto done;

      case EXT_START_CODE:
        gst_flushbits32(gb);
	switch (vid_stream->state) {
          case IN_SEQ_PARSE:
            //fprintf(stderr, "ext code found\n");
            get_ext_data(vid_stream);
	    break;
          case IN_GOP_PARSE:
            get_ext_data(vid_stream);
	    break;
          case IN_PICTURE_PARSE:
            get_ext_data(vid_stream);
	    break;
	}
        goto done;

      case SEQUENCE_ERROR_CODE:
	/* just goto next start code */
        gst_flushbits32(gb);
        goto done;
    
      default:

        /* Check for slice start code. */
        if ((data >= SLICE_MIN_START_CODE) && (data <= SLICE_MAX_START_CODE)) {

	  //* we're parsing slices */
	  vid_stream->state = IN_SLICE_PARSE;
          /* Slice start code. Parse slice header. */
          if (ParseSlice(vid_stream) != PARSE_OK) {
            fprintf(stderr, "mpeg_play: status parce slice != PARSE_OK \n");
            goto error;
	  }
        }
	else goto error;
        break;

    }

    /* parse off all macroblock until start code or end of buffer */
    while ((((unsigned long*)gst_getbits_bufferpos(gb))-vid_stream->swapbuf)<<2 < inlen-4 &&
            !next_bits(23, 0x00000000, vid_stream)) {
      GST_DEBUG (0,"parsing MB (%08lx) %ld %ld\n", data, DATA_IN_BUFFER, inlen);
      if (ParseMacroBlock(vid_stream) != PARSE_OK) {
        fprintf(stderr, "mpeg_play: error parsing macro block \n");
        vid_stream->state = REGULAR_PARSE;
        goto end;
      }
    }
    goto done;

error:
    fprintf(stderr, "Error!!!! %08lx skipping..\n", data);

done:
    //fprintf(stderr, "done: next start code..\n");
    if ((((unsigned long *)gst_getbits_bufferpos(gb))-vid_stream->swapbuf)<<2 < inlen-4) {
      GST_DEBUG (0,"done: searching next start code..\n");
      next_start_code(vid_stream, DATA_IN_BUFFER);
    }
    //(stderr, "done: next start code done..\n");

  } /* end while */
end:

  //(stderr, "mpegVidRsrc end \n");
  //free(vid_stream->swapbuf);
  parse_state |= FINISHED_BLOCK;
  return parse_state;
}

static GstBuffer *mpeg_play_create(GstBufferPool *pool, gpointer data)
{
  mpeg_play_VidStream *vid_stream = (mpeg_play_VidStream *)data;
  GstBuffer *buffer;

  buffer = gst_buffer_new();
  //gst_buffer_ref(vid_stream->current->buffer);
  GST_BUFFER_DATA(buffer) = g_chunk_new(guchar, vid_stream->chunk);

  return buffer;
}

static void mpeg_play_destroy(GstBufferPool *pool, GstBuffer *buf, gpointer data)
{
  mpeg_play_VidStream *vid_stream;

  g_assert(buf != NULL);
  g_assert(data != NULL);
  g_assert(pool != NULL);

  vid_stream = (mpeg_play_VidStream *)data;

  g_chunk_free(GST_BUFFER_DATA(buf), vid_stream->chunk);
  GST_BUFFER_DATA(buf) = NULL;

  gst_buffer_destroy(buf);
}


/*
 *--------------------------------------------------------------
 *
 * ParseSeqHead --
 *
 *      Assumes bit stream is at the begining of the sequence
 *      header start code. Parses off the sequence header.
 *
 * Results:
 *      Fills the vid_stream structure with values derived and
 *      decoded from the sequence header. Allocates the pict image
 *      structures based on the dimensions of the image space
 *      found in the sequence header.
 *
 * Side effects:
 *      Bit stream irreversibly parsed off.
 *
 *--------------------------------------------------------------
 */
static int
ParseSeqHead(vid_stream)
  mpeg_play_VidStream *vid_stream;
{
  int i;
  gst_getbits_t *gb = &vid_stream->gb;

  //fprintf(stderr, "parseSeqHead \n");
  /* Flush off sequence start code. */
  gst_flushbits32(gb);

  /* Get horizontal size of image space. */
  vid_stream->h_size = gst_getbits12(gb);

  /* Get vertical size of image space. */
  vid_stream->v_size = gst_getbits12(gb);

  if (vid_stream->chunk == NULL) {
    vid_stream->chunk = g_mem_chunk_new("mpeg_play_image_pool", vid_stream->h_size * vid_stream->v_size +
                                         vid_stream->h_size * vid_stream->v_size/2, 
                                         (vid_stream->h_size * vid_stream->v_size +
                                         vid_stream->h_size * vid_stream->v_size/2)*4, G_ALLOC_AND_FREE);
  }

  /* Calculate macroblock width and height of image space. */
  vid_stream->mb_width = (vid_stream->h_size + 15) / 16;
  vid_stream->mb_height = (vid_stream->v_size + 15) / 16;

  /*
   * Initialize ring buffer of pict images now that dimensions of image space
   * are known.
   */
  if (vid_stream->ring[0] == NULL) {
    for (i = 0; i < RING_BUF_SIZE; i++) {
      vid_stream->ring[i] = NewPictImage(vid_stream);
    }
  }

  if (vid_stream->pool == NULL) {
    vid_stream->pool = gst_buffer_pool_new();
    gst_buffer_pool_set_create_function(vid_stream->pool, mpeg_play_create, vid_stream);
    gst_buffer_pool_set_destroy_function(vid_stream->pool, mpeg_play_destroy, vid_stream);
  }

  /* Parse of aspect ratio code. */
  vid_stream->aspect_ratio = (unsigned char) gst_getbits4(gb);

  /* Parse off picture rate code. */
  vid_stream->picture_rate = (unsigned char) gst_getbits4(gb);

  /* Parse off bit rate. */
  vid_stream->bit_rate = gst_getbits18(gb);

  /* Flush marker bit. */
  gst_flushbitsn(gb, 1);

  /* Parse off vbv buffer size. */
  vid_stream->vbv_buffer_size = gst_getbits10(gb);

  /* Parse off contrained parameter flag. */
  vid_stream->const_param_flag = gst_getbits1(gb);

  /*
   * If intra_quant_matrix_flag set, parse off intra quant matrix values.
   */
  if (gst_getbits1(gb)) {
    for (i = 0; i < 64; i++) {

      vid_stream->intra_quant_matrix[zigzag[i][1]][zigzag[i][0]] =
	(unsigned char) gst_getbits8(gb);
    }
  }
  /*
   * If non intra quant matrix flag set, parse off non intra quant matrix
   * values.
   */
  if (gst_getbits1(gb)) {
    for (i = 0; i < 64; i++) {

      vid_stream->non_intra_quant_matrix[zigzag[i][1]][zigzag[i][0]] =
	(unsigned char) gst_getbits8(gb);
    }
  }

  GST_DEBUG (0,"parseSeqHead end \n");
  return PARSE_OK;
}

/*
 *--------------------------------------------------------------
 *
 * ParseGOP --
 *
 *      Parses of group of pictures header from bit stream
 *      associated with vid_stream.
 *
 * Results:
 *      Values in gop header placed into video stream structure.
 *
 * Side effects:
 *      Bit stream irreversibly parsed.
 *
 *--------------------------------------------------------------
 */

static int
ParseGOP(vid_stream)
  mpeg_play_VidStream *vid_stream;
{
  gst_getbits_t *gb = &vid_stream->gb;

  /* Flush group of pictures start code. */
  gst_flushbits32(gb);

  /* Parse off drop frame flag. */
  vid_stream->group.drop_flag = gst_getbits1(gb);

  /* Parse off hour component of time code. */
  vid_stream->group.tc_hours = gst_getbits5(gb);

  /* Parse off minute component of time code. */
  vid_stream->group.tc_minutes = gst_getbits6(gb);

  /* Flush marker bit. */
  gst_flushbitsn(gb, 1);

  /* Parse off second component of time code. */
  vid_stream->group.tc_seconds = gst_getbits6(gb);

  /* Parse off picture count component of time code. */
  vid_stream->group.tc_pictures = gst_getbits6(gb);

  /* Parse off closed gop and broken link flags. */
  vid_stream->group.closed_gop = gst_getbits1(gb);
  vid_stream->group.broken_link = gst_getbits1(gb);

  GST_DEBUG (0,"parseGOP \n");
  GST_DEBUG (0,"  drop flag=%d\n", vid_stream->group.drop_flag);
  GST_DEBUG (0,"  timecode %02d:%02d:%02d:%02d\n", vid_stream->group.tc_hours, vid_stream->group.tc_minutes, 
		  vid_stream->group.tc_seconds, vid_stream->group.tc_pictures);
  GST_DEBUG (0,"  closed gop=%d\n", vid_stream->group.closed_gop);
  GST_DEBUG (0,"  broken link=%d\n", vid_stream->group.broken_link);
  return PARSE_OK;
}

/*
 *--------------------------------------------------------------
 *
 * ParsePicture --
 *
 *      Parses picture header. Marks picture to be presented
 *      at particular time given a time stamp.
 *
 * Results:
 *      Values from picture header put into video stream structure.
 *
 * Side effects:
 *      Bit stream irreversibly parsed.
 *
 *--------------------------------------------------------------
 */

static int
ParsePicture(vid_stream, time_stamp)
  mpeg_play_VidStream *vid_stream;
  TimeStamp time_stamp;
{
  int i;
  gst_getbits_t *gb = &vid_stream->gb;
  unsigned int width=vid_stream->mb_width * 16;
  unsigned int height=vid_stream->mb_height * 16;

  GST_DEBUG (0,"parsePicture \n");
  /* Flush header start code. */
  gst_flushbits32(gb);

  /* Parse off temporal reference. */
  vid_stream->picture.temp_ref = gst_getbits10(gb);
  GST_DEBUG (0,"  temporal_reference=%d \n", vid_stream->picture.temp_ref);

  /* Parse of picture type. */
  vid_stream->picture.code_type = gst_getbits3(gb);
  GST_DEBUG (0,"  picture_coding_type=%d \n", vid_stream->picture.code_type);

  if ((vid_stream->picture.code_type == B_TYPE) &&
      (No_B_Flag ||
       (vid_stream->future == NULL) ||
       ((vid_stream->past == NULL) && !(vid_stream->group.closed_gop))))
    /* According to 2-D.5.1 (p D-18) this is ok, if the refereneces are OK */
    return SKIP_PICTURE;

  if ((vid_stream->picture.code_type == P_TYPE) &&
      (No_P_Flag || (vid_stream->future == NULL)))
    return SKIP_PICTURE;

  /* Parse off vbv buffer delay value. */
  vid_stream->picture.vbv_delay = gst_getbits16(gb);
  GST_DEBUG (0,"  vbv_delay=%d \n", vid_stream->picture.vbv_delay);

  /* If P or B type frame... */
  if ((vid_stream->picture.code_type == P_TYPE) || (vid_stream->picture.code_type == B_TYPE)) {

    /* Parse off forward vector full pixel flag. */
    vid_stream->picture.full_pel_forw_vector = gst_getbits1(gb);

    /* Decode forw_r_code into forw_r_size and forw_f. */
    vid_stream->picture.forw_r_size = gst_getbits3(gb) - 1;
    vid_stream->picture.forw_f = (1 << vid_stream->picture.forw_r_size);

    /* If B type frame... */
    if (vid_stream->picture.code_type == B_TYPE) {

      /* Parse off back vector full pixel flag. */
      vid_stream->picture.full_pel_back_vector = gst_getbits1(gb);

      /* Decode back_r_code into back_r_size and back_f. */
      vid_stream->picture.back_r_size = gst_getbits3(gb) - 1;
      vid_stream->picture.back_f = (1 << vid_stream->picture.back_r_size);
    }
  }

  /* Get extra bit picture info. */
  if (vid_stream->picture.extra_info != NULL) {
    free(vid_stream->picture.extra_info);
    vid_stream->picture.extra_info = NULL;
  }
  vid_stream->picture.extra_info = get_extra_bit_info(vid_stream);

  /* Find a pict image structure in ring buffer not currently locked. */
  i = 0;
  while (vid_stream->ring[i]->locked != 0) {
    if (++i >= RING_BUF_SIZE) {
      perror("Fatal error. Ring buffer full.");
      exit(1);
    }
  }

  /* Set current pict image structure to the one just found in ring. */
  vid_stream->current = vid_stream->ring[i];

  if (vid_stream->current->buffer) {
    gst_buffer_unref(vid_stream->current->buffer);
  }
  
  vid_stream->current->buffer = gst_buffer_new_from_pool(vid_stream->pool);

  vid_stream->current->luminance = (unsigned char *) GST_BUFFER_DATA(vid_stream->current->buffer);
  vid_stream->current->Cb = (unsigned char *) &vid_stream->current->luminance[width * height];
  vid_stream->current->Cr = (unsigned char *) &vid_stream->current->luminance[width * height + width * height /4];

  /* Set time stamp. */
  vid_stream->current->show_time = time_stamp;

  GST_DEBUG (0,"coding type: %d, time %llu\n", vid_stream->picture.code_type, time_stamp);

  /* Reset past macroblock address field. */
  vid_stream->mblock.past_mb_addr = -1;

  return PARSE_OK;
}


/*
 *--------------------------------------------------------------
 *
 * ParseSlice --
 *
 *      Parses off slice header.
 *
 * Results:
 *      Values found in slice header put into video stream structure.
 *
 * Side effects:
 *      Bit stream irreversibly parsed.
 *
 *--------------------------------------------------------------
 */

static int
ParseSlice(vid_stream)
  mpeg_play_VidStream *vid_stream;
{
  int slice_picture_id_enable = 0;
  int slice_picture_id = 0;

  gst_getbits_t *gb = &vid_stream->gb;

  /* Flush slice start code. */

  gst_flushbitsn(gb, 24);

  /* Parse off slice vertical position. */

  //fprintf(stderr, "parseSlice %d\n", data);
  vid_stream->slice.vert_pos_ext = (vid_stream->MPEG2 && vid_stream->v_size>2800) ? gst_getbits3(gb) : 0;

  vid_stream->slice.vert_pos = gst_getbits8(gb);

  /* Parse off quantization scale. */
  vid_stream->slice.quant_scale = gst_getbits5(gb);
  if (vid_stream->MPEG2) {
    fprintf(stderr, "FIXME: need to adjust the quantizer scale %d\n", vid_stream->slice.quant_scale);
    //vid_stream->slice.quant_scale <<= 1; 
  }

  /* slice_id introduced in March 1995 as part of the video corridendum
   *      (after the IS was drafted in November 1994) */
  if (gst_getbits1(gb)) {
    vid_stream->slice.intra_slice = gst_getbits1(gb);

    slice_picture_id_enable = gst_getbits1(gb);
    slice_picture_id = gst_getbits6(gb);

    /* Parse off extra bit slice info. */
    if (vid_stream->slice.extra_info != NULL) {
      free(vid_stream->slice.extra_info);
      vid_stream->slice.extra_info = NULL;
    }
    vid_stream->slice.extra_info = get_extra_bit_info(vid_stream);
  }
  else
    vid_stream->slice.intra_slice = 0;


  /* Reset past intrablock address. */
  vid_stream->mblock.past_intra_addr = -2;

  /* Reset previous recon motion vectors. */
  vid_stream->mblock.recon_right_for_prev = 0;
  vid_stream->mblock.recon_down_for_prev = 0;
  vid_stream->mblock.recon_right_back_prev = 0;
  vid_stream->mblock.recon_down_back_prev = 0;

  /* Reset macroblock address. */
  vid_stream->mblock.mb_address = ((vid_stream->slice.vert_pos - 1) *
				   vid_stream->mb_width) - 1;

  /* Reset past dct dc y, cr, and cb values. */
  vid_stream->block.dct_dc_y_past = 1024 << 3;
  vid_stream->block.dct_dc_cr_past = 1024 << 3;
  vid_stream->block.dct_dc_cb_past = 1024 << 3;

  GST_DEBUG (0,"slice header\n");
  GST_DEBUG (0," slice_vertical_position_extension=%d\n", vid_stream->slice.vert_pos_ext);
  GST_DEBUG (0," quantizer_scale=%d\n", vid_stream->slice.quant_scale);
  GST_DEBUG (0," slice_picture_id_enable=%d id=(%d)\n", slice_picture_id_enable, slice_picture_id_enable? slice_picture_id : 0);
  return PARSE_OK;
}

/*
 *--------------------------------------------------------------
 *
 * ParseMacroBlock --
 *
 *      Parseoff macroblock. Reconstructs DCT values. Applies
 *      inverse DCT, reconstructs motion vectors, calculates and
 *      set pixel values for macroblock in current pict image
 *      structure.
 *
 * Results:
 *      Here's where everything really happens. Welcome to the
 *      heart of darkness.
 *
 * Side effects:
 *      Bit stream irreversibly parsed off.
 *
 *--------------------------------------------------------------
 */

static int
ParseMacroBlock(vid_stream)
  mpeg_play_VidStream *vid_stream;
{
  int addr_incr;
  int mask, i, recon_right_for, recon_down_for, recon_right_back,
      recon_down_back;
  int zero_block_flag;
  BOOLEAN mb_quant = 0, mb_motion_forw = 0, mb_motion_back = 0, 
      mb_pattern = 0;
  gst_getbits_t *gb = &vid_stream->gb;

  GST_DEBUG (0,"parseMacroBlock\n");

  /*
   * Parse off macroblock address increment and add to macroblock address.
   */
  do {
    unsigned int ind;				       
    ind = gst_showbits11(gb);				       
    //fprintf(stderr, "ind %d\n", ind);
    DecodeMBAddrInc(gb, addr_incr);
    if (mb_addr_inc[ind].num_bits==0) {
      addr_incr = 1;
    }
    if (addr_incr == MB_ESCAPE) {
      vid_stream->mblock.mb_address += 33;
      addr_incr = MB_STUFFING;
    }
  } while (addr_incr == MB_STUFFING);
  vid_stream->mblock.mb_address += addr_incr;

  if (vid_stream->mblock.mb_address > (vid_stream->mb_height *
				       vid_stream->mb_width - 1))
    return SKIP_TO_START_CODE;

  /*
   * If macroblocks have been skipped, process skipped macroblocks.
   */
  if (vid_stream->mblock.mb_address - vid_stream->mblock.past_mb_addr > 1) {
    if (vid_stream->picture.code_type == P_TYPE)
      ProcessSkippedPFrameMBlocks(vid_stream);
    else if (vid_stream->picture.code_type == B_TYPE)
      ProcessSkippedBFrameMBlocks(vid_stream);
#ifdef HAVE_LIBMMX
    emms();
#endif
  }
  /* Set past macroblock address to current macroblock address. */
  vid_stream->mblock.past_mb_addr = vid_stream->mblock.mb_address;

  /* Based on picture type decode macroblock type. */
  switch (vid_stream->picture.code_type) {
  case I_TYPE:
    DecodeMBTypeI(gb, mb_quant, mb_motion_forw, mb_motion_back, mb_pattern,
		  vid_stream->mblock.mb_intra);
    break;

  case P_TYPE:
    DecodeMBTypeP(gb, mb_quant, mb_motion_forw, mb_motion_back, mb_pattern,
		  vid_stream->mblock.mb_intra);
    break;

  case B_TYPE:
    DecodeMBTypeB(gb, mb_quant, mb_motion_forw, mb_motion_back, mb_pattern,
		  vid_stream->mblock.mb_intra);
    break;
  case D_TYPE:
    fprintf(stderr, "ERROR:  MPEG-1 Streams with D-frames are not supported\n");
    exit(1);
  }
  //GST_DEBUG (0,"mb_motion_forw %d\n", mb_motion_forw);
  //GST_DEBUG (0,"mb_motion_back %d\n", mb_motion_back);
  //GST_DEBUG (0,"mb_pattern %d\n", mb_pattern);

  if ((vid_stream->mblock.mb_intra || mb_pattern) && !vid_stream->picture.frame_pred_frame_dct && 
		  vid_stream->picture.picture_structure == FRAME_PICTURE) {
     int dct_type;

     dct_type = gst_getbits1(gb);
    
     GST_DEBUG (0,"dct_type %d\n", dct_type);
  }

  /* If quantization flag set, parse off new quantization scale. */
  if (mb_quant == TRUE) {
    vid_stream->slice.quant_scale = gst_getbits5(gb);

    if (vid_stream->MPEG2) {
      vid_stream->slice.quant_scale = 
	      vid_stream->picture.q_scale_type ? 
	        Non_Linear_quantizer_scale[vid_stream->slice.quant_scale]
	      : vid_stream->slice.quant_scale << 1;
      GST_DEBUG (0,"adjust quant scale %d\n", vid_stream->slice.quant_scale);
    }
  }
  /* If forward motion vectors exist... */
  if (mb_motion_forw == TRUE) {

    /* Parse off and decode horizontal forward motion vector. */
    DecodeMotionVectors(gb, vid_stream->mblock.motion_h_forw_code);

    /* If horiz. forward r data exists, parse off. */

    if ((vid_stream->picture.forw_f != 1) &&
	(vid_stream->mblock.motion_h_forw_code != 0)) {
      vid_stream->mblock.motion_h_forw_r = gst_getbits_fastn(gb, vid_stream->picture.forw_r_size);
    }
    /* Parse off and decode vertical forward motion vector. */
    DecodeMotionVectors(gb, vid_stream->mblock.motion_v_forw_code);

    /* If vert. forw. r data exists, parse off. */

    if ((vid_stream->picture.forw_f != 1) &&
	(vid_stream->mblock.motion_v_forw_code != 0)) {
      vid_stream->mblock.motion_v_forw_r = gst_getbits_fastn(gb, vid_stream->picture.forw_r_size);
    }
  }
  /* If back motion vectors exist... */
  if (mb_motion_back == TRUE) {

    /* Parse off and decode horiz. back motion vector. */
    DecodeMotionVectors(gb, vid_stream->mblock.motion_h_back_code);

    /* If horiz. back r data exists, parse off. */

    if ((vid_stream->picture.back_f != 1) &&
	(vid_stream->mblock.motion_h_back_code != 0)) {
      vid_stream->mblock.motion_h_back_r = gst_getbits_fastn(gb, vid_stream->picture.back_r_size);
    }
    /* Parse off and decode vert. back motion vector. */
    DecodeMotionVectors(gb, vid_stream->mblock.motion_v_back_code);

    /* If vert. back r data exists, parse off. */

    if ((vid_stream->picture.back_f != 1) &&
	(vid_stream->mblock.motion_v_back_code != 0)) {
      vid_stream->mblock.motion_v_back_r = gst_getbits_fastn(gb, vid_stream->picture.back_r_size);
    }
  }

  /* If mblock pattern flag set, parse and decode CBP (code block pattern). */
  if (mb_pattern == TRUE) {
    DecodeCBP(gb, vid_stream->mblock.cbp);
  }
  /* Otherwise, set CBP to zero. */
  else
    vid_stream->mblock.cbp = 0;

  /* Reconstruct motion vectors depending on picture type. */
  if (vid_stream->picture.code_type == P_TYPE) {

    /*
     * If no forw motion vectors, reset previous and current vectors to 0.
     */
    if (!mb_motion_forw) {
      recon_right_for = 0;
      recon_down_for = 0;
      vid_stream->mblock.recon_right_for_prev = 0;
      vid_stream->mblock.recon_down_for_prev = 0;
    }
    /*
     * Otherwise, compute new forw motion vectors. Reset previous vectors to
     * current vectors.
     */
    else {
      ComputeForwVector(&recon_right_for, &recon_down_for, vid_stream);
    }
  }
  if (vid_stream->picture.code_type == B_TYPE) {

    /* Reset prev. and current vectors to zero if mblock is intracoded. */
    if (vid_stream->mblock.mb_intra) {
      vid_stream->mblock.recon_right_for_prev = 0;
      vid_stream->mblock.recon_down_for_prev = 0;
      vid_stream->mblock.recon_right_back_prev = 0;
      vid_stream->mblock.recon_down_back_prev = 0;
    } else {
      
      /* If no forw vectors, current vectors equal prev. vectors. */
      if (!mb_motion_forw) {
	recon_right_for = vid_stream->mblock.recon_right_for_prev;
	recon_down_for = vid_stream->mblock.recon_down_for_prev;
      }
      /*
       * Otherwise compute forw. vectors. Reset prev vectors to new values.
       */
      else {
	ComputeForwVector(&recon_right_for, &recon_down_for, vid_stream);
      }
      
      /* If no back vectors, set back vectors to prev back vectors. */
      if (!mb_motion_back) {
        recon_right_back = vid_stream->mblock.recon_right_back_prev;
        recon_down_back = vid_stream->mblock.recon_down_back_prev;
      }
      /* Otherwise compute new vectors and reset prev. back vectors. */
      else {
        ComputeBackVector(&recon_right_back, &recon_down_back, vid_stream);
      }
      /*
       * Store vector existence flags in structure for possible skipped
       * macroblocks to follow.
       */
      vid_stream->mblock.bpict_past_forw = mb_motion_forw;
      vid_stream->mblock.bpict_past_back = mb_motion_back;
    }
  }

  for (mask = 32, i = 0; i < 6; mask >>= 1, i++) {
	
    /* If block exists... */
    if ((vid_stream->mblock.mb_intra) || (vid_stream->mblock.cbp & mask)) {
      zero_block_flag = 0;
      GST_DEBUG (0,"parsing ReconBlock %d\n", i);
      ParseReconBlock(i, vid_stream);
    } else {
      zero_block_flag = 1;
    }
	
    /* If macroblock is intra coded... */
    if (vid_stream->mblock.mb_intra) {
      GST_DEBUG (0,"parsing ReconIMBlock %d\n", i);
      ReconIMBlock(vid_stream, i);
    } else if (mb_motion_forw && mb_motion_back) {
      GST_DEBUG (0,"parsing ReconBiMBlock %d\n", i);
      ReconBiMBlock(vid_stream, i, recon_right_for, recon_down_for,
    		recon_right_back, recon_down_back, zero_block_flag);
    } else if (mb_motion_forw || (vid_stream->picture.code_type == P_TYPE)) {
      GST_DEBUG (0,"parsing ReconPMBlock %d\n", i);
      ReconPMBlock(vid_stream, i, recon_right_for, recon_down_for,
    	       zero_block_flag);
    } else if (mb_motion_back) {
      GST_DEBUG (0,"parsing ReconBMBlock %d\n", i);
      ReconBMBlock(vid_stream, i, recon_right_back, recon_down_back,
    	       zero_block_flag);
    }
#ifdef HAVE_LIBMMX
    emms();
#endif
  }

  /* If D Type picture, flush marker bit. */
  if (vid_stream->picture.code_type == D_TYPE)
    gst_flushbitsn(gb, 1);

  /* If macroblock was intracoded, set macroblock past intra address. */
  if (vid_stream->mblock.mb_intra)
    vid_stream->mblock.past_intra_addr =
      vid_stream->mblock.mb_address;

  GST_DEBUG (0,"parseMacroBlock end\n");
  return PARSE_OK;
}


/*
 *--------------------------------------------------------------
 *
 * ReconIMBlock --
 *
 *	Reconstructs intra coded macroblock.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static void
ReconIMBlock(vid_stream, bnum)
  mpeg_play_VidStream *vid_stream;
  int bnum;
{
  int mb_row, mb_col, row, col, row_size, rr;
  unsigned char *dest;

  /* Calculate macroblock row and column from address. */
  mb_row = vid_stream->mblock.mb_address / vid_stream->mb_width;
  mb_col = vid_stream->mblock.mb_address % vid_stream->mb_width;


  /* If block is luminance block... */
  if (bnum < 4) {

    /* Calculate row and col values for upper left pixel of block. */
    row = mb_row << 4;
    col = mb_col << 4;
    if (bnum > 1)
      row += 8;
    if (bnum & 1)
      col += 8;

    /* Set dest to luminance plane of current pict image. */
    dest = vid_stream->current->luminance;

    /* Establish row size. */
    row_size = vid_stream->mb_width << 4;
  }
  /* Otherwise if block is Cr block... */
  /* Cr first because of the earlier mixup */
  else if (bnum == 5) {

    /* Set dest to Cr plane of current pict image. */
    dest = vid_stream->current->Cr;

    /* Establish row size. */
    row_size = vid_stream->mb_width << 3;

    /* Calculate row,col for upper left pixel of block. */
    row = mb_row << 3;
    col = mb_col << 3;
  }
  /* Otherwise block is Cb block, and ... */
  else {

    /* Set dest to Cb plane of current pict image. */
    dest = vid_stream->current->Cb;

    /* Establish row size. */
    row_size = vid_stream->mb_width << 3;

    /* Calculate row,col for upper left pixel value of block. */
    row = mb_row << 3;
    col = mb_col << 3;
  }
  /*
   * For each pixel in block, set to cropped reconstructed value from inverse
   * dct.
   */
  {
    short *sp = &vid_stream->block.dct_recon[0][0];
#ifndef HAVE_LIBMMX
    unsigned char *cm = cropTbl + MAX_NEG_CROP;
#endif

    dest += row * row_size + col;
    for (rr = 8; rr; rr--, sp += 8, dest += row_size) {
#ifdef HAVE_LIBMMX
      movq_m2r(*sp, mm0);
      movq_m2r(*(sp+4), mm1);
      packuswb_r2r(mm1,mm0);
      movq_r2m(mm0, *dest);
#else
      dest[0] = cm[sp[0]];
      dest[1] = cm[sp[1]];
      dest[2] = cm[sp[2]];
      dest[3] = cm[sp[3]];
      dest[4] = cm[sp[4]];
      dest[5] = cm[sp[5]];
      dest[6] = cm[sp[6]];
      dest[7] = cm[sp[7]];
#endif
    }
  }
}


/*
 *--------------------------------------------------------------
 *
 * ReconPMBlock --
 *
 *	Reconstructs forward predicted macroblocks.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *--------------------------------------------------------------
 */

static void
ReconPMBlock(vid_stream, bnum, recon_right_for, recon_down_for, zflag)
  mpeg_play_VidStream *vid_stream;
  int bnum, recon_right_for, recon_down_for, zflag;
{
  int mb_row, mb_col, row, col, row_size;
  unsigned char *dest, *past = NULL;


  /* Calculate macroblock row and column from address. */
  mb_row = vid_stream->mblock.mb_address / vid_stream->mb_width;
  mb_col = vid_stream->mblock.mb_address % vid_stream->mb_width;

  if (bnum < 4) {

    /* Set dest to luminance plane of current pict image. */
    dest = vid_stream->current->luminance;

    if (vid_stream->picture.code_type == B_TYPE) {
      if (vid_stream->past != NULL)
       past = vid_stream->past->luminance;
    } else {

      /* Set predictive frame to current future frame. */
      if (vid_stream->future != NULL)
        past = vid_stream->future->luminance;
    }

    /* Establish row size. */
    row_size = vid_stream->mb_width << 4;

    /* Calculate row,col of upper left pixel in block. */
    row = mb_row << 4;
    col = mb_col << 4;
    if (bnum > 1)
      row += 8;
    if (bnum & 1)
      col += 8;

  }
  /* Otherwise, block is NOT luminance block, ... */
  else {

    /* Construct motion vectors. */
    recon_right_for /= 2;
    recon_down_for /= 2;

    /* Establish row size. */
    row_size = vid_stream->mb_width << 3;

    /* Calculate row,col of upper left pixel in block. */
    row = mb_row << 3;
    col = mb_col << 3;

    /* If block is Cr block... */
    /* 5 first because order was mixed up in earlier versions */
    if (bnum == 5) {

      /* Set dest to Cr plane of current pict image. */
      dest = vid_stream->current->Cr;

      if (vid_stream->picture.code_type == B_TYPE) {
        if (vid_stream->past != NULL)
          past = vid_stream->past->Cr;
      } else {
        if (vid_stream->future != NULL)
          past = vid_stream->future->Cr;
      }
    }
    /* Otherwise, block is Cb block... */
    else {

      /* Set dest to Cb plane of current pict image. */
      dest = vid_stream->current->Cb;

      if (vid_stream->picture.code_type == B_TYPE) {
        if (vid_stream->past != NULL)
          past = vid_stream->past->Cb;
      } else {
        if (vid_stream->future != NULL)
          past = vid_stream->future->Cb;
      }
    }
  }

  /* For each pixel in block... */

  form_component_prediction(past, dest, &(vid_stream->block.dct_recon[0][0]), row_size, row_size, 
		  8, 8, col, row, recon_right_for, recon_down_for, zflag);
}


/*
 *--------------------------------------------------------------
 *
 * ReconBMBlock --
 *
 *	Reconstructs back predicted macroblocks.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *--------------------------------------------------------------
 */

static void
ReconBMBlock(vid_stream, bnum, recon_right_back, recon_down_back, zflag)
  mpeg_play_VidStream *vid_stream;
  int bnum, recon_right_back, recon_down_back, zflag;
{
  int mb_row, mb_col, row, col, row_size;
  unsigned char *dest, *future = NULL;


  /* Calculate macroblock row and column from address. */
  mb_row = vid_stream->mblock.mb_address / vid_stream->mb_width;
  mb_col = vid_stream->mblock.mb_address % vid_stream->mb_width;

  /* If block is luminance block... */
  if (bnum < 4) {

    /* Set dest to luminance plane of current pict image. */
    dest = vid_stream->current->luminance;

    /*
     * If future frame exists, set future to luminance plane of future frame.
     */
    if (vid_stream->future != NULL)
      future = vid_stream->future->luminance;

    /* Establish row size. */
    row_size = vid_stream->mb_width << 4;

    /* Calculate row,col of upper left pixel in block. */
    row = mb_row << 4;
    col = mb_col << 4;
    if (bnum > 1)
      row += 8;
    if (bnum & 1)
      col += 8;

  }
  /* Otherwise, block is NOT luminance block, ... */
  else {

    /* Construct motion vectors. */
    recon_right_back /= 2;
    recon_down_back /= 2;

    /* Establish row size. */
    row_size = vid_stream->mb_width << 3;

    /* Calculate row,col of upper left pixel in block. */
    row = mb_row << 3;
    col = mb_col << 3;

    /* If block is Cr block... */
    /* They were switched earlier, so 5 is first - eyhung */
    if (bnum == 5) {

      /* Set dest to Cr plane of current pict image. */
      dest = vid_stream->current->Cr;

      /*
       * If future frame exists, set future to Cr plane of future image.
       */
      if (vid_stream->future != NULL)
	future = vid_stream->future->Cr;
    }
    /* Otherwise, block is Cb block... */
    else {

      /* Set dest to Cb plane of current pict image. */
      dest = vid_stream->current->Cb;

      /*
       * If future frame exists, set future to Cb plane of future frame.
       */
      if (vid_stream->future != NULL)
	future = vid_stream->future->Cb;
    }
  }

  form_component_prediction(future, dest, &(vid_stream->block.dct_recon[0][0]), row_size, row_size, 
		  8, 8, col, row, recon_right_back, recon_down_back, zflag);

}

/*
 *--------------------------------------------------------------
 *
 * ReconBiMBlock --
 *
 *	Reconstructs bidirectionally predicted macroblocks.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *--------------------------------------------------------------
 */

static void
ReconBiMBlock(vid_stream, bnum, recon_right_for, recon_down_for,
	      recon_right_back, recon_down_back, zflag)
  mpeg_play_VidStream *vid_stream;
  int bnum, recon_right_for, recon_down_for, recon_right_back, recon_down_back;
  int zflag;
{
  int mb_row, mb_col, row, col, row_size;
  unsigned char *dest, *past=NULL, *future=NULL;

  /* Calculate macroblock row and column from address. */
  mb_row = vid_stream->mblock.mb_address / vid_stream->mb_width;
  mb_col = vid_stream->mblock.mb_address % vid_stream->mb_width;

  /* If block is luminance block... */
  if (bnum < 4) {

    /* Set dest to luminance plane of current pict image. */
    dest = vid_stream->current->luminance;

    /* If past frame exists, set past to luminance plane of past frame. */
    if (vid_stream->past != NULL)
      past = vid_stream->past->luminance;

    /*
     * If future frame exists, set future to luminance plane of future frame.
     */
    if (vid_stream->future != NULL)
      future = vid_stream->future->luminance;

    /* Establish row size. */
    row_size = (vid_stream->mb_width << 4);

    /* Calculate row,col of upper left pixel in block. */
    row = (mb_row << 4);
    col = (mb_col << 4);
    if (bnum > 1)
      row += 8;
    if (bnum & 0x01)
      col += 8;

  }
  /* Otherwise, block is NOT luminance block, ... */
  else {

    /* Construct motion vectors. */
    recon_right_for /= 2;
    recon_down_for /= 2;

    recon_right_back /= 2;
    recon_down_back /= 2;

    /* Establish row size. */
    row_size = (vid_stream->mb_width << 3);

    /* Calculate row,col of upper left pixel in block. */
    row = (mb_row << 3);
    col = (mb_col << 3);

    /* If block is Cr block... */
	/* Switched earlier, so we test Cr first - eyhung */
    if (bnum == 5) {

      /* Set dest to Cr plane of current pict image. */
      dest = vid_stream->current->Cr;

      /* If past frame exists, set past to Cr plane of past image. */
      if (vid_stream->past != NULL)
	past = vid_stream->past->Cr;

      /*
       * If future frame exists, set future to Cr plane of future image.
       */
      if (vid_stream->future != NULL)
	future = vid_stream->future->Cr;
    }
    /* Otherwise, block is Cb block... */
    else {

      /* Set dest to Cb plane of current pict image. */
      dest = vid_stream->current->Cb;

      /* If past frame exists, set past to Cb plane of past frame. */
      if (vid_stream->past != NULL)
	past = vid_stream->past->Cb;

      /*
       * If future frame exists, set future to Cb plane of future frame.
       */
      if (vid_stream->future != NULL)
	future = vid_stream->future->Cb;
    }
  }

  if (future == NULL || past == NULL) return;

  form_component_prediction_bidir(past, future, dest, 
		  &(vid_stream->block.dct_recon[0][0]), 
		  row_size, row_size, 8, 8, col, row, 
		  recon_right_for, recon_down_for,
		  recon_right_back, recon_down_back, 
		  zflag);
}

/*
 *--------------------------------------------------------------
 *
 * ProcessSkippedPFrameMBlocks --
 *
 *	Processes skipped macroblocks in P frames.
 *
 * Results:
 *	Calculates pixel values for luminance, Cr, and Cb planes
 *      in current pict image for skipped macroblocks.
 *
 * Side effects:
 *	Pixel values in pict image changed.
 *
 *--------------------------------------------------------------
 */

static void
ProcessSkippedPFrameMBlocks(vid_stream)
  mpeg_play_VidStream *vid_stream;
{
  int row_size, half_row, mb_row, mb_col, row, col;
  int addr, crow, ccol;

  /* Calculate row sizes for luminance and Cr/Cb macroblock areas. */
  row_size = vid_stream->mb_width << 4;
  half_row = (row_size >> 1);

  /* For each skipped macroblock, do... */
  for (addr = vid_stream->mblock.past_mb_addr + 1;
       addr < vid_stream->mblock.mb_address; addr++) {

    /* Calculate macroblock row and col. */
    mb_row = addr / vid_stream->mb_width;
    mb_col = addr % vid_stream->mb_width;

    /* Calculate upper left pixel row,col for luminance plane. */
    row = mb_row << 4;
    col = mb_col << 4;

    crow = row>>1;
    ccol = col>>1;

    form_component_prediction(vid_stream->future->luminance, vid_stream->current->luminance, 
		  NULL, row_size, row_size, 
		  16, 16, col, row, 0, 0, 1);
    form_component_prediction(vid_stream->future->Cr, vid_stream->current->Cr, 
		  NULL, half_row, half_row, 
		  8, 8, ccol, crow, 0, 0, 1);
    form_component_prediction(vid_stream->future->Cb, vid_stream->current->Cb, 
		  NULL, half_row, half_row, 
		  8, 8, ccol, crow, 0, 0, 1);
  }

  vid_stream->mblock.recon_right_for_prev = 0;
  vid_stream->mblock.recon_down_for_prev = 0;
}




/*
 *--------------------------------------------------------------
 *
 * ProcessSkippedBFrameMBlocks --
 *
 *	Processes skipped macroblocks in B frames.
 *
 * Results:
 *	Calculates pixel values for luminance, Cr, and Cb planes
 *      in current pict image for skipped macroblocks.
 *
 * Side effects:
 *	Pixel values in pict image changed.
 *
 *--------------------------------------------------------------
 */

static void
ProcessSkippedBFrameMBlocks(vid_stream)
  mpeg_play_VidStream *vid_stream;
{
  int row_size, half_row, mb_row, mb_col, row, col;
  int addr;
  int recon_right_for, recon_down_for;
  int recon_right_back, recon_down_back;
  int ccol, crow;

  /* Calculate row sizes for luminance and Cr/Cb macroblock areas. */
  row_size = vid_stream->mb_width << 4;
  half_row = (row_size >> 1);

  /* Establish motion vector codes based on full pixel flag. */
  if (vid_stream->picture.full_pel_forw_vector) {
    recon_right_for = vid_stream->mblock.recon_right_for_prev << 1;
    recon_down_for = vid_stream->mblock.recon_down_for_prev << 1;
  } else {
    recon_right_for = vid_stream->mblock.recon_right_for_prev;
    recon_down_for = vid_stream->mblock.recon_down_for_prev;
  }

  if (vid_stream->picture.full_pel_back_vector) {
    recon_right_back = vid_stream->mblock.recon_right_back_prev << 1;
    recon_down_back = vid_stream->mblock.recon_down_back_prev << 1;
  } else {
    recon_right_back = vid_stream->mblock.recon_right_back_prev;
    recon_down_back = vid_stream->mblock.recon_down_back_prev;
  }


  /* For each skipped macroblock, do... */
  for (addr = vid_stream->mblock.past_mb_addr + 1;
       addr < vid_stream->mblock.mb_address; addr++) {
    
    /* Calculate macroblock row and col. */
    mb_row = addr / vid_stream->mb_width;
    mb_col = addr % vid_stream->mb_width;
    
    /* Calculate upper left pixel row,col for luminance plane. */
    row = mb_row << 4;
    col = mb_col << 4;
    crow = row >> 1;
    ccol = col >> 1;
    
    if ( vid_stream->mblock.bpict_past_forw &&
	!vid_stream->mblock.bpict_past_back) {

      form_component_prediction(vid_stream->past->luminance, vid_stream->current->luminance, 
		  NULL, row_size, row_size, 
		  16, 16, col, row, recon_right_for, recon_down_for, 1);
      form_component_prediction(vid_stream->past->Cr, vid_stream->current->Cr, 
		  NULL, half_row, half_row, 
		  8, 8, ccol, crow, recon_right_for/2, recon_down_for/2, 1);
      form_component_prediction(vid_stream->past->Cb, vid_stream->current->Cb, 
		  NULL, half_row, half_row, 
		  8, 8, ccol, crow, recon_right_for/2, recon_down_for/2, 1);
    }
    else 
      if (!vid_stream->mblock.bpict_past_forw &&
  	   vid_stream->mblock.bpict_past_back) {

      form_component_prediction(vid_stream->future->luminance, vid_stream->current->luminance, 
		  NULL, row_size, row_size, 
		  16, 16, col, row, recon_right_back, recon_down_back, 1);
      form_component_prediction(vid_stream->future->Cr, vid_stream->current->Cr, 
		  NULL, half_row, half_row, 
		  8, 8, ccol, crow, recon_right_back/2, recon_down_back/2, 1);
      form_component_prediction(vid_stream->future->Cb, vid_stream->current->Cb, 
		  NULL, half_row, half_row, 
		  8, 8, ccol, crow, recon_right_back/2, recon_down_back/2, 1);
    }
    else {
      form_component_prediction_bidir(
		  vid_stream->past->luminance, 
		  vid_stream->future->luminance, 
		  vid_stream->current->luminance, 
		  NULL, row_size, row_size, 
		  16, 16, col, row, 
		  recon_right_for, recon_down_for,
		  recon_right_back, recon_down_back, 1);
      form_component_prediction_bidir(
		  vid_stream->past->Cr, 
		  vid_stream->future->Cr, 
		  vid_stream->current->Cr, 
		  NULL, half_row, half_row, 
		  8, 8, ccol, crow, 
		  recon_right_for/2, recon_down_for/2,
		  recon_right_back/2, recon_down_back/2, 1);
      form_component_prediction_bidir(
		  vid_stream->past->Cb, 
		  vid_stream->future->Cb, 
		  vid_stream->current->Cb, 
		  NULL, half_row, half_row, 
		  8, 8, ccol, crow, 
		  recon_right_for/2, recon_down_for/2,
		  recon_right_back/2, recon_down_back/2, 1);
    }
  }
}

/*
 *--------------------------------------------------------------
 *
 * DoPictureDisplay --
 *
 *	Converts Updates past and future frame
 *      pointers. 
 *
 * Results:
 *	Pict image structure locked  if frame
 *      is needed as past or future reference.
 *
 * Side effects:
 *	None
 *
 *--------------------------------------------------------------
 */

static void
DoPictureDisplay(vid_stream, frame_num)
  mpeg_play_VidStream *vid_stream;
	int frame_num;
{
  GST_DEBUG (0,"display %llu %d\n", vid_stream->current->show_time, vid_stream->picture.code_type);

  /* Update past and future references if needed. */
  if ((vid_stream->picture.code_type == I_TYPE) || (vid_stream->picture.code_type == P_TYPE)) {
    if (vid_stream->future == NULL) {
      vid_stream->future = vid_stream->current;
      vid_stream->future->locked |= FUTURE_LOCK;
    } else {
      if (vid_stream->past != NULL) {
        vid_stream->past->locked &= ~PAST_LOCK;
      }
      vid_stream->past = vid_stream->future;
      vid_stream->past->locked &= ~FUTURE_LOCK;
      vid_stream->past->locked |= PAST_LOCK;
      vid_stream->future = vid_stream->current;
      vid_stream->future->locked |= FUTURE_LOCK;
      vid_stream->current = vid_stream->past;
    }
  } 
}



/*
 *--------------------------------------------------------------
 *
 * SetBFlag --
 *
 *	Called to set no b frame processing flag.
 *
 * Results:
 *      No_B_Flag flag is set to its argument.
 *
 * Side effects:
 *      None.
 *
 *--------------------------------------------------------------
 */

void
SetBFlag(val)
BOOLEAN val;
{
    No_B_Flag = val;
}



/*
 *--------------------------------------------------------------
 *
 * SetPFlag --
 *
 *	Called to set no p frame processing flag.
 *
 * Results:
 *      No_P_Flag flag is set to value of argument
 *
 * Side effects:
 *      None.
 *
 *--------------------------------------------------------------
 */

void
SetPFlag(val)
BOOLEAN val;
{
    No_P_Flag = val;
}
