/* Gnome-Streamer
 * Copyright (C) <1999> Erik Walthinsen <omega@cse.ogi.edu>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <string.h>

#include <sys/time.h>

//#define GST_DEBUG_FORCE_DISABLE

#include "xvideosink.h"

static GstElementDetails gst_xvideosink_details = {
  "Video sink",
  "Sink/Video",
  "A general X Window video sink",
  VERSION,
  "Wim Taymans <wim.taymans@tvd.be>",
  "(C) 2000",
};

/* xvideosink signals and args */
enum {
  SIGNAL_FRAME_DISPLAYED,
  SIGNAL_HAVE_SIZE,
  LAST_SIGNAL
};


enum {
  ARG_0,
  ARG_WIDTH,
  ARG_HEIGHT,
  ARG_XID,
  ARG_SCALE,
  ARG_FRAMES_DISPLAYED,
  ARG_FRAME_TIME,
  ARG_DISABLE_XV,
};

static void	gst_xvideosink_class_init	(GstXVideoSinkClass *klass);
static void	gst_xvideosink_init		(GstXVideoSink *xvideosink);

static void	gst_xvideosink_chain		(GstPad *pad, GstBuffer *buf);

static void	gst_xvideosink_set_property		(GObject *object, guint prop_id, const GValue *value, GParamSpec *pspec);
static void	gst_xvideosink_get_property		(GObject *object, guint prop_id, GValue *value, GParamSpec *pspec);

static GstCaps* gst_xvideosink_get_padtemplate_caps (gboolean with_xv);

static GstPadTemplate *sink_template;
static GstCaps *formats;

static GstElementClass *parent_class = NULL;
static guint gst_xvideosink_signals[LAST_SIGNAL] = { 0 };

static GSList *image_pool;
static GMutex *pool_lock;

GType
gst_xvideosink_get_type (void)
{
  static GType xvideosink_type = 0;

  if (!xvideosink_type) {
    static const GTypeInfo xvideosink_info = {
      sizeof(GstXVideoSinkClass),      NULL,
      NULL,
      (GClassInitFunc)gst_xvideosink_class_init,
      NULL,
      NULL,
      sizeof(GstXVideoSink),
      0,
      (GInstanceInitFunc)gst_xvideosink_init,
    };
    xvideosink_type = g_type_register_static(GST_TYPE_ELEMENT, "GstXVideoSink", &xvideosink_info, 0);
  }
  return xvideosink_type;
}

static void
gst_xvideosink_class_init (GstXVideoSinkClass *klass)
{
  GObjectClass *gobject_class;
  GstElementClass *gstelement_class;

  gobject_class = (GObjectClass*)klass;
  gstelement_class = (GstElementClass*)klass;

  parent_class = g_type_class_ref (GST_TYPE_ELEMENT);

  g_object_class_install_property(G_OBJECT_CLASS(klass), ARG_WIDTH,
    g_param_spec_int("width","width","width",
                     G_MININT,G_MAXINT,0,G_PARAM_READWRITE)); // CHECKME
  g_object_class_install_property(G_OBJECT_CLASS(klass), ARG_HEIGHT,
    g_param_spec_int("height","height","height",
                     G_MININT,G_MAXINT,0,G_PARAM_READWRITE)); // CHECKME
  g_object_class_install_property(G_OBJECT_CLASS(klass), ARG_XID,
    g_param_spec_int("xid","xid","xid",
                     G_MININT,G_MAXINT,0,G_PARAM_READABLE)); // CHECKME
  g_object_class_install_property(G_OBJECT_CLASS(klass), ARG_FRAMES_DISPLAYED,
    g_param_spec_int("frames_displayed","frames_displayed","frames_displayed",
                     G_MININT,G_MAXINT,0,G_PARAM_READWRITE)); // CHECKME
  g_object_class_install_property(G_OBJECT_CLASS(klass), ARG_FRAME_TIME,
    g_param_spec_int("frame_time","frame_time","frame_time",
                     G_MININT,G_MAXINT,0,G_PARAM_READWRITE)); // CHECKME
  g_object_class_install_property(G_OBJECT_CLASS(klass), ARG_DISABLE_XV,
    g_param_spec_boolean("disable_xv","disable_xv","disable_xv",
                         TRUE,G_PARAM_READWRITE)); // CHECKME

  gobject_class->set_property = gst_xvideosink_set_property;
  gobject_class->get_property = gst_xvideosink_get_property;

  gst_xvideosink_signals[SIGNAL_FRAME_DISPLAYED] =
    g_signal_newc ("frame_displayed", G_TYPE_FROM_CLASS(klass), G_SIGNAL_RUN_LAST,
                   G_STRUCT_OFFSET (GstXVideoSinkClass, frame_displayed), NULL, NULL,
                   g_cclosure_marshal_VOID__VOID, G_TYPE_NONE, 0);
  gst_xvideosink_signals[SIGNAL_HAVE_SIZE] =
    g_signal_newc ("have_size", G_TYPE_FROM_CLASS(klass), G_SIGNAL_RUN_LAST,
                   G_STRUCT_OFFSET (GstXVideoSinkClass, have_size), NULL, NULL,
                   gst_marshal_VOID__INT_INT, G_TYPE_NONE, 2,
                   G_TYPE_UINT, G_TYPE_UINT);


  image_pool = NULL;
  pool_lock = g_mutex_new ();
}


static GstBuffer*
gst_xvideosink_buffer_create (GstBufferPool *pool, gpointer user_data)
{
  GstXVideoSink *xvideosink;
  GstBuffer *buffer;
  
  xvideosink = GST_XVIDEOSINK (user_data);

  switch (xvideosink->format) {
    case GST_MAKE_FOURCC ('R','G','B',' '):
    {
      GstXImage *ximage;
      
      g_mutex_lock (pool_lock);
      if (!image_pool) {
        g_mutex_unlock (pool_lock);
	g_mutex_lock (xvideosink->lock);
        ximage = gst_ximage_new ( 
		      xvideosink->window, 
		      xvideosink->width, xvideosink->height);
	g_mutex_unlock (xvideosink->lock);
      }
      else {
	ximage = image_pool->data;
	image_pool = g_slist_remove (image_pool, ximage);
        g_mutex_unlock (pool_lock);
      }

      buffer = gst_buffer_new ();
      GST_BUFFER_POOL_PRIVATE (buffer) = ximage;
      GST_BUFFER_DATA (buffer) = GST_XIMAGE_DATA (ximage);
      GST_BUFFER_SIZE (buffer) = xvideosink->width * xvideosink->height *2;
      GST_BUFFER_FLAG_SET (buffer, GST_BUFFER_DONTFREE);
      break;
    }
    default:
    {
      GstXvImage *xvimage;

      g_mutex_lock (pool_lock);
      if (!image_pool) {
        g_mutex_unlock (pool_lock);
	g_mutex_lock (xvideosink->lock);
        xvimage = gst_xvimage_new ( 
		      xvideosink->format,
		      xvideosink->window, 
		      xvideosink->width, xvideosink->height);
	g_mutex_unlock (xvideosink->lock);
      }
      else {
	xvimage = image_pool->data;
	image_pool = g_slist_remove (image_pool, xvimage);
        g_mutex_unlock (pool_lock);
      }

      buffer = gst_buffer_new ();
      GST_BUFFER_POOL_PRIVATE (buffer) = xvimage;
      GST_BUFFER_DATA (buffer) = GST_XVIMAGE_DATA (xvimage);
      GST_BUFFER_SIZE (buffer) = xvideosink->width * xvideosink->height *2;
      GST_BUFFER_FLAG_SET (buffer, GST_BUFFER_DONTFREE);
      break;
    }
  }

  return buffer;
}

static void
gst_xvideosink_buffer_destroy (GstBufferPool *pool, GstBuffer *buffer, gpointer user_data)
{
  GstXVideoSink *xvideosink;
  
  xvideosink = GST_XVIDEOSINK (user_data);

  g_mutex_lock (pool_lock);
  image_pool = g_slist_prepend (image_pool, GST_BUFFER_POOL_PRIVATE (buffer));
  g_mutex_unlock (pool_lock);
}

static GstBufferPool*
gst_xvideosink_get_bufferpool (GstPad *pad)
{
  GstXVideoSink *xvideosink;
  
  xvideosink = GST_XVIDEOSINK (gst_pad_get_parent (pad));

  switch (xvideosink->format) {
    case GST_MAKE_FOURCC ('R','G','B',' '):
      if (!xvideosink->bufferpool) {
        GST_DEBUG (0, "xvideosink: creating RGB XImage bufferpool\n");

        xvideosink->bufferpool = gst_buffer_pool_new ();
        gst_buffer_pool_set_create_function (xvideosink->bufferpool,
		    gst_xvideosink_buffer_create, xvideosink);
        gst_buffer_pool_set_destroy_function (xvideosink->bufferpool,
		    gst_xvideosink_buffer_destroy, xvideosink);
      }
      break;
    default:
      if (!xvideosink->bufferpool) {
        GST_DEBUG (0, "xvideosink: creating YUV XvImage bufferpool\n");

        xvideosink->bufferpool = gst_buffer_pool_new ();
        gst_buffer_pool_set_create_function (xvideosink->bufferpool,
		    gst_xvideosink_buffer_create, xvideosink);
        gst_buffer_pool_set_destroy_function (xvideosink->bufferpool,
		    gst_xvideosink_buffer_destroy, xvideosink);
      }
      break;
  }

  return xvideosink->bufferpool;
}

static void
gst_xvideosink_newcaps (GstPad *pad, GstCaps *caps)
{
  GstXVideoSink *xvideosink;
  gulong print_format;

  xvideosink = GST_XVIDEOSINK (gst_pad_get_parent (pad));

  g_return_if_fail(caps != NULL);

  xvideosink->format = gst_caps_get_fourcc_int (caps, "format");
  xvideosink->width =  gst_caps_get_int (caps, "width");
  xvideosink->height =  gst_caps_get_int (caps, "height");

  print_format = GULONG_FROM_LE (xvideosink->format);

  GST_DEBUG (0, "xvideosink: setting %08lx (%4.4s)\n", xvideosink->format, (gchar*)&print_format);

  switch (xvideosink->format) {
    case GST_MAKE_FOURCC ('R','G','B',' '):
      g_mutex_lock (xvideosink->lock);
      xvideosink->ximage = gst_ximage_new ( 
		      xvideosink->window, 
		      xvideosink->width, xvideosink->height);
      g_mutex_unlock (xvideosink->lock);

      xvideosink->xvimage = NULL;
      break;
    default:
      g_mutex_lock (xvideosink->lock);
      xvideosink->xvimage = gst_xvimage_new ( 
		      xvideosink->format,
		      xvideosink->window, 
		      xvideosink->width, xvideosink->height);
      g_mutex_unlock (xvideosink->lock);
      xvideosink->ximage = NULL;
      break;
  }
  g_signal_emit (G_OBJECT (xvideosink), gst_xvideosink_signals[SIGNAL_HAVE_SIZE], 0,
		  xvideosink->width, xvideosink->height);
}

static GstPadNegotiateReturn
gst_xvideosink_negotiate (GstPad *pad, GstCaps **caps, gpointer *data)
{
  GstXVideoSink *xvideosink;

  GST_DEBUG (0, "xvideosink: negotiate %p %p\n", data, *data);

  xvideosink = GST_XVIDEOSINK (gst_pad_get_parent (pad));

  if (*data == NULL) {
    *data = xvideosink->formats;
    xvideosink->nego_formats = *data;
  }

  if (*caps == NULL) {
    GstCaps *newcaps;

    newcaps = xvideosink->nego_formats;

    if (newcaps) {
      xvideosink->nego_formats = xvideosink->nego_formats->next;
      gst_caps_set (newcaps, "width", GST_PROPS_INT (xvideosink->width));
      gst_caps_set (newcaps, "height", GST_PROPS_INT (xvideosink->height));
      *caps = newcaps;

      if (gst_caps_get_fourcc_int (newcaps, "format") == GST_MAKE_FOURCC ('R','G','B',' ')) {
        GST_DEBUG (0, "using RGB image\n");

        xvideosink->ximage = gst_ximage_new ( 
		      xvideosink->window, 
		      xvideosink->width, xvideosink->height);

        xvideosink->xvimage = NULL;
      }
      else {
        if (xvideosink->xvimage)
	  gst_xvimage_destroy (xvideosink->xvimage);
        GST_DEBUG (0, "using Xv image %08lx\n", gst_caps_get_fourcc_int (newcaps, "format"));

        xvideosink->xvimage = gst_xvimage_new ( 
		      gst_caps_get_fourcc_int (newcaps, "format"),
		      xvideosink->window, 
		      xvideosink->width, xvideosink->height);
        xvideosink->ximage = NULL;
      }

      return GST_PAD_NEGOTIATE_TRY;
    }
    else {
      GST_DEBUG (0, "no more caps\n");
    }
  }
  else {
    gulong format, print_format;
    gint width, height;

    format = gst_caps_get_fourcc_int (*caps, "format");
    width =  gst_caps_get_int (*caps, "width");
    height =  gst_caps_get_int (*caps, "height");
    
    print_format = GULONG_FROM_LE (format);

    GST_DEBUG (0, "xvideosink: got %08lx (%4.4s)\n", format, (gchar*)&print_format);

    switch (format) {
      case GST_MAKE_FOURCC ('R','G','B',' '):
        xvideosink->ximage = gst_ximage_new ( 
		      xvideosink->window, 
		      width, height);

        xvideosink->xvimage = NULL;
	if (xvideosink->ximage)
          return GST_PAD_NEGOTIATE_AGREE;
	break;
      default:
        xvideosink->xvimage = gst_xvimage_new ( 
		      format,
		      xvideosink->window, 
		      width, height);
        xvideosink->ximage = NULL;

	if (xvideosink->xvimage)
          return GST_PAD_NEGOTIATE_AGREE;
	break;
    }
  }

  return GST_PAD_NEGOTIATE_FAIL;
}

static void
gst_xvideosink_init (GstXVideoSink *xvideosink)
{
  xvideosink->sinkpad = gst_pad_new_from_template (sink_template, "sink");
  gst_element_add_pad (GST_ELEMENT (xvideosink), xvideosink->sinkpad);
  gst_pad_set_chain_function (xvideosink->sinkpad, gst_xvideosink_chain);
  gst_pad_set_negotiate_function (xvideosink->sinkpad, gst_xvideosink_negotiate);
  gst_pad_set_newcaps_function (xvideosink->sinkpad, gst_xvideosink_newcaps);
  gst_pad_set_bufferpool_function (xvideosink->sinkpad, gst_xvideosink_get_bufferpool);

  xvideosink->clock = gst_clock_get_system();
  gst_clock_register(xvideosink->clock, GST_OBJECT(xvideosink));

  xvideosink->window = gst_xwindow_new (100, 100);

  xvideosink->width = 100;
  xvideosink->height = 100;

  //xvideosink->xvimage = gst_xvimage_new (0x32315659, xvideosink->window, 100, 100);
  xvideosink->ximage = gst_ximage_new (xvideosink->window, 100, 100);
  xvideosink->formats = formats;
  xvideosink->lock = g_mutex_new();
  xvideosink->disable_xv = FALSE;

  GST_FLAG_SET(xvideosink, GST_ELEMENT_THREAD_SUGGESTED);
}

static void
gst_xvideosink_chain (GstPad *pad, GstBuffer *buf)
{
  GstXVideoSink *xvideosink;
  GstClockTimeDiff jitter;

  g_return_if_fail (pad != NULL);
  g_return_if_fail (GST_IS_PAD (pad));
  g_return_if_fail (buf != NULL);

  xvideosink = GST_XVIDEOSINK (gst_pad_get_parent (pad));

  if (!GST_BUFFER_FLAG_IS_SET(buf, GST_BUFFER_FLUSH)) {
    GST_DEBUG (0,"videosink: clock wait: %llu\n", GST_BUFFER_TIMESTAMP(buf));

    jitter = gst_clock_current_diff(xvideosink->clock, GST_BUFFER_TIMESTAMP (buf));

    if (jitter > 500000 || jitter < -500000)
    {
      GST_DEBUG (0, "jitter: %lld\n", jitter);
      gst_clock_set (xvideosink->clock, GST_BUFFER_TIMESTAMP (buf));
    }
    else {
      gst_clock_wait(xvideosink->clock, GST_BUFFER_TIMESTAMP(buf), GST_OBJECT(xvideosink));
    }
  }

  g_mutex_lock (xvideosink->lock);
  if (xvideosink->bufferpool && GST_BUFFER_BUFFERPOOL (buf) == xvideosink->bufferpool) {
    if (xvideosink->format == GST_MAKE_FOURCC ('R','G','B',' ')) 
      gst_ximage_put (xvideosink->window, GST_XIMAGE (GST_BUFFER_POOL_PRIVATE (buf)));
    else {
      gst_xvimage_put (xvideosink->window, GST_XVIMAGE (GST_BUFFER_POOL_PRIVATE (buf)));
    }
  }
  else {
    if (xvideosink->xvimage) {
      memcpy (GST_XVIMAGE_DATA (xvideosink->xvimage), GST_BUFFER_DATA (buf), GST_BUFFER_SIZE (buf));
      gst_xvimage_put (xvideosink->window, xvideosink->xvimage);
    }
    else if (xvideosink->ximage) {
      memcpy (GST_XIMAGE_DATA (xvideosink->ximage), GST_BUFFER_DATA (buf), GST_BUFFER_SIZE (buf));
      gst_ximage_put (xvideosink->window, xvideosink->ximage);
    }
    else {
      g_assert_not_reached ();
    }
  }
  g_mutex_unlock (xvideosink->lock);

  g_signal_emit(G_OBJECT(xvideosink),gst_xvideosink_signals[SIGNAL_FRAME_DISPLAYED],0);

  gst_buffer_unref(buf);
}


static void
gst_xvideosink_set_property (GObject *object, guint prop_id, const GValue *value, GParamSpec *pspec)
{
  GstXVideoSink *xvideosink;

  /* it's not null if we got it, but it might not be ours */
  g_return_if_fail (GST_IS_XVIDEOSINK (object));

  xvideosink = GST_XVIDEOSINK (object);

  switch (prop_id) {
    case ARG_WIDTH:
      xvideosink->width = g_value_get_int (value);
      GST_DEBUG (0,"xvideosink: setting width to %d\n", xvideosink->width);
      break;
    case ARG_HEIGHT:
      xvideosink->height = g_value_get_int (value);
      GST_DEBUG (0,"xvideosink: setting height to %d\n", xvideosink->height);
      break;
    case ARG_FRAMES_DISPLAYED:
      xvideosink->frames_displayed = g_value_get_int (value);
      break;
    case ARG_FRAME_TIME:
      xvideosink->frame_time = g_value_get_int (value);
      break;
    case ARG_DISABLE_XV:
      xvideosink->disable_xv = g_value_get_boolean (value);
      xvideosink->formats = gst_xvideosink_get_padtemplate_caps (!xvideosink->disable_xv);

      // FIXME
      sink_template = gst_padtemplate_new (
		  "sink",
                  GST_PAD_SINK,
  		  GST_PAD_ALWAYS,
		  xvideosink->formats, NULL);
      GST_PAD_PADTEMPLATE (xvideosink->sinkpad) = sink_template;
      break;
    default:
      break;
  }
}

static void
gst_xvideosink_get_property (GObject *object, guint prop_id, GValue *value, GParamSpec *pspec)
{
  GstXVideoSink *xvideosink;

  /* it's not null if we got it, but it might not be ours */
  xvideosink = GST_XVIDEOSINK(object);

  switch (prop_id) {
    case ARG_WIDTH: {
      g_value_set_int (value, xvideosink->width);
      break;
    }
    case ARG_HEIGHT: {
      g_value_set_int (value, xvideosink->height);
      break;
    }
    case ARG_XID: {
      g_value_set_int (value, GST_XWINDOW_XID (xvideosink->window));
      break;
    }
    case ARG_FRAMES_DISPLAYED: {
      g_value_set_int (value, xvideosink->frames_displayed);
      break;
    }
    case ARG_FRAME_TIME: {
      g_value_set_int (value, xvideosink->frame_time/1000000);
      break;
    }
    case ARG_DISABLE_XV:
      g_value_set_boolean (value, xvideosink->disable_xv);
      break;
    default: {
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
    }
  }
}

static GstCaps*
gst_xvideosink_get_padtemplate_caps (gboolean with_xv)
{
  GstXWindow *window;
  GstXImage *ximage;
  GstCaps *caps = NULL;

  window = gst_xwindow_new (100, 100);
  if (window == NULL)
    return NULL;
  
  ximage = gst_ximage_new (window, 100, 100);
  if (ximage) {
    caps = gst_caps_new (
	               "xvideosink_caps",
	               "video/raw",
		       gst_props_new (
		           "format",       GST_PROPS_FOURCC (GST_MAKE_FOURCC ('R', 'G', 'B', ' ')),
		             "bpp",        GST_PROPS_INT (GST_XIMAGE_BPP (ximage)),
		             "depth",      GST_PROPS_INT (GST_XIMAGE_DEPTH (ximage)),
		             "endianness", GST_PROPS_INT (GST_XIMAGE_ENDIANNESS (ximage)),
		             "red_mask",   GST_PROPS_INT (GST_XIMAGE_RED_MASK (ximage)),
		             "green_mask", GST_PROPS_INT (GST_XIMAGE_GREEN_MASK (ximage)),
		             "blue_mask",  GST_PROPS_INT (GST_XIMAGE_BLUE_MASK (ximage)),
		             "width",      GST_PROPS_INT_RANGE (0, G_MAXINT),
		             "height",     GST_PROPS_INT_RANGE (0, G_MAXINT),
		            NULL));

    gst_ximage_destroy (ximage);
  }

  if (with_xv && gst_xvimage_check_xvideo ()) {
    gst_xvimage_init();

    caps = gst_caps_prepend (caps, gst_xvimage_get_capslist ());
  }

  gst_xwindow_destroy (window);

  return caps;
}


static gboolean
plugin_init (GModule *module, GstPlugin *plugin)
{
  GstElementFactory *factory;

  /* create an elementfactory for the xvideosink element */
  factory = gst_elementfactory_new("xvideosink",GST_TYPE_XVIDEOSINK,
                                   &gst_xvideosink_details);
  g_return_val_if_fail(factory != NULL, FALSE);

  formats = gst_xvideosink_get_padtemplate_caps (TRUE);

  sink_template = gst_padtemplate_new (
		  "sink",
                  GST_PAD_SINK,
  		  GST_PAD_ALWAYS,
		  formats, NULL);

  gst_elementfactory_add_padtemplate (factory, sink_template);

  gst_plugin_add_factory(plugin,factory);

  return TRUE;
}

GstPluginDesc plugin_desc = {
  GST_VERSION_MAJOR,
  GST_VERSION_MINOR,
  "xvideosink",
  plugin_init
};
