/* Gnome-Streamer
 * Copyright (C) <1999> Erik Walthinsen <omega@cse.ogi.edu>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */


//#define GST_DEBUG_ENABLED
#include <string.h>

#include "gstavidemux.h"



/* elementfactory information */
static GstElementDetails gst_avi_demux_details = {
  ".avi parser",
  "Parser/Video",
  "Parse a .avi file into audio and video",
  VERSION,
  "Erik Walthinsen <omega@cse.ogi.edu>\n"
  "Wim Taymans <wim.taymans@tvd.be>",
  "(C) 1999",
};

static GstCaps* avi_typefind (GstBuffer *buf, gpointer private);

/* typefactory for 'avi' */
static GstTypeFactory avifactory = {
  "video/avi",
  ".avi",
  avi_typefind,
};

/* AviDemux signals and args */
enum {
  /* FILL ME */
  LAST_SIGNAL
};

enum {
  ARG_0,
  ARG_BITRATE,
  ARG_MEDIA_TIME,
  ARG_CURRENT_TIME,
  /* FILL ME */
};

GST_PADTEMPLATE_FACTORY (sink_templ,
  "sink",
  GST_PAD_SINK,
  GST_PAD_ALWAYS,
  GST_CAPS_NEW (
    "avidemux_sink",
     "video/avi",
     NULL
  )
)

GST_PADTEMPLATE_FACTORY (src_video_templ,
  "video_[00-32]",
  GST_PAD_SRC,
  GST_PAD_ALWAYS,
  GST_CAPS_NEW (
    "src_video",
    "video/avi",
      "format",  GST_PROPS_STRING ("strf_vids")
  )
)

GST_PADTEMPLATE_FACTORY (src_audio_templ,
  "src",
  GST_PAD_SRC,
  GST_PAD_ALWAYS,
  GST_CAPS_NEW (
    "src_audio",
    "video/avi",
      "format", GST_PROPS_STRING ("strf_auds")
  )
)

static void 	gst_avi_demux_class_init	(GstAviDemuxClass *klass);
static void 	gst_avi_demux_init		(GstAviDemux *avi_demux);

static void 	gst_avi_demux_chain 		(GstPad *pad, GstBuffer *buf);
static void 	gst_avi_demux_loop 		(GstElement *element);

static void     gst_avi_demux_get_property      (GObject *object, guint prop_id, GValue *value, GParamSpec *pspec);


static GstElementClass *parent_class = NULL;
//static guint gst_avi_demux_signals[LAST_SIGNAL] = { 0 };

GType
gst_avi_demux_get_type(void) 
{
  static GType avi_demux_type = 0;

  if (!avi_demux_type) {
    static const GTypeInfo avi_demux_info = {
      sizeof(GstAviDemuxClass),      
      NULL,
      NULL,
      (GClassInitFunc)gst_avi_demux_class_init,
      NULL,
      NULL,
      sizeof(GstAviDemux),
      0,
      (GInstanceInitFunc)gst_avi_demux_init,
    };
    avi_demux_type = g_type_register_static(GST_TYPE_ELEMENT, "GstAviDemux", &avi_demux_info, 0);
  }
  return avi_demux_type;
}

static void
gst_avi_demux_class_init (GstAviDemuxClass *klass) 
{
  GObjectClass *gobject_class;
  GstElementClass *gstelement_class;

  gobject_class = (GObjectClass*)klass;
  gstelement_class = (GstElementClass*)klass;

  g_object_class_install_property (G_OBJECT_CLASS(klass), ARG_BITRATE,
    g_param_spec_long ("bitrate","bitrate","bitrate",
                       G_MINLONG, G_MAXLONG, 0, G_PARAM_READABLE)); // CHECKME
  g_object_class_install_property (G_OBJECT_CLASS(klass), ARG_MEDIA_TIME,
    g_param_spec_long ("media_time","media_time","media_time",
                       G_MINLONG, G_MAXLONG, 0, G_PARAM_READABLE)); // CHECKME
  g_object_class_install_property (G_OBJECT_CLASS(klass), ARG_CURRENT_TIME,
    g_param_spec_long ("current_time","current_time","current_time",
                       G_MINLONG, G_MAXLONG, 0, G_PARAM_READABLE)); // CHECKME

  parent_class = g_type_class_ref (GST_TYPE_ELEMENT);
  
  gobject_class->get_property = gst_avi_demux_get_property;
}

static void 
gst_avi_demux_init (GstAviDemux *avi_demux) 
{
  guint i;
				
  avi_demux->sinkpad = gst_pad_new_from_template (
		  GST_PADTEMPLATE_GET (sink_templ), "sink");
  gst_element_add_pad (GST_ELEMENT (avi_demux), avi_demux->sinkpad);

  gst_element_set_loop_function (GST_ELEMENT (avi_demux), gst_avi_demux_loop);

  avi_demux->state = GST_AVI_DEMUX_UNKNOWN;
  avi_demux->riff = NULL;
  avi_demux->num_audio_pads = 0;
  avi_demux->num_video_pads = 0;
  avi_demux->next_time = 0;
  avi_demux->flags = 0;
  avi_demux->index_entries = NULL;
  avi_demux->index_size = 0;
  avi_demux->resync_offset = 0;

  //GST_FLAG_SET( GST_OBJECT (avi_demux), GST_ELEMENT_NO_SEEK);

  for(i=0; i<GST_AVI_DEMUX_MAX_AUDIO_PADS; i++) 
    avi_demux->audio_pad[i] = NULL;

  for(i=0; i<GST_AVI_DEMUX_MAX_VIDEO_PADS; i++) 
    avi_demux->video_pad[i] = NULL;
}

static GstCaps*
avi_typefind (GstBuffer *buf,
              gpointer private)
{
  gchar *data = GST_BUFFER_DATA (buf);
  GstCaps *new;

  GST_DEBUG (0,"avi_demux: typefind\n");
  if (strncmp (&data[0], "RIFF", 4)) return NULL;
  if (strncmp (&data[8], "AVI ", 4)) return NULL;

  new = gst_caps_new ("avi_typefind","video/avi", NULL);

  return new;
}

static void
gst_avi_demux_avih (GstRiffChunk *chunk,
		   GstAviDemux *avi_demux)
{
  gst_riff_avih *avih;
  g_return_if_fail (chunk != NULL);
  g_return_if_fail (GST_IS_AVI_DEMUX (avi_demux));

  avih = (gst_riff_avih *)chunk->data;

  GST_INFO (GST_CAT_PLUGIN_INFO, "gst_avi_demux: avih tag found size %08x\n", chunk->size);
  GST_INFO (GST_CAT_PLUGIN_INFO, "gst_avi_demux:  us_frame    %d\n", avih->us_frame);
  GST_INFO (GST_CAT_PLUGIN_INFO, "gst_avi_demux:  max_bps     %d\n", avih->max_bps);
  GST_INFO (GST_CAT_PLUGIN_INFO, "gst_avi_demux:  pad_gran    %d\n", avih->pad_gran);
  GST_INFO (GST_CAT_PLUGIN_INFO, "gst_avi_demux:  flags       0x%08x\n", avih->flags);
  GST_INFO (GST_CAT_PLUGIN_INFO, "gst_avi_demux:  tot_frames  %d\n", avih->tot_frames);
  GST_INFO (GST_CAT_PLUGIN_INFO, "gst_avi_demux:  init_frames %d\n", avih->init_frames);
  GST_INFO (GST_CAT_PLUGIN_INFO, "gst_avi_demux:  streams     %d\n", avih->streams);
  GST_INFO (GST_CAT_PLUGIN_INFO, "gst_avi_demux:  bufsize     %d\n", avih->bufsize);
  GST_INFO (GST_CAT_PLUGIN_INFO, "gst_avi_demux:  width       %d\n", avih->width);
  GST_INFO (GST_CAT_PLUGIN_INFO, "gst_avi_demux:  height      %d\n", avih->height);
  GST_INFO (GST_CAT_PLUGIN_INFO, "gst_avi_demux:  scale       %d\n", avih->scale);
  GST_INFO (GST_CAT_PLUGIN_INFO, "gst_avi_demux:  rate        %d\n", avih->rate);
  GST_INFO (GST_CAT_PLUGIN_INFO, "gst_avi_demux:  start       %d\n", avih->start);
  GST_INFO (GST_CAT_PLUGIN_INFO, "gst_avi_demux:  length      %d\n", avih->length);

  avi_demux->time_interval = avih->us_frame;
  avi_demux->tot_frames = avih->tot_frames;
  avi_demux->flags = avih->flags;
}

static void 
gst_avi_demux_strh (GstRiffChunk *chunk, 
		   GstAviDemux *avi_demux)
{
  gst_riff_strh *strh;

  strh = (gst_riff_strh *)chunk->data;
  GST_INFO (GST_CAT_PLUGIN_INFO, "gst_avi_demux: strh tag found size %08x\n", chunk->size);
  GST_INFO (GST_CAT_PLUGIN_INFO, "gst_avi_demux:  type        0x%08x (%s)\n", 
		  strh->type, gst_riff_id_to_fourcc (strh->type));
  GST_INFO (GST_CAT_PLUGIN_INFO, "gst_avi_demux:  fcc_handler 0x%08x (%s)\n", 
		  strh->fcc_handler, gst_riff_id_to_fourcc (strh->fcc_handler));
  GST_INFO (GST_CAT_PLUGIN_INFO, "gst_avi_demux:  flags       0x%08x\n", strh->flags);
  GST_INFO (GST_CAT_PLUGIN_INFO, "gst_avi_demux:  priority    %d\n", strh->priority);
  GST_INFO (GST_CAT_PLUGIN_INFO, "gst_avi_demux:  init_frames %d\n", strh->init_frames);
  GST_INFO (GST_CAT_PLUGIN_INFO, "gst_avi_demux:  scale       %d\n", strh->scale);
  GST_INFO (GST_CAT_PLUGIN_INFO, "gst_avi_demux:  rate        %d\n", strh->rate);
  GST_INFO (GST_CAT_PLUGIN_INFO, "gst_avi_demux:  start       %d\n", strh->start);
  GST_INFO (GST_CAT_PLUGIN_INFO, "gst_avi_demux:  length      %d\n", strh->length);
  GST_INFO (GST_CAT_PLUGIN_INFO, "gst_avi_demux:  bufsize     %d\n", strh->bufsize);
  GST_INFO (GST_CAT_PLUGIN_INFO, "gst_avi_demux:  quality     %d\n", strh->quality);
  GST_INFO (GST_CAT_PLUGIN_INFO, "gst_avi_demux:  samplesize  %d\n", strh->samplesize);
}

static void 
gst_avi_demux_strf_vids (GstRiffChunk *chunk, 
		        GstAviDemux *avi_demux)
{
  gst_riff_strf_vids *strf;
  GstPad *srcpad;

  strf = (gst_riff_strf_vids *)chunk->data;
  GST_INFO (GST_CAT_PLUGIN_INFO, "gst_avi_demux: strf tag found in context vids size %08x\n", chunk->size);
  GST_INFO (GST_CAT_PLUGIN_INFO, "gst_avi_demux:  size        %d\n", strf->size);
  GST_INFO (GST_CAT_PLUGIN_INFO, "gst_avi_demux:  width       %d\n", strf->width);
  GST_INFO (GST_CAT_PLUGIN_INFO, "gst_avi_demux:  height      %d\n", strf->height);
  GST_INFO (GST_CAT_PLUGIN_INFO, "gst_avi_demux:  planes      %d\n", strf->planes);
  GST_INFO (GST_CAT_PLUGIN_INFO, "gst_avi_demux:  bit_cnt     %d\n", strf->bit_cnt);
  GST_INFO (GST_CAT_PLUGIN_INFO, "gst_avi_demux:  compression 0x%08x (%s)\n", 
		  strf->compression, gst_riff_id_to_fourcc (strf->compression));
  GST_INFO (GST_CAT_PLUGIN_INFO, "gst_avi_demux:  image_size  %d\n", strf->image_size);
  GST_INFO (GST_CAT_PLUGIN_INFO, "gst_avi_demux:  xpels_meter %d\n", strf->xpels_meter);
  GST_INFO (GST_CAT_PLUGIN_INFO, "gst_avi_demux:  ypels_meter %d\n", strf->ypels_meter);
  GST_INFO (GST_CAT_PLUGIN_INFO, "gst_avi_demux:  num_colors  %d\n", strf->num_colors);
  GST_INFO (GST_CAT_PLUGIN_INFO, "gst_avi_demux:  imp_colors  %d\n", strf->imp_colors);

  srcpad =  gst_pad_new_from_template (
		  GST_PADTEMPLATE_GET (src_video_templ), g_strdup_printf ("video_%02d", 
			  avi_demux->num_video_pads));

  gst_pad_set_caps (srcpad, gst_caps_new (
			  "avidec_video_src",
			  "video/avi",
			  gst_props_new (
			    "format",    	GST_PROPS_STRING ("strf_vids"),
			      "size", 		GST_PROPS_INT (strf->size),
			      "width", 		GST_PROPS_INT (strf->width),
			      "height", 	GST_PROPS_INT (strf->height),
			      "planes", 	GST_PROPS_INT (strf->planes),
			      "bit_cnt", 	GST_PROPS_INT (strf->bit_cnt),
			      "compression", 	GST_PROPS_FOURCC (strf->compression),
			      "image_size", 	GST_PROPS_INT (strf->image_size),
			      "xpels_meter", 	GST_PROPS_INT (strf->xpels_meter),
			      "ypels_meter", 	GST_PROPS_INT (strf->ypels_meter),
			      "num_colors", 	GST_PROPS_INT (strf->num_colors),
			      "imp_colors", 	GST_PROPS_INT (strf->imp_colors),
			      NULL)));

  avi_demux->video_pad[avi_demux->num_video_pads++] = srcpad;
  gst_element_add_pad (GST_ELEMENT (avi_demux), srcpad);
}

static void 
gst_avi_demux_strf_auds (GstRiffChunk *chunk, 
		           GstAviDemux *avi_demux)
{
  gst_riff_strf_auds *strf;
  GstPad *srcpad;

  strf = (gst_riff_strf_auds *)chunk->data;
  GST_INFO (GST_CAT_PLUGIN_INFO, "gst_avi_demux: strf tag found in context auds size %08x\n", chunk->size);
  GST_INFO (GST_CAT_PLUGIN_INFO, "gst_avi_demux:  format      %d\n", strf->format);
  GST_INFO (GST_CAT_PLUGIN_INFO, "gst_avi_demux:  channels    %d\n", strf->channels);
  GST_INFO (GST_CAT_PLUGIN_INFO, "gst_avi_demux:  rate        %d\n", strf->rate);
  GST_INFO (GST_CAT_PLUGIN_INFO, "gst_avi_demux:  av_bps      %d\n", strf->av_bps);
  GST_INFO (GST_CAT_PLUGIN_INFO, "gst_avi_demux:  blockalign  %d\n", strf->blockalign);
  GST_INFO (GST_CAT_PLUGIN_INFO, "gst_avi_demux:  size        %d\n", strf->size);

  srcpad =  gst_pad_new_from_template (
		  GST_PADTEMPLATE_GET (src_audio_templ), g_strdup_printf ("audio_%02d", 
			  avi_demux->num_audio_pads));

  gst_pad_set_caps (srcpad, gst_caps_new (
			  "avidec_audio_src",
			  "video/avi",
			  gst_props_new (
			    "format",    	GST_PROPS_STRING ("strf_auds"),
			      "fmt", 		GST_PROPS_INT (strf->format),
			      "channels", 	GST_PROPS_INT (strf->channels),
			      "rate", 		GST_PROPS_INT (strf->rate),
			      "av_bps",		GST_PROPS_INT (strf->av_bps),
			      "blockalign",	GST_PROPS_INT (strf->blockalign),
			      "size",		GST_PROPS_INT (strf->size),
			      NULL)));

  avi_demux->audio_pad[avi_demux->num_audio_pads++] = srcpad;
  gst_element_add_pad (GST_ELEMENT (avi_demux), srcpad);
}

static void
gst_avidemux_parse_index (GstAviDemux *avi_demux,
		            gulong offset)
{
  GstBuffer *buf;
  gulong index_size;

  buf = gst_pad_pullregion (avi_demux->sinkpad, GST_REGION_OFFSET_LEN, offset, 8);

  if (!buf || GST_BUFFER_OFFSET (buf) != offset || GST_BUFFER_SIZE (buf) != 8) {
    GST_INFO (GST_CAT_PLUGIN_INFO, "avidemux: could not get index\n");
    return;
  }

  if (gst_riff_fourcc_to_id (GST_BUFFER_DATA (buf)) != GST_RIFF_TAG_idx1) {
    GST_INFO (GST_CAT_PLUGIN_INFO, "avidemux: no index found\n");
    return;
  }

  index_size = GUINT32_FROM_LE(*(guint32 *)(GST_BUFFER_DATA (buf) + 4));

  buf = gst_pad_pullregion(avi_demux->sinkpad, GST_REGION_OFFSET_LEN, offset+8, index_size);

  avi_demux->index_size = index_size/sizeof(gst_riff_index_entry);

  GST_INFO (GST_CAT_PLUGIN_INFO, "avidemux: index size %lu\n", avi_demux->index_size);

  avi_demux->index_entries = g_malloc (GST_BUFFER_SIZE (buf));
  memcpy (avi_demux->index_entries, GST_BUFFER_DATA (buf), GST_BUFFER_SIZE (buf));

  buf = gst_pad_pullregion(avi_demux->sinkpad, GST_REGION_OFFSET_LEN, avi_demux->index_offset, 0);
}

static void
gst_new_riff_tag_found (GstRiffChunk *chunk,
		        gpointer data)
{
  GstAviDemux *avi_demux;
  gst_riff_strh *strh;
  GstBuffer *buf;

  GST_DEBUG (0,"gst_avi_demux_chain: new tag found size %08x id %s\n",
		  chunk->size, gst_riff_id_to_fourcc(chunk->id));

  g_return_if_fail (chunk != NULL);
  g_return_if_fail (data != NULL);
  g_return_if_fail (GST_IS_AVI_DEMUX (data));

  avi_demux = GST_AVI_DEMUX (data);

  switch (chunk->id) {
    case GST_RIFF_TAG_LIST:
      switch (chunk->form) {
	case GST_RIFF_LIST_hdrl:
          avi_demux->state = GST_AVI_DEMUX_HDRL;
          break;
	case GST_RIFF_LIST_strl:
          avi_demux->state = GST_AVI_DEMUX_STRL;
          break;
	case GST_RIFF_LIST_movi:
	  GST_INFO (GST_CAT_PLUGIN_INFO, "avidemux: index offset %08lx\n", chunk->offset);
	  avi_demux->index_offset = chunk->offset;
	  gst_avidemux_parse_index (avi_demux, chunk->offset + chunk->size);
          avi_demux->state = GST_AVI_DEMUX_MOVI;
          break;
	default:
          GST_INFO (GST_CAT_PLUGIN_INFO, "gst_avi_demux_chain: invalid LIST tag found size %08x form %s\n",
			  chunk->size, gst_riff_id_to_fourcc (chunk->form));
      }
      break;
    case GST_RIFF_TAG_avih:
      if (avi_demux->state != GST_AVI_DEMUX_HDRL) {
        GST_INFO (GST_CAT_PLUGIN_INFO, "gst_avi_demux_chain: invalid avih found\n");
      }
      gst_avi_demux_avih (chunk, avi_demux);
      avi_demux->state = GST_AVI_DEMUX_AVIH;
      break;
    case GST_RIFF_TAG_strh:
      gst_avi_demux_strh (chunk, avi_demux);
      strh = (gst_riff_strh *)chunk->data;
      switch (strh->type) {
	case GST_RIFF_FCC_vids:
          avi_demux->state = GST_AVI_DEMUX_STRH_VIDS;
          break;
	case GST_RIFF_FCC_auds:
          avi_demux->state = GST_AVI_DEMUX_STRH_AUDS;
	  break;
	case GST_RIFF_FCC_pads:
	case GST_RIFF_FCC_txts:
          GST_INFO (GST_CAT_PLUGIN_INFO, "gst_avi_demux_chain: strh type %d not supported\n", strh->type);
	  break;
	default:
          GST_INFO (GST_CAT_PLUGIN_INFO, "gst_avi_demux_chain: invalid strh type %d\n", strh->type);
      }
      break;
    case GST_RIFF_TAG_strf:
      switch (avi_demux->state) {
	case GST_AVI_DEMUX_STRH_VIDS:
          gst_avi_demux_strf_vids (chunk, avi_demux);
	  break;
	case GST_AVI_DEMUX_STRH_AUDS:
	  gst_avi_demux_strf_auds (chunk, avi_demux);
	  break;
	default:
          GST_INFO (GST_CAT_PLUGIN_INFO, "gst_avi_demux_chain: unknown strf received\n");
      }
      break;
    case GST_RIFF_TAG_JUNK:
      break;
    default:
      if (avi_demux->state != GST_AVI_DEMUX_MOVI) {
        GST_INFO (GST_CAT_PLUGIN_INFO, "gst_avi_demux_chain: unknown tag found %s size %08x\n",
			gst_riff_id_to_fourcc (chunk->id), chunk->size);
      }
      else {
        switch (chunk->id) {
	  case GST_RIFF_00dc:
	  case GST_RIFF_00db:

            GST_DEBUG (0,"gst_avi_demux_chain: tag found %08x size %08x offset %08lx\n",
			    chunk->id, chunk->size, chunk->offset);

	    if (GST_PAD_CONNECTED (avi_demux->video_pad[0])) {
	      buf = gst_buffer_new ();
	      GST_BUFFER_DATA (buf) = g_malloc (chunk->size);
	      GST_BUFFER_SIZE (buf) = chunk->size;
	      memcpy (GST_BUFFER_DATA (buf), chunk->data, chunk->size);
	      GST_BUFFER_TIMESTAMP (buf) = avi_demux->next_time;
	      avi_demux->next_time += avi_demux->time_interval;

	      if (avi_demux->video_need_flush[0]) {
	        avi_demux->video_need_flush[0] = FALSE;
		GST_BUFFER_FLAG_SET (buf, GST_BUFFER_FLUSH);
	      }

              GST_DEBUG (0,"gst_avi_demux_chain: send video buffer %08x\n", chunk->size);
	      gst_pad_push(avi_demux->video_pad[0], buf);
              GST_DEBUG (0,"gst_avi_demux_chain: sent video buffer %08x %p\n",
			      chunk->size, &avi_demux->video_pad[0]);
	      avi_demux->current_frame++;
	    }

	    break;

	  case GST_RIFF_01wb:
	    if (GST_PAD_CONNECTED (avi_demux->audio_pad[0])) {
	      buf = gst_buffer_new ();
	      GST_BUFFER_DATA (buf) = g_malloc (chunk->size);
	      GST_BUFFER_SIZE (buf) = chunk->size;
	      memcpy (GST_BUFFER_DATA (buf), chunk->data, chunk->size);
	      GST_DEBUG (0,"gst_avi_demux_chain: buffer: %08x end %08x\n",
			      *(guint32 *)chunk->data,
			      *(guint32 *)(chunk->data+chunk->size-4));

	      if (avi_demux->audio_need_flush[0]) {
		GST_DEBUG (0,"audio flush\n");
	        avi_demux->audio_need_flush[0] = FALSE;
		GST_BUFFER_FLAG_SET (buf, GST_BUFFER_FLUSH);
	      }

              GST_DEBUG (0,"gst_avi_demux_chain: send audio buffer %08x\n", chunk->size);
	      gst_pad_push (avi_demux->audio_pad[0], buf);
              GST_DEBUG (0,"gst_avi_demux_chain: sent audio buffer %08x\n", chunk->size);
	    }
	    break;
	}
      }
  }
}

static void
gst_avi_demux_loop (GstElement *element)
{
  GstAviDemux *avi_demux;
  GstBuffer *buf;

  g_return_if_fail (element != NULL);
  g_return_if_fail (GST_IS_AVI_DEMUX (element));

  avi_demux = GST_AVI_DEMUX (element);

  do {
    GST_DEBUG (0,"gst_avi_demux_chain: pulling buffer \n");
    buf = gst_pad_pull(avi_demux->sinkpad);
    GST_DEBUG (0,"gst_avi_demux_chain: pulling buffer done\n");

    if (buf) gst_avi_demux_chain(avi_demux->sinkpad, buf);
  } while (!GST_ELEMENT_IS_COTHREAD_STOPPING(element));
}

static void
gst_avi_demux_chain (GstPad *pad,
		       GstBuffer *buf)
{
  GstAviDemux *avi_demux;
  guchar *data;
  gulong size;

  g_return_if_fail (pad != NULL);
  g_return_if_fail (GST_IS_PAD(pad));
  g_return_if_fail (buf != NULL);
  g_return_if_fail (GST_BUFFER_DATA(buf) != NULL);

  avi_demux = GST_AVI_DEMUX (gst_pad_get_parent (pad));
  GST_DEBUG (0,"gst_avi_demux_chain: got buffer in %u\n", GST_BUFFER_OFFSET (buf));
  data = (guchar *)GST_BUFFER_DATA (buf);
  size = GST_BUFFER_SIZE (buf);

  /* we're in null state now, look for the RIFF header, start parsing */
  if (avi_demux->state == GST_AVI_DEMUX_UNKNOWN) {
    gint retval;

    GST_INFO (GST_CAT_PLUGIN_INFO, "GstAviDemux: checking for RIFF format\n");

    avi_demux->state = GST_AVI_DEMUX_REGULAR;

    /* create a new RIFF parser */
    avi_demux->riff = gst_riff_parser_new (gst_new_riff_tag_found, avi_demux);
    /* give it the current buffer to start parsing */
    retval = gst_riff_parser_next_buffer (avi_demux->riff,buf,0);
    if (retval < 0) {
      GST_INFO (GST_CAT_PLUGIN_INFO, "sorry, isn't RIFF\n");
      return;
    }

    /* this has to be a file of form AVI for us to deal with it */
    if (avi_demux->riff->form != GST_RIFF_RIFF_AVI) {
      GST_INFO (GST_CAT_PLUGIN_INFO, "sorry, isn't AVI\n");
      return;
    }
    avi_demux->current_frame = 0;
  } else {

    if (GST_BUFFER_FLAG_IS_SET (buf, GST_BUFFER_FLUSH)) {
      gint i;
      gst_riff_index_entry *entry = NULL;
      gulong offset = 0;

      GST_DEBUG (0,"avidemux: flushing\n");
      
      for (i=0; i<GST_AVI_DEMUX_MAX_VIDEO_PADS; i++) {
	avi_demux->video_need_flush[i] = TRUE;
      }
      for (i=0; i<GST_AVI_DEMUX_MAX_AUDIO_PADS; i++) {
	avi_demux->audio_need_flush[i] = TRUE;
      }
      
      avi_demux->current_frame = 0;
	      
      for (i=0; i<avi_demux->index_size; i++) {

	entry = &avi_demux->index_entries[i];

	offset = entry->offset + avi_demux->index_offset;

	if ((entry->id == GST_RIFF_00dc) || (entry->id == GST_RIFF_00db)) 
	{
	  avi_demux->current_frame++;

	  if (offset >= GST_BUFFER_OFFSET (buf) && (entry->flags & GST_RIFF_IF_KEYFRAME))
            break;
	}
      }

      if (!entry) 
	GST_INFO (GST_CAT_PLUGIN_INFO, "avidemux: cannot sync\n");
      else  {
        avi_demux->resync_offset = offset;

	GST_DEBUG (0,"avidemux: sync from %08x to %08lx\n", GST_BUFFER_OFFSET (buf), offset);

        gst_riff_parser_resync (avi_demux->riff, offset);
      }
    }

    if (avi_demux->resync_offset) {
      if ((avi_demux->resync_offset - GST_BUFFER_OFFSET (buf))
		      < GST_BUFFER_SIZE (buf)) {
	
	GstBuffer *outbuf;
	
	GST_DEBUG (0,"creating subbuffer %08lx %08lx\n", avi_demux->resync_offset - GST_BUFFER_OFFSET (buf), 
			GST_BUFFER_SIZE (buf) - (avi_demux->resync_offset - GST_BUFFER_OFFSET (buf)));

	outbuf = gst_buffer_create_sub (buf, avi_demux->resync_offset - GST_BUFFER_OFFSET (buf),
			GST_BUFFER_SIZE (buf) - (avi_demux->resync_offset - GST_BUFFER_OFFSET (buf)));

        gst_riff_parser_next_buffer (avi_demux->riff, outbuf, avi_demux->resync_offset);

	avi_demux->resync_offset = 0;

        gst_buffer_unref (outbuf);
      }
    }
    else {
      gst_riff_parser_next_buffer (avi_demux->riff, buf, GST_BUFFER_OFFSET (buf));
    }
  }
  gst_buffer_unref (buf);
}

static void
gst_avi_demux_get_property (GObject *object, guint prop_id, GValue *value, GParamSpec *pspec)
{
  GstAviDemux *src;

  g_return_if_fail (GST_IS_AVI_DEMUX (object));

  src = GST_AVI_DEMUX (object);

  switch(prop_id) {
    case ARG_BITRATE:
      break;
    case ARG_MEDIA_TIME:
      g_value_set_long (value, (src->tot_frames*src->time_interval)/1000000);
      break;
    case ARG_CURRENT_TIME:
      g_value_set_long (value, (src->current_frame*src->time_interval)/1000000);
      break;
    default:
      break;
  }
}


static gboolean
plugin_init (GModule *module, GstPlugin *plugin)
{
  GstElementFactory *factory;

  /* this filter needs the riff parser */
  if (!gst_library_load ("gstriff")) {
    gst_info("avidemux: could not load support library: 'gstriff'\n");
    return FALSE;
  }

  /* create an elementfactory for the avi_demux element */
  factory = gst_elementfactory_new ("avidemux",GST_TYPE_AVI_DEMUX,
                                    &gst_avi_demux_details);
  g_return_val_if_fail (factory != NULL, FALSE);

  gst_elementfactory_add_padtemplate (factory, GST_PADTEMPLATE_GET (src_audio_templ));
  gst_elementfactory_add_padtemplate (factory, GST_PADTEMPLATE_GET (src_video_templ));
  gst_elementfactory_add_padtemplate (factory, GST_PADTEMPLATE_GET (sink_templ));

  gst_plugin_add_type(plugin, &avifactory);

  gst_plugin_add_factory (plugin,factory);

  return TRUE;
}

GstPluginDesc plugin_desc = {
  GST_VERSION_MAJOR,
  GST_VERSION_MINOR,
  "avidemux",
  plugin_init
};

