/* Gnome-Streamer
 * Copyright (C) <1999> Erik Walthinsen <omega@cse.ogi.edu>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */


#include <string.h>


#include "config.h"
#include "gstwindec.h"

#include <creators.h>

/* elementfactory information */
GstElementDetails gst_windec_details = {
  "Windows codec decoder",
  "Filter/Decoder/Image",
  "Uses the Avifile library to decode avi video using the windows dlls",
  VERSION,
  "Wim Taymans <wim.taymans@chello.be> "
  "Eugene Kuznetsov (http://divx.euro.ru)",
  "(C) 2000",
};

extern GstPadTemplate *wincodec_src_temp;
extern GstPadTemplate *wincodec_sink_temp;

/* Winec signals and args */
enum {
  /* FILL ME */
  LAST_SIGNAL
};

enum {
  ARG_0,
  /* FILL ME */
};

static void 	gst_windec_class_init		(GstWinDec *klass);
static void 	gst_windec_init			(GstWinDec *windec);

static void 	gst_windec_chain 		(GstPad *pad, GstBuffer *buf);

static void     gst_windec_get_property         (GObject *object, guint prop_id, GValue *value, GParamSpec *pspec);
static void     gst_windec_set_property         (GObject *object, guint prop_id, const GValue *value, GParamSpec *pspec);


//static int formats[] = { 32, 24, 16, 15, 0 };

typedef struct _GstWinLoaderData GstWinLoaderData;

static GstElementClass *parent_class = NULL;
//static guint gst_windec_signals[LAST_SIGNAL] = { 0 };

GType
gst_windec_get_type (void) 
{
  static GType windec_type = 0;

  if (!windec_type) {
    static const GTypeInfo windec_info = {
      sizeof(GstWinDecClass),      
      NULL,
      NULL,
      (GClassInitFunc) gst_windec_class_init,
      NULL,
      NULL,
      sizeof(GstWinDec),
      0,
      (GInstanceInitFunc) gst_windec_init,
      NULL
    };
    windec_type = g_type_register_static (GST_TYPE_ELEMENT, "GstWinDec", &windec_info, (GTypeFlags)0);
  }

  return windec_type;
}

static void
gst_windec_class_init (GstWinDec *klass) 
{
  GObjectClass *gobject_class;
  GstElementClass *gstelement_class;

  gobject_class = (GObjectClass*)klass;
  gstelement_class = (GstElementClass*)klass;

  parent_class = GST_ELEMENT_CLASS (g_type_class_ref (GST_TYPE_ELEMENT));

  gobject_class->set_property = gst_windec_set_property;
  gobject_class->get_property = gst_windec_get_property;
}

static GstCaps* 
gst_windec_bh_to_caps (BITMAPINFOHEADER *bh) 
{
    GstCaps *caps;
    gulong compression;
    gint bitCount;
    gint red_mask, green_mask, blue_mask;

    if (bh->biCompression == 0) 
      compression = GST_MAKE_FOURCC ('R','G','B',' ');
    else
      compression = bh->biCompression;

    if (compression != GST_MAKE_FOURCC ('R','G','B',' ')) {
      caps = gst_caps_new (
		    "windec_caps",
		    "video/raw",
		    gst_props_new (
			  "format",    GST_PROPS_FOURCC (compression),
	                    "width",      GST_PROPS_INT (bh->biWidth),
	                    "height",     GST_PROPS_INT (ABS (bh->biHeight)),
			    NULL));
    }
    else {
      bitCount = bh->biBitCount;

      switch (bitCount) {
        case 15:
          red_mask   = 0x7c00;
          green_mask = 0x03e0;
          blue_mask  = 0x001f;
	  break;
        case 16:
          red_mask   = 0xf800;
          green_mask = 0x07e0;
          blue_mask  = 0x001f;
	  break;
        case 32:
        case 24:
          red_mask   = 0x00ff0000;
          green_mask = 0x0000ff00;
          blue_mask  = 0x000000ff;
	  break;
        default:
	  g_warning ("gstwindec: unknown bitCount found\n");
	  red_mask = green_mask = blue_mask = 0;
	  break;
      }

      caps = gst_caps_new (
		    "windec_caps",
		    "video/raw",
		    gst_props_new (
			  "format",    GST_PROPS_FOURCC (compression),
	                    "bpp",        GST_PROPS_INT (bitCount),
	                    "depth",      GST_PROPS_INT (bitCount),
			    "endianness", GST_PROPS_INT (G_LITTLE_ENDIAN),
			    "red_mask",   GST_PROPS_INT (red_mask),
			    "green_mask", GST_PROPS_INT (green_mask),
			    "blue_mask",  GST_PROPS_INT (blue_mask),
	                    "width",      GST_PROPS_INT (bh->biWidth),
	                    "height",     GST_PROPS_INT (ABS (bh->biHeight)),
			    NULL));
    }
    return caps;
}

static int formats[] = { 32, 24, 16, 15, 0 };

static GstPadNegotiateReturn 
gst_windec_negotiate_src (GstPad *pad, GstCaps **caps, gpointer *user_data) 
{
  GstWinDec *windec;
  static int hack_counter = 0; //FIXME
  BITMAPINFOHEADER obh;
  int result;

  windec = GST_WINDEC (gst_pad_get_parent (pad));

  GST_INFO (GST_CAT_NEGOTIATION, "gstwindec: src negotiate\n");

  if (*caps) {
    gint bit_count = gst_caps_get_int (*caps, "depth");
    glong compression = gst_caps_get_fourcc_int (*caps, "format");

    GST_INFO (GST_CAT_NEGOTIATION, "got %08lx\n", compression);

    if (compression == GST_STR_FOURCC ("YUY2") ||
        compression == GST_STR_FOURCC ("RGB ")) 
    {
      result = windec->decoder->SetDestFmt (bit_count, compression);

      if (result >= 0) {
        return GST_PAD_NEGOTIATE_AGREE;
      }
      else {
        GST_INFO(GST_CAT_NEGOTIATION, "gstwindec:: Set Bit Depth failed\n");
      }
    }
    *caps = NULL;
    return GST_PAD_NEGOTIATE_TRY;
  }
  else {
    gint bit_count = formats[hack_counter++];

    if (bit_count == 0)
      return GST_PAD_NEGOTIATE_FAIL;

    result = windec->decoder->SetDestFmt (bit_count);

    if (result >= 0) {
      obh = windec->decoder->DestFmt();
      obh.biCompression = GST_STR_FOURCC ("RGB ");
      GST_INFO (GST_CAT_NEGOTIATION, "gstwindec: Suggest bit depth %d\n", obh.biBitCount);
      *caps = gst_windec_bh_to_caps (&obh);
    }
    else {
      GST_INFO(GST_CAT_NEGOTIATION, "gstwindec: Set Bit Depth failed\n");
      *caps = NULL;
    }
    return GST_PAD_NEGOTIATE_TRY;
  }

  return GST_PAD_NEGOTIATE_FAIL;
}

static void 
gst_windec_newcaps (GstPad *pad, GstCaps *newcaps) 
{
  GstWinDec *windec;
  BITMAPINFOHEADER bh, obh;
  IVideoDecoder::CAPS caps;

  windec = GST_WINDEC (gst_pad_get_parent (pad));

  bh.biSize 		= gst_caps_get_int (newcaps, "size");
  bh.biWidth 		= gst_caps_get_int (newcaps, "width");
  bh.biHeight 		= gst_caps_get_int (newcaps, "height");
  bh.biPlanes 		= gst_caps_get_int (newcaps, "planes");
  bh.biBitCount 	= gst_caps_get_int (newcaps, "bit_cnt");
  bh.biCompression 	= gst_caps_get_fourcc_int (newcaps, "compression");
  bh.biSizeImage 	= gst_caps_get_int (newcaps, "image_size");
  bh.biXPelsPerMeter 	= gst_caps_get_int (newcaps, "xpels_meter");
  bh.biYPelsPerMeter 	= gst_caps_get_int (newcaps, "ypels_meter");
  bh.biClrUsed 		= gst_caps_get_int (newcaps, "num_colors");
  bh.biClrImportant 	= gst_caps_get_int (newcaps, "imp_colors");

  windec->decoder = Creators::CreateVideoDecoder (bh);
  windec->decoder->Start ();
  obh = windec->decoder->DestFmt();

  caps = windec->decoder->GetCapabilities();

  if (caps == 0) {
    // No YUV output...
    if (gst_pad_set_caps (windec->srcpad, gst_windec_bh_to_caps (&obh))) {
      return;
    }
  }
  if (caps & IVideoDecoder::CAP_YUY2) {
    windec->decoder->SetDestFmt (0, GST_STR_FOURCC ("YUY2"));
    obh = windec->decoder->DestFmt();

    if (gst_pad_set_caps (windec->srcpad, gst_windec_bh_to_caps (&obh)))
      return;
  }
  g_warning ("avidecoder: could not find format\n");
}

static void 
gst_windec_init (GstWinDec *windec) 
{
  GST_DEBUG (0,"gst_windec_init: initializing\n");
  /* create the sink and src pads */
  windec->sinkpad = gst_pad_new_from_template (wincodec_sink_temp, "sink");
  gst_element_add_pad (GST_ELEMENT (windec), windec->sinkpad);
  gst_pad_set_chain_function (windec->sinkpad, gst_windec_chain);
  gst_pad_set_newcaps_function (windec->sinkpad, gst_windec_newcaps);

  windec->srcpad = gst_pad_new_from_template (wincodec_src_temp, "src");
  gst_element_add_pad (GST_ELEMENT (windec), windec->srcpad);
  gst_pad_set_negotiate_function (windec->srcpad, gst_windec_negotiate_src);

  windec->decoder = NULL;
}

static void 
gst_windec_chain (GstPad *pad, GstBuffer *buf)
{
  GstWinDec *windec;
  GstBuffer *outbuf;

  g_return_if_fail(buf != NULL);
  if (GST_BUFFER_DATA(buf) == NULL) return;

  windec = GST_WINDEC (gst_pad_get_parent (pad));

  if (!GST_PAD_CONNECTED(windec->srcpad)) {
    GST_DEBUG (0,"gst_windec: src pad not connected\n");
    gst_buffer_unref(buf);
    return;
  }

  outbuf = NULL;
  if (!windec->pool) {
    windec->pool = gst_pad_get_bufferpool (windec->srcpad);
  }
  if (windec->pool) {
    outbuf = gst_buffer_new_from_pool (windec->pool);
  }

  windec->decoder->DecodeFrame ((char*)GST_BUFFER_DATA (buf), (int)GST_BUFFER_SIZE (buf), 
  				GST_BUFFER_TIMESTAMP (buf), 0, FALSE);

  CImage *img = windec->decoder->GetFrame();

  if (!outbuf) {
    outbuf = gst_buffer_new ();
    GST_BUFFER_SIZE (outbuf) = img->Bytes();
    GST_BUFFER_DATA (outbuf) = (guchar *) g_malloc (GST_BUFFER_SIZE (outbuf));
  }

  memcpy (GST_BUFFER_DATA (outbuf), img->Data(), img->Bytes());
  img->Release();

  GST_BUFFER_TIMESTAMP (outbuf) = GST_BUFFER_TIMESTAMP (buf);

  GST_DEBUG (0,"gst_windec: pushing buffer %llu\n", GST_BUFFER_TIMESTAMP (outbuf));
  gst_pad_push (windec->srcpad, outbuf);
  GST_DEBUG (0,"gst_windec: pushed buffer\n");

  gst_buffer_unref (buf);
}

static void 
gst_windec_set_property (GObject *object, guint prop_id, const GValue *value, GParamSpec *pspec)
{
  GstWinDec *windec;

  /* it's not null if we got it, but it might not be ours */
  g_return_if_fail(GST_IS_WINDEC(object));
  windec = GST_WINDEC(object);

  switch(prop_id) {
    default:
      break;
  }
}

static void 
gst_windec_get_property (GObject *object, guint prop_id, GValue *value, GParamSpec *pspec)
{
  GstWinDec *windec;

  /* it's not null if we got it, but it might not be ours */
  g_return_if_fail(GST_IS_WINDEC(object));
  windec = GST_WINDEC(object);

  switch(prop_id) {
    default:
      break;
  }
}

