/* Gnome-Streamer
 * Copyright (C) <1999> Erik Walthinsen <omega@cse.ogi.edu>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <volume.h>

#ifdef HAVE_LIBMMX
#include <mmx.h>
#endif /* HAVE_LIBMMX */

static GstElementDetails volume_details = {
  "Volume control",
  "Filter/Effect",
  "Digitally control the volume of a signal",
  VERSION,
  "Erik Walthinsen <omega@cse.ogi.edu>",
  "(C) 1999",
};


/* Volume signals and args */
enum {
  /* FILL ME */
  LAST_SIGNAL
};

enum {
  ARG_0,
  ARG_ACTIVE,
  ARG_VOLUME
};


static void	gst_volume_class_init		(GstVolumeClass *klass);
static void	gst_volume_init			(GstVolume *volume);

static void	gst_volume_set_property		(GObject *object, guint prop_id, const GValue *value, GParamSpec *pspec);
static void	gst_volume_get_property		(GObject *object, guint prop_id, GValue *value, GParamSpec *pspec);

/* various optimized volume control functions */
static void	gst_volume_chain_none		(GstPad *pad, GstBuffer *buf);
static void	gst_volume_chain_fp_S16		(GstPad *pad, GstBuffer *buf);
static void	gst_volume_chain_fp_S8		(GstPad *pad, GstBuffer *buf);

static void	gst_volume_set_chain		(GstVolume *volume, gint format);
static void	gst_volume_rechain		(gint format, GstPad *pad, GstBuffer *buf);

static GstElementClass *parent_class = NULL;
//static guint gst_volume_signals[LAST_SIGNAL] = { 0 };

GType
gst_volume_get_type(void) {
  static GType volume_type = 0;

  if (!volume_type) {
    static const GTypeInfo volume_info = {
      sizeof(GstVolumeClass),      NULL,
      NULL,
      (GClassInitFunc)gst_volume_class_init,
      NULL,
      NULL,
      sizeof(GstVolume),
      0,
      (GInstanceInitFunc)gst_volume_init,
    };
    volume_type = g_type_register_static(GST_TYPE_ELEMENT, "GstVolume", &volume_info, 0);
  }
  return volume_type;
}

static void
gst_volume_class_init (GstVolumeClass *klass)
{
  GObjectClass *gobject_class;
  GstElementClass *gstelement_class;

  gobject_class = (GObjectClass*)klass;
  gstelement_class = (GstElementClass*)klass;

  parent_class = g_type_class_ref(GST_TYPE_ELEMENT);

  g_object_class_install_property(G_OBJECT_CLASS(klass), ARG_ACTIVE,
    g_param_spec_int("active","active","active",
                     G_MININT,G_MAXINT,0,G_PARAM_READWRITE)); // CHECKME
  g_object_class_install_property(G_OBJECT_CLASS(klass), ARG_VOLUME,
    g_param_spec_float("volume","volume","volume",
                       0.0,1.0,1.0,G_PARAM_READWRITE)); // CHECKME

  gobject_class->set_property = gst_volume_set_property;  
  gobject_class->get_property = gst_volume_get_property;

#ifdef HAVE_LIBMMX
  //klass->mmx = mm_support();
#endif /* HAVE_LIBMMX */
}

static void
gst_volume_init (GstVolume *volume)
{
  volume->sinkpad = gst_pad_new("sink",GST_PAD_SINK);
  gst_element_add_pad(GST_ELEMENT(volume),volume->sinkpad);
  volume->srcpad = gst_pad_new("src",GST_PAD_SRC);
  gst_element_add_pad(GST_ELEMENT(volume),volume->srcpad);

  volume->active = FALSE;
  volume->volume = 1.0;

  /* decide on the chain function (well, it'll be _none()) */
      gst_pad_set_chain_function(volume->sinkpad,gst_volume_chain_fp_S16);
//  gst_volume_set_chain(volume,AFMT_S16_LE);
}

/* does no volume control whatsoever, just passes the buffer through */
static void
gst_volume_chain_none (GstPad *pad, GstBuffer *buf)
{
  GstVolume *volume = GST_VOLUME (GST_OBJECT_PARENT (pad));

  gst_pad_push(volume->srcpad,buf);
}

/* does arbitrary scaling of S16 data with floating point */
static void
gst_volume_chain_fp_S16 (GstPad *pad, GstBuffer *buf)
{
  GstVolume *volume = GST_VOLUME (GST_OBJECT_PARENT (pad));
  gint16 *data;
  int i;

  /* check the format */
//  if (((MetaAudioRaw *)(buf->meta))->format != AFMT_S16_LE) {
//    gst_volume_rechain(((MetaAudioRaw *)(buf->meta))->format,pad,buf);
//    return;
//  }

  /* FIXME: we modify the buffer in place, no regard to CoW */
  data = (gint16 *)GST_BUFFER_DATA(buf);
  for (i=0;i<GST_BUFFER_SIZE(buf)/2;i++)
    data[i] = (float)data[i] * volume->volume;

  gst_pad_push(volume->srcpad,buf);
}

/* does arbitrary scaling of S8 data with floating point */
static void
gst_volume_chain_fp_S8 (GstPad *pad, GstBuffer *buf)
{
  GstVolume *volume = GST_VOLUME (GST_OBJECT_PARENT (pad));
  gint8 *data;
  int i;

  /* check the format */
//  if (((MetaAudioRaw *)(buf->meta))->format != AFMT_S8) {
//    gst_volume_rechain(((MetaAudioRaw *)(buf->meta))->format,pad,buf);
//    return;
//  }

  /* FIXME: we modify the buffer in place, no regard to CoW */
  data = (gint8 *)GST_BUFFER_DATA(buf);
  for (i=0;i<GST_BUFFER_SIZE(buf)/2;i++)
    data[i] = (float)data[i] * volume->volume;

  gst_pad_push(volume->srcpad,buf);
}

/* set the chain function relative to the format, volume, etc. */
/* NOTE: this will change as I add more specialized chain functions */
/* FIXME: should add shifter functions for 2^ volume changes */
static void
gst_volume_set_chain (GstVolume *volume, gint format)
{
  /* simple cases first */
  if ((volume->volume == 1.0) || !volume->active)
    gst_pad_set_chain_function(volume->sinkpad,gst_volume_chain_none);
  /* if we're lowering the volume... */
  else if (volume->volume < 1.0) {
    //if (format == AFMT_S16_LE)
      gst_pad_set_chain_function(volume->sinkpad,gst_volume_chain_fp_S16);
    //else if (format == AFMT_S8)
    //  gst_pad_set_chain_function(volume->sinkpad,gst_volume_chain_fp_S8);
  /* if we're raising the volume */
  } else if (volume->volume > 1.0) {
    //if (format == AFMT_S16_LE)
      gst_pad_set_chain_function(volume->sinkpad,gst_volume_chain_fp_S16);
    //else if (format == AFMT_S8)
    //  gst_pad_set_chain_function(volume->sinkpad,gst_volume_chain_fp_S8);
  }
}

static void
gst_volume_rechain (gint format, GstPad *pad, GstBuffer *buf)
{
  gst_volume_set_chain(GST_VOLUME(GST_OBJECT_PARENT (pad)),format);
  // FIXME this needs to be rethought.
  (GST_REAL_PAD(pad)->chainfunc)(pad,buf);
}

static void
gst_volume_set_property (GObject *object, guint prop_id, const GValue *value, GParamSpec *pspec)
{
  GstVolume *volume;

  /* it's not null if we got it, but it might not be ours */
  g_return_if_fail(GST_IS_VOLUME(object));
  volume = GST_VOLUME(object);

  switch (prop_id) {
    case ARG_ACTIVE:
      volume->active = g_value_get_int (value);
//      gst_volume_set_chain(volume,0);
      break;
    case ARG_VOLUME:
      volume->volume = g_value_get_float (value);
      /* FIXME: this is cheesy, we set it to the most common, and if we're
         wrong it gets reset when the next buffer comes in.  maybe store
         the current format somewhere so we don't do this? */
//      gst_volume_set_chain(volume,AFMT_S16_LE);
      break;
    default:
      break;
  }
}

static void
gst_volume_get_property (GObject *object, guint prop_id, GValue *value, GParamSpec *pspec)
{
  GstVolume *volume;

  /* it's not null if we got it, but it might not be ours */
  g_return_if_fail(GST_IS_VOLUME(object));
  volume = GST_VOLUME(object);

  switch (prop_id) {
    case ARG_ACTIVE:
      g_value_set_int (value, volume->active);
      break;
    case ARG_VOLUME:
      g_value_set_float (value, volume->volume);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static gboolean
plugin_init (GModule *module, GstPlugin *plugin)
{
  GstElementFactory *factory;

  factory = gst_elementfactory_new("volume",GST_TYPE_VOLUME,
                                   &volume_details);
  g_return_val_if_fail(factory != NULL, FALSE);
  gst_plugin_add_factory(plugin,factory);

  return TRUE;
}

GstPluginDesc plugin_desc = {
  GST_VERSION_MAJOR,
  GST_VERSION_MINOR,
  "volume",
  plugin_init
};

