/* Gnome-Streamer
 * Copyright (C) <2001> Steve Baker <stevebaker_org@yahoo.co.uk>
 *
 * float2int.c
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <float2int.h>


static GstElementDetails float2int_details = {
  "Float to Integer effect",
  "Filter/Effect",
  "Convert from floating point to integer audio data",
  VERSION,
  "Steve Baker <stevebaker_org@yahoo.co.uk>",
  "(C) 2001",
};


enum {
  /* FILL ME */
  LAST_SIGNAL
};

enum {
  ARG_0,
  ARG_PAN,
};

static GstPadTemplate* 
float2int_sink_factory (void) 
{
  return 
    gst_padtemplate_new (
      "sink%d",
      GST_PAD_SINK,
      GST_PAD_REQUEST,
      gst_caps_new (
      "float2int_sink",
        "audio/raw",
        gst_props_new (
          "format",     GST_PROPS_STRING ("float"),
          "layout",     GST_PROPS_STRING ("gfloat"),
          "intercept",  GST_PROPS_FLOAT(0.0),
          "slope",      GST_PROPS_FLOAT(1.0),
          "channels",   GST_PROPS_INT (1),
      NULL)),
    NULL);
}

static GstPadTemplate* 
float2int_src_factory (void) 
{
  return 
    gst_padtemplate_new (
    "src",
    GST_PAD_SRC,
    GST_PAD_ALWAYS,
    gst_caps_new (
    "float2int_src",
      "audio/raw",
      gst_props_new (
        "format",   GST_PROPS_STRING ("int"),
        "law",     GST_PROPS_INT (0),
        "endianness",     GST_PROPS_INT (G_BYTE_ORDER),
        "signed",   GST_PROPS_BOOLEAN (TRUE),
        NULL)),
      NULL);
}

static GstPadTemplate *srctempl, *sinktempl;

static void	gst_float2int_class_init		(GstFloat2IntClass *klass);
static void	gst_float2int_init			(GstFloat2Int *plugin);

static void	gst_float2int_set_property		(GObject *object, guint prop_id, const GValue *value, GParamSpec *pspec);
static void	gst_float2int_get_property		(GObject *object, guint prop_id, GValue *value, GParamSpec *pspec);

static GstPad*  gst_float2int_request_new_pad (GstElement *element, GstPadTemplate *temp);

static void	gst_float2int_chain_gint16		(GstPad *pad, GstBuffer *buf_in);
static inline GstFloat2Int* gst_float2int_get_plugin(GstPad *pad,GstBuffer *buf);

static GstElementClass *parent_class = NULL;

static GstPadNegotiateReturn
gst_float2int_negotiate_src (GstPad *pad, GstCaps **caps, gpointer *data)
{
  GstFloat2Int* plugin = GST_FLOAT2INT (GST_OBJECT_PARENT (pad));
  GstCaps* peercaps;
  GstRealPad *realpeer;
  GstPadNegotiateReturn proxyresult;
  GSList *sinkpads;
  GstPad *sinkpad;
  
  g_print("nego float2int src\n");
  if (*caps== NULL) return GST_PAD_NEGOTIATE_FAIL;
  plugin->width = gst_caps_get_int (*caps, "width");
  plugin->depth = gst_caps_get_int (*caps, "depth");

  if (plugin->trying_proxy_caps) return GST_PAD_NEGOTIATE_AGREE;

  plugin->trying_proxy_caps=TRUE;
  
  peercaps=gst_caps_new (
    "float2int_src_caps",
    "audio/raw",
    gst_props_new (
      "format",     GST_PROPS_STRING ("float"),
      "layout",     GST_PROPS_STRING ("gfloat"),
      "intercept",  GST_PROPS_FLOAT(0.0),
      "slope",      GST_PROPS_FLOAT(1.0),
      "rate",       GST_PROPS_INT (gst_caps_get_int (*caps, "rate")),
      "channels",   GST_PROPS_INT (1),
      NULL
    )
  );

  sinkpads = plugin->sinkpads;
  while(sinkpads){
    sinkpad = GST_FLOAT2INT_SINKPAD(sinkpads);
    realpeer = GST_RPAD_PEER(sinkpad);
    proxyresult = gst_pad_negotiate_proxy(&(realpeer->pad), sinkpad, &peercaps);
    if (proxyresult != GST_PAD_NEGOTIATE_AGREE){
      plugin->trying_proxy_caps=FALSE;
      return GST_PAD_NEGOTIATE_FAIL;
    }
    sinkpads = g_slist_next(sinkpads);
  }
  
  plugin->trying_proxy_caps=FALSE;
  return GST_PAD_NEGOTIATE_AGREE;
}

static GstPadNegotiateReturn
gst_float2int_negotiate_sink (GstPad *pad, GstCaps **caps, gpointer *data)
{
  GstFloat2Int* plugin = GST_FLOAT2INT (GST_OBJECT_PARENT (pad));
  GstCaps* peercaps;
  GstRealPad *realpeer;
  GstPadNegotiateReturn proxyresult;

  g_print("nego float2int sink\n");
  if (*caps== NULL) return GST_PAD_NEGOTIATE_FAIL;  

  if (plugin->trying_proxy_caps) return GST_PAD_NEGOTIATE_AGREE;

  plugin->trying_proxy_caps=TRUE;
  peercaps=gst_caps_new (
    "float2int_sink_caps",
    "audio/raw",
    gst_props_new (
      "format",     GST_PROPS_STRING ("int"),
      "law",        GST_PROPS_INT (0),
      "endianness", GST_PROPS_INT (G_BYTE_ORDER),
      "signed",     GST_PROPS_BOOLEAN (TRUE),
      "width",      GST_PROPS_INT (plugin->width),
      "depth",      GST_PROPS_INT (plugin->depth),
      "rate",       GST_PROPS_INT (gst_caps_get_int (*caps, "rate")),
      "channels",   GST_PROPS_INT (plugin->channels),
      NULL
    )
  );
  realpeer = GST_RPAD_PEER(plugin->srcpad);
  proxyresult = gst_pad_negotiate_proxy(&(realpeer->pad), plugin->srcpad, &peercaps);
  plugin->trying_proxy_caps=FALSE;
  return proxyresult;
}

GType
gst_float2int_get_type(void) {
  static GType float2int_type = 0;

  if (!float2int_type) {
    static const GTypeInfo float2int_info = {
      sizeof(GstFloat2IntClass),      NULL,
      NULL,
      (GClassInitFunc)gst_float2int_class_init,
      NULL,
      NULL,
      sizeof(GstFloat2Int),
      0,
      (GInstanceInitFunc)gst_float2int_init,
    };
    float2int_type = g_type_register_static(GST_TYPE_ELEMENT, "GstFloat2Int", &float2int_info, 0);
  }
  return float2int_type;
}

static void
gst_float2int_class_init (GstFloat2IntClass *klass)
{
  GObjectClass *gobject_class;
  GstElementClass *gstelement_class;

  gobject_class = (GObjectClass*)klass;
  gstelement_class = (GstElementClass*)klass;

  parent_class = g_type_class_ref(GST_TYPE_ELEMENT);

  gobject_class->set_property = gst_float2int_set_property;
  gobject_class->get_property = gst_float2int_get_property;

  gstelement_class->request_new_pad = gst_float2int_request_new_pad;
}

static void
gst_float2int_init (GstFloat2Int *plugin)
{
  plugin->srcpad = gst_pad_new_from_template(srctempl,"src");

  gst_element_add_pad(GST_ELEMENT(plugin),plugin->srcpad);

  gst_pad_set_negotiate_function(plugin->srcpad, gst_float2int_negotiate_src);

  plugin->numsinkpads = 0;
  plugin->numbuffers = 0;
  plugin->sinkpads = NULL;
  plugin->buffers = NULL;
  plugin->minbuffersize=G_MAXINT;

  plugin->channels=1;
  plugin->width=16;
  plugin->depth=16;
  plugin->intercept=0.0;
  plugin->slope=1.0;
  plugin->trying_proxy_caps=FALSE;
}

static GstPad*
gst_float2int_request_new_pad (GstElement *element, GstPadTemplate *templ) 
{
  gchar *name;
  GstPad *sinkpad;
  GstFloat2Int *plugin;

  plugin = GST_FLOAT2INT(element);
  
  g_return_val_if_fail(plugin != NULL, NULL);
  g_return_val_if_fail(GST_IS_FLOAT2INT(plugin), NULL);

  if (templ->direction != GST_PAD_SINK) {
    g_warning ("float2int: request new pad that is not a SINK pad\n");
    return NULL;
  }

  name = g_strdup_printf ("sink%d",plugin->numsinkpads);
  g_print("float2int adding pad %s\n", name);
  sinkpad = gst_pad_new_from_template (templ, name);
  gst_element_add_pad (GST_ELEMENT (plugin), sinkpad);
  gst_pad_set_chain_function(sinkpad,gst_float2int_chain_gint16);
  gst_pad_set_negotiate_function(sinkpad, gst_float2int_negotiate_sink);
  sinkpad->element_private = GINT_TO_POINTER(plugin->numsinkpads);
  
  plugin->sinkpads = g_slist_append (plugin->sinkpads, sinkpad);
  plugin->numsinkpads++;
  plugin->channels = plugin->numsinkpads;
  if (plugin->buffers){
    plugin->buffers = g_renew(GstBuffer*,plugin->buffers,plugin->numsinkpads);
    plugin->buffers[plugin->numsinkpads - 1] = NULL;
  }
  else {
    plugin->buffers = g_new0(GstBuffer*,plugin->numsinkpads);
  }
  
  return sinkpad;
}

static void
gst_float2int_chain_gint16 (GstPad *pad,GstBuffer *buf_in)
{
  GstFloat2Int *plugin;
  GstBuffer *buf_out, *buf_temp, **buffers;
  gfloat *data_in;
  gint16 *data_out;
  gint num_frames, i, j;
  guint padpos;
  guint buffer_byte_size;
    
  g_return_if_fail(plugin = gst_float2int_get_plugin(pad,buf_in));
  
  // find which position this sinkpad is in
  buffers = plugin->buffers;
  padpos = GPOINTER_TO_INT(pad->element_private);
  
  if (buffers[padpos]){
    // g_print("float2int: already have buffer %d\n", padpos);
    
    // append the new buffer to the existing one (not ideal, but blame the scheduler :)
    buffers[padpos] = gst_buffer_append (buffers[padpos], buf_in);
  }
  else {
    // g_print("float2int: setting buffer %d\n", padpos);
    buffers[padpos]=buf_in;
    plugin->numbuffers++;
  } 

  // get the minimum buffer size just in case they vary
  if (GST_BUFFER_SIZE(buffers[padpos]) < plugin->minbuffersize){
    plugin->minbuffersize = GST_BUFFER_SIZE(buffers[padpos]);
  }
      
  // check whether we have a full set of buffers yet 
  if (plugin->numbuffers < plugin->numsinkpads){
    // g_print("float2int: waiting for %d more buffers\n", plugin->numbuffers);
    return;
  }
  
  num_frames = plugin->minbuffersize / sizeof(gfloat);
  buffer_byte_size = 2 * num_frames * plugin->channels;
  
  // g_print("float2int: processing %d frames %d channels\n", num_frames, plugin->channels); 
  
  buf_out = gst_buffer_new_from_pool(gst_buffer_pool_get_default (NULL, buffer_byte_size, 4));
  
  data_out = (gint16*)GST_BUFFER_DATA(buf_out);
  GST_BUFFER_SIZE(buf_out) = buffer_byte_size;
  
  for (i = 0; (i < plugin->channels) && (i < plugin->numsinkpads); i++) {
    
    // chop the buffer if it is too big
    if (GST_BUFFER_SIZE(buffers[i]) == plugin->minbuffersize){
      buf_temp = buffers[i];
    }
    else {
      buf_temp =   gst_buffer_create_sub (buffers[i],0,plugin->minbuffersize);
      buffers[i] = gst_buffer_create_sub (buffers[i],plugin->minbuffersize,
                                          GST_BUFFER_SIZE(buffers[i]-plugin->minbuffersize));
    }
    
    data_in=(gfloat *)GST_BUFFER_DATA(buf_temp);
    
    for (j = 0; j < num_frames; j++) {
      
      // FIXME this is naive, slow and broken - apart from that it works fine
      data_out[(j*plugin->channels) + i] = (gint16)(data_in[j] * 32767.0);
    }
    
    // we are done with this buffer
    gst_buffer_unref(buf_temp);
    if (buf_temp == buffers[i]){
      buffers[i]=NULL;
      plugin->numbuffers--;
    }
    else {
      // get the new minimum buffer size
      if (GST_BUFFER_SIZE(buffers[i]) < plugin->minbuffersize){
        plugin->minbuffersize = GST_BUFFER_SIZE(buffers[i]);
      }
    }
  }
  if (!plugin->numbuffers){
    plugin->minbuffersize=G_MAXINT;
  }
  
  gst_pad_push(plugin->srcpad,buf_out);
}

static inline GstFloat2Int* 
gst_float2int_get_plugin(GstPad *pad,GstBuffer *buf)
{
  GstFloat2Int *plugin;
  g_return_val_if_fail(pad != NULL, NULL);
  g_return_val_if_fail(GST_IS_PAD(pad), NULL);
  g_return_val_if_fail(buf != NULL, NULL);

  plugin = GST_FLOAT2INT(GST_OBJECT_PARENT (pad));
  g_return_val_if_fail(plugin != NULL, NULL);
  g_return_val_if_fail(GST_IS_FLOAT2INT(plugin), NULL);
  return plugin;
}

static void
gst_float2int_set_property (GObject *object, guint prop_id, const GValue *value, GParamSpec *pspec)
{
  GstFloat2Int *plugin;

  /* it's not null if we got it, but it might not be ours */
  g_return_if_fail(GST_IS_FLOAT2INT(object));
  plugin = GST_FLOAT2INT(object);

  switch (prop_id) {
    default:
      break;
  }
}

static void
gst_float2int_get_property (GObject *object, guint prop_id, GValue *value, GParamSpec *pspec)
{
  GstFloat2Int *plugin;

  /* it's not null if we got it, but it might not be ours */
  g_return_if_fail(GST_IS_FLOAT2INT(object));
  plugin = GST_FLOAT2INT(object);

  switch (prop_id) {
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

gboolean 
gst_float2int_factory_init (GstPlugin *plugin) 
{
  GstElementFactory *factory;
  
  factory = gst_elementfactory_new("float2int",GST_TYPE_FLOAT2INT,
                                   &float2int_details);
  g_return_val_if_fail(factory != NULL, FALSE);
  
  sinktempl = float2int_sink_factory();
  gst_elementfactory_add_padtemplate (factory, sinktempl);

  srctempl = float2int_src_factory();
  gst_elementfactory_add_padtemplate (factory, srctempl);
  
  gst_plugin_add_factory(plugin,factory);
   
  return TRUE;
}
