/* Gnome-Streamer
 * Copyright (C) <1999> Erik Walthinsen <omega@cse.ogi.edu>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <gst/gst.h>
#include "../../../include/gstaudio.h"
#include "passthrough.h"



static GstElementDetails passthrough_details = {
  "Passthrough",
  "Filter/Effect",
  "Transparent filter for audio/raw (boilerplate for effects)",
  VERSION,
  "Thomas <thomas@apestaart.org>",
  "(C) 2001",
};


/* Filter signals and args */
enum {
  /* FILL ME */
  LAST_SIGNAL
};

enum {
  ARG_0,
  ARG_SILENT
};

static GstPadTemplate*
passthrough_src_factory (void)
{
  static GstPadTemplate *template = NULL;

  if (!template) {
    template = gst_padtemplate_new (
      "src",
      GST_PAD_SRC,
      GST_PAD_ALWAYS,
      gst_caps_new (
        "passthrough_src",
        "audio/raw",
	GST_AUDIO_INT_PAD_TEMPLATE_PROPS,
	NULL);
  }
  return template;
}

static GstPadTemplate*
passthrough_sink_factory (void)
{
  static GstPadTemplate *template = NULL;

  if (!template) {
    template = gst_padtemplate_new (
      "sink",
      GST_PAD_SINK,
      GST_PAD_ALWAYS,
      gst_caps_new (
        "test_src",
        "audio/raw",
	GST_AUDIO_INT_PAD_TEMPLATE_PROPS,
      NULL);
  }
  return template;
}

static void		gst_passthrough_class_init		(GstPassThroughClass *klass);
static void		gst_passthrough_init			(GstPassThrough *filter);

static void		gst_passthrough_set_property			(GObject *object, guint prop_id, const GValue *value, GParamSpec *pspec);
static void		gst_passthrough_get_property			(GObject *object, guint prop_id, GValue *value, GParamSpec *pspec);

static void		gst_passthrough_chain			(GstPad *pad, GstBuffer *buf);
static void inline 	gst_passthrough_fast_16bit_chain 	(gint16* data, gint16** out_data, 
			         				 guint numsamples);
static void inline 	gst_passthrough_fast_8bit_chain		(gint8* data, gint8** out_data,
                                				 guint numsamples);

static GstElementClass *parent_class = NULL;
//static guint gst_filter_signals[LAST_SIGNAL] = { 0 };

static GstPadNegotiateReturn
passthrough_negotiate_src (GstPad *pad, GstCaps **caps, gpointer *data)
{
  GstPassThrough* filter = GST_PASSTHROUGH (gst_pad_get_parent (pad));
  
  if (*caps==NULL) 
    return GST_PAD_NEGOTIATE_FAIL;
  
  return gst_pad_negotiate_proxy(pad,filter->sinkpad,caps);
}

static GstPadNegotiateReturn
passthrough_negotiate_sink (GstPad *pad, GstCaps **caps, gpointer *data)
{
  GstPassThrough* filter = GST_PASSTHROUGH (gst_pad_get_parent (pad));
  
  if (*caps==NULL) 
    return GST_PAD_NEGOTIATE_FAIL;
  
  return gst_pad_negotiate_proxy(pad,filter->srcpad,caps);
}		

GType
gst_passthrough_get_type(void) {
  static GType passthrough_type = 0;

  if (!passthrough_type) {
    static const GTypeInfo passthrough_info = {
      sizeof(GstPassThroughClass),      NULL,
      NULL,
      (GClassInitFunc)gst_passthrough_class_init,
      NULL,
      NULL,
      sizeof(GstPassThrough),
      0,
      (GInstanceInitFunc)gst_passthrough_init,
    };
    passthrough_type = g_type_register_static(GST_TYPE_ELEMENT, "GstPassThrough", &passthrough_info, 0);
  }
  return passthrough_type;
}

static void
gst_passthrough_class_init (GstPassThroughClass *klass)
{
  GObjectClass *gobject_class;
  GstElementClass *gstelement_class;

  gobject_class = (GObjectClass*)klass;
  gstelement_class = (GstElementClass*)klass;

  parent_class = g_type_class_ref(GST_TYPE_ELEMENT);

  g_object_class_install_property(G_OBJECT_CLASS(klass), ARG_SILENT,
    g_param_spec_boolean("silent","silent","silent",
                         TRUE,G_PARAM_READWRITE)); // CHECKME

  gobject_class->set_property = gst_passthrough_set_property;
  gobject_class->get_property = gst_passthrough_get_property;
}

static void
gst_passthrough_init (GstPassThrough *filter)
{
  filter->sinkpad = gst_pad_new_from_template(passthrough_sink_factory (),"sink");
  filter->srcpad = gst_pad_new_from_template(passthrough_src_factory (),"src");

  gst_pad_set_negotiate_function(filter->sinkpad,passthrough_negotiate_sink);
  gst_pad_set_negotiate_function(filter->srcpad,passthrough_negotiate_src);

  gst_element_add_pad(GST_ELEMENT(filter),filter->sinkpad);
  gst_pad_set_chain_function(filter->sinkpad,gst_passthrough_chain);
  filter->srcpad = gst_pad_new("src",GST_PAD_SRC);
  gst_element_add_pad(GST_ELEMENT(filter),filter->srcpad);
  filter->caps_given = FALSE;
  filter->silent = FALSE;
}

static void
gst_passthrough_chain (GstPad *pad, GstBuffer *buf)
{
  GstPassThrough *filter;
  gint16 *in_data;
  gint16 *out_data;
  GstBuffer* outbuf;
  guint width, rate, channels;
  guint endianness;
  gboolean is_signed;

  GstCaps *caps;

  g_return_if_fail(pad != NULL);
  g_return_if_fail(GST_IS_PAD(pad));
  g_return_if_fail(buf != NULL);

  filter = GST_PASSTHROUGH(GST_OBJECT_PARENT (pad));
  g_return_if_fail(filter != NULL);
  g_return_if_fail(GST_IS_PASSTHROUGH(filter));

  caps = NULL;
  caps = GST_PAD_CAPS(pad);
  
  if (caps == NULL)
  {
    // FIXME : Please change this to a better warning method !
    printf ("WARNING: passthrough chain : Could not get caps of pad !\n");
  }

  /* print out caps of sink pad */

  width      = gst_caps_get_int (caps, "width");
  rate       = gst_caps_get_int (caps, "rate");
  channels   = gst_caps_get_int (caps, "channels");
  endianness = gst_caps_get_int (caps, "endianness");
  is_signed  = gst_caps_get_int (caps, "signed");
  if (!filter->caps_given)
  {
    if (!filter->silent)
    {
      g_print ("PassThrough : bit width %d, channels %d, rate %d\n",  
      		width, channels, rate);
      g_print ("PassThrough : endianness %d, signed %s\n",
    		endianness, is_signed ? "yes" : "no");
    }
    filter->caps_given = TRUE;
  }

  in_data = (gint16 *)GST_BUFFER_DATA(buf);
  outbuf=gst_buffer_new();
  GST_BUFFER_DATA(outbuf) = (gchar*)g_new(gint16,GST_BUFFER_SIZE(buf)/2);
  GST_BUFFER_SIZE(outbuf) = GST_BUFFER_SIZE(buf);

  out_data = (gint16*) GST_BUFFER_DATA(outbuf);
  
  switch (width) {
    case 16:
	gst_passthrough_fast_16bit_chain(in_data,&out_data,GST_BUFFER_SIZE(buf)/2);
	break;
    case 8:
	gst_passthrough_fast_8bit_chain((gint8*)in_data,(gint8**)&out_data,GST_BUFFER_SIZE(buf));
	break;
  }
  gst_buffer_unref(buf);
  gst_pad_push(filter->srcpad,outbuf);
}

static void inline
gst_passthrough_fast_16bit_chain(gint16* in_data, gint16** out_data, 
			         guint num_samples)
#include "filter.func"

static void inline
gst_passthrough_fast_8bit_chain(gint8* in_data, gint8** out_data,
                                guint num_samples)
#include "filter.func"

static void
gst_passthrough_set_property (GObject *object, guint prop_id, const GValue *value, GParamSpec *pspec)
{
  GstPassThrough *filter;

  /* it's not null if we got it, but it might not be ours */
  g_return_if_fail(GST_IS_PASSTHROUGH(object));
  filter = GST_PASSTHROUGH(object);

  switch (prop_id) 
  {
    case ARG_SILENT:
      filter->silent = g_value_get_boolean (value);
      break;
    default:
      break;
  }
}

static void
gst_passthrough_get_property (GObject *object, guint prop_id, GValue *value, GParamSpec *pspec)
{
  GstPassThrough *filter;

  /* it's not null if we got it, but it might not be ours */
  g_return_if_fail(GST_IS_PASSTHROUGH(object));
  filter = GST_PASSTHROUGH(object);

  switch (prop_id) {
    case ARG_SILENT:
      g_value_set_boolean (value, filter->silent);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static gboolean
plugin_init (GModule *module, GstPlugin *plugin)
{
  GstElementFactory *factory;

  factory = gst_elementfactory_new("passthrough",GST_TYPE_PASSTHROUGH,
                                   &passthrough_details);
  g_return_val_if_fail(factory != NULL, FALSE);
  
  gst_elementfactory_add_padtemplate (factory, passthrough_src_factory ());
  gst_elementfactory_add_padtemplate (factory, passthrough_sink_factory ());

  gst_plugin_add_factory(plugin,factory);

  return TRUE;
}

GstPluginDesc plugin_desc = {
  GST_VERSION_MAJOR,
  GST_VERSION_MINOR,
  "passthrough",
  plugin_init
};
