/* Predict.c, motion compensation routines                                    */

/* Copyright (C) 1996, MPEG Software Simulation Group. All Rights Reserved. */

/*
 * Disclaimer of Warranty
 *
 * These software programs are available to the user without any license fee or
 * royalty on an "as is" basis.  The MPEG Software Simulation Group disclaims
 * any and all warranties, whether express, implied, or statuary, including any
 * implied warranties or merchantability or of fitness for a particular
 * purpose.  In no event shall the copyright-holder be liable for any
 * incidental, punitive, or consequential damages of any kind whatsoever
 * arising from the use of these programs.
 *
 * This disclaimer of warranty extends to the user of these programs and user's
 * customers, employees, agents, transferees, successors, and assigns.
 *
 * The MPEG Software Simulation Group does not represent or warrant that the
 * programs furnished hereunder are free of infringement of any third-party
 * patents.
 *
 * Commercial implementations of MPEG-1 and MPEG-2 video, including shareware,
 * are subject to royalty fees to patent holders.  Many of these patents are
 * general enough such that they are unavoidable regardless of implementation
 * design.
 *
 */

#include <stdio.h>

#include "video.h"

#define MAX_NEG_CROP 2048
extern unsigned char cropTbl[];

/* ISO/IEC 13818-2 section 7.6.4: Forming predictions */
/* NOTE: the arithmetic below produces numerically equivalent results
 *  to 7.6.4, yet is more elegant. It differs in the following ways:
 *
 *   1. the vectors (dx, dy) are based on cartesian frame 
 *      coordiantes along a half-pel grid (always positive numbers)
 *      In contrast, vector[r][s][t] are differential (with positive and 
 *      negative values). As a result, deriving the integer vectors 
 *      (int_vec[t]) from dx, dy is accomplished by a simple right shift.
 *
 *   2. Half pel flags (xh, yh) are equivalent to the LSB (Least
 *      Significant Bit) of the half-pel coordinates (dx,dy).
 * 
 *
 *  NOTE: the work of combining predictions (ISO/IEC 13818-2 section 7.6.7)
 *  is distributed among several other stages.  This is accomplished by 
 *  folding line offsets into the source and destination (src,dst)
 *  addresses (note the call arguments to form_prediction() in Predict()),
 *  line stride variables lx and lx2, the block dimension variables (w,h), 
 *  zflag, and by the very order in which Predict() is called.  
 *  This implementation design (implicitly different than the spec) 
 *  was chosen for its elegance.
*/

void form_component_prediction_C(src,dst,blockvals,lx,lx2,w,h,x,y,dx,dy,zflag)
unsigned char *src;
unsigned char *dst;
short int *blockvals;
int lx;          /* raster line increment */ 
int lx2;
int w,h;
int x,y;
int dx,dy;
int zflag;      /* flag that signals bi-directional or Dual-Prime 
                          averaging (7.6.7.1 and 7.6.7.4). if zflag==1,
                          a previously formed prediction has been stored in 
                          pel_pred[] */
{
  int xint;      /* horizontal integer sample vector: analogous to int_vec[0] */
  int yint;      /* vertical integer sample vectors: analogous to int_vec[1] */
  int xh;        /* horizontal half sample flag: analogous to half_flag[0]  */
  int yh;        /* vertical half sample flag: analogous to half_flag[1]  */
  int i, j;
  unsigned char *s;    /* source pointer: analogous to pel_ref[][]   */
  unsigned char *d;    /* destination pointer:  analogous to pel_pred[][]  */
  unsigned char *cm = cropTbl + MAX_NEG_CROP;

  /* half pel scaling for integer vectors */
  xint = dx>>1;
  yint = dy>>1;

  /* derive half pel flags */
  xh = dx & 1;
  yh = dy & 1;

  /* compute the linear address of pel_ref[][] and pel_pred[][] 
     based on cartesian/raster cordinates provided */
  s = src + lx*(y+yint) + x + xint;
  d = dst + lx*y + x;

  if (!xh && !yh) /* no horizontal nor vertical half-pel */
  {
    if (!zflag)
    {
      for (j=0; j<h; j++)
      {
        for (i=0; i<w; i++)
        {
          d[i] = cm[(int)s[i]+(int)blockvals[i]];
        }
      
	blockvals += 8;
        s+= lx2;
        d+= lx2;
      }
    }
    else
    {
      for (j=0; j<h; j++)
      {
        for (i=0; i<w; i++)
        {
          d[i] = s[i];
        }
        
        s+= lx2;
        d+= lx2;
      }
    }
  }
  else if (!xh && yh) /* no horizontal but vertical half-pel */
  {
    if (!zflag)
    {
      for (j=0; j<h; j++)
      {
        for (i=0; i<w; i++)
        {
          d[i] = cm[((int)(s[i]+s[i+lx]+1)>>1) + blockvals[i]];
        }
     
	blockvals += 8;
        s+= lx2;
        d+= lx2;
      }
    }
    else
    {
      for (j=0; j<h; j++)
      {
        for (i=0; i<w; i++)
        {
          d[i] = (unsigned int)(s[i]+s[i+lx]+1)>>1;
        }

        s+= lx2;
        d+= lx2;
      }
    }
  }
  else if (xh && !yh) /* horizontal but no vertical half-pel */
  {
    if (!zflag)
    {
      for (j=0; j<h; j++)
      {
        for (i=0; i<w; i++)
        {
          d[i] = cm[((int)(s[i]+s[i+1]+1)>>1) + blockvals[i]];
        }
     
	blockvals += 8;
        s+= lx2;
        d+= lx2;
      }
    }
    else
    {
      for (j=0; j<h; j++)
      {
        for (i=0; i<w; i++)
        {
          d[i] = (unsigned int)(s[i]+s[i+1]+1)>>1;
        }

        s+= lx2;
        d+= lx2;
      }
    }
  }
  else /* if (xh && yh) horizontal and vertical half-pel */
  {
    if (!zflag)
    {
      for (j=0; j<h; j++)
      {
        for (i=0; i<w; i++)
        {
          d[i] = cm[((int)(s[i]+s[i+1]+s[i+lx]+s[i+lx+1]+2)>>2) +blockvals[i]];
        }
     
	blockvals += 8;
        s+= lx2;
        d+= lx2;
      }
    }
    else
    {
      for (j=0; j<h; j++)
      {
        for (i=0; i<w; i++)
        {
          d[i] = (unsigned int)(s[i]+s[i+1]+s[i+lx]+s[i+lx+1]+2)>>2;
        }

        s+= lx2;
        d+= lx2;
      }
    }
  }
}

void form_component_prediction_bidir_C(srcb,srcf,dst,blockvals,lx,lx2,w,h,x,y,dbx,dby,dfx,dfy,zflag)
unsigned char *srcb;
unsigned char *srcf;
unsigned char *dst;
short int *blockvals;
int lx;        
int lx2;
int w,h;
int x,y;
int dbx,dby;
int dfx,dfy;
int zflag;
{
  int xbint, xfint;  
  int ybint, yfint; 
  int xbh, xfh;  
  int ybh, yfh; 
  int i, j;
  unsigned char *sb,*sf; 
  unsigned char *d;
  unsigned char *cm = cropTbl + MAX_NEG_CROP;

  /* half pel scaling for integer vectors */
  xbint = dbx>>1;
  ybint = dby>>1;
  xfint = dfx>>1;
  yfint = dfy>>1;

  /* derive half pel flags */
  xbh = dbx & 1;
  ybh = dby & 1;
  xfh = dfx & 1;
  yfh = dfy & 1;

  /* compute the linear address of pel_ref[][] and pel_pred[][] 
     based on cartesian/raster cordinates provided */
  sb = srcb + lx*(y+ybint) + x + xbint;
  sf = srcf + lx*(y+yfint) + x + xfint;

  d = dst + lx*y + x;
	
  if (!xbh && !ybh && !xfh && !yfh) /* no horizontal nor vertical half-pel */
  {
    if (!zflag)
    {
      for (j=0; j<h; j++)
      {
        for (i=0; i<w; i++)
        {
          d[i] = cm[((int) (sb[i] + sf[i] + 1) >> 1) + blockvals[i]];
        }
      
	blockvals += 8;
        sb+= lx2;
        sf+= lx2;
        d+= lx2;
      }
    }
    else
    {
      for (j=0; j<h; j++)
      {
        for (i=0; i<w; i++)
        {
          d[i] = (int) (sb[i] + sf[i] + 1) >> 1;
        }
        
        sb+= lx2;
        sf+= lx2;
        d+= lx2;
      }
    }
  }
  else {
    unsigned char *sfa, *sfb, *sfc, *sba, *sbb, *sbc;

    sfa = sf + xfh;
    sfb = sf + lx*yfh;
    sfc = sfb + xfh;

    sba = sb + xbh;
    sbb = sb + lx*ybh;
    sbc = sbb + xbh;

    if (!zflag)
    {
      for (j=0; j<h; j++)
      {
        for (i=0; i<w; i++)
        {
          d[i] = cm[((int) (((sb[i] + sba[i] + sbb[i] + sbc[i] + 2) >> 2) +
                            ((sf[i] + sfa[i] + sfb[i] + sfc[i] + 2) >> 2) + 1) >>1) + blockvals[i]];
        }
      
	blockvals += 8;
        sb+= lx2;
        sba+= lx2;
        sbb+= lx2;
        sbc+= lx2;
        sf+= lx2;
        sfa+= lx2;
        sfb+= lx2;
        sfc+= lx2;
        d+= lx2;
      }
    }
    else
    {
      for (j=0; j<h; j++)
      {
        for (i=0; i<w; i++)
        {
          d[i] = (int) (((sb[i] + sba[i] + sbb[i] + sbc[i] + 2) >> 2) +
                        ((sf[i] + sfa[i] + sfb[i] + sfc[i] + 2) >> 2) + 1) >>1;
        }
        
        sb+= lx2;
        sba+= lx2;
        sbb+= lx2;
        sbc+= lx2;
        sf+= lx2;
        sfa+= lx2;
        sfb+= lx2;
        sfc+= lx2;
        d+= lx2;
      }
    }
  }
}

