/* Gnome-Streamer
 * Copyright (C) <1999> Erik Walthinsen <omega@cse.ogi.edu>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <string.h>

#include "ac3dec.h"


/* elementfactory information */
static GstElementDetails gst_ac3dec_details = {
  "ac3dec mp3 decoder",
  "Filter/Decoder/Audio",
  "Uses modified ac3dec code to decode ac3 streams",
  VERSION,
  "Erik Walthinsen <omega@cse.ogi.edu>",
  "(C) 1999",
};


/* Ac3Dec signals and args */
enum {
  /* FILL ME */
  LAST_SIGNAL
};

enum {
  ARG_0,
  /* FILL ME */
};

static GstPadTemplate*
sink_factory (void) 
{
  return 
   gst_padtemplate_new (
  	"sink",
  	GST_PAD_SINK,
  	GST_PAD_ALWAYS,
  	gst_caps_new (
  	  "ac3dec_sink",
    	  "audio/ac3",
	  NULL),
	NULL);
}

static GstPadTemplate*
src_factory (void) 
{
  return 
   gst_padtemplate_new (
  	"src",
  	GST_PAD_SRC,
  	GST_PAD_ALWAYS,
  	gst_caps_new (
  	  "ac3dec_src",
    	  "audio/raw",
	  gst_props_new (
    	    "format",             GST_PROPS_STRING ("int"),
      	      "law",              GST_PROPS_INT (0),
              "endianness",       GST_PROPS_INT (G_BYTE_ORDER),
      	      "signed",           GST_PROPS_BOOLEAN (TRUE),
      	      "width",            GST_PROPS_INT (16),
      	      "depth",            GST_PROPS_INT (16),
      	      "rate",             GST_PROPS_INT_RANGE (11025, 48000),
      	      "channels",         GST_PROPS_INT (2),
	      NULL)),
	NULL);
}

static void	gst_ac3dec_class_init	(Ac3DecClass *klass);
static void	gst_ac3dec_init		(Ac3Dec *ac3dec);

static gint	gst_ac3dec_open		(guint32 bits, guint32 rate, guint32 channels, void *data);
static void	gst_ac3dec_play		(gint16 *samples, guint32 num_bytes, void *data);
static void	gst_ac3dec_close	(void);

static void	gst_ac3dec_chain	(GstPad *pad, GstBuffer *buf);

static GstElementClass *parent_class = NULL;
//static guint gst_ac3dec_signals[LAST_SIGNAL] = { 0 };

static GstPadTemplate *src_template, *sink_template;

GType
gst_ac3dec_get_type (void)
{
  static GType ac3dec_type = 0;

  if (!ac3dec_type) {
    static const GTypeInfo ac3dec_info = {
      sizeof(Ac3DecClass),      NULL,      NULL,      (GClassInitFunc)gst_ac3dec_class_init,
      NULL,
      NULL,
      sizeof(Ac3Dec),
      0,
      (GInstanceInitFunc)gst_ac3dec_init,
    };
    ac3dec_type = g_type_register_static (GST_TYPE_ELEMENT, "Ac3Dec", &ac3dec_info, 0);
  }
  return ac3dec_type;
}

static void
gst_ac3dec_class_init (Ac3DecClass *klass)
{
  GstElementClass *gstelement_class;

  gstelement_class = (GstElementClass*)klass;

  parent_class = g_type_class_ref (GST_TYPE_ELEMENT);
}

static void
gst_ac3dec_init (Ac3Dec *ac3dec)
{
  /* create the sink and src pads */
  ac3dec->sinkpad = gst_pad_new_from_template (sink_template, "sink");
  gst_element_add_pad (GST_ELEMENT (ac3dec), ac3dec->sinkpad);
  gst_pad_set_caps (ac3dec->sinkpad, gst_pad_get_padtemplate_caps (ac3dec->sinkpad));
  gst_pad_set_chain_function (ac3dec->sinkpad, gst_ac3dec_chain);

  ac3dec->srcpad = gst_pad_new_from_template (src_template, "src");
  gst_element_add_pad (GST_ELEMENT (ac3dec), ac3dec->srcpad);

  ac3dec_init ();

  ac3dec->functions.open = gst_ac3dec_open;
  ac3dec->functions.play = gst_ac3dec_play;
  ac3dec->functions.close = gst_ac3dec_close;
}

static gint
gst_ac3dec_open (guint32 bits, guint32 rate, guint32 channels, void *data)
{
  Ac3Dec *ac3dec;
  GstCaps *caps;

  ac3dec = GST_AC3DEC (data);

  GST_INFO (GST_CAT_PLUGIN_INFO, "ac3dec: open \n");

  caps = gst_caps_new (
		  "ac3dec_src_caps",
		  "audio/raw",
		  gst_props_new (
		  "format",             GST_PROPS_STRING ("int"),
		    "law",              GST_PROPS_INT (0),
		    "endianness",       GST_PROPS_INT (G_BYTE_ORDER),
		    "signed",           GST_PROPS_BOOLEAN (TRUE),
		    "width",            GST_PROPS_INT (16),
		    "depth",            GST_PROPS_INT (16),
		    "channels", 	GST_PROPS_INT (channels),
		    "rate", 		GST_PROPS_INT (rate),
		     NULL
		    )
		  );

  gst_pad_set_caps (ac3dec->srcpad, caps);

  return 0;
}

static void
gst_ac3dec_play (gint16 *samples, guint32 num_bytes, void *data)
{
  Ac3Dec *ac3dec;
  GstBuffer *outbuf;

  ac3dec = GST_AC3DEC (data);

  num_bytes <<= 1;

  outbuf = gst_buffer_new ();
  GST_BUFFER_DATA (outbuf) = g_malloc (num_bytes);
  GST_BUFFER_SIZE (outbuf) = num_bytes;
  memcpy (GST_BUFFER_DATA (outbuf), samples, num_bytes);

  gst_pad_push (ac3dec->srcpad, outbuf);
}

static void
gst_ac3dec_close (void)
{
  GST_INFO (GST_CAT_PLUGIN_INFO, "ac3dec: close \n");
}

static void
gst_ac3dec_chain (GstPad *pad, GstBuffer *buf)
{
  Ac3Dec *ac3dec;
  guchar *data;
  gulong size;

  g_return_if_fail (pad != NULL);
  g_return_if_fail (GST_IS_PAD (pad));
  g_return_if_fail (buf != NULL);

  ac3dec = GST_AC3DEC (GST_OBJECT_PARENT (pad));

  data = (guchar *) GST_BUFFER_DATA (buf);
  size = GST_BUFFER_SIZE (buf);

  ac3dec_decode_data (NULL, &ac3dec->functions, data, data+size, ac3dec);

  gst_buffer_unref (buf);
}


static gboolean
plugin_init (GModule *module, GstPlugin *plugin)
{
  GstElementFactory *factory;

  /* create an elementfactory for the ac3dec element */
  factory = gst_elementfactory_new ("ac3dec", GST_TYPE_AC3DEC,
                                    &gst_ac3dec_details);
  g_return_val_if_fail (factory != NULL, FALSE);

  src_template = src_factory ();
  gst_elementfactory_add_padtemplate (factory, src_template);

  sink_template = sink_factory ();
  gst_elementfactory_add_padtemplate (factory, sink_template);

  gst_plugin_add_factory (plugin, factory);

  return TRUE;
}

GstPluginDesc plugin_desc = {
  GST_VERSION_MAJOR,
  GST_VERSION_MINOR,
  "ac3dec",
  plugin_init
};
