/* Gnome-Streamer
 * Copyright (C) <1999> Erik Walthinsen <omega@cse.ogi.edu>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <string.h>

#include <inttypes.h>

#include <mpeg2dec/mm_accel.h>
#include <mpeg2dec/video_out.h>
#include "gstmpeg2dec.h"

/* elementfactory information */
static GstElementDetails gst_mpeg2dec_details = {
  "mpeg1 and mpeg2 video decoder",
  "Filter/Decoder/Video",
  "Uses libmpeg2 to decode MPEG video streams",
  VERSION,
  "David I. Lehn <dlehn@users.sourceforge.net>",
  "(C) 2000",
};

/* Mpeg2dec signals and args */
enum {
  /* FILL ME */
  LAST_SIGNAL
};

enum {
  ARG_0,
  /* FILL ME */
};

static double video_rates[16] =
{
  0.0,
  24000.0/1001.,
  24.0,
  25.0,
  30000.0/1001.,
  30.0,
  50.0,
  60000.0/1001.,
  60.0,
  1,
  5,
  10,
  12,
  15,
  0,
  0
};

static GstPadTemplate*
src_template_factory (void) 
{
  static GstPadTemplate *template = NULL;
  
  if (!template) {
    template = gst_padtemplate_new (
  	"src",
  	GST_PAD_SRC,
  	GST_PAD_ALWAYS,
  	gst_caps_new (
  	  "mpeg2dec_src",
    	  "video/raw",
	  gst_props_new (
    	    "format",    GST_PROPS_FOURCC (GST_MAKE_FOURCC ('I','4','2','0')),
    	    "width",     GST_PROPS_INT_RANGE (16, 4096),
    	    "height",    GST_PROPS_INT_RANGE (16, 4096),
	    NULL)),
	NULL);
  }
  return template;
}

static GstPadTemplate*
sink_template_factory (void) 
{
  static GstPadTemplate *template = NULL;
  
  if (!template) {
    template = gst_padtemplate_new (
  	"sink",
  	GST_PAD_SINK,
  	GST_PAD_ALWAYS,
  	gst_caps_new (
  	  "mpeg2dec_sink",
    	  "video/mpeg",
	  gst_props_new (
    	    "mpegversion",  GST_PROPS_INT (2),
    	    "systemstream", GST_PROPS_BOOLEAN (FALSE),
	    NULL)),
	NULL);
  }
  return template;
}

static void	gst_mpeg2dec_class_init		(GstMpeg2decClass *klass);
static void	gst_mpeg2dec_init		(GstMpeg2dec *mpeg2dec);

static void	gst_mpeg2dec_set_property		(GObject *object, guint prop_id, const GValue *value, GParamSpec *pspec);
static void	gst_mpeg2dec_get_property		(GObject *object, guint prop_id, GValue *value, GParamSpec *pspec);

static void	gst_mpeg2dec_loop		(GstElement *element);

static GstElementClass *parent_class = NULL;
//static guint gst_mpeg2dec_signals[LAST_SIGNAL] = { 0 };

GType
gst_mpeg2dec_get_type (void)
{
  static GType mpeg2dec_type = 0;

  if (!mpeg2dec_type) {
    static const GTypeInfo mpeg2dec_info = {
      sizeof(GstMpeg2decClass),      NULL,
      NULL,
      (GClassInitFunc)gst_mpeg2dec_class_init,
      NULL,
      NULL,
      sizeof(GstMpeg2dec),
      0,
      (GInstanceInitFunc)gst_mpeg2dec_init,
    };
    mpeg2dec_type = g_type_register_static(GST_TYPE_ELEMENT, "GstMpeg2dec", &mpeg2dec_info, 0);
  }
  return mpeg2dec_type;
}

static void
gst_mpeg2dec_class_init(GstMpeg2decClass *klass)
{
  GObjectClass *gobject_class;
  GstElementClass *gstelement_class;

  gobject_class = (GObjectClass*)klass;
  gstelement_class = (GstElementClass*)klass;

  parent_class = g_type_class_ref(GST_TYPE_ELEMENT);

  gobject_class->set_property = gst_mpeg2dec_set_property;
  gobject_class->get_property = gst_mpeg2dec_get_property;

}

typedef struct gst_mpeg2dec_vo_frame_s {
  vo_frame_t vo;

  GstBuffer *buffer;
} gst_mpeg2dec_vo_frame_t;

typedef struct gst_mpeg2dec_vo_instance_s {
  vo_instance_t vo;

  GstMpeg2dec *mpeg2dec;

  gint prediction_index;
  gst_mpeg2dec_vo_frame_t frames[3];

} gst_mpeg2dec_vo_instance_t;

static void
gst_mpeg2dec_vo_frame_copy (vo_frame_t * frame, uint8_t ** src)
{
  GST_INFO (GST_CAT_PLUGIN_INFO, "VO: copy\n");
}

static void
gst_mpeg2dec_vo_frame_field (vo_frame_t * frame, int flags)
{
  GST_INFO (GST_CAT_PLUGIN_INFO, "VO: field\n");
}

static void
gst_mpeg2dec_vo_frame_draw (vo_frame_t * frame)
{
  gst_mpeg2dec_vo_instance_t *_instance;
  gst_mpeg2dec_vo_frame_t *_frame;

  g_return_if_fail (frame != NULL);
  g_return_if_fail (((gst_mpeg2dec_vo_frame_t *)frame)->buffer != NULL);


  _frame = (gst_mpeg2dec_vo_frame_t *)frame;
  _instance = (gst_mpeg2dec_vo_instance_t *)frame->instance;

  GST_BUFFER_TIMESTAMP(_frame->buffer) = _instance->mpeg2dec->next_time;
  _instance->mpeg2dec->next_time += (1000000LL/video_rates[_instance->mpeg2dec->decoder->frame_rate_code]);

  GST_DEBUG (0, "out: %lld %d %lld\n", GST_BUFFER_TIMESTAMP (_frame->buffer),
		  _instance->mpeg2dec->decoder->frame_rate_code,
                  (long long)(1000000LL/video_rates[_instance->mpeg2dec->decoder->frame_rate_code]));

  GST_BUFFER_FLAG_SET (_frame->buffer, GST_BUFFER_READONLY);
  gst_pad_push (_instance->mpeg2dec->srcpad, _frame->buffer);
}

static int
gst_mpeg2dec_vo_setup (vo_instance_t * instance, int width, int height)
{
  gst_mpeg2dec_vo_instance_t * _instance;
  GstMpeg2dec *mpeg2dec;

  g_return_val_if_fail (instance != NULL, -1);

  GST_INFO (GST_CAT_PLUGIN_INFO, "VO: setup w=%d h=%d\n", width, height);

  _instance = (gst_mpeg2dec_vo_instance_t*)instance;
  mpeg2dec = _instance->mpeg2dec;
  _instance->prediction_index = 1;
  _instance->mpeg2dec->width = width;
  _instance->mpeg2dec->height = height;

  gst_pad_set_caps (mpeg2dec->srcpad, 
		    gst_caps_new (
		      "mpeg2dec_caps",
		      "video/raw",
		      gst_props_new (
			"format",   GST_PROPS_FOURCC (GST_MAKE_FOURCC ('I','4','2','0')),
			  "width",  GST_PROPS_INT (width),
			  "height", GST_PROPS_INT (height),
			  NULL)));

  mpeg2dec->peerpool = gst_pad_get_bufferpool (mpeg2dec->srcpad);
  if (mpeg2dec->peerpool)
    GST_INFO (GST_CAT_PLUGIN_INFO, "got pool %p\n", mpeg2dec->peerpool);

  return 0;
}


static void
gst_mpeg2dec_vo_close (vo_instance_t * instance)
{
  gst_mpeg2dec_vo_instance_t * _instance;

  GST_INFO (GST_CAT_PLUGIN_INFO, "VO: close\n");

  _instance = (gst_mpeg2dec_vo_instance_t*)instance;
  // FIXME
}

static vo_frame_t *
gst_mpeg2dec_vo_get_frame (vo_instance_t * instance, int flags)
{
  gst_mpeg2dec_vo_instance_t * _instance;
  gst_mpeg2dec_vo_frame_t *frame;
  size_t size0, size;
  uint8_t *data = NULL;

  g_return_val_if_fail (instance != NULL, NULL);

  GST_INFO (GST_CAT_PLUGIN_INFO, "VO: get_frame\n");

  _instance = (gst_mpeg2dec_vo_instance_t *)instance;
  if (flags & VO_PREDICTION_FLAG) {
    _instance->prediction_index ^= 1;
    frame = &_instance->frames[_instance->prediction_index];
//    fprintf(stderr, "VO: get_frame f=%d\n", _instance->prediction_index);
  } else {
    frame = &_instance->frames[2];
//    fprintf(stderr, "VO: get_frame f=2\n");
  }

  if (frame->buffer != NULL) {
    gst_buffer_unref (frame->buffer);
  }

  frame->buffer = NULL;

  size0 = _instance->mpeg2dec->width * _instance->mpeg2dec->height / 4;

  if (_instance->mpeg2dec->peerpool) {
    frame->buffer = gst_buffer_new_from_pool (_instance->mpeg2dec->peerpool);

    data = GST_BUFFER_DATA (frame->buffer);
  }

  if (!frame->buffer) {
    frame->buffer = gst_buffer_new ();

    size = 6 * size0;
    data = g_new(uint8_t, size);

    GST_BUFFER_SIZE(frame->buffer) = size;
    GST_BUFFER_DATA(frame->buffer) = data;
  }

  // need ref=2
  // 1 - unref when reusing this frame
  // 2 - unref when other elements done with buffer
  gst_buffer_ref (frame->buffer);

  //_instance->mpeg2dec->next_time += video_rates[_instance->mpeg2dec->decoder->frame_rate_code]*1000LL;
  //GST_BUFFER_TIMESTAMP(frame->buffer) = _instance->mpeg2dec->next_time;

  frame->vo.base[0] = data;
  frame->vo.base[1] = data + 4 * size0;
  frame->vo.base[2] = data + 5 * size0;

  return (vo_frame_t *)frame;
}

static vo_instance_t *
gst_mpeg2dec_vo_open (GstMpeg2dec *mpeg2dec)
{
  gst_mpeg2dec_vo_instance_t * instance;
  int i,j;

  GST_INFO (GST_CAT_PLUGIN_INFO, "VO: open\n");

  instance = g_new (gst_mpeg2dec_vo_instance_t, 1);
  
  instance->vo.setup = gst_mpeg2dec_vo_setup;
  instance->vo.close = gst_mpeg2dec_vo_close;
  instance->vo.get_frame = gst_mpeg2dec_vo_get_frame;
  instance->mpeg2dec = mpeg2dec;

  for (i=0; i<3; i++) {
    for (j=0; j<3; j++) {
      instance->frames[j].vo.base[j] = NULL;
    }
    instance->frames[i].vo.copy = NULL;
    instance->frames[i].vo.field = NULL;
    instance->frames[i].vo.draw = gst_mpeg2dec_vo_frame_draw;
    instance->frames[i].vo.instance = (vo_instance_t *)instance;
    instance->frames[i].buffer = NULL;
  }

  return (vo_instance_t *) instance;
}

static void
gst_mpeg2dec_init (GstMpeg2dec *mpeg2dec)
{
  guint32 accel;
  vo_instance_t *vo;

  /* create the sink and src pads */
  mpeg2dec->sinkpad = gst_pad_new_from_template (sink_template_factory (), "sink");
  gst_element_add_pad (GST_ELEMENT (mpeg2dec), mpeg2dec->sinkpad);
  gst_element_set_loop_function (GST_ELEMENT(mpeg2dec), gst_mpeg2dec_loop);

  mpeg2dec->srcpad = gst_pad_new_from_template (src_template_factory (), "src");
  gst_element_add_pad (GST_ELEMENT (mpeg2dec), mpeg2dec->srcpad);

  /* initialize the mpeg2dec decoder state */
  mpeg2dec->decoder = g_new (mpeg2dec_t, 1);

  accel = mm_accel();

  vo = gst_mpeg2dec_vo_open (mpeg2dec);
  mpeg2_init (mpeg2dec->decoder, accel, vo);
  mpeg2dec->next_time = 0;
  mpeg2dec->peerpool = NULL;

  // reset the initial video state
  mpeg2dec->format = -1;
  mpeg2dec->width = -1;
  mpeg2dec->height = -1;
}

static void
gst_mpeg2dec_free (GstMpeg2dec *mpeg2dec)
{
  g_free (mpeg2dec);
}

static void
gst_mpeg2dec_loop (GstElement *element)
{
  GstMpeg2dec *mpeg2dec = GST_MPEG2DEC(element);
  GstBuffer *buf;
  guint32 size, offset;
  guchar *data;
  guint num_frames;

  GST_DEBUG (0, "MPEG2DEC: loop started\n");

  do {
    // get a new buffer
    buf = gst_pad_pull (mpeg2dec->sinkpad);

    if (GST_BUFFER_FLAG_IS_SET (buf, GST_BUFFER_FLUSH)) {
      //mpeg2dec->decoder->is_picture_needed = 1;
      mpeg2dec->decoder->is_sequence_needed = 1;
      gst_buffer_unref(buf);
      continue; //FIXME  need to pass on flush?
    }

    size = GST_BUFFER_SIZE (buf);
    data = GST_BUFFER_DATA (buf);
    offset = 0;
    if (mpeg2dec->next_time <= GST_BUFFER_TIMESTAMP (buf)) {
      mpeg2dec->next_time = GST_BUFFER_TIMESTAMP (buf);
      GST_DEBUG (0, "in:  %lld\n", GST_BUFFER_TIMESTAMP (buf));
    }

//fprintf(stderr, "MPEG2DEC: in timestamp=%llu\n",GST_BUFFER_TIMESTAMP(buf));
//fprintf(stderr, "MPEG2DEC: have buffer of %d bytes\n",size);
    num_frames = mpeg2_decode_data(mpeg2dec->decoder, data, data + size);
//fprintf(stderr, "MPEG2DEC: decoded %d frames\n", num_frames);

    gst_buffer_unref(buf);
  } while (!GST_ELEMENT_IS_COTHREAD_STOPPING (element));
}

static void
gst_mpeg2dec_set_property (GObject *object, guint prop_id, const GValue *value, GParamSpec *pspec)
{
  GstMpeg2dec *src;

  /* it's not null if we got it, but it might not be ours */
  g_return_if_fail (GST_IS_MPEG2DEC (object));
  src = GST_MPEG2DEC (object);

  switch (prop_id) {
    default:
      break;
  }
}

static void
gst_mpeg2dec_get_property (GObject *object, guint prop_id, GValue *value, GParamSpec *pspec)
{
  GstMpeg2dec *src;

  /* it's not null if we got it, but it might not be ours */
  g_return_if_fail (GST_IS_MPEG2DEC (object));
  src = GST_MPEG2DEC (object);

  switch (prop_id) {
    default:
      break;
  }
}

static gboolean
plugin_init (GModule *module, GstPlugin *plugin)
{
  GstElementFactory *factory;

  /* create an elementfactory for the mpeg2dec element */
  factory = gst_elementfactory_new("mpeg2dec",GST_TYPE_MPEG2DEC,
                                   &gst_mpeg2dec_details);
  g_return_val_if_fail(factory != NULL, FALSE);

  gst_elementfactory_add_padtemplate (factory, src_template_factory ());

  gst_elementfactory_add_padtemplate (factory, sink_template_factory ());

  gst_plugin_add_factory(plugin,factory);

  return TRUE;
}

GstPluginDesc plugin_desc = {
  GST_VERSION_MAJOR,
  GST_VERSION_MINOR,
  "mpeg2dec",
  plugin_init
};
