/* Gnome-Streamer
 * Copyright (C) <1999> Erik Walthinsen <omega@cse.ogi.edu>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */


//#define GST_DEBUG_ENABLED
#include <mpeg2parse.h>

static void gst_mpeg2parse_chain (GstPad *pad, GstBuffer *buf);

/* elementfactory information */
static GstElementDetails mpeg2parse_details = {
  "MPEG System Parser",
  "Filter/Parser/System",
  "Demultiplexes MPEG1 and MPEG2 System Streams",
  VERSION,
  "Erik Walthinsen <omega@cse.ogi.edu>\n"
  "Wim Taymans <wxim.taymans@chello.be>",
  "(C) 1999",
};


/* Mpeg2Parse signals and args */
enum {
  /* FILL ME */
  LAST_SIGNAL
};

enum {
  ARG_0,
  ARG_BIT_RATE,
  /* FILL ME */
};

static GstPadTemplate*
sink_factory (void) 
{
  return
   gst_padtemplate_new (
  	"sink",
  	GST_PAD_SINK,
  	GST_PAD_ALWAYS,
  	gst_caps_new (
  	  "mpeg2parse_sink",
    	  "video/mpeg",
	  gst_props_new (
    	    "mpegversion",  GST_PROPS_INT (2),
    	    "systemstream", GST_PROPS_BOOLEAN (TRUE),
	    NULL)),
	NULL);
}

static GstPadTemplate*
audio_factory (void) 
{
  return
   gst_padtemplate_new (
  	"audio_[1-32]",
  	GST_PAD_SRC,
  	GST_PAD_SOMETIMES,
  	gst_caps_new (
  	  "mpeg2parse_audio",
    	  "audio/mp3",
	  NULL),
	NULL);
}

static GstPadTemplate* 
video_factory (void)
{
  return
   gst_padtemplate_new (
  	"video_[0-15]",
  	GST_PAD_SRC,
  	GST_PAD_SOMETIMES,
  	gst_caps_new (
  	  "mpeg2parse_video",
     	  "video/mpeg",
	  gst_props_new (
    	    "mpegversion",  GST_PROPS_INT (2),
    	    "systemstream",  GST_PROPS_BOOLEAN (FALSE),
	    NULL)),
	NULL);
}

static GstPadTemplate* 
private1_factory (void)
{
  return
   gst_padtemplate_new (
  	"private_stream_1.[0-7]",
  	GST_PAD_SRC,
  	GST_PAD_SOMETIMES,
  	gst_caps_new (
  	  "mpeg2parse_private1",
    	  "audio/ac3",
	  NULL),
	NULL);
};

static GstPadTemplate* 
private2_factory (void)
{
  return
   gst_padtemplate_new (
  	"private_stream_2",
  	GST_PAD_SRC,
  	GST_PAD_SOMETIMES,
  	gst_caps_new (
  	  "mpeg2parse_private2",
    	  "unknown/unknown",
	  NULL),
	NULL);
};

static GstPadTemplate* 
subtitle_factory (void)
{
  return
   gst_padtemplate_new (
  	"subtitle_stream_[0-15]",
  	GST_PAD_SRC,
  	GST_PAD_SOMETIMES,
  	gst_caps_new (
  	  "mpeg2parse_subtitle",
    	  "video/mpeg",
	  NULL),
	NULL);
};

static void gst_mpeg2parse_class_init(Mpeg2ParseClass *klass);
static void gst_mpeg2parse_init(Mpeg2Parse *mpeg2parse);

static void gst_mpeg2parse_get_property(GObject *object, guint prop_id, GValue *value, GParamSpec *pspec);

static GstPadTemplate *sink_template,
                      *audio_template,
		      *video_template,
		      *private1_template,
		      *private2_template,
		      *subtitle_template;

static GstElementClass *parent_class = NULL;
//static guint gst_mpeg2parse_signals[LAST_SIGNAL] = { 0 };

GType
mpeg2parse_get_type (void)
{
  static GType mpeg2parse_type = 0;

  if (!mpeg2parse_type) {
    static const GTypeInfo mpeg2parse_info = {
      sizeof(Mpeg2ParseClass),      NULL,
      NULL,
      (GClassInitFunc)gst_mpeg2parse_class_init,
      NULL,
      NULL,
      sizeof(Mpeg2Parse),
      0,
      (GInstanceInitFunc)gst_mpeg2parse_init,
    };
    mpeg2parse_type = g_type_register_static(GST_TYPE_ELEMENT, "Mpeg2Parse", &mpeg2parse_info, 0);
  }
  return mpeg2parse_type;
}

static void
gst_mpeg2parse_class_init(Mpeg2ParseClass *klass) {
  GObjectClass *gobject_class;
  GstElementClass *gstelement_class;

  gobject_class = (GObjectClass*)klass;
  gstelement_class = (GstElementClass*)klass;

  g_object_class_install_property(G_OBJECT_CLASS(klass), ARG_BIT_RATE,
    g_param_spec_uint("bit_rate","bit_rate","bit_rate",
                     0,G_MAXUINT,0,G_PARAM_READABLE)); // CHECKME

  gobject_class->get_property = gst_mpeg2parse_get_property;

  parent_class = g_type_class_ref(GST_TYPE_ELEMENT);
}

static void
gst_mpeg2parse_init (Mpeg2Parse *mpeg2parse)
{
  gint i;

  mpeg2parse->sinkpad = gst_pad_new_from_template(sink_template, "sink");
  gst_element_add_pad(GST_ELEMENT(mpeg2parse),mpeg2parse->sinkpad);
  gst_pad_set_chain_function(mpeg2parse->sinkpad,gst_mpeg2parse_chain);

  // set up getbits
  gst_getbits_init(&mpeg2parse->gb, NULL, NULL);

  // initialize parser state
  mpeg2parse->have_sync = FALSE;
  mpeg2parse->sync_zeros = 0;
  mpeg2parse->id = 0;

  // zero counters (should be done at RUNNING?)
  mpeg2parse->packs = 0;
  mpeg2parse->prevbuf = NULL;
  mpeg2parse->bit_rate = 0;

  mpeg2parse->last_pts = 0;

  for (i=0;i<8;i++) {
    mpeg2parse->private_1_pad[i] = NULL;
    mpeg2parse->private_1_offset[i] = 0;
    mpeg2parse->private_1_need_flush[i] = FALSE;
  }
  for (i=0;i<16;i++) {
    mpeg2parse->subtitle_pad[i] = NULL;
    mpeg2parse->subtitle_offset[i] = 0;
    mpeg2parse->subtitle_need_flush[i] = FALSE;
  }
  mpeg2parse->private_2_pad = NULL;
  mpeg2parse->private_2_offset = 0;
  mpeg2parse->private_2_need_flush = FALSE;
  for (i=0;i<16;i++) {
    mpeg2parse->video_pad[i] = NULL;
    mpeg2parse->video_offset[i] = 0;
    mpeg2parse->video_need_flush[i] = FALSE;
  }
  for (i=0;i<32;i++) {
    mpeg2parse->audio_pad[i] = NULL;
    mpeg2parse->audio_offset[i] = 0;
    mpeg2parse->audio_need_flush[i] = FALSE;
  }
}

/* NOTE: all chunk parsers will return one of three things:
   Negative number: there are -N bytes too few to parse this chunk
   0: failed to correctly parse chunk
   Positive number: there were N bytes total in the chunk
*/
#define calc_time_stamp(gb, res)  \
      res = (gst_getbits3(gb) << 30); \
      if (gst_getbits1(gb) != 1) { \
        printf("mpeg2parse: expecting marker bit '1' in calc_time_stamp (1)\n");return FALSE; } \
      res |= (gst_getbits15(gb) << 15); \
      if (gst_getbits1(gb) != 1) { \
        printf("mpeg2parse: expecting marker bit '1' in calc_time_stamp (2)\n");return FALSE; } \
      res |= gst_getbits15(gb); \
      if (gst_getbits1(gb) != 1) { \
        printf("mpeg2parse: expecting marker bit '1' in calc_time_stamp (3)\n");return FALSE; }

/* This checks for and parses the pack header.  If at any point something
   looks odd (marker bits not correct), it will return FALSE and not set
   the values in the mpeg2parse structure. */
static gint
parse_packhead (Mpeg2Parse *mpeg2parse, guchar *buf, gint size)
{
  gint stuffing = 0;
  guint64 scr_base;
  guint16 scr_extension = 0;
  guint16 bit_rate;
  gst_getbits_t *gb = &mpeg2parse->gb;
  int code;
  int length = 8;

  GST_DEBUG (0,"mpeg2parse: in parse_packhead\n");

  // check length first
  if (size < 8)
    // return how many bytes missing
    return -(8-size);

  // set up getbits
  gst_getbits_newbuf(gb,buf, size);

  // start parsing the stream
  if ((code = gst_getbits2(gb)) != 0x1){
    if (!code && (gst_getbits2(gb) != 0x2)){ /* this is MPEG1 */
      printf("mpeg2parse::parse_packhead invalid header\n");
      return FALSE;
    }
    GST_DEBUG (0,"mpeg2parse::parse_packhead setting mpeg1\n");
    mpeg2parse->MPEG2 = FALSE;
  }
  else {
    GST_DEBUG (0,"mpeg2parse::parse_packhead setting mpeg2\n");
    mpeg2parse->MPEG2 = TRUE;
    // check length first
    if (size < 10)
      // return how many bytes missing
      return -(10-size);
  }
  calc_time_stamp(gb, scr_base);

  GST_DEBUG (0,"mpeg2parse::parse_packhead: SCR = %lld\n",scr_base);
  mpeg2parse->scr_base = scr_base;

  if (mpeg2parse->MPEG2) {
    scr_extension = gst_getbits9(gb); /* MPEG2 extension */
  }
  // parse the bit_rate
  if (gst_getbits1(gb) != 1) {
    printf("mpeg2parse::parse_packhead: expecting marker bit '1' in calc mux (1)\n");return FALSE; }

  bit_rate = gst_getbits22(gb);
  GST_DEBUG (0,"mpeg2parse: stream is %1.3fMbps\n",((float)(bit_rate*50))/1000000);

  if (gst_getbits1(gb) != 1) {
      printf("mpeg2parse::parse_packhead: expecting marker bit '1' in calc mux (2)\n");return FALSE; }

  mpeg2parse->bit_rate = bit_rate;

  if (mpeg2parse->MPEG2) { /* MPEG2 extensions */
    if (gst_getbits1(gb) != 1)
      return FALSE;
    gst_flushbitsn(gb, 5);
    stuffing = gst_getbits3(gb);
    // make sure we've got enough bytes here to include stuffing
    if (size < (10 + stuffing))
      return -((10 + stuffing)-size);
    gst_flushbitsn(gb, stuffing*8);

    mpeg2parse->scr_extension = scr_extension;
    length = 10;
  }
  mpeg2parse->have_packhead = TRUE;
  mpeg2parse->packs++;
  // return total number of bytes used in this chunk
  GST_DEBUG (0,"mpeg2parse: out parse_packhead %d\n", length+stuffing);
  return (length+stuffing);
}

static gint
parse_syshead (Mpeg2Parse *mpeg2parse, guchar *buf, gint size)
{
  guint16 header_length;
  guint32 rate_bound;
  guint8 audio_bound;
  gboolean fixed;
  gboolean constrained;
  gboolean audio_lock;
  gboolean video_lock;
  guint8 video_bound;
  guint8 ps_id_code;
  gboolean packet_rate_restriction;
  guint stream_count, i, j;
  gst_getbits_t *gb = &mpeg2parse->gb;
  GstPadTemplate *newtemp = NULL;

  GST_DEBUG (0,"mpeg2parse: in parse_syshead\n");

  // make sure we have enough bytes to know how long this is first
  if (size < 2)
    // return how many bytes missing
    return -(2-size);

  // set up getbits
  gst_getbits_newbuf(gb,buf,size);

  // start parsing
  header_length = gst_getbits16(gb);
  // check known length
  if (size < (2 + header_length))
    return -((2 + header_length)-size);

  if (gst_getbits1(gb) != 1) {
    printf("mpeg2parse::parse_syshead: expecting marker bit '1' before rate bound \n");return FALSE; }
  rate_bound = gst_getbits22(gb);
  if (gst_getbits1(gb) != 1) {
    printf("mpeg2parse::parse_syshead: expecting marker bit '1' after rate bound\n");return FALSE; }
  GST_DEBUG (0,"mpeg2parse::parse_syshead: Rate bound = %d \n",rate_bound);

  audio_bound = gst_getbits6(gb);
  fixed = gst_getbits1(gb);
  constrained = gst_getbits1(gb);
  GST_DEBUG (0,"mpeg2parse::parse_syshead: Audio bound=%d, Fixed=%d, constrained=%d \n",audio_bound, fixed, constrained);
  
  audio_lock = gst_getbits1(gb);
  video_lock = gst_getbits1(gb);
  if (gst_getbits1(gb) != 1) {
    printf("mpeg2parse::parse_syshead: expecting marker bit '1' after video lock\n");return FALSE; }

  video_bound = gst_getbits5(gb);
  GST_DEBUG (0,"mpeg2parse::parse_syshead: audio lock=%d, video lock=%d, video bound=%d \n",audio_lock, video_lock, video_bound);
	
  packet_rate_restriction = gst_getbits1(gb);
  if (gst_getbits7(gb) != 0x7f) {
    printf("mpeg2parse::parse_syshead: reserved bytes not set to default\n");return FALSE; }

  mpeg2parse->have_syshead = TRUE;
  mpeg2parse->header_length = header_length;
  mpeg2parse->rate_bound = rate_bound;
  mpeg2parse->audio_bound = audio_bound;
  mpeg2parse->fixed = fixed;
  mpeg2parse->constrained = constrained;
  mpeg2parse->audio_lock = audio_lock;
  mpeg2parse->video_lock = video_lock;
  mpeg2parse->video_bound = video_bound;
  mpeg2parse->packet_rate_restriction = packet_rate_restriction;
  // if MPEG2, we are done, return total number of bytes used in this chunk
  if (mpeg2parse->MPEG2) 
    return (2 + header_length);
  // MPEG1 steam parsing
  stream_count = (header_length - 6) / 3;
  GST_DEBUG (0,"mpeg2parse::parse_syshead: number of streams=%d \n", stream_count);

  j=0;
  for (i=0; i < stream_count; i++) {
    gint stream_num;
    guint8 stream_id;
    gboolean STD_buffer_bound_scale;
    guint16 STD_buffer_size_bound;
    guint32 buf_byte_size_bound;
    gchar *name = NULL;
    GstPad **outpad = NULL;

    if (gst_getbits1(gb) != 1) {
      printf("mpeg2parse::parse_syshead: error in system header length\n");return FALSE; }
    stream_id = 0x80 | gst_getbits7(gb);

	 // check marker bits
    if (gst_getbits2(gb) != 0x3) {
      printf("mpeg2parse::parse_syshead: expecting placeholder bit values '11' after stream id\n");return FALSE; }

     STD_buffer_bound_scale = gst_getbits1(gb);
     STD_buffer_size_bound = gst_getbits5(gb)<<8;
     STD_buffer_size_bound |= gst_getbits8(gb);

     if (STD_buffer_bound_scale == 0) {
       buf_byte_size_bound = STD_buffer_size_bound * 128;
     } else {
       buf_byte_size_bound = STD_buffer_size_bound * 1024;
     }

    // private_stream_1
    if (stream_id == 0xBD) {
      // first find the track code
      ps_id_code = *(guint8 *)(buf+header_length);
      // make sure it's valid
      if ((ps_id_code >= 0x80) && (ps_id_code <= 0x87)) {
        GST_DEBUG (0,"mpeg2parse::parse_syshead: 0x%02X: we have a private_stream_1 (AC3) packet, track %d\n",
              stream_id, ps_id_code - 0x80);
        // scrap first 4 bytes (so-called "mystery AC3 tag")
        //headerlen += 4;
        //datalen -= 4;
        name = g_strdup_printf("private_stream_1_%02d",ps_id_code - 0x80);
        stream_num = stream_id & 0x07;
        outpad = &mpeg2parse->private_1_pad[ps_id_code - 0x80];
	newtemp = private1_template;
      }
      else if ((ps_id_code >= 0x20) && (ps_id_code <= 0x2f)) {
        GST_DEBUG (0,"mpeg2parse::parse_syshead: 0x%02X: we have a subtitle_stream packet, track %d\n",
              stream_id, ps_id_code - 0x20);
        name = g_strdup_printf("subtitle_stream_%02d",ps_id_code - 0x20);
        stream_num = stream_id & 0x0f;
        outpad = &mpeg2parse->subtitle_pad[ps_id_code - 0x20];
        newtemp = subtitle_template;
      }
    // private_stream_2
    } else if (stream_id == 0xBF) {
      name = g_strdup_printf("private_stream_2");
      stream_num = 0;
      outpad = &mpeg2parse->private_2_pad;
      newtemp = private2_template;
    // Audio
    } else if ((stream_id >= 0xC0) && (stream_id <= 0xDF)) {
      name = g_strdup_printf("audio_%02d",stream_id & 0x1F);
      stream_num = stream_id & 0x1F;
      outpad = &mpeg2parse->audio_pad[stream_num];
      newtemp = audio_template;
    // Video
    } else if ((stream_id >= 0xE0) && (stream_id <= 0xEF)) {
      name = g_strdup_printf("video_%02d",stream_id & 0x0F);
      stream_num = stream_id & 0x0F;
      outpad = &mpeg2parse->video_pad[stream_num];
      newtemp = video_template;
    }

    GST_DEBUG (0,"mpeg2parse::parse_syshead: stream ID 0x%02X (%s)\n", stream_id, name);
    GST_DEBUG (0,"mpeg2parse::parse_syshead: STD_buffer_bound_scale %d\n", STD_buffer_bound_scale);
    GST_DEBUG (0,"mpeg2parse::parse_syshead: STD_buffer_size_bound %d or %d bytes\n", STD_buffer_size_bound, buf_byte_size_bound);

    // create the pad and add it to self if it does not yet exist
    // this should trigger the NEW_PAD signal, which should be caught by
    // the app and used to attach to desired streams.
    if (*outpad == NULL) {
      (*outpad) = gst_pad_new_from_template(newtemp, name);
      gst_pad_set_caps((*outpad), gst_pad_get_padtemplate_caps (*outpad));
      gst_element_add_pad(GST_ELEMENT(mpeg2parse),(*outpad));
    }
    else {
      // we won't be needing this.
      if (name) g_free(name);
    }

    mpeg2parse->STD_buffer_info[j].stream_id = stream_id;
    mpeg2parse->STD_buffer_info[j].STD_buffer_bound_scale = STD_buffer_bound_scale;
    mpeg2parse->STD_buffer_info[j].STD_buffer_size_bound = STD_buffer_size_bound;

    j++;
  }
  // return total number of bytes used in this chunk
  return (2 + header_length);
}

static gint
parse_packet(Mpeg2Parse *mpeg2parse,guchar *data,gulong size,
             GstBuffer *buf,gulong offset)
{
  guint8 id = mpeg2parse->id;
  guint8 bits;
  guint16 headerlen;
  gboolean *need_flush = NULL;

  guint16 packet_length;
  gboolean std_buffer_scale;
  guint16 std_buffer_size;
  guint64 dts;

  guint16 datalen;
  gulong outoffset = 0;
  guint8 ps_id_code;

  GstPad **outpad = NULL;
  GstBuffer *outbuf;
  gst_getbits_t *gb = &mpeg2parse->gb;

  GST_DEBUG (0,"mpeg2parse::parse_packet: in parse_packet\n");

  // make sure we have enough bytes to know how long this is first
  if (size < 2)
    // return how many bytes missing
    return -(2-size);

  // set up getbits
  gst_getbits_newbuf(gb,data, size);

  // start parsing
  packet_length = gst_getbits16(gb);
  headerlen = 2;
  GST_DEBUG (0,"mpeg2parse::parse_packet: got packet_length %d\n",packet_length);
  // constant is 2 bytes packet_length
  if (size < (2 + packet_length)) {
    GST_DEBUG (0,"mpeg2parse::parse_packet: have not enough data\n");
    return -((2 + packet_length)-size);
  }

  // loop through looping for stuffing bits, STD, PTS, DTS, etc
  do {
    bits = gst_getbits2(gb);
    // stuffing bytes
    if (bits == 0x3) {
      if (gst_getbits6(gb) == 0x3f) {
        GST_DEBUG (0,"mpeg2parse::parse_packet: have stuffing byte\n");
      } else {
        GST_DEBUG (0,"mpeg2parse::parse_packet: expected stuffing byte\n");
      }
      headerlen++;
    } else if (bits == 0x1) {
      GST_DEBUG (0,"mpeg2parse::parse_packet: have STD\n");
      std_buffer_scale = gst_getbits1(gb);
      std_buffer_size = gst_getbits13(gb);
      headerlen += 2;
    } else if (bits == 0) {
      bits = gst_getbits2(gb);
      if (bits == 0x2) {
        calc_time_stamp(gb, mpeg2parse->last_pts);
        GST_DEBUG (0,"mpeg2parse::parse_packet: PTS = %llu\n", mpeg2parse->last_pts);
        headerlen += 5;
	break;
      } else if (bits == 0x3) {
        calc_time_stamp(gb, mpeg2parse->last_pts);
        if (gst_getbits4(gb) != 1)
          return FALSE;
        calc_time_stamp(gb, dts);
        GST_DEBUG (0,"mpeg2parse::parse_packet: PTS = %llu, DTS = %llu\n", mpeg2parse->last_pts, dts);
        headerlen += 10;
        break;
      } else if (bits == 0) {
        GST_DEBUG (0,"mpeg2parse::parse_packet: have no pts/dts\n");
        bits = gst_getbits4(gb);
        GST_DEBUG (0,"mpeg2parse::parse_packet: got trailer bits %x\n",bits);
        if (bits != 0xf) {
          printf("mpeg2parse::parse_packet: not a valid packet time sequence\n");
          return FALSE;
        }
        headerlen++;
        break;
      } else break;
    } else break;
  } while (1);
  GST_DEBUG (0,"mpeg2parse::parse_packet: done with header loop\n");

  // calculate the amount of real data in this packet
  datalen = packet_length - headerlen+2;
  GST_DEBUG (0,"mpeg2parse::parse_packet: headerlen is %d, datalen is %d, size is %lu\n",
        headerlen,datalen,size);

  // private_stream_1
  if (id == 0xBD) {
    // first find the track code
    ps_id_code = *(guint8 *)(data+headerlen);
    // make sure it's valid
    if ((ps_id_code >= 0x80) && (ps_id_code <= 0x87)) {
      GST_DEBUG (0,"mpeg2parse::parse_packet: 0x%02X: we have a private_stream_1 (AC3) packet, track %d\n",
            id, ps_id_code - 0x80);
      outpad = &mpeg2parse->private_1_pad[ps_id_code - 0x80];
      // scrap first 4 bytes (so-called "mystery AC3 tag")
      headerlen += 4;
      datalen -= 4;
      outoffset = mpeg2parse->private_1_offset[ps_id_code - 0x80];
      mpeg2parse->private_1_offset[ps_id_code - 0x80] += datalen;
      need_flush = &mpeg2parse->private_1_need_flush[ps_id_code - 0x80];
    }
    if ((ps_id_code >= 0x20) && (ps_id_code <= 0x2f)) {
      GST_DEBUG (0,"mpeg2parse::parse_syshead: 0x%02X: we have a subtitle_stream packet, track %d\n",
            id, ps_id_code - 0x80);
      outpad = &mpeg2parse->subtitle_pad[ps_id_code - 0x20];
      headerlen += 1;
      datalen -= 1;
      outoffset = mpeg2parse->subtitle_offset[ps_id_code - 0x20];
      mpeg2parse->subtitle_offset[ps_id_code - 0x20] += datalen;
      need_flush = &mpeg2parse->subtitle_need_flush[ps_id_code - 0x20];
    }
  // private_stream_1
  } else if (id == 0xBF) {
    GST_DEBUG (0,"mpeg2parse::parse_packet: 0x%02X: we have a private_stream_2 packet\n", id);
    outpad = &mpeg2parse->private_2_pad;
    outoffset = mpeg2parse->private_2_offset;
    mpeg2parse->private_2_offset += datalen;
    need_flush = &mpeg2parse->private_2_need_flush;
  // audio
  } else if ((id >= 0xC0) && (id <= 0xDF)) {
    GST_DEBUG (0,"mpeg2parse::parse_packet: 0x%02X: we have an audio packet\n", id);
    outpad = &mpeg2parse->audio_pad[id & 0x1F];
    outoffset = mpeg2parse->audio_offset[id & 0x1F];
    mpeg2parse->audio_offset[id & 0x1F] += datalen;
    need_flush = &mpeg2parse->audio_need_flush[id & 0x1F];
  // video
  } else if ((id >= 0xE0) && (id <= 0xEF)) {
    GST_DEBUG (0,"mpeg2parse::parse_packet: 0x%02X: we have a video packet\n", id);
    outpad = &mpeg2parse->video_pad[id & 0x0F];
    outoffset = mpeg2parse->video_offset[id & 0x0F];
    mpeg2parse->video_offset[id & 0x0F] += datalen;
    need_flush = &mpeg2parse->video_need_flush[id & 0x0F];

  }

  // if we don't know what it is, bail
  if (outpad == NULL) {
    printf("mpeg2parse::parse_packet: unknown packet id 0x%02X !!\n", id);
    // return total number of bytes
    return (2 + packet_length);
  }

  // FIXME, this should be done in parse_syshead
  if ((*outpad) == NULL) {
    printf("mpeg2parse::parse_packet: unexpected packet id 0x%02X!!\n", id);
    // return total number of bytes
    return (2 + packet_length);
  }

  // create the buffer and send it off to the Other Side
  if (GST_PAD_CONNECTED(*outpad)) {
    // if this is part of the buffer, create a subbuffer
    GST_DEBUG (0,"mpeg2parse::parse_packet: creating subbuffer at offset %lu, len %d\n",
            offset+headerlen,datalen);
    outbuf = gst_buffer_create_sub(buf,offset+headerlen,datalen);
    g_assert(outbuf != NULL);

    GST_DEBUG (0,"mpeg2parse::parse_packet: pushing buffer of len %d id %d\n", datalen, id);
    GST_BUFFER_OFFSET(outbuf) = outoffset;
    if (need_flush && *need_flush) {
      GST_BUFFER_FLAG_SET(outbuf, GST_BUFFER_FLUSH);
      *need_flush = FALSE;
    }
    gst_pad_push((*outpad),outbuf);
  }

  // return total number of bytes
  return (2 + packet_length);
}

static gint
parse_pes (Mpeg2Parse *mpeg2parse,guchar *data,gulong size,
           GstBuffer *buf,gulong offset)
{
  guint8 id;

  guint16 packet_length;
  guint8 header_data_length = 0;
  gboolean *need_flush = NULL;

  guint16 datalen;
  gulong outoffset = 0;
  guint16 headerlen;
  guint8 ps_id_code = 0x80;

  GstPad **outpad = NULL;
  GstBuffer *outbuf;
  GstPadTemplate *newtemp = NULL;

  GST_DEBUG (0,"mpeg2parse: in parse_pes\n");

  // make sure we have enough bytes to know how long this is first
  if (size < 2)
    // return how many bytes missing
    return -(2-size);

  // start parsing
  packet_length = (data[0]<<8) | data[1];

  GST_DEBUG (0,"mpeg2parse: got packet_length %d\n",packet_length);
  // constant is 2 bytes packet_length
  if (size < (2 + packet_length))
    return -((2 + packet_length)-size);

  // we don't operate on: program_stream_map, padding_stream,
  // private_stream_2, ECM, EMM, or program_stream_directory
  id = mpeg2parse->id;
  if ((id != 0xBC) && (id != 0xBE) && (id != 0xBF) && (id != 0xF0) &&
      (id != 0xF1) && (id != 0xFF)) 
  {
    gint pos = 2;
    guchar flags1 = data[pos++];
    guchar flags2 = data[pos++];
    header_data_length = data[pos++];

    if ((flags1 & 0xC0) != 0x80)
      return FALSE;

    GST_DEBUG (0,"mpeg2parse: header_data_length is %d\n",header_data_length);

    // check for PTS
    if ((flags2 & 0x80) && id == 0xe0) {
      mpeg2parse->last_pts  = (data[pos++] & 0x0E) << 29;
      mpeg2parse->last_pts |=  data[pos++]         << 22;
      mpeg2parse->last_pts |= (data[pos++] & 0xFE) << 14;
      mpeg2parse->last_pts |=  data[pos++]         <<  7;
      mpeg2parse->last_pts |= (data[pos++] & 0xFE) >>  1;
      GST_DEBUG (0, "mpeg2parse::parse_packet: %x PTS = %llu\n", id, (mpeg2parse->last_pts*1000000LL)/90000LL);

    }
    if ((flags2 & 0x40)) {
      //g_print("dts found %d %d\n", pos, flags2);
      pos += 5;
    }
    if ((flags2 & 0x20)) {
      //g_print("escr found %d\n", pos);
      pos += 6;
    }
    if ((flags2 & 0x10)) {
      guint32 es_rate;

      es_rate  = (data[pos++] & 0x07) << 14;
      es_rate |= (data[pos++]       ) << 7;
      es_rate |= (data[pos++] & 0xFE) >> 1;
      //g_print("es rate %d %u\n", pos, es_rate);
    }
    // FIXME: lots of PES parsing missing here...

  }

  // calculate the amount of real data in this PES packet
  // constant is 2 bytes packet_length, 2 bytes of bits, 1 byte header len
  headerlen = 5 + header_data_length;
  // constant is 2 bytes of bits, 1 byte header len
  datalen = packet_length - (3 + header_data_length);
  GST_DEBUG (0,"mpeg2parse: headerlen is %d, datalen is %d, size is %ld\n",
        headerlen,datalen,size);

  // private_stream_1
  if (id == 0xBD) {
    // first find the track code
    ps_id_code = *(guint8 *)(data+headerlen);
    // make sure it's valid
    if ((ps_id_code >= 0x80) && (ps_id_code <= 0x87)) {
      GST_DEBUG (0,"mpeg2parse: we have a private_stream_1 (AC3) packet, track %d\n",
            ps_id_code - 0x80);
      outpad = &mpeg2parse->private_1_pad[ps_id_code - 0x80];
      // scrap first 4 bytes (so-called "mystery AC3 tag")
      headerlen += 4;
      datalen -= 4;
      outoffset = mpeg2parse->private_1_offset[ps_id_code - 0x80];
      mpeg2parse->private_1_offset[ps_id_code - 0x80] += datalen;
      need_flush = &mpeg2parse->private_1_need_flush[ps_id_code - 0x80];
    }
    else if ((ps_id_code >= 0x20) && (ps_id_code <= 0x2f)) {
      GST_DEBUG (0,"mpeg2parse: we have a subtitle_stream packet, track %d\n",
            ps_id_code - 0x20);
      outpad = &mpeg2parse->subtitle_pad[ps_id_code - 0x20];
      headerlen += 1;
      datalen -= 1;
      outoffset = mpeg2parse->subtitle_offset[ps_id_code - 0x20];
      mpeg2parse->subtitle_offset[ps_id_code - 0x20] += datalen;
      need_flush = &mpeg2parse->subtitle_need_flush[ps_id_code - 0x20];
    }
  // private_stream_1
  } else if (id == 0xBF) {
    GST_DEBUG (0,"mpeg2parse: we have a private_stream_2 packet\n");
    outpad = &mpeg2parse->private_2_pad;
    outoffset = mpeg2parse->private_2_offset;
    mpeg2parse->private_2_offset += datalen;
    need_flush = &mpeg2parse->private_2_need_flush;
  // audio
  } else if ((id >= 0xC0) && (id <= 0xDF)) {
    GST_DEBUG (0,"mpeg2parse: we have an audio packet\n");
    outpad = &mpeg2parse->audio_pad[id - 0xC0];
    outoffset = mpeg2parse->audio_offset[id & 0x1F];
    mpeg2parse->audio_offset[id & 0x1F] += datalen;
    need_flush = &mpeg2parse->audio_need_flush[id & 0x1F];
  // video
  } else if ((id >= 0xE0) && (id <= 0xEF)) {
    GST_DEBUG (0,"mpeg2parse: we have a video packet\n");
    outpad = &mpeg2parse->video_pad[id - 0xE0];
    outoffset = mpeg2parse->video_offset[id & 0x0F];
    mpeg2parse->video_offset[id & 0x0F] += datalen;
    need_flush = &mpeg2parse->video_need_flush[id & 0x0F];
  }

  // if we don't know what it is, bail
  if (outpad == NULL) {
    // return total number of bytes
    return (2 + packet_length);
  }

  // create the pad and add it if we don't already have one.
  // this should trigger the NEW_PAD signal, which should be caught by
  // the app and used to attach to desired streams.
  if ((*outpad) == NULL) {
    gchar *name;

    // we have to name the stream approriately
    if (id == 0xBD) {
      if (ps_id_code >= 0x80 && ps_id_code <= 0x87) {
        name = g_strdup_printf("private_stream_1.%d",ps_id_code - 0x80);
	newtemp = private1_template;
      }
      else if (ps_id_code >= 0x20 && ps_id_code <= 0x2f) {
        name = g_strdup_printf("subtitle_stream_%d",ps_id_code - 0x20);
        newtemp = subtitle_template;
      }
      else {
        name = g_strdup_printf("unknown_stream_%d",ps_id_code);
      }
    }
    else if (id == 0xBF) {
      name = "private_stream_2";
      newtemp = private2_template;
    }
    else if ((id >= 0xC0) && (id <= 0xDF)) {
      name = g_strdup_printf("audio_%d",id - 0xC0);
      newtemp = audio_template;
    }
    else if ((id >= 0xE0) && (id <= 0xEF)) {
      name = g_strdup_printf("video_%d",id - 0xE0);
      newtemp = video_template;
    }
    else {
      name = g_strdup_printf("unknown");
    }

    if (newtemp) {
      // create the pad and add it to self
      (*outpad) = gst_pad_new_from_template(newtemp, name);
      gst_pad_set_caps((*outpad), gst_pad_get_padtemplate_caps (*outpad));
      gst_element_add_pad(GST_ELEMENT(mpeg2parse),(*outpad));
    }
    else {
      g_print ("mpeg2parse: cannot create pad %s, no template for %02x\n", name, id);
    }
  }

  // create the buffer and send it off to the Other Side
  if (GST_PAD_CONNECTED(*outpad)) {
    // if this is part of the buffer, create a subbuffer
    GST_DEBUG (0,"mpeg2parse: creating subbuffer at offset %ld, len %d\n",
            offset+headerlen,datalen);
    outbuf = gst_buffer_create_sub(buf,offset+headerlen,datalen);
    g_assert(outbuf != NULL);

    GST_BUFFER_OFFSET(outbuf) = outoffset;
    if (need_flush && *need_flush) {
      GST_BUFFER_FLAG_SET(outbuf, GST_BUFFER_FLUSH);
      *need_flush = FALSE;
    }
    else {
      GST_BUFFER_FLAG_UNSET(outbuf, GST_BUFFER_FLUSH);
    }
    GST_BUFFER_TIMESTAMP(outbuf) = (mpeg2parse->last_pts*1000000LL)/90000LL;

    gst_pad_push((*outpad),outbuf);
  }

  // return total number of bytes
  return (2 + packet_length);
}

/* Some assumptions: buffers coming in are at least as big as a given
   header or PES packet.  This means we can get away without copying like
   mad to reassemble many-buffer splits. */
static void
gst_mpeg2parse_chain (GstPad *pad, GstBuffer *buf)
{
  Mpeg2Parse *mpeg2parse;
  guchar *data;
  gulong size,offset, bufoffset;
  gint i;
  GstBuffer *temp;

  g_return_if_fail(pad != NULL);
  g_return_if_fail(GST_IS_PAD(pad));
  g_return_if_fail(buf != NULL);

  mpeg2parse = GST_MPEG2PARSE(GST_OBJECT_PARENT (pad));

  bufoffset = GST_BUFFER_OFFSET(buf);
  size = GST_BUFFER_SIZE(buf);
  offset = 0;

  mpeg2parse->in_flush = GST_BUFFER_FLAG_IS_SET(buf, GST_BUFFER_FLUSH);

  if (mpeg2parse->in_flush) {
    GST_DEBUG (0,"mpeg2parse::chain: mpeg2parse: flush\n");

    if (mpeg2parse->prevbuf) {
      gst_buffer_unref(mpeg2parse->prevbuf);
      mpeg2parse->prevbuf = NULL;
    }
    for (i=0;i<8;i++) {
      mpeg2parse->private_1_offset[i] = 0;
      mpeg2parse->private_1_need_flush[i] = FALSE;
    }
    for (i=0;i<16;i++) {
      mpeg2parse->subtitle_offset[i] = 0;
      mpeg2parse->subtitle_need_flush[i] = FALSE;
    }
    mpeg2parse->private_2_offset = 0;
    mpeg2parse->private_2_need_flush = FALSE;
    for (i=0;i<16;i++) {
      mpeg2parse->video_offset[i] = 0;
      mpeg2parse->video_need_flush[i] = TRUE;
    }
    for (i=0;i<32;i++) {
      mpeg2parse->audio_offset[i] = 0;
      mpeg2parse->audio_need_flush[i] = TRUE;
    }
    mpeg2parse->have_sync = FALSE;
    mpeg2parse->sync_zeros = 0;
    mpeg2parse->id = 0;
  }

  GST_DEBUG (0,"mpeg2parse::chain: mpeg2parse: have buffer of size %lu\n",size);

  if (mpeg2parse->prevbuf) {
    GST_DEBUG (0,"mpeg2parse::chain: we have some data left\n");
    mpeg2parse->prevbuf = gst_buffer_append(mpeg2parse->prevbuf, buf);
    gst_buffer_unref(buf);
  }
  else {
    mpeg2parse->prevbuf = buf;
  }

  data = GST_BUFFER_DATA(mpeg2parse->prevbuf);
  size = GST_BUFFER_SIZE(mpeg2parse->prevbuf);

  // main loop
  while (offset < size) {
    GST_DEBUG (0,"mpeg2parse: offset is %lu\n",offset);
    // first step is to find sync, which will take at least three rounds
    if (!mpeg2parse->have_sync) {
      guchar byte = *(data+offset);
      offset++;
      // if it's zero, increment the zero count
      if (byte == 0) {
        mpeg2parse->sync_zeros++;
        GST_DEBUG (0,"mpeg2parse: found zero #%d at %ld\n",mpeg2parse->sync_zeros,offset-1);
      }
      // if it's a one and we have two previous zeros, we have sync
      else if ((byte == 1) && (mpeg2parse->sync_zeros >= 2)) {
        mpeg2parse->have_sync = TRUE;
        mpeg2parse->sync_zeros = 0;
      }
      // if it's anything else, we've lost it completely
      else mpeg2parse->sync_zeros = 0;

    // then snag the chunk ID
    } else if (mpeg2parse->id == 0) {
      mpeg2parse->id = *(data+offset);
      GST_DEBUG (0,"mpeg2parse: got id 0x%02X at offset %ld+%d=%ld\n",mpeg2parse->id,offset,
            mpeg2parse->prevbuf->offset,offset+mpeg2parse->prevbuf->offset);
      offset++;
      // check to see if it's valid
      if (mpeg2parse->MPEG2 && mpeg2parse->id < 0xB9) {
        // invalid, so we reset the parse state
        mpeg2parse->have_sync = FALSE;
        mpeg2parse->id = 0;
      }

    // and lastly get the chunk contents dealt with
    } else {
      gint ret = 0;
      // switchin the ID and run the right chunk parser
      GST_DEBUG (0,"mpeg2parse: have chunk 0x%02X at offset %ld\n",mpeg2parse->id,offset);
      switch (mpeg2parse->id) {
        case 0xBA:
          ret = parse_packhead(mpeg2parse,data+offset,size-offset);
          break;
        case 0xBB:
          ret = parse_syshead(mpeg2parse,data+offset,size-offset);
          break;
        default:
          if (mpeg2parse->MPEG2 && ((mpeg2parse->id < 0xBD) || (mpeg2parse->id > 0xFE))) {
            printf("mpeg2parse: ******** unknown id 0x%02X\n",mpeg2parse->id);
	  }
	  else {
            // if we're on the original buffer, we send subbuffers
            ret = mpeg2parse->MPEG2 ?
                  parse_pes(mpeg2parse,data+offset,size-offset,mpeg2parse->prevbuf,offset)
                : parse_packet(mpeg2parse,data+offset,size-offset,mpeg2parse->prevbuf,offset);
          }
      }
      // if we got a successful return out
      if (ret >= 0) {
        offset += ret;
        GST_DEBUG (0,"mpeg2parse: chunk is %d bytes\n",ret);
        // we have to reset the state
        mpeg2parse->have_sync = FALSE;
        mpeg2parse->id = 0;
        if (offset >= size-4) break; // no way we can find a sync
      }
      // or we didn't have enough data
      else {
	break;
      }
    }
  }
  if (offset < size) {
    GST_DEBUG (0,"mpeg2parse::chain: we're short for this chunk\n");
    temp = gst_buffer_create_sub(mpeg2parse->prevbuf, offset, size-offset);
    g_assert(temp != NULL);
    gst_buffer_unref(mpeg2parse->prevbuf);
    mpeg2parse->prevbuf = temp;
  }
  else {
    GST_DEBUG (0,"mpeg2parse::chain: cleaning previous buffer\n");
    gst_buffer_unref(mpeg2parse->prevbuf);
    mpeg2parse->prevbuf = NULL;
  }
}

static void
gst_mpeg2parse_get_property (GObject *object, guint prop_id, GValue *value, GParamSpec *pspec)
{
  Mpeg2Parse *mpeg2parse;

  /* it's not null if we got it, but it might not be ours */
  mpeg2parse = GST_MPEG2PARSE(object);

  switch (prop_id) {
    case ARG_BIT_RATE: {
      g_value_set_uint (value, mpeg2parse->bit_rate*316); // hu?
      break;
    }
    default: {
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
    }
  }
}


static gboolean
plugin_init (GModule *module, GstPlugin *plugin)
{
  GstElementFactory *factory;

  /* this filter needs the getbits package */
  if (!gst_library_load("gstgetbits")) {
    gst_info("mpeg2parse:: could not load support library: 'gstgetbits'\n");
    return FALSE;
  }

  /* create an elementfactory for the mpeg2parse element */
  factory = gst_elementfactory_new("mpeg2parse",GST_TYPE_MPEG2PARSE,
                                   &mpeg2parse_details);
  g_return_val_if_fail(factory != NULL, FALSE);

  sink_template = sink_factory ();
  gst_elementfactory_add_padtemplate (factory, sink_template);

  audio_template = audio_factory ();
  gst_elementfactory_add_padtemplate (factory, audio_template);

  video_template = video_factory ();
  gst_elementfactory_add_padtemplate (factory, video_template);

  private1_template = private1_factory ();
  gst_elementfactory_add_padtemplate (factory, private1_template);

  private2_template = private2_factory ();
  gst_elementfactory_add_padtemplate (factory, private2_template);

  subtitle_template = subtitle_factory ();
  gst_elementfactory_add_padtemplate (factory, subtitle_template);

  gst_plugin_add_factory(plugin,factory);

  return TRUE;
}

GstPluginDesc plugin_desc = {
  GST_VERSION_MAJOR,
  GST_VERSION_MINOR,
  "mpeg2parse",
  plugin_init
};
