/* getblk.c, DCT block decoding                                             */

/* Copyright (C) 1996, MPEG Software Simulation Group. All Rights Reserved. */

/*
 * Disclaimer of Warranty
 *
 * These software programs are available to the user without any license fee or
 * royalty on an "as is" basis.  The MPEG Software Simulation Group disclaims
 * any and all warranties, whether express, implied, or statuary, including any
 * implied warranties or merchantability or of fitness for a particular
 * purpose.  In no event shall the copyright-holder be liable for any
 * incidental, punitive, or consequential damages of any kind whatsoever
 * arising from the use of these programs.
 *
 * This disclaimer of warranty extends to the user of these programs and user's
 * customers, employees, agents, transferees, successors, and assigns.
 *
 * The MPEG Software Simulation Group does not represent or warrant that the
 * programs furnished hereunder are free of infringement of any third-party
 * patents.
 *
 * Commercial implementations of MPEG-1 and MPEG-2 video, including shareware,
 * are subject to royalty fees to patent holders.  Many of these patents are
 * general enough such that they are unavoidable regardless of implementation
 * design.
 *
 */

#include <stdio.h>

//#define TRACE
#include "mpeg2dec.h"

/* defined in getvlc.h */
typedef struct {
  char run, level, len;
} DCTtab;

extern DCTtab DCTtabfirst[],DCTtabnext[],DCTtab0[],DCTtab1[];
extern DCTtab DCTtab2[],DCTtab3[],DCTtab4[],DCTtab5[],DCTtab6[];
extern DCTtab DCTtab0a[],DCTtab1a[];


/* decode one intra coded MPEG-1 block */
void Decode_MPEG1_Intra_Block(vid_stream, comp, dc_dct_pred)
mpeg2play_vid_stream *vid_stream;
int comp;
int dc_dct_pred[];
{
  int val, i, j, sign;
  unsigned int code;
  DCTtab *tab;
  short *bp;
  gst_getbits_t *gb = &vid_stream->ld->gb;
  mpeg2play_layer_data *ld = vid_stream->ld;

  bp = ld->block[comp];

  /* ISO/IEC 11172-2 section 2.4.3.7: Block layer. */
  /* decode DC coefficients */
  if (comp<4)
    bp[0] = (dc_dct_pred[0]+=Get_Luma_DC_dct_diff(gb)) << 3;
  else if (comp==4)
    bp[0] = (dc_dct_pred[1]+=Get_Chroma_DC_dct_diff(gb)) << 3;
  else
    bp[0] = (dc_dct_pred[2]+=Get_Chroma_DC_dct_diff(gb)) << 3;

  if (vid_stream->Fault_Flag) return;

  /* D-pictures do not contain AC coefficients */
  if(vid_stream->picture.picture_coding_type == D_TYPE)
    return;

  /* decode AC coefficients */
  for (i=1; ; i++)
  {
    code = gst_showbits16(gb);
    if (code>=16384)
      tab = &DCTtabnext[(code>>12)-4];
    else if (code>=1024)
      tab = &DCTtab0[(code>>8)-4];
    else if (code>=512)
      tab = &DCTtab1[(code>>6)-8];
    else if (code>=256)
      tab = &DCTtab2[(code>>4)-16];
    else if (code>=128)
      tab = &DCTtab3[(code>>3)-16];
    else if (code>=64)
      tab = &DCTtab4[(code>>2)-16];
    else if (code>=32)
      tab = &DCTtab5[(code>>1)-16];
    else if (code>=16)
      tab = &DCTtab6[code-16];
    else
    {
      printf("invalid Huffman code in Decode_MPEG1_Intra_Block()\n");
      break;
    }

    gst_flushbitsn(gb, tab->len);

    if (tab->run==64)  {/* end_of_block */
      /* not a sparse matrice ! */
      ld->sparse[comp] = (i==0);
      return;
    }

    if (tab->run==65) /* escape */
    {
      i+= gst_getbits6(gb);

      val = gst_getbits8(gb);
      if (val==0)
        val = gst_getbits8(gb);
      else if (val==128)
        val = gst_getbits8(gb) - 256;
      else if (val>128)
        val -= 256;

      if((sign = (val<0)))
        val = -val;
    }
    else
    {
      i+= tab->run;
      val = tab->level;
      sign = gst_getbits1(gb);
    }

    if (i>=64)
    {
      fprintf(stderr,"DCT coeff index (i) out of bounds (intra)\n");
      break;
    }

    j = vid_stream->scan[ZIG_ZAG][i];
    val = (val*ld->quantizer_scale*ld->intra_quantizer_matrix[j]) >> 3;
    /* mismatch control ('oddification') */
    if (val!=0) /* should always be true, but it's not guaranteed */
      val = (val-1) | 1; /* equivalent to: if ((val&1)==0) val = val - 1; */
    /* saturation */
    if (!sign)
      bp[j] = (val>2047) ?  2047 :  val; /* positive */
    else
      bp[j] = (val>2048) ? -2048 : -val; /* negative */
    
  }
  vid_stream->Fault_Flag = 1;
}


/* decode one non-intra coded MPEG-1 block */

void Decode_MPEG1_Non_Intra_Block(vid_stream, comp)
mpeg2play_vid_stream *vid_stream;
int comp;
{
  int val, i, j, sign;
  unsigned int code;
  DCTtab *tab;
  short *bp;
  gst_getbits_t *gb = &vid_stream->ld->gb;
  mpeg2play_layer_data *ld = vid_stream->ld;

  bp = ld->block[comp];

  /* decode AC coefficients */
  for (i=0; ; i++)
  {
    code = gst_showbits16(gb);
    if (code>=16384)
    {
      if (i==0)
        tab = &DCTtabfirst[(code>>12)-4];
      else
        tab = &DCTtabnext[(code>>12)-4];
    }
    else if (code>=1024)
      tab = &DCTtab0[(code>>8)-4];
    else if (code>=512)
      tab = &DCTtab1[(code>>6)-8];
    else if (code>=256)
      tab = &DCTtab2[(code>>4)-16];
    else if (code>=128)
      tab = &DCTtab3[(code>>3)-16];
    else if (code>=64)
      tab = &DCTtab4[(code>>2)-16];
    else if (code>=32)
      tab = &DCTtab5[(code>>1)-16];
    else if (code>=16)
      tab = &DCTtab6[code-16];
    else
    {
      printf("invalid Huffman code in Decode_MPEG1_Non_Intra_Block()\n");
      break;
    }

    gst_flushbitsn(gb, tab->len);

    if (tab->run==64)  {/* end_of_block */
      ld->sparse[comp] = (i==0);
      return;
    }

    if (tab->run==65) /* escape */
    {
      i+= gst_getbits6(gb);

      val = gst_getbits8(gb);
      if (val==0)
        val = gst_getbits8(gb);
      else if (val==128)
        val = gst_getbits8(gb) - 256;
      else if (val>128)
        val -= 256;

      if((sign = (val<0)))
        val = -val;
    }
    else
    {
      i+= tab->run;
      val = tab->level;
      sign = gst_getbits1(gb);
    }

    if (i>=64)
    {
      fprintf(stderr,"DCT coeff index (i) out of bounds (inter)\n");
      break;
    }

    j = vid_stream->scan[ZIG_ZAG][i];
    val = (((val<<1)+1)*ld->quantizer_scale*ld->non_intra_quantizer_matrix[j]) >> 4;

    /* mismatch control ('oddification') */
    if (val!=0) /* should always be true, but it's not guaranteed */
      val = (val-1) | 1; /* equivalent to: if ((val&1)==0) val = val - 1; */
    /* saturation */
    if (!sign)
      bp[j] = (val>2047) ?  2047 :  val; /* positive */
    else
      bp[j] = (val>2048) ? -2048 : -val; /* negative */

  }
  vid_stream->Fault_Flag = 1;
}


/* decode one intra coded MPEG-2 block */
void Decode_MPEG2_Intra_Block(vid_stream, comp,dc_dct_pred)
mpeg2play_vid_stream *vid_stream;
int comp;
int dc_dct_pred[];
{
  int val, i, j, sign, nc, cc, run;
  unsigned int code;
  DCTtab *tab;
  short *bp;
  int *qmat;
  struct layer_data *ld1;
  gst_getbits_t *gb = &vid_stream->ld->gb;

  /* with data partitioning, data always goes to base layer */
#ifdef ENHANCE
  ld1 = (vid_stream->ld->scalable_mode==SC_DP) ? &vid_stream->base : vid_stream->ld;
  bp = ld1->block[comp];

  if (vid_stream->base.scalable_mode==SC_DP) {
    if (vid_stream->base.priority_breakpoint<64) {
      vid_stream->ld = &vid_stream->enhan;
    }
    else {
      vid_stream->ld = &vid_stream->base;
    }
  }
#else
  ld1 = &vid_stream->base;
  bp = ld1->block[comp];
#endif

  cc = (comp<4) ? 0 : (comp&1)+1;

  qmat = (comp<4 || vid_stream->sequence.chroma_format==CHROMA420)
         ? ld1->intra_quantizer_matrix
         : ld1->chroma_intra_quantizer_matrix;

  /* ISO/IEC 13818-2 section 7.2.1: decode DC coefficients */
  if (cc==0)
    val = (dc_dct_pred[0]+= Get_Luma_DC_dct_diff(gb));
  else if (cc==1)
    val = (dc_dct_pred[1]+= Get_Chroma_DC_dct_diff(gb));
  else
    val = (dc_dct_pred[2]+= Get_Chroma_DC_dct_diff(gb));

  if (vid_stream->Fault_Flag) return;
  bp[0] = val << (3-vid_stream->picture.intra_dc_precision);

  nc=0;

#ifdef TRACE
  printf("DCT(%d)i:",comp);
#endif /* TRACE */

  /* decode AC coefficients */
  for (i=1; ; i++)
  {
    code = gst_showbits16(gb);
    if (code>=16384 && !vid_stream->picture.intra_vlc_format)
      tab = &DCTtabnext[(code>>12)-4];
    else if (code>=1024)
    {
      if (vid_stream->picture.intra_vlc_format)
        tab = &DCTtab0a[(code>>8)-4];
      else
        tab = &DCTtab0[(code>>8)-4];
    }
    else if (code>=512)
    {
      if (vid_stream->picture.intra_vlc_format)
        tab = &DCTtab1a[(code>>6)-8];
      else
        tab = &DCTtab1[(code>>6)-8];
    }
    else if (code>=256)
      tab = &DCTtab2[(code>>4)-16];
    else if (code>=128)
      tab = &DCTtab3[(code>>3)-16];
    else if (code>=64)
      tab = &DCTtab4[(code>>2)-16];
    else if (code>=32)
      tab = &DCTtab5[(code>>1)-16];
    else if (code>=16)
      tab = &DCTtab6[code-16];
    else
    {
      printf("invalid Huffman code in Decode_MPEG2_Intra_Block()\n");
      break;
    }

    gst_flushbitsn(gb,tab->len);

#ifdef TRACE
    printf(" (");
#endif /* TRACE */

    if (tab->run==64) /* end_of_block */
    {
#ifdef TRACE
      printf("): EOB\n");
#endif /* TRACE */
      vid_stream->ld->sparse[comp] = (i==0);
      return;
    }

    if (tab->run==65) /* escape */
    {
#ifdef TRACE
      putchar(' ');
#endif /* TRACE */

      i+= run = gst_getbits6(gb);

#ifdef TRACE
      putchar(' ');
#endif /* TRACE */

      val = gst_getbits12(gb);
      if ((val&2047)==0)
      {
        printf("invalid escape in Decode_MPEG2_Intra_Block()\n");
        break;
      }
      if((sign = (val>=2048)))
        val = 4096 - val;
    }
    else
    {
      i+= run = tab->run;
      val = tab->level;
      sign = gst_getbits1(gb);

#ifdef TRACE
      printf("%d",sign);
#endif /* TRACE */
    }

    if (i>=64)
    {
      fprintf(stderr,"DCT coeff index (i) out of bounds (intra2)\n");
      break;
    }

#ifdef TRACE
    printf("): %d/%d",run,sign ? -val : val);
#endif /* TRACE */

    j = vid_stream->scan[ld1->alternate_scan][i];
    val = (val * ld1->quantizer_scale * qmat[j]) >> 4;
    bp[j] = sign ? -val : val;
    nc++;

#ifdef ENHANCE
    if (vid_stream->base.scalable_mode==SC_DP && nc==vid_stream->base.priority_breakpoint-63)
      vid_stream->ld = &vid_stream->enhan;
#endif
  }
  vid_stream->Fault_Flag = 1;
}


/* decode one non-intra coded MPEG-2 block */

void Decode_MPEG2_Non_Intra_Block(vid_stream, comp)
mpeg2play_vid_stream *vid_stream;
int comp;
{
  int val, i, j, sign, nc, run;
  unsigned int code;
  DCTtab *tab;
  short *bp;
  int *qmat;
  struct layer_data *ld1;
  gst_getbits_t *gb = &vid_stream->ld->gb;

#ifdef ENHANCE
  /* with data partitioning, data always goes to base layer */
  ld1 = (vid_stream->ld->scalable_mode==SC_DP) ? &vid_stream->base : vid_stream->ld;
  bp = ld1->block[comp];

  if (vid_stream->base.scalable_mode==SC_DP) {
    if (vid_stream->base.priority_breakpoint<64) {
      vid_stream->ld = &vid_stream->enhan;
    }
    else {
      vid_stream->ld = &vid_stream->base;
    }
  }
#else
  ld1 = &vid_stream->base;
  bp = ld1->block[comp];
#endif

  qmat = (comp<4 || vid_stream->sequence.chroma_format==CHROMA420)
         ? ld1->non_intra_quantizer_matrix
         : ld1->chroma_non_intra_quantizer_matrix;

  nc = 0;

#ifdef TRACE
  printf("DCT(%d)n:",comp);
#endif /* TRACE */

  /* decode AC coefficients */
  for (i=0; ; i++)
  {
    code = gst_showbits16(gb);
    if (code>=16384)
    {
      if (i==0)
        tab = &DCTtabfirst[(code>>12)-4];
      else
        tab = &DCTtabnext[(code>>12)-4];
    }
    else if (code>=1024)
      tab = &DCTtab0[(code>>8)-4];
    else if (code>=512)
      tab = &DCTtab1[(code>>6)-8];
    else if (code>=256)
      tab = &DCTtab2[(code>>4)-16];
    else if (code>=128)
      tab = &DCTtab3[(code>>3)-16];
    else if (code>=64)
      tab = &DCTtab4[(code>>2)-16];
    else if (code>=32)
      tab = &DCTtab5[(code>>1)-16];
    else if (code>=16)
      tab = &DCTtab6[code-16];
    else
    {
      printf("invalid Huffman code in Decode_MPEG2_Non_Intra_Block()\n");
      break;
    }

    gst_flushbitsn(gb, tab->len);

#ifdef TRACE
    printf(" (");
#endif /* TRACE */

    if (tab->run==64) /* end_of_block */
    {
#ifdef TRACE
      printf("): EOB\n");
#endif /* TRACE */
      vid_stream->ld->sparse[comp] = (i==0);
      return;
    }

    if (tab->run==65) /* escape */
    {
#ifdef TRACE
      putchar(' ');
#endif /* TRACE */

      i+= run = gst_getbits6(gb);

#ifdef TRACE
      putchar(' ');
#endif /* TRACE */

      val = gst_getbits12(gb);
      if ((val&2047)==0)
      {
        printf("invalid escape in Decode_MPEG2_Intra_Block()\n");
        break;
      }
      if((sign = (val>=2048)))
        val = 4096 - val;
    }
    else
    {
      i+= run = tab->run;
      val = tab->level;
      sign = gst_getbits1(gb);

#ifdef TRACE
      printf("%d",sign);
#endif /* TRACE */
    }

    if (i>=64)
    {
      fprintf(stderr,"DCT coeff index (i) out of bounds (inter2)\n");
      break;
    }

#ifdef TRACE
    printf("): %d/%d",run,sign?-val:val);
#endif /* TRACE */

    j = vid_stream->scan[ld1->alternate_scan][i];
    val = (((val<<1)+1) * ld1->quantizer_scale * qmat[j]) >> 5;
    bp[j] = sign ? -val : val;
    nc++;

#ifdef ENHANCE
    if (vid_stream->base.scalable_mode==SC_DP && nc==vid_stream->base.priority_breakpoint-63)
      vid_stream->ld = &vid_stream->enhan;
#endif
  }
  vid_stream->Fault_Flag = 1;
}
