/* getpic.c, picture decoding                                               */

/* Copyright (C) 1996, MPEG Software Simulation Group. All Rights Reserved. */

/*
 * Disclaimer of Warranty
 *
 * These software programs are available to the user without any license fee or
 * royalty on an "as is" basis.  The MPEG Software Simulation Group disclaims
 * any and all warranties, whether express, implied, or statuary, including any
 * implied warranties or merchantability or of fitness for a particular
 * purpose.  In no event shall the copyright-holder be liable for any
 * incidental, punitive, or consequential damages of any kind whatsoever
 * arising from the use of these programs.
 *
 * This disclaimer of warranty extends to the user of these programs and user's
 * customers, employees, agents, transferees, successors, and assigns.
 *
 * The MPEG Software Simulation Group does not represent or warrant that the
 * programs furnished hereunder are free of infringement of any third-party
 * patents.
 *
 * Commercial implementations of MPEG-1 and MPEG-2 video, including shareware,
 * are subject to royalty fees to patent holders.  Many of these patents are
 * general enough such that they are unavoidable regardless of implementation
 * design.
 *
 */

#include <stdio.h>

#include "mpeg2dec.h"
#ifdef HAVE_LIBMMX
#include "mmx.h"
#endif /* HAVE_LIBMMX */

//#undef HAVE_LIBMMX
//#define TRACE

#ifdef HAVE_LIBMMX
G_GNUC_UNUSED static unsigned long long MMX_128 = 0x80008000800080LL;
#endif


/* private prototypes*/
static void picture_data _ANSI_ARGS_((mpeg2play_vid_stream *vid_stream, int framenum));
static void macroblock_modes _ANSI_ARGS_((mpeg2play_vid_stream *vid_stream, int *pmacroblock_type, int *pstwtype,
  int *pstwclass, int *pmotion_type, int *pmotion_vector_count, int *pmv_format, int *pdmv,
  int *pmvscale, int *pdct_type));
static void Clear_Block _ANSI_ARGS_((mpeg2play_vid_stream *vid_stream, int comp));
#ifdef ENHANCE
static void Sum_Block _ANSI_ARGS_((mpeg2play_vid_stream *vid_stream, int comp));
static void Decode_SNR_Macroblock _ANSI_ARGS_((mpeg2play_vid_stream *vid_stream, int *SNRMBA, int *SNRMBAinc, 
  int MBA, int MBAmax, int *dct_type));
#endif
#ifndef HAVE_LIBMMX
static void Saturate _ANSI_ARGS_((mpeg2play_vid_stream *vid_stream, short *bp));
#endif
static void Add_Block _ANSI_ARGS_((mpeg2play_vid_stream *vid_stream, int comp, int bx, int by,
  int dct_type, int addflag));
static void Update_Picture_Buffers _ANSI_ARGS_((mpeg2play_vid_stream *vid_stream));
static void frame_reorder _ANSI_ARGS_((mpeg2play_vid_stream *vid_stream, int bitstream_framenum, 
  int sequence_framenum));

static void motion_compensation _ANSI_ARGS_((mpeg2play_vid_stream *vid_stream, int MBA, int macroblock_type, 
 int motion_type, int PMV[2][2][2], int motion_vertical_field_select[2][2], 
 int dmvector[2], int stwtype, int dct_type));

static void skipped_macroblock _ANSI_ARGS_((mpeg2play_vid_stream *vid_stream, int dc_dct_pred[3], 
  int PMV[2][2][2], int *motion_type, int motion_vertical_field_select[2][2],
  int *stwtype, int *macroblock_type));

static int start_of_slice _ANSI_ARGS_ ((mpeg2play_vid_stream *vid_stream, int MBAmax, int *MBA,
  int *MBAinc, int dc_dct_pred[3], int PMV[2][2][2]));

static int decode_macroblock _ANSI_ARGS_((mpeg2play_vid_stream *vid_stream, int *macroblock_type, 
  int *stwtype, int *stwclass, int *motion_type, int *dct_type,
  int PMV[2][2][2], int dc_dct_pred[3], 
  int motion_vertical_field_select[2][2], int dmvector[2]));

/* decode one frame or field picture */
void Decode_Picture(vid_stream, bitstream_framenum, sequence_framenum)
mpeg2play_vid_stream *vid_stream;
int bitstream_framenum, sequence_framenum;
{

  if (vid_stream->picture.picture_structure==FRAME_PICTURE && vid_stream->Second_Field)
  {
    /* recover from illegal number of field pictures */
    printf("odd number of field pictures\n");
    vid_stream->Second_Field = 0;
  }

  /* IMPLEMENTATION: update picture buffer pointers */
  Update_Picture_Buffers(vid_stream);

#ifdef VERIFY 
  Check_Headers(bitstream_framenum, sequence_framenum);
#endif /* VERIFY */

  /* ISO/IEC 13818-4 section 2.4.5.4 "frame buffer intercept method" */
  /* (section number based on November 1995 (Dallas) draft of the 
      conformance document) */
  //if(Ersatz_Flag)
   // Substitute_Frame_Buffer(vid_stream, bitstream_framenum, sequence_framenum);

  /* form spatial scalable picture */
 
  /* form spatial scalable picture */
  /* ISO/IEC 13818-2 section 7.7: Spatial scalability */
  if (vid_stream->base.pict_scal && !vid_stream->Second_Field) 
  {
    Spatial_Prediction(vid_stream);
  }

  /* decode picture data ISO/IEC 13818-2 section 6.2.3.7 */
  //picture_data(vid_stream, bitstream_framenum);

  /* write or display current or previously decoded reference frame */
  /* ISO/IEC 13818-2 section 6.1.1.11: Frame reordering */
  /*
  frame_reorder(vid_stream, bitstream_framenum, sequence_framenum);

  if (vid_stream->picture.picture_structure!=FRAME_PICTURE)
    vid_stream->Second_Field = !vid_stream->Second_Field;
    */
#ifdef HAVE_LIBMMX
  emms();
#endif
}


/* decode all macroblocks of the current picture */
/* stages described in ISO/IEC 13818-2 section 7 */
static void picture_data(vid_stream, framenum)
mpeg2play_vid_stream *vid_stream;
int framenum;
{
  int MBAmax;
  int ret;

  /* number of macroblocks per picture */
  MBAmax = vid_stream->mb_width*vid_stream->mb_height;

  if (vid_stream->picture.picture_structure!=FRAME_PICTURE)
    MBAmax>>=1; /* field picture has half as mnay macroblocks as frame */

  for(;;)
  {
    if((ret=slice(vid_stream, framenum, MBAmax))<0)
      break;
  }
#ifdef HAVE_LIBMMX
  emms();
#endif

}

void Decode_Picture_End(vid_stream, bitstream_framenum, sequence_framenum)
mpeg2play_vid_stream *vid_stream;
int bitstream_framenum, sequence_framenum;
{
  frame_reorder(vid_stream, bitstream_framenum, sequence_framenum);

  if (vid_stream->picture.picture_structure!=FRAME_PICTURE)
    vid_stream->Second_Field = !vid_stream->Second_Field;
#ifdef HAVE_LIBMMX
  emms();
#endif
}


/* decode all macroblocks of the current picture */
/* ISO/IEC 13818-2 section 6.3.16 */
int slice(vid_stream, framenum, MBAmax)
mpeg2play_vid_stream *vid_stream;
int framenum, MBAmax;
{
  int MBA; 
  int MBAinc, macroblock_type, motion_type, dct_type;
  int dc_dct_pred[3];
  int PMV[2][2][2], motion_vertical_field_select[2][2];
  int dmvector[2];
  int stwtype, stwclass;
#ifdef ENHANCE
  int SNRMBA, SNRMBAinc;
#endif
  int ret;
  gst_getbits_t *gb = &vid_stream->ld->gb;

  MBA = 0; /* macroblock address */
  MBAinc = 0;

  if((ret=start_of_slice(vid_stream, MBAmax, &MBA, &MBAinc, dc_dct_pred, PMV))!=1)
    return(ret);

#ifdef ENHANCE
  if (vid_stream->Two_Streams && vid_stream->enhan.scalable_mode==SC_SNR)
  {
    SNRMBA=0;
    SNRMBAinc=0;
  }
#endif

  vid_stream->Fault_Flag=0;

  for (;;)
  {

    /* this is how we properly exit out of picture */
    if (MBA>=MBAmax)
      return(-1); /* all macroblocks decoded */

    vid_stream->ld = &vid_stream->base;

    if (MBAinc==0)
    {
#ifdef ENHANCE
      if (vid_stream->base.scalable_mode==SC_DP && vid_stream->base.priority_breakpoint==1)
          vid_stream->ld = &vid_stream->enhan;
#endif

      //printf("bytes left %d %ld bits %ld\n", gst_getbits_bytesleft(gb), gst_getbits_bitoffset(gb), gst_getbits_bitsleft(gb));
      //if (gst_getbits_bitsleft(gb) < 23 || !gst_showbits23(gb) || vid_stream->Fault_Flag) /* next_start_code or fault */
      if (!gst_showbits23(gb) || vid_stream->Fault_Flag) /* next_start_code or fault */
      {
resync: /* if vid_stream->Fault_Flag: resynchronize to next next_start_code */
        vid_stream->Fault_Flag = 0;
        return(0);     /* trigger: go to next slice */
      }
      else /* neither next_start_code nor vid_stream->Fault_Flag */
      {
#ifdef ENHANCE
        if (vid_stream->base.scalable_mode==SC_DP && vid_stream->base.priority_breakpoint==1)
          vid_stream->ld = &vid_stream->enhan;
#endif

        //printf("bits left %ld\n", gst_getbits_bitsleft(gb));
        /* decode macroblock address increment */
        MBAinc = Get_macroblock_address_increment(vid_stream, gb);
        //printf("bits left %ld\n", gst_getbits_bitsleft(gb));

        if (vid_stream->Fault_Flag) {
          //printf("bytes left %d\n", gst_getbits_bytesleft(gb));
          goto resync;
	}
      }
    }

    if (MBA>=MBAmax)
    {
      /* MBAinc points beyond picture dimensions */
      printf("Too many macroblocks in picture\n");
      return(-1);
    }

    if (MBAinc==1) /* not skipped */
    {
      ret = decode_macroblock(vid_stream, &macroblock_type, &stwtype, &stwclass,
              &motion_type, &dct_type, PMV, dc_dct_pred, 
              motion_vertical_field_select, dmvector);

      if(ret==-1)
        return(-1);
   
      if(ret==0)
        goto resync;

    }
    else /* MBAinc!=1: skipped macroblock */
    {      
      /* ISO/IEC 13818-2 section 7.6.6 */
      skipped_macroblock(vid_stream, dc_dct_pred, PMV, &motion_type, 
        motion_vertical_field_select, &stwtype, &macroblock_type);
    }

#ifdef ENHANCE
    /* SCALABILITY: SNR */
    /* ISO/IEC 13818-2 section 7.8 */
    /* NOTE: we currently ignore faults encountered in this routine */
    if (vid_stream->Two_Streams && vid_stream->enhan.scalable_mode==SC_SNR)
      Decode_SNR_Macroblock(vid_stream, &SNRMBA, &SNRMBAinc, MBA, MBAmax, &dct_type);
#endif

    /* ISO/IEC 13818-2 section 7.6 */
    motion_compensation(vid_stream, MBA, macroblock_type, motion_type, PMV, 
      motion_vertical_field_select, dmvector, stwtype, dct_type);

    /* advance to next macroblock */
    MBA++;
    MBAinc--;
 
#ifdef ENHANCE
    /* SCALABILITY: SNR */
    if (vid_stream->Two_Streams && vid_stream->enhan.scalable_mode==SC_SNR)
    {
      SNRMBA++;
      SNRMBAinc--;
    }
#endif

    if (MBA>=MBAmax)
      return(-1); /* all macroblocks decoded */
  }
}

 
/* ISO/IEC 13818-2 section 6.3.17.1: Macroblock modes */
static void macroblock_modes(vid_stream, pmacroblock_type,pstwtype,pstwclass,
  pmotion_type,pmotion_vector_count,pmv_format,pdmv,pmvscale,pdct_type)
mpeg2play_vid_stream *vid_stream;
int *pmacroblock_type, *pstwtype, *pstwclass;
int *pmotion_type, *pmotion_vector_count, *pmv_format, *pdmv, *pmvscale;
int *pdct_type;
{
  int macroblock_type;
  int stwtype, stwcode, stwclass;
  int motion_type = 0;
  int motion_vector_count, mv_format, dmv, mvscale;
  int dct_type;
  static unsigned char stwc_table[3][4]
    = { {6,3,7,4}, {2,1,5,4}, {2,5,7,4} };
  static unsigned char stwclass_table[9]
    = {0, 1, 2, 1, 1, 2, 3, 3, 4};
  gst_getbits_t *gb = &vid_stream->ld->gb;

  /* get macroblock_type */
  macroblock_type = Get_macroblock_type(vid_stream);

  //if (vid_stream->Fault_Flag) return;

  /* get spatial_temporal_weight_code */
  if (macroblock_type & MB_WEIGHT)
  {
    if (vid_stream->picture.spatial_temporal_weight_code_table_index==0)
      stwtype = 4;
    else
    {
      stwcode = gst_getbits2(gb);
#ifdef TRACE
      printf("spatial_temporal_weight_code (");
      //Print_Bits(stwcode,2,2);
      printf("): %d\n",stwcode);
#endif /* TRACE */
      stwtype = stwc_table[vid_stream->picture.spatial_temporal_weight_code_table_index-1][stwcode];
    }
  }
  else
    stwtype = (macroblock_type & MB_CLASS4) ? 8 : 0;

  /* SCALABILITY: derive spatial_temporal_weight_class (Table 7-18) */
  stwclass = stwclass_table[stwtype];

  /* get frame/field motion type */
  if (macroblock_type & (MACROBLOCK_MOTION_FORWARD|MACROBLOCK_MOTION_BACKWARD))
  {
    if (vid_stream->picture.picture_structure==FRAME_PICTURE) /* frame_motion_type */
    {
      motion_type = vid_stream->picture.frame_pred_frame_dct ? MC_FRAME : gst_getbits2(gb);
#ifdef TRACE
      if (!vid_stream->picture.frame_pred_frame_dct)
      {
        printf("frame_motion_type (");
        //Print_Bits(motion_type,2,2);
        printf("): %s\n",motion_type==MC_FIELD?"Field":
                         motion_type==MC_FRAME?"Frame":
                         motion_type==MC_DMV?"Dual_Prime":"Invalid");
      }
#endif /* TRACE */
    }
    else /* field_motion_type */
    {
      motion_type = gst_getbits2(gb);
#ifdef TRACE
      printf("field_motion_type (");
      //Print_Bits(motion_type,2,2);
      printf("): %s\n",motion_type==MC_FIELD?"Field":
                       motion_type==MC_16X8?"16x8 MC":
                       motion_type==MC_DMV?"Dual_Prime":"Invalid");
     
#endif /* TRACE */
    }
  }
  else if ((macroblock_type & MACROBLOCK_INTRA) && vid_stream->picture.concealment_motion_vectors)
  {
    /* concealment motion vectors */
    motion_type = (vid_stream->picture.picture_structure==FRAME_PICTURE) ? MC_FRAME : MC_FIELD;
  }
#if 0
  else
  {
    printf("maroblock_modes(): unknown macroblock type\n");
    motion_type = -1;
  }
#endif

  /* derive motion_vector_count, mv_format and dmv, (table 6-17, 6-18) */
  if (vid_stream->picture.picture_structure==FRAME_PICTURE)
  {
    motion_vector_count = (motion_type==MC_FIELD && stwclass<2) ? 2 : 1;
    mv_format = (motion_type==MC_FRAME) ? MV_FRAME : MV_FIELD;
  }
  else
  {
    motion_vector_count = (motion_type==MC_16X8) ? 2 : 1;
    mv_format = MV_FIELD;
  }

  dmv = (motion_type==MC_DMV); /* dual prime */

  /* field mv predictions in frame pictures have to be scaled
   * ISO/IEC 13818-2 section 7.6.3.1 Decoding the motion vectors
   * IMPLEMENTATION: mvscale is derived for later use in motion_vectors()
   * it displaces the stage:
   *
   *    if((mv_format=="field")&&(t==1)&&(picture_structure=="Frame picture"))
   *      prediction = PMV[r][s][t] DIV 2;
   */

  mvscale = ((mv_format==MV_FIELD) && (vid_stream->picture.picture_structure==FRAME_PICTURE));

  /* get dct_type (frame DCT / field DCT) */
  dct_type = (vid_stream->picture.picture_structure==FRAME_PICTURE)
             && (!vid_stream->picture.frame_pred_frame_dct)
             && (macroblock_type & (MACROBLOCK_PATTERN|MACROBLOCK_INTRA))
             ? gst_getbits1(gb)
             : 0;

#ifdef TRACE
  if ((vid_stream->picture.picture_structure==FRAME_PICTURE)
             && (!vid_stream->picture.frame_pred_frame_dct)
             && (macroblock_type & (MACROBLOCK_PATTERN|MACROBLOCK_INTRA)))
    printf("dct_type (%d): %s\n",dct_type,dct_type?"Field":"Frame");
#endif /* TRACE */

  /* return values */
  *pmacroblock_type = macroblock_type;
  *pstwtype = stwtype;
  *pstwclass = stwclass;
  *pmotion_type = motion_type;
  *pmotion_vector_count = motion_vector_count;
  *pmv_format = mv_format;
  *pdmv = dmv;
  *pmvscale = mvscale;
  *pdct_type = dct_type;
}


/* move/add 8x8-Block from block[comp] to backward_reference_frame */
/* copy reconstructed 8x8 block from block[comp] to current_frame[]
 * ISO/IEC 13818-2 section 7.6.8: Adding prediction and coefficient data
 * This stage also embodies some of the operations implied by:
 *   - ISO/IEC 13818-2 section 7.6.7: Combining predictions
 *   - ISO/IEC 13818-2 section 6.1.3: Macroblock
*/
static void Add_Block(vid_stream, comp,bx,by,dct_type,addflag)
mpeg2play_vid_stream *vid_stream;
int comp,bx,by,dct_type,addflag;
{
  int cc, iincr;
#ifndef HAVE_LIBMMX
  int i;
#endif
  unsigned char *rfp;
  short *bp;
  int sparse = vid_stream->ld->sparse[comp];

  
  /* derive color component index */
  /* equivalent to ISO/IEC 13818-2 Table 7-1 */
  cc = (comp<4) ? 0 : (comp&1)+1; /* color component index */

  if (cc==0)
  {
    /* luminance */

    if (vid_stream->picture.picture_structure==FRAME_PICTURE)
      if (dct_type)
      {
        /* field DCT coding */
        rfp = vid_stream->current_frame[0]
              + vid_stream->Coded_Picture_Width*(by+((comp&2)>>1)) + bx + ((comp&1)<<3);
        iincr = (vid_stream->Coded_Picture_Width<<1);
      }
      else
      {
        /* frame DCT coding */
        rfp = vid_stream->current_frame[0]
              + vid_stream->Coded_Picture_Width*(by+((comp&2)<<2)) + bx + ((comp&1)<<3);
        iincr = vid_stream->Coded_Picture_Width;
      }
    else
    {
      /* field picture */
      rfp = vid_stream->current_frame[0]
            + (vid_stream->Coded_Picture_Width<<1)*(by+((comp&2)<<2)) + bx + ((comp&1)<<3);
      iincr = (vid_stream->Coded_Picture_Width<<1);
    }
  }
  else
  {
    /* chrominance */

    /* scale coordinates */
    if (vid_stream->sequence.chroma_format!=CHROMA444)
      bx >>= 1;
    if (vid_stream->sequence.chroma_format==CHROMA420)
      by >>= 1;
    if (vid_stream->picture.picture_structure==FRAME_PICTURE)
    {
      if (dct_type && (vid_stream->sequence.chroma_format!=CHROMA420))
      {
        /* field DCT coding */
        rfp = vid_stream->current_frame[cc]
              + vid_stream->Chroma_Width*(by+((comp&2)>>1)) + bx + (comp&8);
        iincr = (vid_stream->Chroma_Width<<1);
      }
      else
      {
        /* frame DCT coding */
        rfp = vid_stream->current_frame[cc]
              + vid_stream->Chroma_Width*(by+((comp&2)<<2)) + bx + (comp&8);
        iincr = vid_stream->Chroma_Width;
      }
    }
    else
    {
      /* field picture */
      rfp = vid_stream->current_frame[cc]
            + (vid_stream->Chroma_Width<<1)*(by+((comp&2)<<2)) + bx + (comp&8);
      iincr = (vid_stream->Chroma_Width<<1);
    }
  }

  bp = vid_stream->ld->block[comp];

  if (addflag)
  {
#ifdef HAVE_LIBMMX
    if (sparse) {
      __asm__ __volatile__(
           "movq       (%2),  %%mm6\n"  /* 4 blockvals */
           "pxor       %%mm4, %%mm4\n"
           "punpcklwd  %%mm6, %%mm6\n"
           "punpcklwd  %%mm6, %%mm6\n"
           ".align 8\n"
           "1:"
              "movq       (%1),  %%mm0\n"     /* 8 rindex1 */
              "movq       %%mm0, %%mm2\n"
              "punpcklbw  %%mm4, %%mm0\n"
              "punpckhbw  %%mm4, %%mm2\n"
              "paddw      %%mm6, %%mm0\n"
              "paddw      %%mm6, %%mm2\n"

              "packuswb   %%mm2, %%mm0\n"
              "movq       %%mm0, (%1)\n"

              "leal       (%1, %3), %1\n"
           "loop       1b\n"
           :              /* scr   dest */
           : "c" (8),"r" (rfp), "r" (bp), "r" (iincr)
      );
    }
    else {
      __asm__ __volatile__(
         "pxor    %%mm4,%%mm4\n"

         ".align 8\n"
         "1:"
           "movq       (%2), %%mm0\n"   /* 8 rfp 0 1 2 3 4 5 6 7*/
           "movq       (%1), %%mm6\n"   /* 4 blockvals 0 1 2 3 */

           "movq       %%mm0, %%mm2\n"
           "movq       8(%1), %%mm5\n"  /* 4 blockvals 0 1 2 3 */
           "punpcklbw  %%mm4, %%mm0\n"  /* 0 2 4 6 */
           "punpckhbw  %%mm4, %%mm2\n"  /* 1 3 5 7 */

           "paddw      %%mm6, %%mm0\n"
	   "paddw      %%mm5, %%mm2\n"
	   "packuswb   %%mm2, %%mm0\n"

	   "addl       $16, %1\n"
	   "movq       %%mm0, (%2)\n"

	   "leal       (%2,%3), %2\n"
	 "loop       1b\n"
	 :              /* scr   dest */
         : "c" (8),"r" (bp), "r" (rfp), "r" (iincr)
      );
    }
#else
    for (i=0; i<8; i++){
      rfp[0] = Clip[bp[0] + rfp[0]];
      rfp[1] = Clip[bp[1] + rfp[1]];
      rfp[2] = Clip[bp[2] + rfp[2]];
      rfp[3] = Clip[bp[3] + rfp[3]];
      rfp[4] = Clip[bp[4] + rfp[4]];
      rfp[5] = Clip[bp[5] + rfp[5]];
      rfp[6] = Clip[bp[6] + rfp[6]];
      rfp[7] = Clip[bp[7] + rfp[7]];
      rfp+= iincr;
      bp += 8;
    }
#endif
  }
  else
  {
#ifdef HAVE_LIBMMX
	  
	  /*
    if (sparse) {
      __asm__ __volatile__(
           "movd       (%2),           %%mm0\n"    // " 0 0 0  v1"
           "punpcklwd  %%mm0,          %%mm0\n"    // " 0 0 v1 v1"
           "punpcklwd  %%mm0,          %%mm0\n"
           "paddw      MMX_128,        %%mm0\n"
           "packuswb   %%mm0,          %%mm0\n"
           "leal       (%0,%1,2),      %%eax\n"

           "movq        %%mm0,         (%0, %1)\n"
           "movq        %%mm0,         (%%eax)\n"
           "leal        (%%eax,%1,2),  %0\n"
           "movq        %%mm0,         (%%eax, %1)\n"

           "movq        %%mm0,         (%0)\n"
           "leal        (%0,%1,2),     %%eax\n"
           "movq        %%mm0,         (%0, %1)\n"

           "movq        %%mm0,         (%%eax)\n"
           "movq        %%mm0,         (%%eax, %1)\n"
           :
           : "D" (rfp), "c" (iincr), "b" (bp)
           : "eax");
    }
    else {
    */
    
      __asm__ __volatile__(
             "movq        MMX_128,%%mm4\n"
             ".align 8\n"
             "1:"
               "movq      (%1),   %%mm0\n"
               "movq      8(%1),  %%mm1\n"
               "paddw     %%mm4,  %%mm0\n"

               "movq      16(%1), %%mm2\n"
               "paddw     %%mm4,  %%mm1\n"

               "movq      24(%1), %%mm3\n"
               "paddw     %%mm4,  %%mm2\n"

               "packuswb  %%mm1,  %%mm0\n"
               "paddw     %%mm4,  %%mm3\n"

               "addl $32, %1\n"
               "packuswb  %%mm3,  %%mm2\n"

               "movq   %%mm0, (%2)\n"

               "movq   %%mm2, (%2,%3)\n"

               "leal       (%2,%3,2), %2\n"
             "loop       1b\n"
             :
             : "c" (4), "r" (bp), "r" (rfp), "r" (iincr)
         );
    //}
#else
    for (i=0; i<8; i++){
      rfp[0] = Clip[bp[0]+128];
      rfp[1] = Clip[bp[1]+128];
      rfp[2] = Clip[bp[2]+128];
      rfp[3] = Clip[bp[3]+128];
      rfp[4] = Clip[bp[4]+128];
      rfp[5] = Clip[bp[5]+128];
      rfp[6] = Clip[bp[6]+128];
      rfp[7] = Clip[bp[7]+128];
      rfp+= iincr;
      bp += 8;
    }
#endif
  }
}


#ifdef ENHANCE
/* ISO/IEC 13818-2 section 7.8 */
static void Decode_SNR_Macroblock(vid_stream, SNRMBA, SNRMBAinc, MBA, MBAmax, dct_type)
mpeg2play_vid_stream *vid_stream;
  int *SNRMBA, *SNRMBAinc;
  int MBA, MBAmax;
  int *dct_type;
{
  int SNRmacroblock_type, SNRcoded_block_pattern, SNRdct_type, dummy; 
  int slice_vert_pos_ext, quantizer_scale_code, comp, code;
  gst_getbits_t *gb = &vid_stream->ld->gb;

  vid_stream->ld = &vid_stream->enhan;

  if (*SNRMBAinc==0)
  {
    if (!gst_showbits23(gb)) /* next_start_code */
    {
      mpeg2play_next_start_code(gb);
      code = gst_showbits32(gb);

      if (code<SLICE_START_CODE_MIN || code>SLICE_START_CODE_MAX)
      {
        /* only slice headers are allowed in picture_data */
        printf("SNR: Premature end of picture\n");
        return;
      }

      gst_flushbits32(gb);

      /* decode slice header (may change quantizer_scale) */
      slice_vert_pos_ext = slice_header(vid_stream);

      /* decode macroblock address increment */
      *SNRMBAinc = Get_macroblock_address_increment(gb);

      /* set current location */
      *SNRMBA =
        ((slice_vert_pos_ext<<7) + (code&255) - 1)*vid_stream->mb_width + *SNRMBAinc - 1;

      *SNRMBAinc = 1; /* first macroblock in slice: not skipped */
    }
    else /* not mpeg2play_next_start_code */
    {
      if (*SNRMBA>=MBAmax)
      {
        printf("Too many macroblocks in picture\n");
        return;
      }

      /* decode macroblock address increment */
      *SNRMBAinc = Get_macroblock_address_increment(gb);
    }
  }

  if (*SNRMBA!=MBA)
  {
    /* streams out of sync */
    printf("Cant't synchronize streams\n");
    return;
  }

  if (*SNRMBAinc==1) /* not skipped */
  {
    macroblock_modes(vid_stream, &SNRmacroblock_type, &dummy, &dummy,
      &dummy, &dummy, &dummy, &dummy, &dummy,
      &SNRdct_type);

    if (SNRmacroblock_type & MACROBLOCK_PATTERN)
      *dct_type = SNRdct_type;

    if (SNRmacroblock_type & MACROBLOCK_QUANT)
    {
      quantizer_scale_code = gst_getbits5(gb);
      vid_stream->ld->quantizer_scale =
        vid_stream->ld->q_scale_type ? Non_Linear_quantizer_scale[quantizer_scale_code] : quantizer_scale_code<<1;
    }

    /* macroblock_pattern */
    if (SNRmacroblock_type & MACROBLOCK_PATTERN)
    {
      SNRcoded_block_pattern = Get_coded_block_pattern(gb);

      if (vid_stream->sequence.chroma_format==CHROMA422)
        SNRcoded_block_pattern = (SNRcoded_block_pattern<<2) | gst_getbits2(gb); /* coded_block_pattern_1 */
      else if (vid_stream->sequence.chroma_format==CHROMA444)
        SNRcoded_block_pattern = (SNRcoded_block_pattern<<6) | gst_getbits6(gb); /* coded_block_pattern_2 */
    }
    else
      SNRcoded_block_pattern = 0;

    /* decode blocks */
    for (comp=0; comp<vid_stream->block_count; comp++)
    {
      Clear_Block(vid_stream, comp);

      if (SNRcoded_block_pattern & (1<<(vid_stream->block_count-1-comp)))
        Decode_MPEG2_Non_Intra_Block(vid_stream, comp);
    }
  }
  else /* SNRMBAinc!=1: skipped macroblock */
  {
    for (comp=0; comp<vid_stream->block_count; comp++)
      Clear_Block(vid_stream, comp);
  }

  vid_stream->ld = &vid_stream->base;
}
#endif



/* IMPLEMENTATION: set scratch pad macroblock to zero */
static void Clear_Block(vid_stream, comp)
mpeg2play_vid_stream *vid_stream;
int comp;
{
  long *Block_Ptr;
  int i;

  Block_Ptr = (long *)vid_stream->ld->block[comp];

  vid_stream->ld->sparse[comp] = 1;
  for (i=32; i; i--)
    *Block_Ptr++ = 0;
}


#ifdef ENHANCE
/* SCALABILITY: add SNR enhancement layer block data to base layer */
/* ISO/IEC 13818-2 section 7.8.3.4: Addition of coefficients from the two layes */
static void Sum_Block(vid_stream, comp)
mpeg2play_vid_stream *vid_stream;
int comp;
{
  short *Block_Ptr1, *Block_Ptr2;
  int i;

  Block_Ptr1 = vid_stream->base.block[comp];
  Block_Ptr2 = vid_stream->enhan.block[comp];

  for (i=0; i<64; i++)
    *Block_Ptr1++ += *Block_Ptr2++;
}
#endif


#ifndef HAVE_LIBMMX
/* limit coefficients to -2048..2047 */
/* ISO/IEC 13818-2 section 7.4.3 and 7.4.4: Saturation and Mismatch control */
static void Saturate(vid_stream, Block_Ptr)
mpeg2play_vid_stream *vid_stream;
short *Block_Ptr;
{
  int i, sum, val;

  sum = 0;

  /* ISO/IEC 13818-2 section 7.4.3: Saturation */
  for (i=0; i<64; i++)
  {
    val = Block_Ptr[i];

    if (val>2047)
      val = 2047;
    else if (val<-2048)
      val = -2048;

    Block_Ptr[i] = val;
    sum+= val;
  }

  /* ISO/IEC 13818-2 section 7.4.4: Mismatch control */
  if ((sum&1)==0)
    Block_Ptr[63]^= 1;

}
#endif


/* reuse old picture buffers as soon as they are no longer needed 
   based on life-time axioms of MPEG */
static void Update_Picture_Buffers(vid_stream)
mpeg2play_vid_stream *vid_stream;
{                           
  int cc;              /* color component index */
  unsigned char *tmp;  /* temporary swap pointer */

  for (cc=0; cc<3; cc++)
  {
    /* B pictures do not need to be save for future reference */
    if (vid_stream->picture.picture_coding_type==B_TYPE)
    {
      vid_stream->current_frame[cc] = vid_stream->auxframe[cc];
    }
    else
    {
      /* only update at the beginning of the coded frame */
      if (!vid_stream->Second_Field)
      {
        tmp = vid_stream->forward_reference_frame[cc];

        /* the previously decoded reference frame is stored
           coincident with the location where the backward 
           reference frame is stored (backwards prediction is not
           needed in P pictures) */
        vid_stream->forward_reference_frame[cc] = vid_stream->backward_reference_frame[cc];
        
        /* update pointer for potential future B pictures */
        vid_stream->backward_reference_frame[cc] = tmp;
      }

      /* can erase over old backward reference frame since it is not used
         in a P picture, and since any subsequent B pictures will use the 
         previously decoded I or P frame as the backward_reference_frame */
      vid_stream->current_frame[cc] = vid_stream->backward_reference_frame[cc];
    }

    /* IMPLEMENTATION:
       one-time folding of a line offset into the pointer which stores the
       memory address of the current frame saves offsets and conditional 
       branches throughout the remainder of the picture processing loop */
    if (vid_stream->picture.picture_structure==BOTTOM_FIELD)
      vid_stream->current_frame[cc]+= (cc==0) ? vid_stream->Coded_Picture_Width : vid_stream->Chroma_Width;
  }

  vid_stream->output_frame = vid_stream->current_frame[0];
}


/* store last frame */

void Output_Last_Frame_of_Sequence(vid_stream, Framenum)
mpeg2play_vid_stream *vid_stream;
int Framenum;
{
  if (vid_stream->Second_Field)
    printf("last frame incomplete, not stored\n");
  else {
    //Write_Frame(vid_stream->backward_reference_frame,Framenum-1);
  }
}



static void frame_reorder(vid_stream, Bitstream_Framenum, Sequence_Framenum)
mpeg2play_vid_stream *vid_stream;
int Bitstream_Framenum, Sequence_Framenum;
{
  /* tracking variables to insure proper output in spatial scalability */
  static int Oldref_progressive_frame, Newref_progressive_frame;

  if (Sequence_Framenum!=0)
  {
    if (vid_stream->picture.picture_structure==FRAME_PICTURE || vid_stream->Second_Field)
    {
      if (vid_stream->picture.picture_coding_type==B_TYPE) {
        //Write_Frame(vid_stream->auxframe,Bitstream_Framenum-1);
	vid_stream->output_frame = vid_stream->auxframe[0];
      }
      else
      {
        Newref_progressive_frame = vid_stream->picture.progressive_frame;
        vid_stream->picture.progressive_frame = Oldref_progressive_frame;

        //Write_Frame(vid_stream->forward_reference_frame,Bitstream_Framenum-1);
	vid_stream->output_frame = vid_stream->forward_reference_frame[0];

        Oldref_progressive_frame = vid_stream->picture.progressive_frame = Newref_progressive_frame;
      }
    }
  }
  else
    Oldref_progressive_frame = vid_stream->picture.progressive_frame;

}


/* ISO/IEC 13818-2 section 7.6 */
static void motion_compensation(vid_stream, MBA, macroblock_type, motion_type, PMV, 
  motion_vertical_field_select, dmvector, stwtype, dct_type)
mpeg2play_vid_stream *vid_stream;
int MBA;
int macroblock_type;
int motion_type;
int PMV[2][2][2];
int motion_vertical_field_select[2][2];
int dmvector[2];
int stwtype;
int dct_type;
{
  int bx, by;
  int comp;

  /* derive current macroblock position within picture */
  /* ISO/IEC 13818-2 section 6.3.1.6 and 6.3.1.7 */
  bx = 16*(MBA%vid_stream->mb_width);
  by = 16*(MBA/vid_stream->mb_width);

  /* motion compensation */
  if (!(macroblock_type & MACROBLOCK_INTRA))
    mpeg2play_form_predictions(vid_stream, bx,by,macroblock_type,motion_type,PMV,
      motion_vertical_field_select,dmvector,stwtype);
  
#ifdef ENHANCE
  /* SCALABILITY: Data Partitioning */
  if (vid_stream->base.scalable_mode==SC_DP)
    vid_stream->ld = &vid_stream->base;
#endif

  /* copy or add block data into picture */
  for (comp=0; comp<vid_stream->block_count; comp++)
  {
#ifdef ENHANCE
    /* SCALABILITY: SNR */
    /* ISO/IEC 13818-2 section 7.8.3.4: Addition of coefficients from 
       the two a layers */
    if (vid_stream->Two_Streams && vid_stream->enhan.scalable_mode==SC_SNR)
      Sum_Block(vid_stream, comp); /* add SNR enhancement layer data to base layer */
#endif

    if (vid_stream->ld->sparse[comp]) {
      gst_idct_convert_sparse(vid_stream->idct, vid_stream->ld->block[comp]);
    } else {
#ifndef HAVE_LIBMMX
      /* MPEG-2 saturation and mismatch control */
      /* base layer could be MPEG-1 stream, enhancement MPEG-2 SNR */
      /* ISO/IEC 13818-2 section 7.4.3 and 7.4.4: Saturation and Mismatch control */
      if ((vid_stream->Two_Streams && vid_stream->enhan.scalable_mode==SC_SNR) || vid_stream->ld->MPEG2_Flag)
        Saturate(vid_stream, vid_stream->ld->block[comp]);
#endif

      gst_idct_convert(vid_stream->idct, vid_stream->ld->block[comp]);
    }
    
    /* ISO/IEC 13818-2 section 7.6.8: Adding prediction and coefficient data */
    Add_Block(vid_stream, comp,bx,by,dct_type,(macroblock_type & MACROBLOCK_INTRA)==0);
  }

}



/* ISO/IEC 13818-2 section 7.6.6 */
static void skipped_macroblock(vid_stream, dc_dct_pred, PMV, motion_type, 
  motion_vertical_field_select, stwtype, macroblock_type)
mpeg2play_vid_stream *vid_stream;
int dc_dct_pred[3];
int PMV[2][2][2];
int *motion_type;
int motion_vertical_field_select[2][2];
int *stwtype;
int *macroblock_type;
{
  int comp;
  
  /* SCALABILITY: Data Paritioning */
  if (vid_stream->base.scalable_mode==SC_DP)
    vid_stream->ld = &vid_stream->base;

  for (comp=0; comp<vid_stream->block_count; comp++)
    Clear_Block(vid_stream, comp);

  /* reset intra_dc predictors */
  /* ISO/IEC 13818-2 section 7.2.1: DC coefficients in intra blocks */
  dc_dct_pred[0]=dc_dct_pred[1]=dc_dct_pred[2]=0;

  /* reset motion vector predictors */
  /* ISO/IEC 13818-2 section 7.6.3.4: Resetting motion vector predictors */
  if (vid_stream->picture.picture_coding_type==P_TYPE)
    PMV[0][0][0]=PMV[0][0][1]=PMV[1][0][0]=PMV[1][0][1]=0;

  /* derive motion_type */
  if (vid_stream->picture.picture_structure==FRAME_PICTURE)
    *motion_type = MC_FRAME;
  else
  {
    *motion_type = MC_FIELD;

    /* predict from field of same parity */
    /* ISO/IEC 13818-2 section 7.6.6.1 and 7.6.6.3: P field picture and B field
       picture */
    motion_vertical_field_select[0][0]=motion_vertical_field_select[0][1] = 
      (vid_stream->picture.picture_structure==BOTTOM_FIELD);
  }

  /* skipped I are spatial-only predicted, */
  /* skipped P and B are temporal-only predicted */
  /* ISO/IEC 13818-2 section 7.7.6: Skipped macroblocks */
  *stwtype = (vid_stream->picture.picture_coding_type==I_TYPE) ? 8 : 0;

 /* IMPLEMENTATION: clear MACROBLOCK_INTRA */
  *macroblock_type&= ~MACROBLOCK_INTRA;

}


/* return==-1 means go to next picture */
/* the expression "start of slice" is used throughout the normative
   body of the MPEG specification */
static int start_of_slice(vid_stream, MBAmax, MBA, MBAinc, 
  dc_dct_pred, PMV)
mpeg2play_vid_stream *vid_stream;
int MBAmax;
int *MBA;
int *MBAinc;
int dc_dct_pred[3];
int PMV[2][2][2];
{
  unsigned int code;
  int slice_vert_pos_ext;
  gst_getbits_t *gb = &vid_stream->ld->gb;

  vid_stream->ld = &vid_stream->base;

  vid_stream->Fault_Flag = 0;

  if (mpeg2play_next_start_code(gb) == NO_STARTCODE) {
    printf("start_of_slice(): end of picture\n");
    return -1;
  }

  code = gst_showbits32(gb);

  if (code<SLICE_START_CODE_MIN || code>SLICE_START_CODE_MAX)
  {
    /* only slice headers are allowed in picture_data */
    printf("start_of_slice(): Premature end of picture\n");

    return(-1);  /* trigger: go to next picture */
  }

  gst_flushbits32(gb); 

  /* decode slice header (may change quantizer_scale) */
  slice_vert_pos_ext = slice_header(vid_stream);

 
#ifdef ENHANCE
  /* SCALABILITY: Data Partitioning */
  if (vid_stream->base.scalable_mode==SC_DP)
  {
    vid_stream->ld = &vid_stream->enhan;
    mpeg2play_next_start_code(gb);
    code = gst_showbits32(gb);

    if (code<SLICE_START_CODE_MIN || code>SLICE_START_CODE_MAX)
    {
      /* only slice headers are allowed in picture_data */
      printf("DP: Premature end of picture\n");
      return(-1);    /* trigger: go to next picture */
    }

    gst_flushbits32(gb);

    /* decode slice header (may change quantizer_scale) */
    slice_vert_pos_ext = slice_header(vid_stream);

    if (vid_stream->base.priority_breakpoint!=1)
      vid_stream->ld = &vid_stream->base;
  }
#endif

  /* decode macroblock address increment */
  *MBAinc = Get_macroblock_address_increment(vid_stream, gb);

  if (vid_stream->Fault_Flag) 
  {
    printf("start_of_slice(): MBAinc unsuccessful\n");
    return(0);   /* trigger: go to next slice */
  }

  /* set current location */
  /* NOTE: the arithmetic used to derive macroblock_address below is
   *       equivalent to ISO/IEC 13818-2 section 6.3.17: Macroblock
   */
  *MBA = ((slice_vert_pos_ext<<7) + (code&255) - 1)*vid_stream->mb_width + *MBAinc - 1;
  *MBAinc = 1; /* first macroblock in slice: not skipped */

  /* reset all DC coefficient and motion vector predictors */
  /* reset all DC coefficient and motion vector predictors */
  /* ISO/IEC 13818-2 section 7.2.1: DC coefficients in intra blocks */
  dc_dct_pred[0]=dc_dct_pred[1]=dc_dct_pred[2]=0;
  
  /* ISO/IEC 13818-2 section 7.6.3.4: Resetting motion vector predictors */
  PMV[0][0][0]=PMV[0][0][1]=PMV[1][0][0]=PMV[1][0][1]=0;
  PMV[0][1][0]=PMV[0][1][1]=PMV[1][1][0]=PMV[1][1][1]=0;

  /* successfull: trigger decode macroblocks in slice */
  return(1);
}


/* ISO/IEC 13818-2 sections 7.2 through 7.5 */
static int decode_macroblock(vid_stream, macroblock_type, stwtype, stwclass,
  motion_type, dct_type, PMV, dc_dct_pred, 
  motion_vertical_field_select, dmvector)
mpeg2play_vid_stream *vid_stream;
int *macroblock_type; 
int *stwtype;
int *stwclass;
int *motion_type; 
int *dct_type;
int PMV[2][2][2]; 
int dc_dct_pred[3]; 
int motion_vertical_field_select[2][2];
int dmvector[2];
{
  /* locals */
  int quantizer_scale_code; 
  int comp;

  int motion_vector_count; 
  int mv_format; 
  int dmv; 
  int mvscale;
  int coded_block_pattern;
  gst_getbits_t *gb = &vid_stream->ld->gb;

#ifdef ENHANCE
  /* SCALABILITY: Data Patitioning */
  if (vid_stream->base.scalable_mode==SC_DP)
  {
    if (vid_stream->base.priority_breakpoint<=2)
      vid_stream->ld = &vid_stream->enhan;
    else
      vid_stream->ld = &vid_stream->base;

    gb = &vid_stream->ld->gb;
  }
#endif

  /* ISO/IEC 13818-2 section 6.3.17.1: Macroblock modes */
  macroblock_modes(vid_stream, macroblock_type, stwtype, stwclass,
    motion_type, &motion_vector_count, &mv_format, &dmv, &mvscale,
    dct_type);

  if (vid_stream->Fault_Flag) return(0);  /* trigger: go to next slice */

  if (*macroblock_type & MACROBLOCK_QUANT)
  {
    quantizer_scale_code = gst_getbits5(gb);

#ifdef TRACE
    printf("quantiser_scale_code (");
    //Print_Bits(quantizer_scale_code,5,5);
    printf("): %d\n",quantizer_scale_code);
   
#endif /* TRACE */

    /* ISO/IEC 13818-2 section 7.4.2.2: Quantizer scale factor */
    if (vid_stream->ld->MPEG2_Flag)
      vid_stream->ld->quantizer_scale =
      vid_stream->ld->q_scale_type ? Non_Linear_quantizer_scale[quantizer_scale_code] 
       : (quantizer_scale_code << 1);
    else
      vid_stream->ld->quantizer_scale = quantizer_scale_code;

#ifdef ENHANCE
    /* SCALABILITY: Data Partitioning */
    if (vid_stream->base.scalable_mode==SC_DP)
      /* make sure base.quantizer_scale is valid */
      vid_stream->base.quantizer_scale = vid_stream->ld->quantizer_scale;
#endif
  }

  /* motion vectors */

  /* ISO/IEC 13818-2 section 6.3.17.2: Motion vectors */

  /* decode forward motion vectors */
  if ((*macroblock_type & MACROBLOCK_MOTION_FORWARD) 
    || ((*macroblock_type & MACROBLOCK_INTRA) 
    && vid_stream->picture.concealment_motion_vectors))
  {
    if (vid_stream->ld->MPEG2_Flag)
      motion_vectors(vid_stream, PMV,dmvector,motion_vertical_field_select,
        0,motion_vector_count,mv_format,vid_stream->picture.f_code[0][0]-1,vid_stream->picture.f_code[0][1]-1,
        dmv,mvscale);
    else
      motion_vector(vid_stream, PMV[0][0],dmvector,
      vid_stream->picture.forward_f_code-1,vid_stream->picture.forward_f_code-1,0,0,vid_stream->picture.full_pel_forward_vector);
  }

  if (vid_stream->Fault_Flag) return(0);  /* trigger: go to next slice */

  /* decode backward motion vectors */
  if (*macroblock_type & MACROBLOCK_MOTION_BACKWARD)
  {
    if (vid_stream->ld->MPEG2_Flag)
      motion_vectors(vid_stream, PMV,dmvector,motion_vertical_field_select,
        1,motion_vector_count,mv_format,vid_stream->picture.f_code[1][0]-1,vid_stream->picture.f_code[1][1]-1,0,
        mvscale);
    else
      motion_vector(vid_stream, PMV[0][1],dmvector,
        vid_stream->picture.backward_f_code-1,vid_stream->picture.backward_f_code-1,0,0,vid_stream->picture.full_pel_backward_vector);
  }

  if (vid_stream->Fault_Flag) return(0);  /* trigger: go to next slice */

  if ((*macroblock_type & MACROBLOCK_INTRA) && vid_stream->picture.concealment_motion_vectors)
    gst_flushbitsn(gb, 1); /* remove marker_bit */

#ifdef ENHANCE
  if (vid_stream->base.scalable_mode==SC_DP && vid_stream->base.priority_breakpoint==3)
    vid_stream->ld = &vid_stream->enhan;
#endif

  /* macroblock_pattern */
  /* ISO/IEC 13818-2 section 6.3.17.4: Coded block pattern */
  if (*macroblock_type & MACROBLOCK_PATTERN)
  {
    coded_block_pattern = Get_coded_block_pattern(gb);

    if (vid_stream->sequence.chroma_format==CHROMA422)
    {
      /* coded_block_pattern_1 */
      coded_block_pattern = (coded_block_pattern<<2) | gst_getbits2(gb); 

#ifdef TRACE
         printf("coded_block_pattern_1: ");
         //Print_Bits(coded_block_pattern,2,2);
         printf(" (%d)\n",coded_block_pattern&3);
#endif /* TRACE */
     }
     else if (vid_stream->sequence.chroma_format==CHROMA444)
     {
      /* coded_block_pattern_2 */
      coded_block_pattern = (coded_block_pattern<<6) | gst_getbits6(gb); 

#ifdef TRACE
        printf("coded_block_pattern_2: ");
        //Print_Bits(coded_block_pattern,6,6);
        printf(" (%d)\n",coded_block_pattern&63);
#endif /* TRACE */
    }
  }
  else
    coded_block_pattern = (*macroblock_type & MACROBLOCK_INTRA) ? 
      (1<<vid_stream->block_count)-1 : 0;

  if (vid_stream->Fault_Flag) return(0);  /* trigger: go to next slice */

  /* decode blocks */
  for (comp=0; comp<vid_stream->block_count; comp++)
  {
#ifdef ENHANCE
    /* SCALABILITY: Data Partitioning */
    if (vid_stream->base.scalable_mode==SC_DP)
      vid_stream->ld = &vid_stream->base;
#endif

    Clear_Block(vid_stream, comp);
    //if (coded_block_pattern == 0) printf("sparse block\n");

    if (coded_block_pattern & (1<<(vid_stream->block_count-1-comp)))
    {
      if (*macroblock_type & MACROBLOCK_INTRA)
      {
        if (vid_stream->ld->MPEG2_Flag)
          Decode_MPEG2_Intra_Block(vid_stream, comp,dc_dct_pred);
        else
          Decode_MPEG1_Intra_Block(vid_stream, comp,dc_dct_pred);
      }
      else
      {
        if (vid_stream->ld->MPEG2_Flag)
          Decode_MPEG2_Non_Intra_Block(vid_stream, comp);
        else
          Decode_MPEG1_Non_Intra_Block(vid_stream, comp);
      }

      if (vid_stream->Fault_Flag) return(0);  /* trigger: go to next slice */
    }
  }

  if(vid_stream->picture.picture_coding_type==D_TYPE)
  {
    /* remove end_of_macroblock (always 1, prevents startcode emulation) */
    /* ISO/IEC 11172-2 section 2.4.2.7 and 2.4.3.6 */
    marker_bit(gb, "D picture end_of_macroblock bit");
  }

  /* reset intra_dc predictors */
  /* ISO/IEC 13818-2 section 7.2.1: DC coefficients in intra blocks */
  if (!(*macroblock_type & MACROBLOCK_INTRA))
    dc_dct_pred[0]=dc_dct_pred[1]=dc_dct_pred[2]=0;

  /* reset motion vector predictors */
  if ((*macroblock_type & MACROBLOCK_INTRA) && !vid_stream->picture.concealment_motion_vectors)
  {
    /* intra mb without concealment motion vectors */
    /* ISO/IEC 13818-2 section 7.6.3.4: Resetting motion vector predictors */
    PMV[0][0][0]=PMV[0][0][1]=PMV[1][0][0]=PMV[1][0][1]=0;
    PMV[0][1][0]=PMV[0][1][1]=PMV[1][1][0]=PMV[1][1][1]=0;
  }

  /* special "No_MC" macroblock_type case */
  /* ISO/IEC 13818-2 section 7.6.3.5: Prediction in P pictures */
  if ((vid_stream->picture.picture_coding_type==P_TYPE) 
    && !(*macroblock_type & (MACROBLOCK_MOTION_FORWARD|MACROBLOCK_INTRA)))
  {
    /* non-intra mb without forward mv in a P picture */
    /* ISO/IEC 13818-2 section 7.6.3.4: Resetting motion vector predictors */
    PMV[0][0][0]=PMV[0][0][1]=PMV[1][0][0]=PMV[1][0][1]=0;

    /* derive motion_type */
    /* ISO/IEC 13818-2 section 6.3.17.1: Macroblock modes, frame_motion_type */
    if (vid_stream->picture.picture_structure==FRAME_PICTURE)
      *motion_type = MC_FRAME;
    else
    {
      *motion_type = MC_FIELD;
      /* predict from field of same parity */
      motion_vertical_field_select[0][0] = (vid_stream->picture.picture_structure==BOTTOM_FIELD);
    }
  }

  if (*stwclass==4)
  {
    /* purely spatially predicted macroblock */
    /* ISO/IEC 13818-2 section 7.7.5.1: Resetting motion vector predictions */
    PMV[0][0][0]=PMV[0][0][1]=PMV[1][0][0]=PMV[1][0][1]=0;
    PMV[0][1][0]=PMV[0][1][1]=PMV[1][1][0]=PMV[1][1][1]=0;
  }

  /* successfully decoded macroblock */
  return(1);

} /* decode_macroblock */


