/* Gnome-Streamer
 * Copyright (C) <1999> Erik Walthinsen <omega@cse.ogi.edu>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

//#define DEBUG_ENABLED
#include <mp2videoparse.h>

static void	gst_mp2videoparse_class_init	(Mp2VideoParseClass *klass);
static void	gst_mp2videoparse_init		(Mp2VideoParse *mp2videoparse);

static void	gst_mp2videoparse_chain		(GstPad *pad, GstBuffer *buf);
static void	gst_mp2videoparse_real_chain	(Mp2VideoParse *mp2videoparse, GstBuffer *buf, GstPad *outpad);

/* Start codes. */
#define SEQ_START_CODE 0x000001b3
#define GOP_START_CODE 0x000001b8
#define PICTURE_START_CODE 0x00000100
#define SLICE_MIN_START_CODE 0x00000101
#define SLICE_MAX_START_CODE 0x000001af
#define EXT_START_CODE 0x000001b5
#define USER_START_CODE 0x000001b2
#define SEQUENCE_ERROR_CODE 0x000001b4
#define SEQ_END_CODE 0x000001b7

/* elementfactory information */
static GstElementDetails mp2videoparse_details = {
  "MPEG 2 Video Parser",
  "Filter/Parser/Video",
  "Parses and frames MPEG 2 video streams, provides seek",
  VERSION,
  "Wim Taymans <wim.taymans@chello.be>",
  "(C) 2000",
};

/* Mp2VideoParse signals and args */
enum {
  /* FILL ME */
  LAST_SIGNAL
};

enum {
  ARG_0,
  /* FILL ME */
};

static GstPadTemplate*
src_factory (void) 
{
  return 
    gst_padtemplate_new (
  	"src",
  	GST_PAD_SRC,
  	GST_PAD_ALWAYS,
  	gst_caps_new (
  	  "mp2videoparse_src",
    	  "video/mpeg",
	  gst_props_new (
    	    "mpegversion",   GST_PROPS_INT (2),
    	    "systemstream",  GST_PROPS_BOOLEAN (FALSE),
    	    "sliced",        GST_PROPS_BOOLEAN (TRUE),
	    NULL)),
	NULL);
}

static GstPadTemplate*
sink_factory (void) 
{
  return 
    gst_padtemplate_new (
  	"sink",
  	GST_PAD_SINK,
  	GST_PAD_ALWAYS,
  	gst_caps_new (
  	  "mp2videoparse_sink",
    	  "video/mpeg",
	  gst_props_new (
    	    "mpegversion",   GST_PROPS_INT (2),
    	    "systemstream",  GST_PROPS_BOOLEAN (FALSE),
	    NULL)),
	NULL);
}


static GstPadTemplate *src_template, *sink_template;

static GstElementClass *parent_class = NULL;
//static guint gst_mp2videoparse_signals[LAST_SIGNAL] = { 0 };

GType
mp2videoparse_get_type (void)
{
  static GType mp2videoparse_type = 0;

  if (!mp2videoparse_type) {
    static const GTypeInfo mp2videoparse_info = {
      sizeof(Mp2VideoParseClass),      NULL,
      NULL,
      (GClassInitFunc)gst_mp2videoparse_class_init,
      NULL,
      NULL,
      sizeof(Mp2VideoParse),
      0,
      (GInstanceInitFunc)gst_mp2videoparse_init,
    };
    mp2videoparse_type = g_type_register_static(GST_TYPE_ELEMENT, "Mp2VideoParse", &mp2videoparse_info, 0);
  }
  return mp2videoparse_type;
}

static void
gst_mp2videoparse_class_init (Mp2VideoParseClass *klass)
{
  GstElementClass *gstelement_class;

  gstelement_class = (GstElementClass*)klass;

  parent_class = g_type_class_ref(GST_TYPE_ELEMENT);
}

static void
gst_mp2videoparse_init (Mp2VideoParse *mp2videoparse)
{
  mp2videoparse->sinkpad = gst_pad_new_from_template (sink_template, "sink");
  gst_element_add_pad(GST_ELEMENT(mp2videoparse),mp2videoparse->sinkpad);
  gst_pad_set_chain_function(mp2videoparse->sinkpad,gst_mp2videoparse_chain);

  mp2videoparse->srcpad = gst_pad_new_from_template (src_template, "src");
  gst_element_add_pad(GST_ELEMENT(mp2videoparse),mp2videoparse->srcpad);

  mp2videoparse->partialbuf = NULL;
  mp2videoparse->need_resync = FALSE;
  mp2videoparse->last_pts = 0;
  mp2videoparse->picture_in_buffer = 0;
}

static
gboolean mp2videoparse_valid_sync (gulong head)
{
  if (head == SEQ_START_CODE)
    return TRUE;
  if (head == GOP_START_CODE)
    return TRUE;
  if (head == PICTURE_START_CODE)
    return TRUE;
  if (head >= SLICE_MIN_START_CODE &&
      head <= SLICE_MAX_START_CODE)
    return TRUE;
  if (head == USER_START_CODE)
    return TRUE;
  if (head == EXT_START_CODE)
    return TRUE;

  return FALSE;
}

static gint
mp2videoparse_find_next_gop (Mp2VideoParse *mp2videoparse, GstBuffer *buf)
{
  guchar *data = GST_BUFFER_DATA(buf);
  gulong size = GST_BUFFER_SIZE(buf);
  gulong offset = 0;
  gint sync_zeros = 0;
  gboolean have_sync = FALSE;

  while (offset < size) {
    guchar byte = *(data+offset);
    offset++;
    if (byte == 0) {
      sync_zeros++;
    }
    else if (byte == 1 && sync_zeros >=2 ) {
      sync_zeros = 0;
      have_sync = TRUE;
    }
    else if (have_sync) {
      if (byte == (SEQ_START_CODE & 0xff) || byte == (GOP_START_CODE & 0xff)) return offset-4;
      else {
        sync_zeros = 0;
	have_sync = FALSE;
      }
    }
    else {
      sync_zeros = 0;
    }
  }

  return -1;
}

static void
gst_mp2videoparse_chain (GstPad *pad, GstBuffer *buf)
{
  Mp2VideoParse *mp2videoparse;

  g_return_if_fail(pad != NULL);
  g_return_if_fail(GST_IS_PAD(pad));
  g_return_if_fail(buf != NULL);

  mp2videoparse = GST_MP2VIDEOPARSE (GST_OBJECT_PARENT (pad));

  gst_mp2videoparse_real_chain(mp2videoparse, buf, mp2videoparse->srcpad);
}

static void
gst_mp2videoparse_real_chain (Mp2VideoParse *mp2videoparse, GstBuffer *buf, GstPad *outpad)
{
  guchar *data;
  gulong size, offset = 0;
  GstBuffer *outbuf;
  gint sync_state;
  gboolean have_sync;
  guchar sync_byte;
  gulong head;
  gint sync_pos;
  guint64 time_stamp;
  GstBuffer *temp;

//  g_return_if_fail(GST_IS_BUFFER(buf));

  time_stamp = GST_BUFFER_TIMESTAMP(buf);
  size = GST_BUFFER_SIZE(buf);

  GST_DEBUG (0,"mp2videoparse: received buffer of %ld bytes %lld\n",size, GST_BUFFER_TIMESTAMP(buf));

  if (GST_BUFFER_FLAG_IS_SET(buf, GST_BUFFER_FLUSH)) {
    GST_DEBUG (0,"mp2videoparse: flushing\n");
    if (mp2videoparse->partialbuf) {
      gst_buffer_unref(mp2videoparse->partialbuf);
      mp2videoparse->partialbuf= NULL;
    }
    mp2videoparse->need_resync = TRUE;
    mp2videoparse->in_flush = TRUE;
    mp2videoparse->picture_in_buffer = 0;
  }


  if (mp2videoparse->partialbuf) {
    offset = GST_BUFFER_SIZE(mp2videoparse->partialbuf);
    mp2videoparse->partialbuf = gst_buffer_append(mp2videoparse->partialbuf, buf);
    gst_buffer_unref(buf);
  }
  else {
    mp2videoparse->partialbuf = buf;
    offset = 0;
  }

  data = GST_BUFFER_DATA(mp2videoparse->partialbuf);
  size = GST_BUFFER_SIZE(mp2videoparse->partialbuf);

  head = GULONG_FROM_BE(*((gulong *)data));

  GST_DEBUG (0,"mp2videoparse: head is %08lx\n", head);

  if (!mp2videoparse_valid_sync(head) || mp2videoparse->need_resync) {
    sync_pos = mp2videoparse_find_next_gop(mp2videoparse, mp2videoparse->partialbuf);
    if (sync_pos != -1) {
      mp2videoparse->need_resync = FALSE;
      GST_DEBUG (0,"mp2videoparse: found new gop at %d\n", sync_pos);

      if (sync_pos != 0) {
        temp = gst_buffer_create_sub(mp2videoparse->partialbuf, sync_pos, size-sync_pos);
	g_assert(temp != NULL);
	gst_buffer_unref(mp2videoparse->partialbuf);
	mp2videoparse->partialbuf = temp;
        data = GST_BUFFER_DATA(mp2videoparse->partialbuf);
        size = GST_BUFFER_SIZE(mp2videoparse->partialbuf);
	offset = 0;
      }
    }
    else {
      gst_buffer_unref(mp2videoparse->partialbuf);
      mp2videoparse->partialbuf = NULL;
      return;
    }
  }

  if (mp2videoparse->picture_in_buffer == 1) {
    mp2videoparse->last_pts = time_stamp;
  }

  sync_state = 0;
  have_sync = FALSE;

  GST_DEBUG (0,"mp2videoparse: searching sync\n");

  while (offset < size-1) {
    sync_byte = *(data + offset);
    //printf(" %d %02x\n", offset, sync_byte);
    if (sync_byte == 0) {
      sync_state++;
    }
    else if ((sync_byte == 1) && (sync_state >=2)) {
      GST_DEBUG (0,"mp2videoparse: code 0x000001%02x\n",data[offset+1]);
      if (data[offset+1] == (PICTURE_START_CODE & 0xff)) {
	mp2videoparse->picture_in_buffer++;
	if (mp2videoparse->picture_in_buffer == 1) {
          mp2videoparse->last_pts = time_stamp;
	  sync_state = 0;
	}
	else if (mp2videoparse->picture_in_buffer == 2) {
          have_sync = TRUE;
          break;
	}
	else {
          g_assert_not_reached();
	}
      }
      else sync_state = 0;
    }
    // something else...
    else sync_state = 0;
    // go down the buffer
    offset++;
  }

  if (have_sync) {
    offset -= 2;
    GST_DEBUG (0,"mp2videoparse: synced at %ld code 0x000001%02x\n",offset,data[offset+3]);

    outbuf = gst_buffer_create_sub(mp2videoparse->partialbuf, 0, offset+4);
    g_assert(outbuf != NULL);
    GST_BUFFER_TIMESTAMP(outbuf) = mp2videoparse->last_pts;

    if (mp2videoparse->in_flush) {
      GST_BUFFER_FLAG_SET(outbuf, GST_BUFFER_FLUSH);
      mp2videoparse->in_flush = FALSE;
    }

    GST_DEBUG (0,"mp2videoparse: pushing  %d bytes %llu\n", GST_BUFFER_SIZE(outbuf), GST_BUFFER_TIMESTAMP(outbuf));
    gst_pad_push(outpad, outbuf);
    GST_DEBUG (0,"mp2videoparse: pushing  done\n");
    mp2videoparse->picture_in_buffer = 0;

    temp = gst_buffer_create_sub(mp2videoparse->partialbuf, offset, size-offset);
    gst_buffer_unref(mp2videoparse->partialbuf);
    mp2videoparse->partialbuf = temp;
  }
  else {
    mp2videoparse->last_pts = time_stamp;
  }
}

static gboolean
plugin_init (GModule *module, GstPlugin *plugin)
{
  GstElementFactory *factory;

  /* create an elementfactory for the mp2videoparse element */
  factory = gst_elementfactory_new("mp2videoparse",GST_TYPE_MP2VIDEOPARSE,
                                   &mp2videoparse_details);
  g_return_val_if_fail(factory != NULL, FALSE);

  src_template = src_factory ();
  gst_elementfactory_add_padtemplate (factory, src_template);

  sink_template = sink_factory ();
  gst_elementfactory_add_padtemplate (factory, sink_template);

  gst_plugin_add_factory(plugin,factory);

  return TRUE;
}

GstPluginDesc plugin_desc = {
  GST_VERSION_MAJOR,
  GST_VERSION_MINOR,
  "mp2videoparse",
  plugin_init
};
