/*
    Copyright (C) 2001 CodeFactory AB
    Copyright (C) 2001 Thomas Nyberg <thomas@codefactory.se>
                            
    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include "alsa.h"

static GstPlugin *alsa_sink_plugin = NULL;
static GstElement *parent_class = NULL;
static GHashTable *alsa_cards = NULL;

static void gst_alsasink_set_property(GObject *object, guint prop_id, const GValue *value, GParamSpec *pspec);
static void gst_alsasink_get_property(GObject *object, guint prop_id, GValue *value, GParamSpec *pspec);
static GstElementStateReturn gst_alsasink_change_state(GstElement *element);
static void gst_alsasink_chain(GstPad *pad, GstBuffer *buffer);


enum {
	ARG_0,
	ARG_MUTE,
	ARG_FORMAT,
	ARG_CHANNELS,
	ARG_FREQUENCY,
};

/*
 * signals
 * more generalized code
 * testing of multiple soundcards
 * Support for ALSA > 0.5.x
 */

static void
gst_alsasink_class_init(GstALSAClass *klass)
{
	GObjectClass *object_class;
	GstElementClass *element_class;
	gchar *name;

	object_class = (GObjectClass *)klass;
	element_class = (GstElementClass *)klass;

	if (parent_class == NULL)
		parent_class = g_type_class_ref(GST_TYPE_ELEMENT);

	/* add set_property, get_property-functions */

	name = gtk_type_name(object_class->type); /* get name */
	/* create the arg-name */

  g_object_class_install_property(G_OBJECT_CLASS(klass), GTK_ARG_READWRITE,
    g_param_spec_enum("%s::mute","%s::mute","%s::mute",
                      name),0,G_TYPE_BOOL)); // CHECKME!
  g_object_class_install_property(G_OBJECT_CLASS(klass), GTK_ARG_READWRITE,
    g_param_spec_enum("%s::format","%s::format","%s::format",
                      name),0,GST_TYPE_ALSA_FORMATS)); // CHECKME!
  g_object_class_install_property(G_OBJECT_CLASS(klass), GTK_ARG_READWRITE,
    g_param_spec_enum("%s::channels","%s::channels","%s::channels",
                      name),0,GST_TYPE_ALSA_CHANNELS)); // CHECKME!
  g_object_class_install_property(G_OBJECT_CLASS(klass), GTK_ARG_READWRITE,
    g_param_spec_enum("%s::frequency","%s::frequency","%s::frequency",
                      name),0,G_TYPE_INT)); // CHECKME!

	g_free(name); /* free name */
	object_class->get_property = gst_alsasink_get_property;
	object_class->set_property = gst_alsasink_set_property;

	element_class->change_state = gst_alsasink_change_state;
}

static void
gst_alsa_create_pad(GstALSA *this)
{
	GstPadTemplate *gst_template = NULL;

	g_assert(this != NULL);

	gst_template = gst_padtemplate_new("sink", GST_PAD_SINK, GST_PAD_ALWAYS, _gst_alsa_create_caps(this), NULL);
	g_assert(gst_template != NULL);

	this->pad = gst_pad_new_from_template(gst_template, "sink");
	gst_element_add_pad(GST_ELEMENT(this), this->pad);
}

static void
gst_alsasink_init(GstALSA *this)
{
	ALSA_DEBUG_MSG(__FUNCTION__);

	/* init values */
	this->handle = NULL;
	this->clock = gst_clock_get_system();

	/* set some nice default-values */
	this->format = 16;
	this->rate = 44100;
	this->channels = 2;

	/* buffer-stuff */
	this->buffer = NULL;
	this->count = 0;

	/* fetch the card-info */
	this->card = (ALSACard *)g_hash_table_lookup(alsa_cards, GINT_TO_POINTER(G_OBJECT(this)->klass->type));
	g_assert(this->card != NULL);

	this->cap = _gst_alsa_probe_capabilities(this->card); /* probe for card-capabilities */

	gst_clock_register(this->clock, GST_OBJECT(this));
	GST_FLAG_SET(this, GST_ELEMENT_THREAD_SUGGESTED);

	/* create the pad */
	gst_alsa_create_pad(this);

	g_assert(this->pad != NULL);
	gst_pad_set_negotiate_function(this->pad, _gst_alsa_negotiate);
	gst_pad_set_chain_function(this->pad, gst_alsasink_chain);
}

static void
gst_alsasink_set_property(GObject *object, guint prop_id, const GValue *value, GParamSpec *pspec)
{
	GstALSA *this;
	gboolean prepare_channel = FALSE;

	this = (GstALSA *)object;

	switch (prop_id) {
	case ARG_MUTE:
		ALSA_DEBUG_MSG("ARG_MUTE not supported");
		break;
	case ARG_FORMAT:
		ALSA_DEBUG_MSG("ARG_FORMAT not supported");
		/* formats could be of any kind, signed or unsigned or anything... */
		break;
	case ARG_CHANNELS:
		this->channels = g_value_get_enum (value);
		prepare_channel = TRUE;
		break;
	case ARG_FREQUENCY:
		this->rate = g_value_get_int (value);
		prepare_channel = TRUE;
		break;
	default:
		ALSA_DEBUG_MSG("Unknown arg");
		break;
	}

	if (prepare_channel) {
		/* if we are playing, we need to do some additional stuff */
		if (GST_FLAG_IS_SET(this, GST_ALSA_OPEN))
			snd_pcm_channel_flush(this->handle, this->card->channel);

		_gst_alsa_channel_prepare(this);

		if (GST_FLAG_IS_SET(this, GST_ALSA_OPEN))
			snd_pcm_channel_prepare(this->handle, this->card->channel);
		/* ready to play again */
	}
}

static void
gst_alsasink_get_property(GObject *object, guint prop_id, GValue *value, GParamSpec *pspec)
{
	GstALSA *this;

	this = (GstALSA *)object;

	switch (prop_id) {
	case ARG_MUTE:
		ALSA_DEBUG_MSG("ARG_MUTE not supported");
		break;
	case ARG_FORMAT:
		ALSA_DEBUG_MSG("ARG_FORMAT not supported");
		break;
	case ARG_CHANNELS:
		g_value_set_int (value, this->channels);
		break;
	case ARG_FREQUENCY:
		g_value_set_int (value, this->rate);
		break;
	default:
		ALSA_DEBUG_MSG("Unknown arg");
		break;
	}
}

static gint
gst_alsa_write(GstALSA *this, const gchar *buffer, gint length)
{
	gint offset = 0;
	gint ret;

	while (length - offset > 0) {

		ret = snd_pcm_write(this->handle, buffer + offset, length - offset);

		if (ret == -EAGAIN) {
			ALSA_DEBUG_MSG("Sleeping");
			usleep(ALSA_SLEEP_DELAY);
		} else if (ret == -EPIPE) {
			ALSA_ERROR("buffer underrun occured");

			/* try to recover */
			if ((ret = snd_pcm_channel_prepare(this->handle, this->card->channel))) {
				ALSA_ERROR(snd_strerror(ret));
				return ret;
			}
			/* ready to receive audio again */
		} else if (ret > 0) {
			offset += ret;
		} else {
			ALSA_ERROR(snd_strerror(ret));

			/* try to recover */
			if ((ret = snd_pcm_channel_prepare(this->handle, this->card->channel))) {
				ALSA_ERROR(snd_strerror(ret));
				return ret;
			}
			/* ready to receive audio again */
		}
	}

	return offset;	
}

static void
gst_alsasink_chain(GstPad *pad, GstBuffer *buffer)
{
	GstALSA *this;
	//MetaAudioRaw *meta;

	this = (GstALSA *)gst_pad_get_parent(pad);
	
	//meta = (MetaAudioRaw *)gst_buffer_get_first_meta(buffer);

	if (GST_BUFFER_FLAG_IS_SET(buffer, GST_BUFFER_FLUSH)) {
		/* failure or not doesn't matter */
		snd_pcm_channel_flush(this->handle, this->card->channel);
		snd_pcm_channel_prepare(this->handle, this->card->channel);	
	}

	/*
	if (meta != NULL) {
		if ((meta->format != this->format) ||
		    (meta->channels != this->channels) ||
		    (meta->frequency != this->rate)) {
			this->format = meta->format;
			this->channels = meta->channels;
			this->rate = meta->frequency;
			*/

			/* this is bad, reset channel-params */
			//snd_pcm_channel_flush(this->handle, this->card->channel);

			//_gst_alsa_channel_prepare(this);

			//snd_pcm_channel_prepare(this->handle, this->card->channel);	
			/* ready to play audio again */
	//	}
	//}

	if (GST_BUFFER_DATA(buffer) != NULL) {
		if (this->handle) {
			/* should not wait for clock */
			gst_clock_wait(this->clock, GST_BUFFER_TIMESTAMP(buffer), GST_OBJECT(this));
			if (this->cap->stream_support) { /* we are using stream-mode by default */
				if (gst_alsa_write(this, GST_BUFFER_DATA(buffer), GST_BUFFER_SIZE(buffer)) < 0) {
					ALSA_ERROR("Error during write");
				}
			} else if (this->cap->block_support) { /* block-mode is used, this sucks */
				gint size;
				gint offset = 0;
				gint ret;

				g_assert(this->buffer != NULL);

				ALSA_DEBUG_PRINT("alsasink: playing %dbytes\n", GST_BUFFER_SIZE(buffer));
				while (GST_BUFFER_SIZE(buffer) - offset > 0) {
					/* fill buffer */
					size = GST_BUFFER_SIZE(buffer) - offset + this->count >= ALSA_BUFFER_SIZE ? 
						ALSA_BUFFER_SIZE - this->count : GST_BUFFER_SIZE(buffer) - offset;

					g_memmove(this->buffer + this->count, GST_BUFFER_DATA(buffer) + offset, size);
					this->count += size;
					offset += size;

					if (this->count == ALSA_BUFFER_SIZE) { /* play buffer only if buffer is full */
						if ((ret = gst_alsa_write(this, this->buffer, this->count)) < 0) {
							ALSA_ERROR("Error during write");
						} else
							this->count -= ret;
					}
				}
				ALSA_DEBUG_PRINT("alsasink: buffer: %dbytes\n", this->count);
			}
		}
	}

	gst_buffer_unref(buffer);
}

static GstElementStateReturn
gst_alsasink_change_state(GstElement *element)
{
	g_return_val_if_fail(element != NULL, FALSE);
	ALSA_DEBUG_MSG(__FUNCTION__);

	if (GST_STATE_PENDING(element) == GST_STATE_NULL) {
		if (GST_FLAG_IS_SET(element, GST_ALSA_OPEN))
			_gst_alsa_close_audio((GstALSA *)element);
	} else {
		if (GST_FLAG_IS_SET(element, GST_ALSA_OPEN) == FALSE)
			if (_gst_alsa_open_audio((GstALSA *)element) == FALSE)
				return GST_STATE_FAILURE;
	}

	if (GST_ELEMENT_CLASS(parent_class)->change_state)
		return GST_ELEMENT_CLASS(parent_class)->change_state(element);

	return GST_STATE_SUCCESS;
}



static void
add_card(gint card, gint dev, const gchar *name)
{
	GType type;
	ALSACard *alsacard;
	GstElementDetails *details;
	GstElementFactory *factory;
	gchar *tmp_name;
  gint j;
  
	GTypeInfo alsa_type_info = {
      sizeof(GstALSAClass),      NULL,
      NULL,
      (GClassInitFunc)gst_alsasink_class_init,
      NULL,
      NULL,
      sizeof(GstALSA),
      0,
      (GInstanceInitFunc)gst_alsasink_init,
	};

	/* 
	 * needs to create a unique name. Since we have multiple soundcards and multiple devices for
	 * each card - this is not so easy. "name_card_dev_ direction" will have to do for now.
	 * My card becomes: cs461x_0_0_src and cs461x_0_0_sink
	 */
	alsa_type_info.type_name = g_strdup_printf("%s_%d_%d_%s", name, card, dev, "sink");
	
	/* replace spaces in the device name 
	 * FIXME may need something better than this
	 */
	for (j=0;j<strlen(alsa_type_info.type_name);j++){
    if (alsa_type_info.type_name[j] == ' ') alsa_type_info.type_name[j] = '_';
  }

	if (gtk_type_from_name(alsa_type_info.type_name)) { /* type already exists */
		g_free(alsa_type_info.type_name);
		return;
	}
  
	type = g_type_register_static(GST_TYPE_ELEMENT, 		NULL, /* name */, , &alsa_type_info, 0);

	/* fill in details */
	details = g_new0(GstElementDetails, 1);
	snd_card_get_longname(card, &tmp_name);
	details->longname = g_strdup(tmp_name);
	free(tmp_name); /* g_xxx may have it's own structure - do not mix these two! */

	details->klass = g_strdup("Sink/Alsasink");
	details->description = g_strdup("Plays the audio on a soundcard");
	details->version = g_strdup(GST_ALSA_VERSION);
	details->author = g_strdup("Thomas Nyberg");
	details->copyright = g_strdup("Copyright (C) 2000 CodeFactory AB");

	factory = gst_elementfactory_new(alsa_type_info.type_name, type, details);
	g_return_if_fail(factory != NULL);

	ALSA_DEBUG_PRINT("Adding '%s' to plugin\n", alsa_type_info.type_name);
	gst_plugin_add_factory(alsa_sink_plugin, factory);


	/* insert card in hash */
	alsacard = g_new0(ALSACard, 1);
	alsacard->card = card;
	alsacard->device = dev;
	alsacard->subdevice = 0; /* not used */
	alsacard->mode = SND_PCM_OPEN_PLAYBACK;
	alsacard->channel = SND_PCM_CHANNEL_PLAYBACK;

	if (alsa_cards == NULL)
		alsa_cards = g_hash_table_new(g_direct_hash, g_direct_equal);

	g_hash_table_insert(alsa_cards, GINT_TO_POINTER(type), alsacard);

	g_free (alsa_type_info.type_name);

	/* check to see if this is the default-card */
	if (card == snd_defaults_pcm_card() && dev == snd_defaults_pcm_device()) {
		/*
		 * Create a dummy-factory, this makes it easier for quick creation of
		 * an alsasink. Instead of knowing name of soundcard and such you can
		 * use "alsasink".
		 */
		ALSA_DEBUG_PRINT("Adding default-card(%d/%d)\n", card, dev);
		if ((factory = gst_elementfactory_new("alsasink", type, details)) != NULL)
			gst_plugin_add_factory(alsa_sink_plugin, factory);
	}
}


static gboolean
plugin_init (GModule *module, GstPlugin *plugin)
{
	ALSA_DEBUG_MSG("Loading alsasink-plugin");

	alsa_sink_plugin = plugin;
	_gst_alsa_probe_cards(add_card, NULL);

	return TRUE;
}

GstPluginDesc plugin_desc = {
	GST_VERSION_MAJOR,
	GST_VERSION_MINOR,
	"alsasink",
	plugin_init
};

