/* Gnome-Streamer
 * Copyright (C) <2001> Steve Baker <stevebaker_org@yahoo.co.uk>
 *
 * int2float.c
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <int2float.h>


static GstElementDetails int2float_details = {
  "Integer to Float effect",
  "Filter/Effect",
  "Convert from integer to floating point audio data",
  VERSION,
  "Steve Baker <stevebaker_org@yahoo.co.uk>",
  "(C) 2001",
};

/* Stereo signals and args */
enum {
  /* FILL ME */
  LAST_SIGNAL
};

enum {
  ARG_0,
  ARG_PAN,
};

static GstPadTemplate* 
int2float_src_factory (void)
{
  return 
    gst_padtemplate_new (
    "src%d",
    GST_PAD_SRC,
    GST_PAD_REQUEST,
    gst_caps_new (
    "int2float_src",
      "audio/raw",
      gst_props_new (
        "format",     GST_PROPS_STRING ("float"),
        "layout",     GST_PROPS_STRING ("gfloat"),
        "intercept",  GST_PROPS_FLOAT(0.0),
        "slope",      GST_PROPS_FLOAT(1.0),
        "channels",   GST_PROPS_INT (1),
      NULL)),
    NULL);
}

static GstPadTemplate* 
int2float_sink_factory (void)
{
  return 
    gst_padtemplate_new (
    "sink",
    GST_PAD_SINK,
    GST_PAD_ALWAYS,
    gst_caps_new (
    "int2float_sink",
      "audio/raw",
      gst_props_new (
        "format",   GST_PROPS_STRING ("int"),
        "law",     GST_PROPS_INT (0),
        "endianness",     GST_PROPS_INT (G_BYTE_ORDER),
        "signed",   GST_PROPS_BOOLEAN (TRUE),
        "width", GST_PROPS_INT (16),
        "depth", GST_PROPS_INT (16),
      NULL)),
    NULL);
}

static GstPadTemplate *srctempl, *sinktempl;

static void	gst_int2float_class_init		(GstInt2FloatClass *klass);
static void	gst_int2float_init			(GstInt2Float *plugin);
static void	gst_int2float_set_property		(GObject *object, guint prop_id, const GValue *value, GParamSpec *pspec);
static void	gst_int2float_get_property		(GObject *object, guint prop_id, GValue *value, GParamSpec *pspec);
static GstPad*  gst_int2float_request_new_pad (GstElement *element, GstPadTemplate *temp);
static void	gst_int2float_chain_gint16  (GstPad *pad, GstBuffer *buf_in);
static inline GstInt2Float* gst_int2float_get_plugin(GstPad *pad,GstBuffer *buf);

static GstElementClass *parent_class = NULL;

static GstPadNegotiateReturn
gst_int2float_negotiate_sink (GstPad *pad, GstCaps **caps, gpointer *data)
{
  GstInt2Float* plugin = GST_INT2FLOAT (GST_OBJECT_PARENT (pad));
  GstCaps* peercaps;
  GstRealPad *realpeer;
  GstPadNegotiateReturn proxyresult;
  GSList *srcpads;
  GstPad *srcpad;
  
  g_print("nego int2float sink\n");
  if (*caps== NULL) return GST_PAD_NEGOTIATE_FAIL;
  
  plugin->width = gst_caps_get_int (*caps, "width");
  plugin->depth = gst_caps_get_int (*caps, "depth");
  plugin->channels = gst_caps_get_int (*caps, "channels");
  
  if (plugin->trying_proxy_caps) return GST_PAD_NEGOTIATE_AGREE;

  plugin->trying_proxy_caps=TRUE;
  
  peercaps=gst_caps_new (
    "int2float_sink_caps",
    "audio/raw",
    gst_props_new (
      "format",     GST_PROPS_STRING ("float"),
      "layout",     GST_PROPS_STRING ("gfloat"),
      "intercept",  GST_PROPS_FLOAT(0.0),
      "slope",      GST_PROPS_FLOAT(1.0),
      "rate",       GST_PROPS_INT (gst_caps_get_int (*caps, "rate")),
      "channels",   GST_PROPS_INT (1),
      NULL
    )
  );
  
  srcpads = plugin->srcpads;
  while(srcpads){
    srcpad = GST_INT2FLOAT_SRCPAD(srcpads);
    realpeer = GST_RPAD_PEER(srcpad);
    proxyresult = gst_pad_negotiate_proxy(&(realpeer->pad), srcpad, &peercaps);
    if (proxyresult != GST_PAD_NEGOTIATE_AGREE){
      plugin->trying_proxy_caps=FALSE;
      return GST_PAD_NEGOTIATE_FAIL;
    }
    srcpads = g_slist_next(srcpads);
  }
  
  plugin->trying_proxy_caps=FALSE;
  return GST_PAD_NEGOTIATE_AGREE;
}

static GstPadNegotiateReturn
gst_int2float_negotiate_src (GstPad *pad, GstCaps **caps, gpointer *data)
{
  GstInt2Float* plugin = GST_INT2FLOAT (GST_OBJECT_PARENT (pad));
  GstCaps* peercaps;
  GstRealPad *realpeer;
  GstPadNegotiateReturn proxyresult;

  g_print("nego int2float src\n");
  if (*caps== NULL) return GST_PAD_NEGOTIATE_FAIL;

  if (plugin->trying_proxy_caps) return GST_PAD_NEGOTIATE_AGREE;

  plugin->trying_proxy_caps=TRUE;
  peercaps=gst_caps_new (
    "int2float_src_caps",
    "audio/raw",
    gst_props_new (
      "format",     GST_PROPS_STRING ("int"),
      "law",        GST_PROPS_INT (0),
      "endianness", GST_PROPS_INT (G_BYTE_ORDER),
      "signed",     GST_PROPS_BOOLEAN (TRUE),
      "width",      GST_PROPS_INT (plugin->width),
      "depth",      GST_PROPS_INT (plugin->depth),
      "rate",       GST_PROPS_INT (gst_caps_get_int (*caps, "rate")),
      "channels",   GST_PROPS_INT (1),
      NULL
    )
  );
  realpeer = GST_RPAD_PEER(plugin->sinkpad);
  proxyresult = gst_pad_negotiate_proxy(&(realpeer->pad), plugin->sinkpad, &peercaps);
  plugin->trying_proxy_caps=FALSE;
  return proxyresult;
}

GType
gst_int2float_get_type(void) {
  static GType int2float_type = 0;

  if (!int2float_type) {
    static const GTypeInfo int2float_info = {
      sizeof(GstInt2FloatClass),      NULL,
      NULL,
      (GClassInitFunc)gst_int2float_class_init,
      NULL,
      NULL,
      sizeof(GstInt2Float),
      0,
      (GInstanceInitFunc)gst_int2float_init,
    };
    int2float_type = g_type_register_static(GST_TYPE_ELEMENT, "GstInt2Float", &int2float_info, 0);
  }
  return int2float_type;
}

static void
gst_int2float_class_init (GstInt2FloatClass *klass)
{
  GObjectClass *gobject_class;
  GstElementClass *gstelement_class;

  gobject_class = (GObjectClass*)klass;
  gstelement_class = (GstElementClass*)klass;

  parent_class = g_type_class_ref(GST_TYPE_ELEMENT);

  gobject_class->set_property = gst_int2float_set_property;
  gobject_class->get_property = gst_int2float_get_property;
  gstelement_class->request_new_pad = gst_int2float_request_new_pad;
}

static void
gst_int2float_init (GstInt2Float *plugin)
{
  plugin->sinkpad = gst_pad_new_from_template(sinktempl,"sink");

  gst_element_add_pad(GST_ELEMENT(plugin),plugin->sinkpad);
  gst_pad_set_chain_function(plugin->sinkpad,gst_int2float_chain_gint16);
  
  gst_pad_set_negotiate_function(plugin->sinkpad, gst_int2float_negotiate_sink);

  plugin->numsrcpads = 0;
  plugin->srcpads = NULL;
  
  plugin->channels=1;
  plugin->width=16;
  plugin->depth=16;
  plugin->intercept=0.0;
  plugin->slope=1.0;
  plugin->trying_proxy_caps=FALSE;
}

static GstPad*
gst_int2float_request_new_pad (GstElement *element, GstPadTemplate *templ) 
{
  gchar *name;
  GstPad *srcpad;
  GstInt2Float *plugin;

  plugin = GST_INT2FLOAT(element);
  
  g_return_val_if_fail(plugin != NULL, NULL);
  g_return_val_if_fail(GST_IS_INT2FLOAT(plugin), NULL);

  if (templ->direction != GST_PAD_SRC) {
    g_warning ("int2float: request new pad that is not a SRC pad\n");
    return NULL;
  }

  name = g_strdup_printf ("src%d",plugin->numsrcpads);
  g_print("int2float adding pad %s\n", name);
  srcpad = gst_pad_new_from_template (templ, name);
  gst_element_add_pad (GST_ELEMENT (plugin), srcpad);
  gst_pad_set_negotiate_function(srcpad, gst_int2float_negotiate_src);
  
  plugin->srcpads = g_slist_append (plugin->srcpads, srcpad);
  plugin->numsrcpads++;
  
  return srcpad;
}

static void
gst_int2float_chain_gint16 (GstPad *pad,GstBuffer *buf_in)
{
  GstInt2Float *plugin;
  gint16 *data_in;
  gfloat *data_out;
  gint num_frames, i, j;
  GSList *srcpads;
  GstBuffer **buffers;
  guint buffer_byte_size;
  GstBufferPool *pool;
  
  g_return_if_fail(plugin = gst_int2float_get_plugin(pad,buf_in));

  num_frames = GST_BUFFER_SIZE(buf_in)/(2*plugin->channels);
  data_in=(gint16 *)GST_BUFFER_DATA(buf_in);
  buffers = g_new0(GstBuffer*,plugin->channels);
  buffer_byte_size = sizeof(gfloat) * num_frames;
  pool = gst_buffer_pool_get_default (NULL, buffer_byte_size, 4);
  
  for (i = 0; (i < plugin->channels) && (i < plugin->numsrcpads); i++) {

    buffers[i] = gst_buffer_new_from_pool(pool);
    
    data_out = (gfloat*)GST_BUFFER_DATA(buffers[i]);
    GST_BUFFER_SIZE(buffers[i]) = buffer_byte_size;
    
    for (j = 0; j < num_frames; j++) {
      data_out[j] = ((gfloat)data_in[(j*plugin->channels) + i]) / 32767.0;
    }
  }

  gst_buffer_unref(buf_in);
  srcpads=plugin->srcpads;
   
  for (i = 0; (i < plugin->channels) && (srcpads != NULL) ; i++) {
    // g_print("pushing to %s:%s\n", GST_DEBUG_PAD_NAME(GST_INT2FLOAT_SRCPAD(srcpads)));

    gst_pad_push(GST_INT2FLOAT_SRCPAD(srcpads),buffers[i]);
    srcpads = g_slist_next(srcpads);
  }
  g_free(buffers);
}

static inline GstInt2Float* 
gst_int2float_get_plugin(GstPad *pad,GstBuffer *buf)
{
  GstInt2Float *plugin;
  g_return_val_if_fail(pad != NULL, NULL);
  g_return_val_if_fail(GST_IS_PAD(pad), NULL);
  g_return_val_if_fail(buf != NULL, NULL);

  plugin = GST_INT2FLOAT(GST_OBJECT_PARENT (pad));
  g_return_val_if_fail(plugin != NULL, NULL);
  g_return_val_if_fail(GST_IS_INT2FLOAT(plugin), NULL);
  return plugin;
}

static void
gst_int2float_set_property (GObject *object, guint prop_id, const GValue *value, GParamSpec *pspec)
{
  GstInt2Float *plugin;

  /* it's not null if we got it, but it might not be ours */
  g_return_if_fail(GST_IS_INT2FLOAT(object));
  plugin = GST_INT2FLOAT(object);

  switch (prop_id) {
    default:
      break;
  }
}

static void
gst_int2float_get_property (GObject *object, guint prop_id, GValue *value, GParamSpec *pspec)
{
  GstInt2Float *plugin;

  /* it's not null if we got it, but it might not be ours */
  g_return_if_fail(GST_IS_INT2FLOAT(object));
  plugin = GST_INT2FLOAT(object);

  switch (prop_id) {
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

gboolean 
gst_int2float_factory_init (GstPlugin *plugin) 
{
  GstElementFactory *factory;
  
  factory = gst_elementfactory_new("int2float",GST_TYPE_INT2FLOAT,
                                   &int2float_details);
  g_return_val_if_fail(factory != NULL, FALSE);
  
  sinktempl = int2float_sink_factory();
  gst_elementfactory_add_padtemplate (factory, sinktempl);

  srctempl = int2float_src_factory();
  gst_elementfactory_add_padtemplate (factory, srctempl);
  
  gst_plugin_add_factory(plugin,factory);
  
  return TRUE;
}
