/* Gnome-Streamer
 * Copyright (C) <2001> Steve Baker <stevebaker_org@yahoo.co.uk>
 *
 * mono2stereo.c
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <mono2stereo.h>


static GstElementDetails mono2stereo_details = {
  "Mono to Stereo effect",
  "Filter/Effect",
  "Take a single channel and send it over 2 channels with pan",
  VERSION,
  "Steve Baker <stevebaker_org@yahoo.co.uk>",
  "(C) 2001",
};


/* Stereo signals and args */
enum {
  /* FILL ME */
  LAST_SIGNAL
};

enum {
  ARG_0,
  ARG_PAN,
};

static GstPadTemplate* 
mono2stereo_src_factory (void)
{
  return 
    gst_padtemplate_new (
    "src",
    GST_PAD_SRC,
    GST_PAD_ALWAYS,
    gst_caps_new (
    "sinesrc_src",
      "audio/raw",
      gst_props_new (
        "format",   GST_PROPS_STRING ("int"),
        "channels", GST_PROPS_INT (2),
      NULL)),
    NULL);
}

static GstPadTemplate* 
mono2stereo_sink_factory (void)
{
  return 
    gst_padtemplate_new (
      "sink",
      GST_PAD_SINK,
      GST_PAD_ALWAYS,
      gst_caps_new (
      "sinesrc_sink",
        "audio/raw",
        gst_props_new (
          "format",   GST_PROPS_STRING ("int"),
          "channels", GST_PROPS_INT (1),
        NULL)),
      NULL);
}

static GstPadTemplate *srctempl, *sinktempl;

static void	gst_mono2stereo_class_init		(GstMono2StereoClass *klass);
static void	gst_mono2stereo_init			(GstMono2Stereo *plugin);
static inline void gst_mono2stereo_update_pan(GstMono2Stereo *plugin);
static void	gst_mono2stereo_set_property		(GObject *object, guint prop_id, const GValue *value, GParamSpec *pspec);
static void	gst_mono2stereo_get_property		(GObject *object, guint prop_id, GValue *value, GParamSpec *pspec);

static void	gst_mono2stereo_chain		(GstPad *pad, GstBuffer *buf);

static GstElementClass *parent_class = NULL;

static GstPadNegotiateReturn
gst_mono2stereo_negotiate_src (GstPad *pad, GstCaps **caps, gpointer *data)
{
  GstMono2Stereo* plugin=GST_MONO2STEREO (GST_OBJECT_PARENT (pad));
  if (*caps==NULL) return GST_PAD_NEGOTIATE_FAIL;
  *caps = gst_caps_copy_on_write (*caps);
  gst_caps_set(*caps,"channels",GST_PROPS_INT(1));
  plugin->width=gst_caps_get_int (*caps, "width");
  return gst_pad_negotiate_proxy(pad,plugin->sinkpad,caps);
}

static GstPadNegotiateReturn
gst_mono2stereo_negotiate_sink (GstPad *pad, GstCaps **caps, gpointer *data)
{
  GstMono2Stereo* plugin=GST_MONO2STEREO (GST_OBJECT_PARENT (pad));
  if (*caps==NULL) return GST_PAD_NEGOTIATE_FAIL;
  *caps = gst_caps_copy_on_write (*caps);
  gst_caps_set(*caps,"channels",GST_PROPS_INT(2));
  plugin->width=gst_caps_get_int (*caps, "width");
  return gst_pad_negotiate_proxy(pad,plugin->srcpad,caps);
}	

GType
gst_mono2stereo_get_type(void) {
  static GType mono2stereo_type = 0;

  if (!mono2stereo_type) {
    static const GTypeInfo mono2stereo_info = {
      sizeof(GstMono2StereoClass),      NULL,
      NULL,
      (GClassInitFunc)gst_mono2stereo_class_init,
      NULL,
      NULL,
      sizeof(GstMono2Stereo),
      0,
      (GInstanceInitFunc)gst_mono2stereo_init,
    };
    mono2stereo_type = g_type_register_static(GST_TYPE_ELEMENT, "GstMono2Stereo", &mono2stereo_info, 0);
  }
  return mono2stereo_type;
}

static void
gst_mono2stereo_class_init (GstMono2StereoClass *klass)
{
  GObjectClass *gobject_class;
  GstElementClass *gstelement_class;

  gobject_class = (GObjectClass*)klass;
  gstelement_class = (GstElementClass*)klass;

  parent_class = g_type_class_ref(GST_TYPE_ELEMENT);

  g_object_class_install_property(G_OBJECT_CLASS(klass), ARG_PAN,
    g_param_spec_float("pan","pan","pan",
                       -1.0,1.0,0.0,G_PARAM_READWRITE)); // CHECKME

  gobject_class->set_property = gst_mono2stereo_set_property;
  gobject_class->get_property = gst_mono2stereo_get_property;
}

static void
gst_mono2stereo_init (GstMono2Stereo *plugin)
{
  plugin->sinkpad = gst_pad_new_from_template(sinktempl,"sink");
  plugin->srcpad = gst_pad_new_from_template(srctempl,"src");

  gst_element_add_pad(GST_ELEMENT(plugin),plugin->sinkpad);
  gst_pad_set_chain_function(plugin->sinkpad,gst_mono2stereo_chain);
  gst_element_add_pad(GST_ELEMENT(plugin),plugin->srcpad);

  gst_pad_set_negotiate_function(plugin->sinkpad, gst_mono2stereo_negotiate_sink);
  gst_pad_set_negotiate_function(plugin->srcpad, gst_mono2stereo_negotiate_src);
  
  plugin->pan=0.0;
  gst_mono2stereo_update_pan(plugin);
}

static inline void gst_mono2stereo_update_pan(GstMono2Stereo *plugin)
{
  plugin->pan_right = (plugin->pan+1.0) / 2.0;
  plugin->pan_left = 1.0 - plugin->pan_right;
  g_print("%f %f %f\n", plugin->pan, plugin->pan_left, plugin->pan_right);
}

static void
gst_mono2stereo_chain (GstPad *pad,GstBuffer *buf_in)
{
  GstMono2Stereo *plugin;
  GstBuffer *buf_out;
  gint16 *data_in, *data_out;
  gint i;
  gint num_frames;
  
  g_return_if_fail(pad != NULL);
  g_return_if_fail(GST_IS_PAD(pad));
  g_return_if_fail(buf_in != NULL);

  plugin = GST_MONO2STEREO(GST_OBJECT_PARENT (pad));
  g_return_if_fail(plugin != NULL);
  g_return_if_fail(GST_IS_MONO2STEREO(plugin));

  data_in = (gint16 *)GST_BUFFER_DATA(buf_in);
  num_frames = GST_BUFFER_SIZE(buf_in)/2;

  buf_out = gst_buffer_new();
  g_return_if_fail (buf_out);
  data_out = g_new(gint16, num_frames * 2);
  GST_BUFFER_DATA(buf_out) = (gpointer) data_out;
  GST_BUFFER_SIZE(buf_out) = num_frames * 4;

  for (i = 0; i < num_frames; i++) {
    data_out[i*2] = data_in[i] * plugin->pan_left;
    data_out[i*2+1] = data_in[i] * plugin->pan_right;
  }
  
  gst_buffer_unref(buf_in);
  gst_pad_push(plugin->srcpad,buf_out);
}

static void
gst_mono2stereo_set_property (GObject *object, guint prop_id, const GValue *value, GParamSpec *pspec)
{
  GstMono2Stereo *plugin;

  /* it's not null if we got it, but it might not be ours */
  g_return_if_fail(GST_IS_MONO2STEREO(object));
  plugin = GST_MONO2STEREO(object);

  switch (prop_id) {
    case ARG_PAN:
      if (g_value_get_double (value) < -1.0){
        plugin->pan = -1.0;
      } else if (g_value_get_double (value) > 1.0){
        plugin->pan = 1.0;
      } else {
        plugin->pan = g_value_get_float (value);
      }
      gst_mono2stereo_update_pan(plugin);
      break;
    default:
      break;
  }
}

static void
gst_mono2stereo_get_property (GObject *object, guint prop_id, GValue *value, GParamSpec *pspec)
{
  GstMono2Stereo *plugin;

  /* it's not null if we got it, but it might not be ours */
  g_return_if_fail(GST_IS_MONO2STEREO(object));
  plugin = GST_MONO2STEREO(object);

  switch (prop_id) {
    case ARG_PAN:
      g_value_set_float (value, plugin->pan);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static gboolean
plugin_init (GModule *module, GstPlugin *plugin)
{
  GstElementFactory *factory;

  factory = gst_elementfactory_new("mono2stereo",GST_TYPE_MONO2STEREO,
                                   &mono2stereo_details);
  g_return_val_if_fail(factory != NULL, FALSE);
  
  srctempl = mono2stereo_src_factory();
  gst_elementfactory_add_padtemplate (factory, srctempl);

  sinktempl = mono2stereo_sink_factory();
  gst_elementfactory_add_padtemplate (factory, sinktempl);
  
  gst_plugin_add_factory(plugin,factory);

  return TRUE;
}

GstPluginDesc plugin_desc = {
  GST_VERSION_MAJOR,
  GST_VERSION_MINOR,
  "mono2stereo",
  plugin_init
};
