/* Gnome-Streamer
 * Copyright (C) <1999> Erik Walthinsen <omega@cse.ogi.edu>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

//#define DEBUG_ENABLED
#include <gst/gstclock.h>
#include "gstmpg123.h"


/* elementfactory information */
static GstElementDetails gst_mpg123_details = {
  "mpg123 mp3 decoder",
  "Filter/Decoder/Audio",
  "Uses modified mpg123 code to decode mp3 streams",
  VERSION,
  "Erik Walthinsen <omega@cse.ogi.edu>",
  "(C) 1999",
};


/* Mpg123 signals and args */
enum {
  /* FILL ME */
  LAST_SIGNAL
};

enum {
  ARG_0,
  /* FILL ME */
};

static GstPadTemplate*
mpg123_src_factory (void) 
{
  return 
   gst_padtemplate_new (
  	"src",
  	GST_PAD_SRC,
  	GST_PAD_ALWAYS,
  	gst_caps_new (
  	  "mpg123_src",
    	  "audio/raw",
	  gst_props_new (
    	    "format",   GST_PROPS_STRING ("int"),
      	     "law",         GST_PROPS_INT (0),
      	     "endianness",  GST_PROPS_INT (G_BYTE_ORDER),
      	     "signed",      GST_PROPS_BOOLEAN (TRUE),
      	     "width",       GST_PROPS_INT (16),
      	     "depth",       GST_PROPS_INT (16),
      	     "rate",        GST_PROPS_INT_RANGE (11025, 48000),
      	     "channels",    GST_PROPS_LIST (
                  		GST_PROPS_INT (1),
                  		GST_PROPS_INT (2)
	        	    ),
	     NULL)),
	NULL);
};

static GstPadTemplate*
mpg123_sink_factory (void) 
{
  return 
   gst_padtemplate_new (
  	"sink",
  	GST_PAD_SINK,
  	GST_PAD_ALWAYS,
  	gst_caps_new (
  	  "mpg123_sink",
    	  "audio/mp3",
	  gst_props_new (
    	    "layer",   GST_PROPS_INT_RANGE (1, 3),
            "bitrate", GST_PROPS_INT_RANGE (8, 320),
    	    "framed",  GST_PROPS_BOOLEAN (TRUE),
	    NULL)),
	NULL);
};

static GstPadTemplate *sink_temp, *src_temp;

static void gst_mpg123_class_init	(GstMpg123Class *klass);
static void gst_mpg123_init		(GstMpg123 *mpg123);

static void gst_mpg123_chain		(GstPad *pad, GstBuffer *buf);

static GstElementClass *parent_class = NULL;
//static guint gst_mpg123_signals[LAST_SIGNAL] = { 0 };

GType
gst_mpg123_get_type (void)
{
  static GType mpg123_type = 0;

  if (!mpg123_type) {
    static const GTypeInfo mpg123_info = {
      sizeof(GstMpg123Class),      NULL,
      NULL,
      (GClassInitFunc)gst_mpg123_class_init,
      NULL,
      NULL,
      sizeof(GstMpg123),
      0,
      (GInstanceInitFunc)gst_mpg123_init,
    };
    mpg123_type = g_type_register_static(GST_TYPE_ELEMENT, "GstMpg123", &mpg123_info, 0);
  }
  return mpg123_type;
}

static void
gst_mpg123_class_init (GstMpg123Class *klass)
{
  GstElementClass *gstelement_class;

  gstelement_class = (GstElementClass*)klass;

  parent_class = g_type_class_ref(GST_TYPE_ELEMENT);
}

static void
gst_mpg123_init (GstMpg123 *mpg123)
{
  /* create the sink and src pads */
  mpg123->sinkpad = gst_pad_new_from_template(sink_temp, "sink");
  gst_element_add_pad(GST_ELEMENT(mpg123),mpg123->sinkpad);
  gst_pad_set_chain_function(mpg123->sinkpad,gst_mpg123_chain);

  mpg123->srcpad = gst_pad_new_from_template(src_temp, "src");
  gst_element_add_pad(GST_ELEMENT(mpg123),mpg123->srcpad);

  /* initialize the mpg123 decoder state */
  mpg123->decoder = mpg123_init_decoder();
  mpg123->in_flush = FALSE;
}

static void
gst_mpg123_chain (GstPad *pad,GstBuffer *buf)
{
  GstMpg123 *mpg123;
  guchar *data;
  gulong size;
  gulong header;
  GstBuffer *outbuf;

  g_return_if_fail(pad != NULL);
  g_return_if_fail(GST_IS_PAD(pad));
  g_return_if_fail(buf != NULL);
//  g_return_if_fail(GST_IS_BUFFER(buf));

  mpg123 = GST_MPG123 (gst_pad_get_parent (pad));

  if (GST_BUFFER_FLAG_IS_SET(buf, GST_BUFFER_FLUSH)) {
    //mpg123->next_time = -gst_clock_current_diff(gst_clock_get_system(), 0);
    mpg123->in_flush = TRUE;
    GST_DEBUG (0,"mpg123: flush\n");
  }

  data = (guchar *)GST_BUFFER_DATA(buf);
  size = GST_BUFFER_SIZE(buf);
  GST_DEBUG (0,"gst_mpg123_chain: got buffer of %ld bytes in '%s'\n",size,
          gst_element_get_name (GST_ELEMENT (mpg123)));

  //gst_util_dump_mem(data, size);

  //g_print("about to decode a frame\n");
  header = GULONG_TO_BE(*(gulong *)data);
  outbuf = gst_buffer_new();
  GST_BUFFER_DATA(outbuf) = g_malloc(1152*4);
  GST_BUFFER_SIZE(outbuf) = 1152*4;
  // if the audio parameters changed
  if (mpg123_decode_frame(mpg123->decoder,data,size,
             GST_BUFFER_DATA(outbuf),(long *)&GST_BUFFER_SIZE(outbuf))) {

    GstCaps *caps;

    caps = gst_caps_new (
		    "mpg123_src_caps",
		    "audio/raw",
		     gst_props_new (
		           "format", 		GST_PROPS_STRING ("int"),
      			     "law",     	GST_PROPS_INT (0),
      			     "endianness",     	GST_PROPS_INT (G_BYTE_ORDER),
      			     "signed",   	GST_PROPS_BOOLEAN (TRUE),
      			     "width",   	GST_PROPS_INT (16),
			     "depth", 		GST_PROPS_INT (16),
			     "rate", 		GST_PROPS_INT (mpg123->decoder->samplerate),
			     "channels", 	GST_PROPS_INT (mpg123->decoder->channels),
			     NULL
			     )
		    );

    gst_pad_set_caps (mpg123->srcpad, caps);

    GST_DEBUG (0,"mpg123: setting interval \n");
  }

  mpg123->time_interval = ((1000000LL*GST_BUFFER_SIZE(outbuf))/(mpg123->decoder->channels*
			         2*(mpg123->decoder->samplerate)));

  if (mpg123->in_flush) {
    GST_BUFFER_FLAG_SET(outbuf, GST_BUFFER_FLUSH);
    GST_DEBUG (0,"mpg123: flush \n");
    mpg123->in_flush = FALSE;
  }
  GST_BUFFER_TIMESTAMP(outbuf) = GST_BUFFER_TIMESTAMP(buf);

  GST_DEBUG (0,"mpg123: pushing %d bytes %llu %llu\n", GST_BUFFER_SIZE(outbuf), GST_BUFFER_TIMESTAMP(outbuf),
		  mpg123->time_interval);

  if (GST_PAD_CONNECTED(mpg123->srcpad))
    gst_pad_push(mpg123->srcpad,outbuf);
  else
    gst_buffer_unref(outbuf);

  mpg123->next_time += mpg123->time_interval;

  gst_buffer_unref(buf);
}


static gboolean
plugin_init (GModule *module, GstPlugin *plugin)
{
  GstElementFactory *factory;

  if (!gst_library_load("gstgetbits")) {
    gst_info("mpg123:: could not load support library: 'gstgetbits'\n");
    return FALSE;
  }

  /* create an elementfactory for the mpg123 element */
  factory = gst_elementfactory_new("mpg123",GST_TYPE_MPG123,
                                   &gst_mpg123_details);
  g_return_val_if_fail(factory != NULL, FALSE); 

  sink_temp = mpg123_sink_factory ();
  gst_elementfactory_add_padtemplate (factory, sink_temp);

  src_temp = mpg123_src_factory ();
  gst_elementfactory_add_padtemplate (factory, src_temp);

  gst_plugin_add_factory(plugin,factory);

  return TRUE;
}

GstPluginDesc plugin_desc = {
  GST_VERSION_MAJOR,
  GST_VERSION_MINOR,
  "mpg123",
  plugin_init
};
