/*
 * util.c --
 *
 *      Miscellaneous utility procedures.
 *
 */

/*
 * Copyright (c) 1995 The Regents of the University of California.
 * All rights reserved.
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose, without fee, and without written agreement is
 * hereby granted, provided that the above copyright notice and the following
 * two paragraphs appear in all copies of this software.
 * 
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY FOR
 * DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
 * OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF THE UNIVERSITY OF
 * CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND THE UNIVERSITY OF CALIFORNIA HAS NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */

/*
 * Portions of this software Copyright (c) 1995 Brown University.
 * All rights reserved.
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose, without fee, and without written agreement
 * is hereby granted, provided that the above copyright notice and the
 * following two paragraphs appear in all copies of this software.
 * 
 * IN NO EVENT SHALL BROWN UNIVERSITY BE LIABLE TO ANY PARTY FOR
 * DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
 * OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF BROWN
 * UNIVERSITY HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * BROWN UNIVERSITY SPECIFICALLY DISCLAIMS ANY WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 * PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS ON AN "AS IS"
 * BASIS, AND BROWN UNIVERSITY HAS NO OBLIGATION TO PROVIDE MAINTENANCE,
 * SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */

#include <stdlib.h>
#include <libs/getbits/gstgetbits.h>
#include "video.h"
#include "proto.h"
#include "util.h"

//#define DEBUG_ENABLED
#undef DEBUG

/* debugging */
#ifndef DEBUG
#ifdef DEBUG_ENABLED
#define DEBUG(format,args...) g_print("DEBUG: " format, ##args)
#else
#define DEBUG(format,args...)
#endif
#endif


/*
   Changes to make the code reentrant:
     de-globalized: totNumFrames, realTimeStart, vid_stream, sys_layer,
        bitOffset, bitLength, bitBuffer, curVidStream
     setjmp/longjmp replaced

   Additional changes:
     only call DestroyVidStream up in mpegVidRsrc, not in correct_underflow

   -lsh@cs.brown.edu (Loring Holden)
 */

/*
 *--------------------------------------------------------------
 *
 * next_bits --
 *
 *	Compares next num bits to low order position in mask.
 *      Buffer pointer is NOT advanced.
 *
 * Results:
 *	TRUE, FALSE, or error code.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

int next_bits(num, mask, vid_stream)
int num;
unsigned int mask;
mpeg_play_VidStream *vid_stream;
{
  unsigned int stream;
  int ret_value;

  /* Get next num bits, no buffer pointer advance. */

  stream = gst_showbitsn(&vid_stream->gb, num);

  /* Compare bit stream and mask. Set return value toTRUE if equal, FALSE if
     differs. 
  */

  if (mask == stream) {
    ret_value = TRUE;
  } else ret_value = FALSE;

  /* Return return value. */
  return ret_value;
}


/*
 *--------------------------------------------------------------
 *
 * get_ext_data --
 *
 *	Assumes that bit stream is at begining of extension
 *      data. Parses off extension data into dynamically 
 *      allocated space until start code is hit. 
 *
 * Results:
 *	Pointer to dynamically allocated memory containing
 *      extension data.
 *
 * Side effects:
 *	Bit stream irreversibly parsed.
 *
 *--------------------------------------------------------------
 */

void get_ext_data (vid_stream)
   mpeg_play_VidStream *vid_stream;
{
  unsigned int ext_ID;

  ext_ID = gst_getbits4(&vid_stream->gb);

  GST_DEBUG (0,"mpeg_play:MPEG2 detected, disabling video\n");
  vid_stream->MPEG2 = 1;

  switch(ext_ID) {
    case SEQUENCE_EXTENSION_ID:
      //get_sequence_extension(vid_stream);
      break;
    case SEQUENCE_DISPLAY_EXTENSION_ID:
      //sequence_display_extension(vid_stream);
      break;
    case QUANT_MATRIX_EXTENSION_ID:
      //quant_matrix_extension(vid_stream);
      break;
    case SEQUENCE_SCALABLE_EXTENSION_ID:
      //sequence_scalable_extension(vid_stream);
      break;
    case PICTURE_DISPLAY_EXTENSION_ID:
      //picture_display_extension(vid_stream);
      break;
    case PICTURE_CODING_EXTENSION_ID:
      //get_picture_coding_extension(vid_stream);
      break;
    case PICTURE_SPATIAL_SCALABLE_EXTENSION_ID:
      //picture_spatial_scalable_extension(vid_stream);
      break;
    case PICTURE_TEMPORAL_SCALABLE_EXTENSION_ID:
      //picture_temporal_scalable_extension(vid_stream);
      break;
    case COPYRIGHT_EXTENSION_ID:
      //copyright_extension(vid_stream);
      break;
    default:
      fprintf(stderr,"reserved extension start code ID %d\n",ext_ID);
      break;
  }
  return; 
}

/*
 *--------------------------------------------------------------
 *
 * get_user_data --
 *
 *	Assumes that bit stream is at begining of extension
 *      data. Parses off extension data into dynamically 
 *      allocated space until start code is hit. 
 *
 * Results:
 *	Pointer to dynamically allocated memory containing
 *      extension data.
 *
 * Side effects:
 *	Bit stream irreversibly parsed.
 *
 *--------------------------------------------------------------
 */

char *get_user_data (vid_stream)
   mpeg_play_VidStream *vid_stream;
{
  unsigned int size, marker;
  char *dataPtr;
  unsigned int data;


  /* Set initial ext data buffer size. */
  size = USER_BUF_SIZE;

  /* Allocate ext data buffer. */
  dataPtr = (char *) malloc(size);

  /* Initialize marker to keep place in ext data buffer. */
  marker = 0;

  /* While next data is not start code... */
  while (!next_bits(24, 0x000001, vid_stream)) {

    /* Get next byte of ext data. */
    data = gst_getbyte(&vid_stream->gb);

    /* Put ext data into ext data buffer. Advance marker. */
    dataPtr[marker] = (char) data;
    marker++;

    /* If end of ext data buffer reached, resize data buffer. */
    if (marker == size) {
      size += USER_BUF_SIZE;
      dataPtr = (char *) realloc(dataPtr, size);
    }
  }

  /* Realloc data buffer to free any extra space. */
  dataPtr = (char *) realloc(dataPtr, marker+1);
  dataPtr[marker] = '\0';

  GST_DEBUG (0,"mpeg_play:util: user_data=\"%s\"\n", dataPtr);

  /* Return pointer to ext data buffer. */
  return dataPtr;
}

/*
 *--------------------------------------------------------------
 *
 * next_start_code --
 *
 *	Parses off bitstream until start code reached. When done
 *      next 4 bytes of bitstream will be start code. Bit offset
 *      reset to 0.
 *
 * Results:
 *	Status code.
 *
 * Side effects:
 *	Bit stream irreversibly parsed.
 *
 *--------------------------------------------------------------
 */

int next_start_code(vid_stream, maxdata)
   mpeg_play_VidStream *vid_stream;
   int maxdata;
{
  int state;
  int byteoff;
  unsigned long data;
  gst_getbits_t *gb = &vid_stream->gb;

  GST_DEBUG (0,"next start code %d\n", maxdata);

  /* If bit offset not zero, reset and advance buffer pointer. */

 byteoff = gst_getbits_bitoffset(gb);

  if (byteoff != 0) {
    //fprintf(stderr, "flush %p %d\n", vid_stream->buffer, byteoff);
    gst_flushbitsn(gb, byteoff);
  }

  /* Set state = 0. */

  state = 0;

  /* While buffer has data ...  we append an end code so this will always finish */
  while(maxdata--) {

    /* If next byte is zero... */
    data = gst_getbyte(gb);

    if (data == 0) {
      /* If state < 2, advance state. */
      if (state < 2) state++;
    }
    /* If next byte is one... */
    else if (data == 1) {
      /* If state == 2, advance state (i.e. start code found). */
      if (state == 2) state++;
      /* Otherwise, reset state to zero. */
      else state = 0;
    }
    /* Otherwise byte is neither 1 or 0, reset state to 0. */
    else {
      state = 0;
    }

    /* If state == 3 (i.e. start code found)... */
    if (state == 3) {
       //gst_showbitsn(gb, 8, data);
       //fprintf(stderr, "next start code: buffer %02x\n", data);

      /* Set buffer pointer back and reset length & bit offsets so
       * next bytes will be beginning of start code. 
       */
      gst_backbits24(gb);

      /* Return success. */
      return OK;
    }
  }

  /* Return underflow error. */
  return STREAM_UNDERFLOW;
}


/*
 *--------------------------------------------------------------
 *
 * get_extra_bit_info --
 *
 *	Parses off extra bit info stream into dynamically 
 *      allocated memory. Extra bit info is indicated by
 *      a flag bit set to 1, followed by 8 bits of data.
 *      This continues until the flag bit is zero. Assumes
 *      that bit stream set to first flag bit in extra
 *      bit info stream.
 *
 * Results:
 *	Pointer to dynamically allocated memory with extra
 *      bit info in it. Flag bits are NOT included.
 *
 * Side effects:
 *	Bit stream irreversibly parsed.
 *
 *--------------------------------------------------------------
 */

char *get_extra_bit_info(vid_stream)
mpeg_play_VidStream *vid_stream;
{
  unsigned int size, marker;
  char *dataPtr;
  unsigned int data;
  gst_getbits_t *gb = &vid_stream->gb;

  /* Get first flag bit. */
  data = gst_getbits1(gb);

  /* If flag is false, return NULL pointer (i.e. no extra bit info). */

  if (!data) return NULL;

  /* Initialize size of extra bit info buffer and allocate. */

  size = EXT_BUF_SIZE;
  dataPtr = (char *) malloc(size);

  /* Reset marker to hold place in buffer. */

  marker = 0;

  /* While flag bit is true. */

  while (data) {

    /* Get next 8 bits of data. */
    data = gst_getbits8(gb);

    /* Place in extra bit info buffer. */

    dataPtr[marker] = (char) data;
    marker++;

    /* If buffer is full, reallocate. */

    if (marker == size) {
      size += EXT_BUF_SIZE;
      dataPtr = (char *) realloc(dataPtr, size);
    }

    /* Get next flag bit. */
    data = gst_getbits1(gb);
  }

  /* Reallocate buffer to free extra space. */

  dataPtr = (char *) realloc(dataPtr, marker);

  /* Return pointer to extra bit info buffer. */
  return dataPtr;
}

