/* Gnome-Streamer
 * Copyright (C) <1999> Erik Walthinsen <omega@cse.ogi.edu>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

//#define GST_DEBUG_ENABLED
#include <mpeg1parse.h>


/* elementfactory information */
static GstElementDetails mpeg1parse_details = {
  "MPEG1 Parser",
  "Filter/Parser/System",
  "Demultiplexes MPEG-1 System Streams",
  VERSION,
  "Erik Walthinsen <omega@cse.ogi.edu>" "\n"
  "Wim Taymans <wim.taymans@tvd.be>",
  "(C) 1999",
};

/* Mpeg1Parse signals and args */
enum {
  /* FILL ME */
  LAST_SIGNAL
};

enum {
  ARG_0,
  ARG_BIT_RATE,
  ARG_SYNC,
  /* FILL ME */
};

static GstPadTemplate*
sink_factory (void)
{
  return 
    gst_padtemplate_new (
  	"sink",
  	GST_PAD_SINK,
  	GST_PAD_ALWAYS,
  	gst_caps_new (
  	  "mpeg1parse_sink",
    	  "video/mpeg",
	  gst_props_new (
    	    "mpegversion",  GST_PROPS_INT (1),
    	    "systemstream", GST_PROPS_BOOLEAN (TRUE),
	    NULL)),
	NULL);
}

static GstPadTemplate*
audio_factory (void)
{
  return 
    gst_padtemplate_new (
  	"audio_[1-32]",
  	GST_PAD_SRC,
  	GST_PAD_SOMETIMES,
  	gst_caps_new (
  	  "mpeg1parse_audio",
    	  "audio/mp3",
	  NULL),
	NULL);
}

static GstPadTemplate*
video_factory (void)
{
  return 
    gst_padtemplate_new (
  	"video_[1-16]",
  	GST_PAD_SRC,
  	GST_PAD_SOMETIMES,
  	gst_caps_new (
  	  "mpeg1parse_video",
     	  "video/mpeg",
	  gst_props_new (
    	    "mpegversion",  GST_PROPS_INT (1),
    	    "systemstream",  GST_PROPS_BOOLEAN (FALSE),
	    NULL)),
	NULL);
}

static GstPadTemplate*
private1_factory (void)
{
  return 
    gst_padtemplate_new (
  	"private_stream_1_[1-8]",
  	GST_PAD_SRC,
  	GST_PAD_SOMETIMES,
  	gst_caps_new (
  	  "mpeg1parse_private1",
    	  "audio/ac3",
	  NULL),
	NULL);
}

static GstPadTemplate*
private2_factory (void)
{
  return 
    gst_padtemplate_new (
  	"private_stream_2",
  	GST_PAD_SRC,
  	GST_PAD_SOMETIMES,
  	gst_caps_new (
  	  "mpeg1parse_private2",
    	  "unknown/unknown",
	  NULL),
	NULL);
}


static void			gst_mpeg1parse_class_init	(Mpeg1ParseClass *klass);
static void			gst_mpeg1parse_init		(Mpeg1Parse *mpeg1parse);

static void			gst_mpeg1parse_chain		(GstPad *pad, GstBuffer *buf);

static void			gst_mpeg1parse_set_property	(GObject *object, guint prop_id, const GValue *value, GParamSpec *pspec);
static void			gst_mpeg1parse_get_property	(GObject *object, guint prop_id, GValue *value, GParamSpec *pspec);

static GstElementStateReturn	gst_mpeg1parse_change_state	(GstElement *element);

static GstPadTemplate *sink_template, *audio_template, *video_template, *private1_template, *private2_template;


static GstElementClass *parent_class = NULL;
//static guint gst_mpeg1parse_signals[LAST_SIGNAL] = { 0 };

GType
mpeg1parse_get_type (void)
{
  static GType mpeg1parse_type = 0;

  if (!mpeg1parse_type) {
    static const GTypeInfo mpeg1parse_info = {
      sizeof(Mpeg1ParseClass),      NULL,
      NULL,
      (GClassInitFunc)gst_mpeg1parse_class_init,
      NULL,
      NULL,
      sizeof(Mpeg1Parse),
      0,
      (GInstanceInitFunc)gst_mpeg1parse_init,
    };
    mpeg1parse_type = g_type_register_static(GST_TYPE_ELEMENT, "Mpeg1Parse", &mpeg1parse_info, 0);
  }
  return mpeg1parse_type;
}

static void
gst_mpeg1parse_class_init (Mpeg1ParseClass *klass)
{
  GObjectClass *gobject_class;
  GstElementClass *gstelement_class;

  gobject_class = (GObjectClass*)klass;
  gstelement_class = (GstElementClass*)klass;

  g_object_class_install_property(G_OBJECT_CLASS(klass), ARG_SYNC,
    g_param_spec_boolean("sync","sync","sync",
                         TRUE,G_PARAM_READWRITE)); // CHECKME
  g_object_class_install_property(G_OBJECT_CLASS(klass), ARG_BIT_RATE,
    g_param_spec_int("bit_rate","bit_rate","bit_rate",
                     G_MININT,G_MAXINT,0,G_PARAM_READABLE)); // CHECKME

  gobject_class->set_property = gst_mpeg1parse_set_property;
  gobject_class->get_property = gst_mpeg1parse_get_property;

  gstelement_class->change_state = gst_mpeg1parse_change_state;

  parent_class = g_type_class_ref(GST_TYPE_ELEMENT);
}

static void
gst_mpeg1parse_init (Mpeg1Parse *mpeg1parse)
{
  gint i;

  mpeg1parse->sinkpad = gst_pad_new_from_template (sink_template, "sink");
  gst_element_add_pad(GST_ELEMENT(mpeg1parse),mpeg1parse->sinkpad);
  gst_pad_set_chain_function(mpeg1parse->sinkpad,gst_mpeg1parse_chain);

  // set up getbits
  gst_getbits_init(&mpeg1parse->gb, NULL, NULL);

  // initialize parser state
  mpeg1parse->have_sync = FALSE;
  mpeg1parse->sync_zeros = 0;
  mpeg1parse->id = 0;

  // zero counters (should be done at RUNNING?)
  mpeg1parse->packs = 0;
  mpeg1parse->prevbuf = NULL;
  mpeg1parse->sync = TRUE;
  mpeg1parse->bit_rate = 0;

  mpeg1parse->last_pts = 0;

  for (i=0;i<8;i++) mpeg1parse->private_1_pad[i] = NULL;
  mpeg1parse->private_2_pad = NULL;
  for (i=0;i<16;i++) {
    mpeg1parse->video_pad[i] = NULL;
    mpeg1parse->video_offset[i] = 0;
    mpeg1parse->video_need_flush[i] = FALSE;
  }
  for (i=0;i<32;i++) {
    mpeg1parse->audio_pad[i] = NULL;
    mpeg1parse->audio_offset[i] = 0;
    mpeg1parse->audio_need_flush[i] = FALSE;
  }

  mpeg1parse->clock = gst_clock_get_system();
  gst_clock_register(mpeg1parse->clock, GST_OBJECT(mpeg1parse));
}

static GstCaps*
mpeg1parse_typefind (GstBuffer *buf,gpointer private)
{
  gulong head = GULONG_FROM_BE(*((gulong *)GST_BUFFER_DATA(buf)));

  if (head  != 0x000001ba)
    return NULL;
  if ((*(GST_BUFFER_DATA(buf)+4) & 0xC0) == 0x40)
    return NULL;

  return NULL;
}

/* NOTE: all chunk parsers will return one of three things:
   Negative number: there are -N bytes too few to parse this chunk
   0: failed to correctly parse chunk
   Positive number: there were N bytes total in the chunk
*/
#define calc_time_stamp(gb, res)  \
      res = (gst_getbits3(gb) << 30); \
      if (gst_getbits1(gb) != 1) { \
        GST_DEBUG (0,"expecting marker bit '1' in calc_time_stamp (1)\n");return FALSE; } \
      res |= (gst_getbits15(gb) << 15); \
      if (gst_getbits1(gb) != 1) { \
        GST_DEBUG (0,"expecting marker bit '1' in calc_time_stamp (2)\n");return FALSE; } \
      res |= gst_getbits15(gb); \
      if (gst_getbits1(gb) != 1) { \
        GST_DEBUG (0,"expecting marker bit '1' in calc_time_stamp (3)\n");return FALSE; }

/* This checks for and parses the pack header.  If at any point something
   looks odd (marker bits not correct), it will return FALSE and not set
   the values in the mpeg1parse structure. */
static gint
parse_packhead (Mpeg1Parse *mpeg1parse, guchar *buf, gint size)
{
  guint64 scr = 0;
  guint32 bit_rate;

  GST_DEBUG (0,"mpeg1parse::parse_packhead: enter\n");

  // check length first
  if (size < 8)
    // return how many bytes missing
    return -(8-size);

  // set up getbits
  gst_getbits_newbuf(&mpeg1parse->gb,buf, size);

  // start parsing the stream start with the SCR
  if (gst_getbits4(&mpeg1parse->gb) != 0x2) {
    GST_DEBUG (0,"mpeg1parse::parse_packhead: expecting 0010\n");return FALSE; }

  calc_time_stamp(&mpeg1parse->gb, scr);

  GST_DEBUG (0,"mpeg1parse::parse_packhead: SCR = %lld\n",scr);


  if (mpeg1parse->sync) {
    GstClockTimeDiff clock_diff;

    clock_diff = gst_clock_current_diff(gst_clock_get_system(), (scr*100LL)/9LL);
  //printf("%lld\n", clock_diff);

    if (clock_diff > 500000 || clock_diff < -100000) {
      gst_clock_set(gst_clock_get_system(), (scr*100LL)/9LL);
    }
    else {
      gst_clock_wait(gst_clock_get_system(), (scr*100LL)/9LL, GST_OBJECT(mpeg1parse));
    }
  }


  // parse the mux_rate
  if (gst_getbits1(&mpeg1parse->gb) != 1) {
    GST_DEBUG (0,"mpeg1parse::parse_packhead: expecting marker bit '1' in calc mux (1)\n");return FALSE; }

  bit_rate = gst_getbits22(&mpeg1parse->gb);

  if (gst_getbits1(&mpeg1parse->gb) != 1) {
    GST_DEBUG (0,"mpeg1parse::parse_packhead: expecting marker bit '1' in calc mux (2)\n");return FALSE; }

  GST_DEBUG (0,"mpeg1parse::parse_packhead: Mux Rate is %d bps\n", bit_rate*50);

  mpeg1parse->have_packhead = TRUE;
  mpeg1parse->scr = scr;
  mpeg1parse->bit_rate = bit_rate;
  mpeg1parse->packs++;
  // return total number of bytes used in this chunk
  return 8;
}

static gint
parse_syshead (Mpeg1Parse *mpeg1parse, guchar *buf, gint size)
{
  guint16 header_length;
  guint32 rate_bound;
  guint8 audio_bound;
  gboolean fixed;
  gboolean constrained;
  gboolean audio_lock;
  gboolean video_lock;
  guint8 video_bound;
  guint8 ac3_track_code;
  gboolean packet_rate_restriction;
  guint stream_count, i,j;
  GstPadTemplate *newtemp = NULL;

  GST_DEBUG (0,"mpeg1parse::parse_syshead: in parse_syshead\n");

  // make sure we have enough bytes to know how long this is first
  if (size < 2)
    // return how many bytes missing
    return -(2-size);

  // set up getbits
  gst_getbits_newbuf(&mpeg1parse->gb,buf, size);

  // start parsing
  header_length = gst_getbits16(&mpeg1parse->gb);

  // check known length
  if (size < (2 + header_length))
    return -((2 + header_length)-size);

  if (gst_getbits1(&mpeg1parse->gb) != 1){
    GST_DEBUG (0,"mpeg1parse::parse_syshead: expecting marker bit '1' before rate bound \n");return FALSE; }
  rate_bound = gst_getbits22(&mpeg1parse->gb);
  if (gst_getbits1(&mpeg1parse->gb) != 1) {
    GST_DEBUG (0,"mpeg1parse::parse_syshead: expecting marker bit '1' after rate bound\n");return FALSE; }
  GST_DEBUG (0,"mpeg1parse::parse_syshead: Rate bound = %d \n",rate_bound);

  audio_bound = gst_getbits6(&mpeg1parse->gb);
  fixed = gst_getbits1(&mpeg1parse->gb);
  constrained = gst_getbits1(&mpeg1parse->gb);
  GST_DEBUG (0,"mpeg1parse::parse_syshead: Audio bound=%d, Fixed=%d, constrained=%d \n",audio_bound, fixed, constrained);

  audio_lock = gst_getbits1(&mpeg1parse->gb);
  video_lock = gst_getbits1(&mpeg1parse->gb);
  if (gst_getbits1(&mpeg1parse->gb) != 1) {
    GST_DEBUG (0,"mpeg1parse::parse_syshead: expecting marker bit '1' after video local\n");return FALSE; }
  video_bound = gst_getbits5(&mpeg1parse->gb);
  GST_DEBUG (0,"mpeg1parse::parse_syshead: audio lock=%d, video lock=%d, video bound=%d \n",audio_lock, video_lock, video_bound);

  packet_rate_restriction = gst_getbits1(&mpeg1parse->gb);
  if (gst_getbits7(&mpeg1parse->gb) != 0x7f) {
    GST_DEBUG (0,"mpeg1parse::parse_syshead: reserved bytes not set to default\n");return FALSE; }
  GST_DEBUG (0,"mpeg1parse::parse_syshead: packet rate restriction=%d \n", packet_rate_restriction);

  mpeg1parse->have_syshead = TRUE;
  mpeg1parse->header_length = header_length;
  mpeg1parse->rate_bound = rate_bound;
  mpeg1parse->audio_bound = audio_bound;
  mpeg1parse->fixed = fixed;
  mpeg1parse->constrained = constrained;
  mpeg1parse->audio_lock = audio_lock;
  mpeg1parse->video_lock = video_lock;
  mpeg1parse->video_bound = video_bound;
  mpeg1parse->packet_rate_restriction = packet_rate_restriction;

  stream_count = (header_length - 6) / 3;
  GST_DEBUG (0,"mpeg1parse::parse_syshead: number of streams=%d \n", stream_count);

  j=0;
  for (i=0; i < stream_count; i++) {
    gint stream_num;
    guint8 stream_id;
    gboolean STD_buffer_bound_scale;
    guint16 STD_buffer_size_bound;
    guint32 buf_byte_size_bound;
    gchar *name = NULL;
    GstPad **outpad = NULL;

    if (gst_getbits1(&mpeg1parse->gb) != 1) {
      GST_DEBUG (0,"mpeg1parse::parse_syshead: error in system header length\n");return FALSE; }
    stream_id = 0x80 | gst_getbits7(&mpeg1parse->gb);

	 // check marker bits
    if (gst_getbits2(&mpeg1parse->gb) != 0x3) {
      GST_DEBUG (0,"mpeg1parse::parse_syshead: expecting placeholder bit values '11' after stream id\n");return FALSE; }

     STD_buffer_bound_scale = gst_getbits1(&mpeg1parse->gb);
     STD_buffer_size_bound = gst_getbits5(&mpeg1parse->gb)<<8;
     STD_buffer_size_bound |= gst_getbits8(&mpeg1parse->gb);

     if (STD_buffer_bound_scale == 0) {
       buf_byte_size_bound = STD_buffer_size_bound * 128;
     } else {
       buf_byte_size_bound = STD_buffer_size_bound * 1024;
     }

    // private_stream_1
    if (stream_id == 0xBD) {
      // first find the track code
      ac3_track_code = *(guint8 *)(buf+header_length);
      // make sure it's valid
      if ((ac3_track_code >= 0x80) && (ac3_track_code <= 0x87)) {
        GST_DEBUG (0,"mpeg1parse::parse_syshead: 0x%02X: we have a private_stream_1 (AC3) packet, track %d\n",
              stream_id, ac3_track_code - 0x80);
        // scrap first 4 bytes (so-called "mystery AC3 tag")
        //headerlen += 4;
        //datalen -= 4;
        name = g_strdup_printf("private_stream_1_%02d",ac3_track_code - 0x80);
        stream_num = stream_id & 0x07;
        outpad = &mpeg1parse->private_1_pad[ac3_track_code - 0x80];
        newtemp = private1_template;
      }
    // private_stream_2
    } else if (stream_id == 0xBF) {
      name = g_strdup_printf("private_stream_2");
      stream_num = 0;
      outpad = &mpeg1parse->private_2_pad;
      newtemp = private2_template;
    // Audio
    } else if ((stream_id >= 0xC0) && (stream_id <= 0xDF)) {
      name = g_strdup_printf("audio_%02d",stream_id & 0x1F);
      stream_num = stream_id & 0x1F;
      outpad = &mpeg1parse->audio_pad[stream_num];
      newtemp = audio_template;
    // Video
    } else if ((stream_id >= 0xE0) && (stream_id <= 0xEF)) {
      name = g_strdup_printf("video_%02d",stream_id & 0x0F);
      stream_num = stream_id & 0x0F;
      outpad = &mpeg1parse->video_pad[stream_num];
      newtemp = video_template;
    }

    GST_DEBUG (0,"mpeg1parse::parse_syshead: stream ID 0x%02X (%s)\n", stream_id, name);
    GST_DEBUG (0,"mpeg1parse::parse_syshead: STD_buffer_bound_scale %d\n", STD_buffer_bound_scale);
    GST_DEBUG (0,"mpeg1parse::parse_syshead: STD_buffer_size_bound %d or %d bytes\n", STD_buffer_size_bound, buf_byte_size_bound);

    // create the pad and add it to self if it does not yet exist
    // this should trigger the NEW_PAD signal, which should be caught by
    // the app and used to attach to desired streams.
    if (*outpad == NULL) {
      (*outpad) = gst_pad_new_from_template (newtemp, name);
      gst_pad_set_caps (*outpad, gst_pad_get_padtemplate_caps (*outpad));
      gst_element_add_pad(GST_ELEMENT(mpeg1parse),(*outpad));
    }
    else {
      // we won't be needing this.
      if (name) g_free(name);
    }

    mpeg1parse->STD_buffer_info[j].stream_id = stream_id;
    mpeg1parse->STD_buffer_info[j].STD_buffer_bound_scale = STD_buffer_bound_scale;
    mpeg1parse->STD_buffer_info[j].STD_buffer_size_bound = STD_buffer_size_bound;

    j++;
  }
  // return total number of bytes used in this chunk
  return (2 + header_length);
}

static gint
parse_packet (Mpeg1Parse *mpeg1parse, guchar *data, gulong size,
              GstBuffer *buf, gulong offset)
{
  guint8 id = mpeg1parse->id;
  guint8 bits;
  guint16 headerlen;
  gboolean *need_flush = NULL;

  guint16 packet_length;
  gboolean std_buffer_scale;
  guint16 std_buffer_size;
  guint64 dts;

  guint16 datalen;
  gulong outoffset = 0;
  guint8 ac3_track_code;

  GstPad **outpad = NULL;
  GstBuffer *outbuf;

  GST_DEBUG (0,"mpeg1parse::parse_packet: in parse_packet\n");


  // make sure we have enough bytes to know how long this is first
  if (size < 2)
    // return how many bytes missing
    return -(2-size);

  // set up getbits
  gst_getbits_newbuf(&mpeg1parse->gb,data, size);

  // start parsing
  packet_length = gst_getbits16(&mpeg1parse->gb);
  headerlen = 2;
  GST_DEBUG (0,"mpeg1parse::parse_packet: got packet_length %d\n",packet_length);
  // constant is 2 bytes packet_length
  if (size < (2 + packet_length)) {
    GST_DEBUG (0,"mpeg1parse::parse_packet: have not enough data\n");
    return -((2 + packet_length)-size);
  }

  // loop through looping for stuffing bits, STD, PTS, DTS, etc
  do {
    bits = gst_getbits2(&mpeg1parse->gb);
    // stuffing bytes
    if (bits == 0x3) {
      if (gst_getbits6(&mpeg1parse->gb) == 0x3f) {
        GST_DEBUG (0,"mpeg1parse::parse_packet: have stuffing byte\n");
      } else {
        GST_DEBUG (0,"mpeg1parse::parse_packet: expected stuffing byte\n");
      }
      headerlen++;
    } else if (bits == 0x1) {
      GST_DEBUG (0,"mpeg1parse::parse_packet: have STD\n");
      std_buffer_scale = gst_getbits1(&mpeg1parse->gb);
      std_buffer_size = gst_getbits13(&mpeg1parse->gb);
      headerlen += 2;
    } else if (bits == 0) {
      bits = gst_getbits2(&mpeg1parse->gb);
      if (bits == 0x2) {
        calc_time_stamp(&mpeg1parse->gb, mpeg1parse->last_pts);
        GST_DEBUG (0,"mpeg1parse::parse_packet: PTS = %llu\n", mpeg1parse->last_pts);
        headerlen += 5;
	break;
      } else if (bits == 0x3) {
        calc_time_stamp(&mpeg1parse->gb, mpeg1parse->last_pts);
        if (gst_getbits4(&mpeg1parse->gb) != 1)
          return FALSE;
        calc_time_stamp(&mpeg1parse->gb, dts);
        GST_DEBUG (0,"mpeg1parse::parse_packet: PTS = %llu, DTS = %llu\n", mpeg1parse->last_pts, dts);
        headerlen += 10;
        break;
      } else if (bits == 0) {
        GST_DEBUG (0,"mpeg1parse::parse_packet: have no pts/dts\n");
        bits = gst_getbits4(&mpeg1parse->gb);
        GST_DEBUG (0,"mpeg1parse::parse_packet: got trailer bits %x\n",bits);
        if (bits != 0xf) {
          GST_DEBUG (0,"mpeg1parse::parse_packet: not a valid packet time sequence\n");
          return FALSE;
        }
        headerlen++;
        break;
      } else break;
    } else break;
  } while (1);
  GST_DEBUG (0,"mpeg1parse::parse_packet: done with header loop\n");

  // calculate the amount of real data in this packet
  datalen = packet_length - headerlen+2;
  GST_DEBUG (0,"mpeg1parse::parse_packet: headerlen is %d, datalen is %d, size is %lu\n",
        headerlen,datalen,size);

  // private_stream_1
  if (id == 0xBD) {
    // first find the track code
    ac3_track_code = *(guint8 *)(data+headerlen);
    // make sure it's valid
    if ((ac3_track_code >= 0x80) && (ac3_track_code <= 0x87)) {
      GST_DEBUG (0,"mpeg1parse::parse_packet: 0x%02X: we have a private_stream_1 (AC3) packet, track %d\n",
            id, ac3_track_code - 0x80);
      outpad = &mpeg1parse->private_1_pad[ac3_track_code - 0x80];
      // scrap first 4 bytes (so-called "mystery AC3 tag")
      headerlen += 4;
      datalen -= 4;
    }
  // private_stream_1
  } else if (id == 0xBF) {
    GST_DEBUG (0,"mpeg1parse::parse_packet: 0x%02X: we have a private_stream_2 packet\n", id);
    outpad = &mpeg1parse->private_2_pad;
  // audio
  } else if ((id >= 0xC0) && (id <= 0xDF)) {
    GST_DEBUG (0,"mpeg1parse::parse_packet: 0x%02X: we have an audio packet\n", id);
    outpad = &mpeg1parse->audio_pad[id & 0x1F];
    outoffset = mpeg1parse->audio_offset[id & 0x1F];
    mpeg1parse->audio_offset[id & 0x1F] += datalen;
    need_flush = &mpeg1parse->audio_need_flush[id & 0x1F];
  // video
  } else if ((id >= 0xE0) && (id <= 0xEF)) {
    GST_DEBUG (0,"mpeg1parse::parse_packet: 0x%02X: we have a video packet\n", id);
    outpad = &mpeg1parse->video_pad[id & 0x0F];
    outoffset = mpeg1parse->video_offset[id & 0x1F];
    mpeg1parse->video_offset[id & 0x1F] += datalen;
    need_flush = &mpeg1parse->video_need_flush[id & 0x1F];
  }

  // if we don't know what it is, bail
  if (outpad == NULL) {
    GST_DEBUG (0,"mpeg1parse::parse_packet: unknown packet id 0x%02X !!\n", id);
    // return total number of bytes
    return (2 + packet_length);
  }

  // FIXME, this should be done in parse_syshead
  if ((*outpad) == NULL) {
    GST_DEBUG (0,"mpeg1parse::parse_packet: unexpected packet id 0x%02X!!\n", id);
    // return total number of bytes
    return (2 + packet_length);
  }

  //if (!GST_STATE_IS_SET(mpeg1parse, GST_STATE_COMPLETE)) {
  //  gst_element_set_state(GST_ELEMENT(mpeg1parse), GST_STATE_COMPLETE);
  //}

  // create the buffer and send it off to the Other Side
  if (GST_PAD_CONNECTED(*outpad) && datalen > 0) {
    // if this is part of the buffer, create a subbuffer
    GST_DEBUG (0,"mpeg1parse::parse_packet: creating subbuffer at offset %lu, len %d\n",
          offset+headerlen,datalen);
    outbuf = gst_buffer_create_sub(buf,offset+headerlen,datalen);
    g_assert(outbuf != NULL);

    GST_DEBUG (0,"mpeg1parse::parse_packet: pushing buffer of len %d id %d\n", datalen, id);
    GST_BUFFER_OFFSET(outbuf) = outoffset;
    if (need_flush && *need_flush) {
      GST_BUFFER_FLAG_SET(outbuf, GST_BUFFER_FLUSH);
      *need_flush = FALSE;
    }
    else {
      GST_BUFFER_FLAG_UNSET(outbuf, GST_BUFFER_FLUSH);
    }
    GST_BUFFER_TIMESTAMP(outbuf) = (mpeg1parse->last_pts*1000000LL)/90000LL;
    gst_pad_push((*outpad),outbuf);
  }

  // return total number of bytes
  return (2 + packet_length);
}

static void
gst_mpeg1parse_flush (Mpeg1Parse *mpeg1parse)
{
  gint i;
  GST_DEBUG (0,"mpeg1parse:: flush\n");

  if (mpeg1parse->prevbuf) {
    gst_buffer_unref(mpeg1parse->prevbuf);
    mpeg1parse->prevbuf = NULL;
  }
  for (i=0;i<16;i++) {
    mpeg1parse->video_offset[i] = 0;
    mpeg1parse->video_need_flush[i] = TRUE;
  }
  for (i=0;i<32;i++) {
    mpeg1parse->audio_offset[i] = 0;
    mpeg1parse->audio_need_flush[i] = TRUE;
  }
  mpeg1parse->have_sync = FALSE;
  mpeg1parse->sync_zeros = 0;
  mpeg1parse->id = 0;
}

/* Some assumptions: buffers coming in are at least as big as a given
   header or PES packet.  This means we can get away without copying like
   mad to reassemble many-buffer splits. */
static void
gst_mpeg1parse_chain (GstPad *pad, GstBuffer *buf)
{
  Mpeg1Parse *mpeg1parse;
  guchar *data;
  gulong size,offset, bufoffset;
  GstBuffer *temp;

  g_return_if_fail(pad != NULL);
  g_return_if_fail(GST_IS_PAD(pad));
  g_return_if_fail(buf != NULL);

  mpeg1parse = GST_MPEG1PARSE (GST_OBJECT_PARENT (pad));

  bufoffset = GST_BUFFER_OFFSET(buf);
  offset = 0;

  mpeg1parse->in_flush = GST_BUFFER_FLAG_IS_SET(buf, GST_BUFFER_FLUSH);

  if (mpeg1parse->in_flush) {
    gst_mpeg1parse_flush(mpeg1parse);
  }

  GST_DEBUG (0,"mpeg1parse::chain: mpeg1parse: have buffer of size %d\n",GST_BUFFER_SIZE(buf));

  if (mpeg1parse->prevbuf) {
    GST_DEBUG (0,"mpeg1parse::chain: we have some data left\n");
    mpeg1parse->prevbuf = gst_buffer_append(mpeg1parse->prevbuf, buf);
    gst_buffer_unref(buf);
  }
  else {
    mpeg1parse->prevbuf = buf;
  }

  data = GST_BUFFER_DATA(mpeg1parse->prevbuf);
  size = GST_BUFFER_SIZE(mpeg1parse->prevbuf);

  // main loop
  while (offset < size) {
    GST_DEBUG (0,"mpeg1parse::chain: offset is %lu\n",offset);
    // first step is to find sync, which will take at least three rounds
    if (!mpeg1parse->have_sync) {
      guchar byte = *(data+offset);
      GST_DEBUG (0,"mpeg1parse::chain: found #%d at %lu\n",byte,offset);
      offset++;
      // if it's zero, increment the zero count
      if (byte == 0) {
        mpeg1parse->sync_zeros++;
        GST_DEBUG (0,"mpeg1parse::chain: found zero #%d at %lu\n",mpeg1parse->sync_zeros,offset-1);
      }
      // if it's a one and we have two previous zeros, we have sync
      else if ((byte == 1) && (mpeg1parse->sync_zeros >= 2)) {
        GST_DEBUG (0,"mpeg1parse::chain: synced at %lu\n",offset-1);
        mpeg1parse->have_sync = TRUE;
        mpeg1parse->sync_zeros = 0;
      }
      // if it's anything else, we've lost it completely
      else mpeg1parse->sync_zeros = 0;

    // then snag the chunk ID
    } else if (mpeg1parse->id == 0) {
      mpeg1parse->id = *(data+offset);
      GST_DEBUG (0,"mpeg1parse::chain: got id 0x%02X at offset %lu+%d=%lu\n",mpeg1parse->id,offset,
            mpeg1parse->prevbuf->offset,offset+mpeg1parse->prevbuf->offset);
      offset++;
      // check to see if it's valid
      if (mpeg1parse->id < 0xB9) {
        // invalid, so we reset the parse state
        mpeg1parse->have_sync = FALSE;
        mpeg1parse->id = 0;
      }

    // and lastly get the chunk contents dealt with
    } else {
      gint ret = 0;
      // switchin the ID and run the right chunk parser
      GST_DEBUG (0,"mpeg1parse::chain: have chunk 0x%02X at offset %lu\n",mpeg1parse->id,offset);
      switch (mpeg1parse->id) {
        case 0xBA:
          ret = parse_packhead(mpeg1parse,data+offset,size-offset);
          break;
        case 0xBB:
          ret = parse_syshead(mpeg1parse,data+offset,size-offset);
          break;
        default:
          ret = parse_packet(mpeg1parse,data+offset,size-offset,mpeg1parse->prevbuf,offset);
      }
      // if we got a successful return out
      if (ret >= 0) {
        offset += ret;
        GST_DEBUG (0,"mpeg1parse::chain: chunk is %d bytes \n",ret);
        // we have to reset the state
        mpeg1parse->have_sync = FALSE;
        mpeg1parse->id = 0;
	if (offset >= size-4) break; // no way we can find a sync
      }
      // or we didn't have enough data
      else {
	break;
      }
    }
  }

  if (offset < size) {
    GST_DEBUG (0,"mpeg1parse::chain: we're short for this chunk\n");
    temp = gst_buffer_create_sub(mpeg1parse->prevbuf, offset, size-offset);
    g_assert(temp != NULL);
    gst_buffer_unref(mpeg1parse->prevbuf);
    mpeg1parse->prevbuf = temp;
  }
  else {
    GST_DEBUG (0,"mpeg1parse::chain: cleaning previous buffer\n");
    gst_buffer_unref(mpeg1parse->prevbuf);
    mpeg1parse->prevbuf = NULL;
  }
}

static void
gst_mpeg1parse_set_property (GObject *object, guint prop_id, const GValue *value, GParamSpec *pspec)
{
  Mpeg1Parse *mpeg1parse;

  /* it's not null if we got it, but it might not be ours */
  mpeg1parse = GST_MPEG1PARSE(object);

  switch (prop_id) {
    case ARG_SYNC: {
      mpeg1parse->sync = g_value_get_boolean (value);
      break;
    }
    default: {
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
    }
  }
}

static void
gst_mpeg1parse_get_property (GObject *object, guint prop_id, GValue *value, GParamSpec *pspec)
{
  Mpeg1Parse *mpeg1parse;

  /* it's not null if we got it, but it might not be ours */
  mpeg1parse = GST_MPEG1PARSE(object);

  switch (prop_id) {
    case ARG_BIT_RATE: {
      g_value_set_int (value, mpeg1parse->bit_rate*400);
      break;
    }
    case ARG_SYNC: {
      g_value_set_boolean (value, mpeg1parse->sync);
      break;
    }
    default: {
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
    }
  }
}

static GstElementStateReturn
gst_mpeg1parse_change_state (GstElement *element)
{
  Mpeg1Parse *mpeg1parse;
  g_return_val_if_fail(GST_IS_MPEG1PARSE(element),GST_STATE_FAILURE);

  mpeg1parse = GST_MPEG1PARSE(element);
  GST_DEBUG (0,"gstmpeg_play: state pending %d\n", GST_STATE_PENDING(element));

  /* if going down into NULL state, clear out buffers*/
  if (GST_STATE_PENDING(element) == GST_STATE_READY) {
    /* otherwise (READY or higher) we need to open the file */
    gst_mpeg1parse_flush(mpeg1parse);
  }

  /* if we haven't failed already, give the parent class a chance to ;-) */
  if (GST_ELEMENT_CLASS(parent_class)->change_state)
    return GST_ELEMENT_CLASS(parent_class)->change_state(element);

  return GST_STATE_SUCCESS;
}


static gboolean
plugin_init (GModule *module, GstPlugin *plugin)
{
  GstElementFactory *factory;

  /* this filter needs the getbits functions */
  if (!gst_library_load ("gstgetbits")) {
    gst_info ("mpeg1parse:: could not load support library: 'gstgetbits'\n");
    return FALSE;
  }

  /* create an elementfactory for the mpeg1parse element */
  factory = gst_elementfactory_new ("mpeg1parse", GST_TYPE_MPEG1PARSE,
                                    &mpeg1parse_details);
  g_return_val_if_fail (factory != NULL, FALSE);

  sink_template = sink_factory ();
  gst_elementfactory_add_padtemplate (factory, sink_template);

  audio_template = audio_factory ();
  gst_elementfactory_add_padtemplate (factory, audio_template);

  video_template = video_factory ();
  gst_elementfactory_add_padtemplate (factory, video_template);

  gst_plugin_add_factory (plugin, factory);

  return TRUE;
}

GstPluginDesc plugin_desc = {
  GST_VERSION_MAJOR,
  GST_VERSION_MINOR,
  "mpeg1parse",
  plugin_init
};
