/* gethdr.c, header decoding                                                */

/* Copyright (C) 1996, MPEG Software Simulation Group. All Rights Reserved. */

/*
 * Disclaimer of Warranty
 *
 * These software programs are available to the user without any license fee or
 * royalty on an "as is" basis.  The MPEG Software Simulation Group disclaims
 * any and all warranties, whether express, implied, or statuary, including any
 * implied warranties or merchantability or of fitness for a particular
 * purpose.  In no event shall the copyright-holder be liable for any
 * incidental, punitive, or consequential damages of any kind whatsoever
 * arising from the use of these programs.
 *
 * This disclaimer of warranty extends to the user of these programs and user's
 * customers, employees, agents, transferees, successors, and assigns.
 *
 * The MPEG Software Simulation Group does not represent or warrant that the
 * programs furnished hereunder are free of infringement of any third-party
 * patents.
 *
 * Commercial implementations of MPEG-1 and MPEG-2 video, including shareware,
 * are subject to royalty fees to patent holders.  Many of these patents are
 * general enough such that they are unavoidable regardless of implementation
 * design.
 *
 */

#include <stdio.h>

//#define DEBUG_ENABLED
#include "mpeg2dec.h"

/* default intra quantization matrix */
static unsigned char default_intra_quantizer_matrix[64]
=
{
  8, 16, 19, 22, 26, 27, 29, 34,
  16, 16, 22, 24, 27, 29, 34, 37,
  19, 22, 26, 27, 29, 34, 34, 38,
  22, 22, 26, 27, 29, 34, 37, 40,
  22, 26, 27, 29, 32, 35, 40, 48,
  26, 27, 29, 32, 35, 40, 48, 58,
  26, 27, 29, 34, 38, 46, 56, 69,
  27, 29, 35, 38, 46, 56, 69, 83
}
;


/* private prototypes */
static void sequence_extension _ANSI_ARGS_((mpeg2play_vid_stream *vid_stream));
static void sequence_display_extension _ANSI_ARGS_((mpeg2play_vid_stream *vid_stream));
static void quant_matrix_extension _ANSI_ARGS_((mpeg2play_vid_stream *vid_stream));
static void sequence_scalable_extension _ANSI_ARGS_((mpeg2play_vid_stream *vid_stream));
static void picture_display_extension _ANSI_ARGS_((mpeg2play_vid_stream *vid_stream));
static void picture_coding_extension _ANSI_ARGS_((mpeg2play_vid_stream *vid_stream));
static void picture_spatial_scalable_extension _ANSI_ARGS_((mpeg2play_vid_stream *vid_stream));
static void picture_temporal_scalable_extension _ANSI_ARGS_((mpeg2play_vid_stream *vid_stream));
static int  extra_bit_information _ANSI_ARGS_((mpeg2play_vid_stream *vid_stream));
static void copyright_extension _ANSI_ARGS_((mpeg2play_vid_stream *vid_stream));
static void user_data _ANSI_ARGS_((mpeg2play_vid_stream *vid_stream));

/* introduced in September 1995 to assist spatial scalable decoding */
static void Update_Temporal_Reference_Tacking_Data _ANSI_ARGS_((mpeg2play_vid_stream *vid_stream));
/* private variables */
static int Temporal_Reference_Base = 0;
static int True_Framenum_max  = -1;
static int Temporal_Reference_GOP_Reset = 0;

#define RESERVED    -1 
static double frame_rate_Table[16] =
{
  0.0,
  ((23.0*1000.0)/1001.0),
  24.0,
  25.0,
  ((30.0*1000.0)/1001.0),
  30.0,
  50.0,
  ((60.0*1000.0)/1001.0),
  60.0,
 
  1,                  /* these are unofficial libmpeg2 rates */
  5,
  10,
  12,
  15,
  RESERVED,
  RESERVED
};

/*
 * decode headers from one input stream
 * until an End of Sequence or picture start code
 * is found
 */
int Get_Hdr(mpeg2play_vid_stream *vid_stream)
{
  unsigned int code;
  gst_getbits_t *gb = &vid_stream->ld->gb;

  for (;;)
  {
    /* look for mpeg2play_next_start_code */
    mpeg2play_next_start_code(gb);
    code = gst_showbits32(gb);
    gst_flushbits32(gb);
  
    switch (code)
    {
    case SEQUENCE_HEADER_CODE:
      sequence_header(vid_stream);
      break;
    case GROUP_START_CODE:
      group_of_pictures_header(vid_stream);
      break;
    case PICTURE_START_CODE:
      picture_header(vid_stream);
      return 1;
      break;
    case SEQUENCE_END_CODE:
      return 0;
      break;
    default:
      fprintf(stderr,"Unexpected next_start_code %08x (ignored)\n",code);
      break;
    }
  }
}


/* align to start of next next_start_code */
int mpeg2play_next_start_code(gst_getbits_t *gb)
{
  int flush;
  /* byte align */
  //printf("bit offset %d\n", gst_getbits_bitoffset(gb));
  //printf("bits %08x\n", gst_showbits32(gb));
  if (gst_getbits_bitsleft(gb)<24) return NO_STARTCODE;
  flush = gst_getbits_bitoffset(gb);
  gst_flushbitsn(gb, flush);
  while (gst_showbits24(gb)!=0x01L) {
    gst_getbyte(gb);
    if (gst_getbits_bitsleft(gb)<24) return NO_STARTCODE;
  }
  return STARTCODE_FOUND;
}


/* decode sequence header */
int sequence_header(mpeg2play_vid_stream *vid_stream)
{
  int i;
  gst_getbits_t *gb = &vid_stream->ld->gb;
  mpeg2play_Sequence *seq = &vid_stream->sequence;
  mpeg2play_layer_data *ld = vid_stream->ld;

  vid_stream->horizontal_size      = gst_getbits12(gb);
  vid_stream->vertical_size        = gst_getbits12(gb);
  seq->aspect_ratio_information    = gst_getbits4(gb);
  seq->frame_rate_code             = gst_getbits4(gb);
  seq->bit_rate_value              = gst_getbits18(gb);
  marker_bit(gb, "sequence_header()");
  seq->vbv_buffer_size             = gst_getbits10(gb);
  seq->constrained_parameters_flag = gst_getbits1(gb);

  if((ld->load_intra_quantizer_matrix = gst_getbits1(gb)))
  {
    for (i=0; i<64; i++)
      ld->intra_quantizer_matrix[vid_stream->scan[ZIG_ZAG][i]] = gst_getbits8(gb);
  }
  else
  {
    for (i=0; i<64; i++)
      ld->intra_quantizer_matrix[i] = default_intra_quantizer_matrix[i];
  }

  if((ld->load_non_intra_quantizer_matrix = gst_getbits1(gb)))
  {
    for (i=0; i<64; i++)
      ld->non_intra_quantizer_matrix[vid_stream->scan[ZIG_ZAG][i]] = gst_getbits8(gb);
  }
  else
  {
    for (i=0; i<64; i++)
      ld->non_intra_quantizer_matrix[i] = 16;
  }

  /* copy luminance to chrominance matrices */
  for (i=0; i<64; i++)
  {
    ld->chroma_intra_quantizer_matrix[i] =
      ld->intra_quantizer_matrix[i];

    ld->chroma_non_intra_quantizer_matrix[i] =
      ld->non_intra_quantizer_matrix[i];
  }
  vid_stream->frame_rate = frame_rate_Table[seq->frame_rate_code];

  GST_DEBUG (0,"sequence header\n");
  GST_DEBUG (0,"  horizontal_size=%d\n",vid_stream->horizontal_size);
  GST_DEBUG (0,"  vertical_size=%d\n",vid_stream->vertical_size);
  GST_DEBUG (0,"  aspect_ratio_information=%d\n",seq->aspect_ratio_information);
  GST_DEBUG (0,"  frame_rate_code=%d\n",seq->frame_rate_code);
  GST_DEBUG (0,"  bit_rate_value=%d\n",seq->bit_rate_value);
  GST_DEBUG (0,"  vbv_buffer_size=%d\n",seq->vbv_buffer_size);
  GST_DEBUG (0,"  constrained_parameters_flag=%d\n",seq->constrained_parameters_flag);
  GST_DEBUG (0,"  load_intra_quantizer_matrix=%d\n",ld->load_intra_quantizer_matrix);
  GST_DEBUG (0,"  load_non_intra_quantizer_matrix=%d\n",ld->load_non_intra_quantizer_matrix);

#ifdef VERIFY
  verify_sequence_header++;
#endif /* VERIFY */

  return PARSE_OK;
}



/* decode group of pictures header */
/* ISO/IEC 13818-2 section 6.2.2.6 */
int group_of_pictures_header(mpeg2play_vid_stream *vid_stream)
{
  gst_getbits_t *gb = &vid_stream->ld->gb;
  mpeg2play_GoP *gop = &vid_stream->group;

  if (vid_stream->ld == &vid_stream->base)
  {
    Temporal_Reference_Base = True_Framenum_max + 1; 	/* *CH* */
    Temporal_Reference_GOP_Reset = 1;
  }
  gop->drop_flag   = gst_getbits1(gb);
  gop->hour        = gst_getbits5(gb);
  gop->minute      = gst_getbits6(gb);
  marker_bit(gb, "group_of_pictures_header()");
  gop->sec         = gst_getbits6(gb);
  gop->frame       = gst_getbits6(gb);
  gop->closed_gop  = gst_getbits1(gb);
  gop->broken_link = gst_getbits1(gb);

  GST_DEBUG (0,"group of pictures\n");
  GST_DEBUG (0,"  drop_flag=%d\n",gop->drop_flag);
  GST_DEBUG (0,"  timecode %d:%02d:%02d:%02d\n",gop->hour,gop->minute,gop->sec,gop->frame);
  GST_DEBUG (0,"  closed_gop=%d\n",gop->closed_gop);
  GST_DEBUG (0,"  broken_link=%d\n",gop->broken_link);


#ifdef VERIFY
  verify_group_of_pictures_header++;
#endif /* VERIFY */

  return PARSE_OK;

}


/* decode picture header */

/* ISO/IEC 13818-2 section 6.2.3 */
int picture_header(mpeg2play_vid_stream *vid_stream)
{
  int Extra_Information_Byte_Count;
  gst_getbits_t *gb = &vid_stream->ld->gb;
  mpeg2play_layer_data *ld = vid_stream->ld;
  mpeg2play_Pict *pict = &vid_stream->picture;

  /* unless later overwritten by picture_spatial_scalable_extension() */
  ld->pict_scal = 0; 
  
  pict->temporal_reference  = gst_getbits10(gb);
  pict->picture_coding_type = gst_getbits3(gb);
  pict->vbv_delay           = gst_getbits16(gb);

  if (pict->picture_coding_type==P_TYPE || pict->picture_coding_type==B_TYPE)
  {
    pict->full_pel_forward_vector = gst_getbits1(gb);
    pict->forward_f_code = gst_getbits3(gb);
  }
  if (pict->picture_coding_type==B_TYPE)
  {
    pict->full_pel_backward_vector = gst_getbits1(gb);
    pict->backward_f_code = gst_getbits3(gb);
  }

  GST_DEBUG (0,"picture header \n");
  GST_DEBUG (0,"  temporal_reference=%d\n",pict->temporal_reference);
  GST_DEBUG (0,"  picture_coding_type=%d\n",pict->picture_coding_type);
  GST_DEBUG (0,"  vbv_delay=%d\n",pict->vbv_delay);
  if (pict->picture_coding_type==P_TYPE || pict->picture_coding_type==B_TYPE)
  {
    GST_DEBUG (0,"  full_pel_forward_vector=%d\n",pict->full_pel_forward_vector);
    GST_DEBUG (0,"  forward_f_code =%d\n",pict->forward_f_code);
  }
  if (pict->picture_coding_type==B_TYPE)
  {
    GST_DEBUG (0,"  full_pel_backward_vector=%d\n",pict->full_pel_backward_vector);
    GST_DEBUG (0,"  backward_f_code =%d\n",pict->backward_f_code);
  }

#ifdef VERIFY
  verify_picture_header++;
#endif /* VERIFY */

  Extra_Information_Byte_Count = 
    extra_bit_information(vid_stream);
  
  /* update tracking information used to assist spatial scalability */
  Update_Temporal_Reference_Tacking_Data(vid_stream);

  return PARSE_OK;
}

/* decode slice header */

/* ISO/IEC 13818-2 section 6.2.4 */
int slice_header(mpeg2play_vid_stream *vid_stream)
{
  int slice_vertical_position_extension;
  int quantizer_scale_code;
  int slice_picture_id_enable = 0;
  int slice_picture_id = 0;
  int extra_information_slice = 0;
  gst_getbits_t *gb = &vid_stream->ld->gb;
  mpeg2play_layer_data *ld = vid_stream->ld;

  slice_vertical_position_extension =
    (ld->MPEG2_Flag && vid_stream->vertical_size>2800) ? gst_getbits3(gb) : 0;

  if (ld->scalable_mode==SC_DP)
    ld->priority_breakpoint = gst_getbits7(gb);

  quantizer_scale_code = gst_getbits5(gb);
  ld->quantizer_scale =
    ld->MPEG2_Flag ? (ld->q_scale_type ? Non_Linear_quantizer_scale[quantizer_scale_code] : quantizer_scale_code<<1) : quantizer_scale_code;

  /* slice_id introduced in March 1995 as part of the video corridendum
     (after the IS was drafted in November 1994) */
  if (gst_getbits1(gb))
  {
    ld->intra_slice = gst_getbits1(gb);

    slice_picture_id_enable = gst_getbits1(gb);
	slice_picture_id = gst_getbits6(gb);

    extra_information_slice = extra_bit_information(vid_stream);
  }
  else
    ld->intra_slice = 0;

  GST_DEBUG (0,"slice header \n");
  if (ld->MPEG2_Flag && vid_stream->vertical_size>2800)
    GST_DEBUG (0,"  slice_vertical_position_extension=%d\n",slice_vertical_position_extension);
  
  if (ld->scalable_mode==SC_DP)
    GST_DEBUG (0,"  priority_breakpoint=%d\n",ld->priority_breakpoint);

  GST_DEBUG (0,"  quantizer_scale_code=%d\n",quantizer_scale_code);
  GST_DEBUG (0,"  slice_picture_id_enable = %d\n", slice_picture_id_enable);

  if(slice_picture_id_enable)
    GST_DEBUG (0,"  slice_picture_id = %d\n", slice_picture_id);

#ifdef VERIFY
  verify_slice_header++;
#endif /* VERIFY */


  return slice_vertical_position_extension;
}


/* decode extension and user data */
/* ISO/IEC 13818-2 section 6.2.2.2 */
int extension_and_user_data(mpeg2play_vid_stream *vid_stream, int code)
{
  int ext_ID;
  gst_getbits_t *gb = &vid_stream->ld->gb;


  if (code==EXTENSION_START_CODE)
  {
    ext_ID = gst_getbits4(gb);
    switch (ext_ID)
    {
      case SEQUENCE_EXTENSION_ID:
        sequence_extension(vid_stream);
        break;
      case SEQUENCE_DISPLAY_EXTENSION_ID:
        sequence_display_extension(vid_stream);
        break;
      case QUANT_MATRIX_EXTENSION_ID:
        quant_matrix_extension(vid_stream);
        break;
      case SEQUENCE_SCALABLE_EXTENSION_ID:
        sequence_scalable_extension(vid_stream);
        break;
      case PICTURE_DISPLAY_EXTENSION_ID:
        picture_display_extension(vid_stream);
        break;
      case PICTURE_CODING_EXTENSION_ID:
        picture_coding_extension(vid_stream);
        break;
      case PICTURE_SPATIAL_SCALABLE_EXTENSION_ID:
        picture_spatial_scalable_extension(vid_stream);
        break;
      case PICTURE_TEMPORAL_SCALABLE_EXTENSION_ID:
        picture_temporal_scalable_extension(vid_stream);
        break;
      case COPYRIGHT_EXTENSION_ID:
        copyright_extension(vid_stream);
        break;
     default:
        fprintf(stderr,"reserved extension start code ID %d\n",ext_ID);
        break;
      }
      mpeg2play_next_start_code(gb);
  }
  else
  {
#ifdef VERBOSE
    if (vid_stream->Verbose_Flag>NO_LAYER)
      GST_DEBUG (0,"user data\n");
#endif /* VERBOSE */
    gst_flushbits32(gb);
    user_data(vid_stream);
  }
  return PARSE_OK;
}


/* decode sequence extension */

/* ISO/IEC 13818-2 section 6.2.2.3 */
static void sequence_extension(mpeg2play_vid_stream *vid_stream)
{
  int horizontal_size_extension;
  int vertical_size_extension;
  int bit_rate_extension;
  int vbv_buffer_size_extension;
  mpeg2play_Sequence *seq = &vid_stream->sequence;
  gst_getbits_t *gb = &vid_stream->ld->gb;

  vid_stream->ld->MPEG2_Flag = 1;

  vid_stream->ld->scalable_mode = SC_NONE; /* unless overwritten by sequence_scalable_extension() */
  seq->layer_id = 0;                /* unless overwritten by sequence_scalable_extension() */
  
  seq->profile_and_level_indication = gst_getbits8(gb);
  seq->progressive_sequence         = gst_getbits1(gb);
  seq->chroma_format                = gst_getbits2(gb);
  horizontal_size_extension    = gst_getbits2(gb);
  vertical_size_extension      = gst_getbits2(gb);
  bit_rate_extension           = gst_getbits12(gb);
  marker_bit(gb, "sequence_extension");
  vbv_buffer_size_extension    = gst_getbits8(gb);
  seq->low_delay                    = gst_getbits1(gb);
  seq->frame_rate_extension_n       = gst_getbits2(gb);
  seq->frame_rate_extension_d       = gst_getbits5(gb);

  vid_stream->frame_rate = frame_rate_Table[seq->frame_rate_code] *
    ((seq->frame_rate_extension_n+1)/(seq->frame_rate_extension_d+1));

  /* special case for 422 profile & level must be made */
  if((seq->profile_and_level_indication>>7) & 1)
  {  /* escape bit of profile_and_level_indication set */
  
    /* 4:2:2 Profile @ Main Level */
    if((seq->profile_and_level_indication&15)==5)
    {
      vid_stream->profile = PROFILE_422;
      vid_stream->level   = MAIN_LEVEL;  
    }
  }
  else
  {
    vid_stream->profile = seq->profile_and_level_indication >> 4;  /* Profile is upper nibble */
    vid_stream->level   = seq->profile_and_level_indication & 0xF;  /* Level is lower nibble */
  }
  
 
  vid_stream->horizontal_size = (horizontal_size_extension<<12) | (vid_stream->horizontal_size&0x0fff);
  vid_stream->vertical_size = (vertical_size_extension<<12) | (vid_stream->vertical_size&0x0fff);


  /* ISO/IEC 13818-2 does not define bit_rate_value to be composed of
   * both the original bit_rate_value parsed in sequence_header() and
   * the optional bit_rate_extension in sequence_extension_header(). 
   * However, we use it for bitstream verification purposes. 
   */

  seq->bit_rate_value += (bit_rate_extension << 18);
  vid_stream->bit_rate = ((double) seq->bit_rate_value) * 400.0;
  seq->vbv_buffer_size += (vbv_buffer_size_extension << 10);

  GST_DEBUG (0,"sequence extension \n");

  GST_DEBUG (0,"  profile_and_level_indication=%d\n",seq->profile_and_level_indication);

  if (seq->profile_and_level_indication<128)
  {
    GST_DEBUG (0,"    profile=%d, level=%d\n",vid_stream->profile,vid_stream->level);
  }

  GST_DEBUG (0,"  progressive_sequence=%d\n",seq->progressive_sequence);
  GST_DEBUG (0,"  chroma_format=%d\n",seq->chroma_format);
  GST_DEBUG (0,"  horizontal_size_extension=%d\n",horizontal_size_extension);
  GST_DEBUG (0,"  vertical_size_extension=%d\n",vertical_size_extension);
  GST_DEBUG (0,"  bit_rate_extension=%d\n",bit_rate_extension);
  GST_DEBUG (0,"  vbv_buffer_size_extension=%d\n",vbv_buffer_size_extension);
  GST_DEBUG (0,"  low_delay=%d\n",seq->low_delay);
  GST_DEBUG (0,"  frame_rate_extension_n=%d\n",seq->frame_rate_extension_n);
  GST_DEBUG (0,"  frame_rate_extension_d=%d\n",seq->frame_rate_extension_d);

#ifdef VERIFY
  verify_sequence_extension++;
#endif /* VERIFY */


}


/* decode sequence display extension */

static void sequence_display_extension(mpeg2play_vid_stream *vid_stream)
{
  gst_getbits_t *gb = &vid_stream->ld->gb;
  mpeg2play_Sequence *seq = &vid_stream->sequence;

  seq->video_format      = gst_getbits3(gb);
  seq->color_description = gst_getbits1(gb);

  if (seq->color_description)
  {
    seq->color_primaries          = gst_getbits8(gb);
    seq->transfer_characteristics = gst_getbits8(gb);
    seq->matrix_coefficients      = gst_getbits8(gb);
  }

  seq->display_horizontal_size = gst_getbits14(gb);
  marker_bit(gb, "sequence_display_extension");
  seq->display_vertical_size   = gst_getbits14(gb);

  GST_DEBUG (0,"sequence display extension\n");

  GST_DEBUG (0,"  video_format=%d\n",seq->video_format);
  GST_DEBUG (0,"  color_description=%d\n",seq->color_description);

  if (seq->color_description)
  {
    GST_DEBUG (0,"    color_primaries=%d\n",seq->color_primaries);
    GST_DEBUG (0,"    transfer_characteristics=%d\n",seq->transfer_characteristics);
    GST_DEBUG (0,"    matrix_coefficients=%d\n",seq->matrix_coefficients);
  }
  GST_DEBUG (0,"  display_horizontal_size=%d\n",seq->display_horizontal_size);
  GST_DEBUG (0,"  display_vertical_size=%d\n",seq->display_vertical_size);

#ifdef VERIFY
  verify_sequence_display_extension++;
#endif /* VERIFY */

}


/* decode quant matrix entension */
/* ISO/IEC 13818-2 section 6.2.3.2 */
static void quant_matrix_extension(mpeg2play_vid_stream *vid_stream)
{
  int i;
  gst_getbits_t *gb = &vid_stream->ld->gb;
  mpeg2play_layer_data *ld = vid_stream->ld;

  if((ld->load_intra_quantizer_matrix = gst_getbits1(gb)))
  {
    for (i=0; i<64; i++)
    {
      ld->chroma_intra_quantizer_matrix[vid_stream->scan[ZIG_ZAG][i]]
      = ld->intra_quantizer_matrix[vid_stream->scan[ZIG_ZAG][i]]
      = gst_getbits8(gb);
    }
  }

  if((ld->load_non_intra_quantizer_matrix = gst_getbits1(gb)))
  {
    for (i=0; i<64; i++)
    {
      ld->chroma_non_intra_quantizer_matrix[vid_stream->scan[ZIG_ZAG][i]]
      = ld->non_intra_quantizer_matrix[vid_stream->scan[ZIG_ZAG][i]]
      = gst_getbits8(gb);
    }
  }

  if((ld->load_chroma_intra_quantizer_matrix = gst_getbits1(gb)))
  {
    for (i=0; i<64; i++)
      ld->chroma_intra_quantizer_matrix[vid_stream->scan[ZIG_ZAG][i]] = gst_getbits8(gb);
  }

  if((ld->load_chroma_non_intra_quantizer_matrix = gst_getbits1(gb)))
  {
    for (i=0; i<64; i++)
      ld->chroma_non_intra_quantizer_matrix[vid_stream->scan[ZIG_ZAG][i]] = gst_getbits8(gb);
  }

  GST_DEBUG (0,"quant matrix extension \n");
  GST_DEBUG (0,"  load_intra_quantizer_matrix=%d\n",
    ld->load_intra_quantizer_matrix);
  GST_DEBUG (0,"  load_non_intra_quantizer_matrix=%d\n",
    ld->load_non_intra_quantizer_matrix);
  GST_DEBUG (0,"  load_chroma_intra_quantizer_matrix=%d\n",
    ld->load_chroma_intra_quantizer_matrix);
  GST_DEBUG (0,"  load_chroma_non_intra_quantizer_matrix=%d\n",
    ld->load_chroma_non_intra_quantizer_matrix);

#ifdef VERIFY
  verify_quant_matrix_extension++;
#endif /* VERIFY */

}


/* decode sequence scalable extension */
/* ISO/IEC 13818-2   section 6.2.2.5 */
static void sequence_scalable_extension(mpeg2play_vid_stream *vid_stream)
{
  mpeg2play_Sequence *seq = &vid_stream->sequence;
  gst_getbits_t *gb = &vid_stream->ld->gb;

  /* values (without the +1 offset) of scalable_mode are defined in 
     Table 6-10 of ISO/IEC 13818-2 */
  vid_stream->ld->scalable_mode = gst_getbits2(gb) + 1; /* add 1 to make SC_DP != SC_NONE */

  seq->layer_id = gst_getbits4(gb);

  if (vid_stream->ld->scalable_mode==SC_SPAT)
  {
    seq->lower_layer_prediction_horizontal_size = gst_getbits14(gb);
    marker_bit(gb, "sequence_scalable_extension()");
    seq->lower_layer_prediction_vertical_size   = gst_getbits14(gb); 
    seq->horizontal_subsampling_factor_m        = gst_getbits5(gb);
    seq->horizontal_subsampling_factor_n        = gst_getbits5(gb);
    seq->vertical_subsampling_factor_m          = gst_getbits5(gb);
    seq->vertical_subsampling_factor_n          = gst_getbits5(gb);
  }

  if (vid_stream->ld->scalable_mode==SC_TEMP)
    Error("temporal scalability not implemented\n");

  GST_DEBUG (0,"sequence scalable extension\n");
  GST_DEBUG (0,"  scalable_mode=%d\n",vid_stream->ld->scalable_mode-1);
  GST_DEBUG (0,"  layer_id=%d\n",seq->layer_id);
  if (vid_stream->ld->scalable_mode==SC_SPAT)
  {
    GST_DEBUG (0,"    lower_layer_prediction_horiontal_size=%d\n",
      seq->lower_layer_prediction_horizontal_size);
    GST_DEBUG (0,"    lower_layer_prediction_vertical_size=%d\n",
      seq->lower_layer_prediction_vertical_size);
    GST_DEBUG (0,"    horizontal_subsampling_factor_m=%d\n",
      seq->horizontal_subsampling_factor_m);
    GST_DEBUG (0,"    horizontal_subsampling_factor_n=%d\n",
      seq->horizontal_subsampling_factor_n);
    GST_DEBUG (0,"    vertical_subsampling_factor_m=%d\n",
      seq->vertical_subsampling_factor_m);
    GST_DEBUG (0,"    vertical_subsampling_factor_n=%d\n",
      seq->vertical_subsampling_factor_n);
  }

#ifdef VERIFY
  verify_sequence_scalable_extension++;
#endif /* VERIFY */

}


/* decode picture display extension */
/* ISO/IEC 13818-2 section 6.2.3.3. */
static void picture_display_extension(mpeg2play_vid_stream *vid_stream)
{
  int i;
  int number_of_frame_center_offsets;
  gst_getbits_t *gb = &vid_stream->ld->gb;
  mpeg2play_Pict *pict = &vid_stream->picture;

  /* based on ISO/IEC 13818-2 section 6.3.12 
    (November 1994) Picture display extensions */

  /* derive number_of_frame_center_offsets */
  if(vid_stream->sequence.progressive_sequence)
  {
    if(pict->repeat_first_field)
    {
      if(pict->top_field_first)
        number_of_frame_center_offsets = 3;
      else
        number_of_frame_center_offsets = 2;
    }
    else
    {
      number_of_frame_center_offsets = 1;
    }
  }
  else
  {
    if(pict->picture_structure!=FRAME_PICTURE)
    {
      number_of_frame_center_offsets = 1;
    }
    else
    {
      if(pict->repeat_first_field)
        number_of_frame_center_offsets = 3;
      else
        number_of_frame_center_offsets = 2;
    }
  }


  /* now parse */
  for (i=0; i<number_of_frame_center_offsets; i++)
  {
    pict->frame_center_horizontal_offset[i] = gst_getbits16(gb);
    marker_bit(gb, "picture_display_extension, first marker bit");
    
    pict->frame_center_vertical_offset[i]   = gst_getbits16(gb);
    marker_bit(gb, "picture_display_extension, second marker bit");
  }

  GST_DEBUG (0,"picture display extension\n");

  for (i=0; i<number_of_frame_center_offsets; i++)
  {
    GST_DEBUG (0,"  frame_center_horizontal_offset[%d]=%d\n",i,
      pict->frame_center_horizontal_offset[i]);
    GST_DEBUG (0,"  frame_center_vertical_offset[%d]=%d\n",i,
      pict->frame_center_vertical_offset[i]);
  }

#ifdef VERIFY
  verify_picture_display_extension++;
#endif /* VERIFY */

}


/* decode picture coding extension */
static void picture_coding_extension(mpeg2play_vid_stream *vid_stream)
{
  gst_getbits_t *gb = &vid_stream->ld->gb;
  mpeg2play_Pict *pict = &vid_stream->picture;

  pict->f_code[0][0] = gst_getbits4(gb);
  pict->f_code[0][1] = gst_getbits4(gb);
  pict->f_code[1][0] = gst_getbits4(gb);
  pict->f_code[1][1] = gst_getbits4(gb);

  pict->intra_dc_precision         = gst_getbits2(gb);
  pict->picture_structure          = gst_getbits2(gb);
  pict->top_field_first            = gst_getbits1(gb);
  pict->frame_pred_frame_dct       = gst_getbits1(gb);
  pict->concealment_motion_vectors = gst_getbits1(gb);
  vid_stream->ld->q_scale_type           = gst_getbits1(gb);
  pict->intra_vlc_format           = gst_getbits1(gb);
  vid_stream->ld->alternate_scan         = gst_getbits1(gb);
  pict->repeat_first_field         = gst_getbits1(gb);
  pict->chroma_420_type            = gst_getbits1(gb);
  pict->progressive_frame          = gst_getbits1(gb);
  pict->composite_display_flag     = gst_getbits1(gb);

  if (pict->composite_display_flag)
  {
    pict->v_axis            = gst_getbits1(gb);
    pict->field_sequence    = gst_getbits3(gb);
    pict->sub_carrier       = gst_getbits1(gb);
    pict->burst_amplitude   = gst_getbits7(gb);
    pict->sub_carrier_phase = gst_getbits8(gb);
  }

  GST_DEBUG (0,"picture coding extension\n");
  GST_DEBUG (0,"  forward horizontal f_code=%d\n", pict->f_code[0][0]);
  GST_DEBUG (0,"  forward vertical f_code=%d\n", pict->f_code[0][1]);
  GST_DEBUG (0,"  backward horizontal f_code=%d\n", pict->f_code[1][0]);
  GST_DEBUG (0,"  backward_vertical f_code=%d\n", pict->f_code[1][1]);
  GST_DEBUG (0,"  intra_dc_precision=%d\n",pict->intra_dc_precision);
  GST_DEBUG (0,"  picture_structure=%d\n",pict->picture_structure);
  GST_DEBUG (0,"  top_field_first=%d\n",pict->top_field_first);
  GST_DEBUG (0,"  frame_pred_frame_dct=%d\n",pict->frame_pred_frame_dct);
  GST_DEBUG (0,"  concealment_motion_vectors=%d\n",pict->concealment_motion_vectors);
  GST_DEBUG (0,"  q_scale_type=%d\n",vid_stream->ld->q_scale_type);
  GST_DEBUG (0,"  intra_vlc_format=%d\n",pict->intra_vlc_format);
  GST_DEBUG (0,"  alternate_scan=%d\n",vid_stream->ld->alternate_scan);
  GST_DEBUG (0,"  repeat_first_field=%d\n",pict->repeat_first_field);
  GST_DEBUG (0,"  chroma_420_type=%d\n",pict->chroma_420_type);
  GST_DEBUG (0,"  progressive_frame=%d\n",pict->progressive_frame);
  GST_DEBUG (0,"  composite_display_flag=%d\n",pict->composite_display_flag);

  if (pict->composite_display_flag)
  {
    GST_DEBUG (0,"    v_axis=%d\n",pict->v_axis);
    GST_DEBUG (0,"    field_sequence=%d\n",pict->field_sequence);
    GST_DEBUG (0,"    sub_carrier=%d\n",pict->sub_carrier);
    GST_DEBUG (0,"    burst_amplitude=%d\n",pict->burst_amplitude);
    GST_DEBUG (0,"    sub_carrier_phase=%d\n",pict->sub_carrier_phase);
  }

#ifdef VERIFY
  verify_picture_coding_extension++;
#endif /* VERIFY */
}


/* decode picture spatial scalable extension */
/* ISO/IEC 13818-2 section 6.2.3.5. */
static void picture_spatial_scalable_extension(mpeg2play_vid_stream *vid_stream)
{
  gst_getbits_t *gb = &vid_stream->ld->gb;

  vid_stream->ld->pict_scal = 1; /* use spatial scalability in this picture */

  vid_stream->picture.lower_layer_temporal_reference = gst_getbits10(gb);
  marker_bit(gb, "picture_spatial_scalable_extension(), first marker bit");
  vid_stream->picture.lower_layer_horizontal_offset = gst_getbits15(gb);
  if (vid_stream->picture.lower_layer_horizontal_offset>=16384)
    vid_stream->picture.lower_layer_horizontal_offset-= 32768;
  marker_bit(gb, "picture_spatial_scalable_extension(), second marker bit");
  vid_stream->picture.lower_layer_vertical_offset = gst_getbits15(gb);
  if (vid_stream->picture.lower_layer_vertical_offset>=16384)
    vid_stream->picture.lower_layer_vertical_offset-= 32768;
  vid_stream->picture.spatial_temporal_weight_code_table_index = gst_getbits2(gb);
  vid_stream->picture.lower_layer_progressive_frame = gst_getbits1(gb);
  vid_stream->picture.lower_layer_deinterlaced_field_select = gst_getbits1(gb);

  GST_DEBUG (0,"picture spatial scalable extension \n");
  GST_DEBUG (0,"  lower_layer_temporal_reference=%d\n",vid_stream->picture.lower_layer_temporal_reference);
  GST_DEBUG (0,"  lower_layer_horizontal_offset=%d\n",vid_stream->picture.lower_layer_horizontal_offset);
  GST_DEBUG (0,"  lower_layer_vertical_offset=%d\n",vid_stream->picture.lower_layer_vertical_offset);
  GST_DEBUG (0,"  spatial_temporal_weight_code_table_index=%d\n",
    vid_stream->picture.spatial_temporal_weight_code_table_index);
  GST_DEBUG (0,"  lower_layer_progressive_frame=%d\n",vid_stream->picture.lower_layer_progressive_frame);
  GST_DEBUG (0,"  lower_layer_deinterlaced_field_select=%d\n",vid_stream->picture.lower_layer_deinterlaced_field_select);

#ifdef VERIFY
  verify_picture_spatial_scalable_extension++;
#endif /* VERIFY */

}


/* decode picture temporal scalable extension
 *
 * not implemented
 */
/* ISO/IEC 13818-2 section 6.2.3.4. */
static void picture_temporal_scalable_extension(mpeg2play_vid_stream *vid_stream)
{
  Error("temporal scalability not supported\n");

#ifdef VERIFY
  verify_picture_temporal_scalable_extension++;
#endif /* VERIFY */
}


/* decode extra bit information */
/* ISO/IEC 13818-2 section 6.2.3.4. */
static int extra_bit_information(mpeg2play_vid_stream *vid_stream)
{
  int Byte_Count = 0;
  gst_getbits_t *gb = &vid_stream->ld->gb;

  while (gst_getbits1(gb))
  {
    gst_flushbitsn(gb, 8);
    Byte_Count++;
  }

  return(Byte_Count);
}



/* ISO/IEC 13818-2 section 5.3 */
/* Purpose: this function is mainly designed to aid in bitstream conformance
   testing.  A simple Flush_Buffer(1) would do */
void marker_bit(gb, text)
gst_getbits_t *gb;
char *text;
{
  int marker;

  marker = gst_getbits1(gb);

#ifdef VERIFY  
  if(!marker)
    GST_DEBUG (0,"ERROR: %s--marker_bit set to 0",text);
#endif
}


/* ISO/IEC 13818-2  sections 6.3.4.1 and 6.2.2.2.2 */
static void user_data(mpeg2play_vid_stream *vid_stream)
{
  /* skip ahead to the next start code */
  mpeg2play_next_start_code(&vid_stream->ld->gb);
}



/* Copyright extension */
/* ISO/IEC 13818-2 section 6.2.3.6. */
/* (header added in November, 1994 to the IS document) */


static void copyright_extension(mpeg2play_vid_stream *vid_stream)
{
  int reserved_data;
  gst_getbits_t *gb = &vid_stream->ld->gb;

  vid_stream->copyright.copyright_flag =       gst_getbits1(gb); 
  vid_stream->copyright.copyright_identifier = gst_getbits8(gb);
  vid_stream->copyright.original_or_copy =     gst_getbits1(gb);
  
  /* reserved */
  reserved_data = gst_getbits7(gb);

  marker_bit(gb, "copyright_extension(), first marker bit");
  vid_stream->copyright.copyright_number_1 =   gst_getbits20(gb);
  marker_bit(gb, "copyright_extension(), second marker bit");
  vid_stream->copyright.copyright_number_2 =   gst_getbits22(gb);
  marker_bit(gb, "copyright_extension(), third marker bit");
  vid_stream->copyright.copyright_number_3 =   gst_getbits22(gb);

  GST_DEBUG (0,"copyright_extension \n");
  GST_DEBUG (0,"  copyright_flag =%d\n",vid_stream->copyright.copyright_flag);
        
  GST_DEBUG (0,"  copyright_identifier=%d\n",vid_stream->copyright.copyright_identifier);
        
  GST_DEBUG (0,"  original_or_copy = %d (original=1, copy=0)\n",
    vid_stream->copyright.original_or_copy);
        
  GST_DEBUG (0,"  copyright_number_1=%d\n",vid_stream->copyright.copyright_number_1);
  GST_DEBUG (0,"  copyright_number_2=%d\n",vid_stream->copyright.copyright_number_2);
  GST_DEBUG (0,"  copyright_number_3=%d\n",vid_stream->copyright.copyright_number_3);

#ifdef VERIFY
  verify_copyright_extension++;
#endif /* VERIFY */
}



/* introduced in September 1995 to assist Spatial Scalability */
static void Update_Temporal_Reference_Tacking_Data(mpeg2play_vid_stream *vid_stream)
{
  static int temporal_reference_wrap  = 0;
  static int temporal_reference_old   = 0;
  mpeg2play_layer_data *ld = vid_stream->ld;

  if (ld == &vid_stream->base)			/* *CH* */
  {
    if (vid_stream->picture.picture_coding_type!=B_TYPE && 
		    vid_stream->picture.temporal_reference!=temporal_reference_old) 	
    /* check first field of */
    {							
       /* non-B-frame */
      if (temporal_reference_wrap) 		
      {/* wrap occured at previous I- or P-frame */	
       /* now all intervening B-frames which could 
          still have high temporal_reference values are done  */
        Temporal_Reference_Base += 1024;
	    temporal_reference_wrap = 0;
      }
      
      /* distinguish from a reset */
      if (vid_stream->picture.temporal_reference<temporal_reference_old && !Temporal_Reference_GOP_Reset)	
	    temporal_reference_wrap = 1;  /* we must have just passed a GOP-Header! */
      
      temporal_reference_old = vid_stream->picture.temporal_reference;
      Temporal_Reference_GOP_Reset = 0;
    }

    vid_stream->True_Framenum = Temporal_Reference_Base + vid_stream->picture.temporal_reference;
    
    /* temporary wrap of TR at 1024 for M frames */
    if (temporal_reference_wrap && vid_stream->picture.temporal_reference <= temporal_reference_old)	
      vid_stream->True_Framenum += 1024;				

    True_Framenum_max = (vid_stream->True_Framenum > True_Framenum_max) ?
                        vid_stream->True_Framenum : True_Framenum_max;
  }
}
