/* Gnome-Streamer
 * Copyright (C) <1999> Erik Walthinsen <omega@cse.ogi.edu>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <string.h>

//#define DEBUG_ENABLED
#include "gstmpeg2play.h"

/* elementfactory information */
static GstElementDetails gst_mpeg2play_details = {
  "mpeg1 and mpeg2 video decoder",
  "Filter/Decoder/Video",
  "Uses modified mpeg2decode V1.2a to decode MPEG video streams",
  VERSION,
  "(C) 1996, MPEG Software Simulation Group\n"
  "Wim Taymans <wim.taymans@tvd.be>",
  "(C) 2000",
};

/* Mpeg2play signals and args */
enum {
  /* FILL ME */
  LAST_SIGNAL
};

enum {
  ARG_0,
  ARG_FPS,
  /* FILL ME */
};

static GstPadTemplate*
src_factory (void) 
{
  return 
   gst_padtemplate_new (
  	"src",
  	GST_PAD_SRC,
  	GST_PAD_ALWAYS,
  	gst_caps_new (
  	  "mpeg2play_src",
    	  "video/raw",
	  gst_props_new (
    	    "format",    GST_PROPS_LIST (
            		       	GST_PROPS_FOURCC (GST_MAKE_FOURCC ('Y','V','1','2')),
                   		GST_PROPS_FOURCC (GST_MAKE_FOURCC ('I','Y','U','V')),
                   		GST_PROPS_FOURCC (GST_MAKE_FOURCC ('I','4','2','0'))
                 	 ),
    	    "width",     GST_PROPS_INT_RANGE (16, 4096),
    	    "height",    GST_PROPS_INT_RANGE (16, 4096),
	    NULL)),
	NULL);
}

static GstPadTemplate*
sink_factory (void) 
{
  return 
   gst_padtemplate_new (
  	"sink",
  	GST_PAD_SINK,
  	GST_PAD_ALWAYS,
  	gst_caps_new (
  	  "mpeg2play_sink",
    	  "video/mpeg",
	  gst_props_new (
    	    "mpegversion",  GST_PROPS_INT (2),
    	    "systemstream", GST_PROPS_BOOLEAN (FALSE),
    	    "sliced",       GST_PROPS_BOOLEAN (TRUE),
	    NULL)),
	NULL);
}

static void	gst_mpeg2play_class_init	(GstMpeg2playClass *klass);
static void	gst_mpeg2play_init		(GstMpeg2play *mpeg2play);

static void	gst_mpeg2play_set_property		(GObject *object, guint prop_id, const GValue *value, GParamSpec *pspec);
static void	gst_mpeg2play_get_property		(GObject *object, guint prop_id, GValue *value, GParamSpec *pspec);

static void	gst_mpeg2play_chain		(GstPad *pad, GstBuffer *buf);

static GstPadTemplate *src_template, *sink_template;

static GstElementClass *parent_class = NULL;
//static guint gst_mpeg2play_signals[LAST_SIGNAL] = { 0 };

GType
gst_mpeg2play_get_type (void)
{
  static GType mpeg2play_type = 0;

  if (!mpeg2play_type) {
    static const GTypeInfo mpeg2play_info = {
      sizeof(GstMpeg2playClass),      NULL,
      NULL,
      (GClassInitFunc)gst_mpeg2play_class_init,
      NULL,
      NULL,
      sizeof(GstMpeg2play),
      0,
      (GInstanceInitFunc)gst_mpeg2play_init,
    };
    mpeg2play_type = g_type_register_static(GST_TYPE_ELEMENT, "GstMpeg2play", &mpeg2play_info, 0);
  }
  return mpeg2play_type;
}

static void
gst_mpeg2play_class_init (GstMpeg2playClass *klass)
{
  GObjectClass *gobject_class;
  GstElementClass *gstelement_class;

  gobject_class = (GObjectClass*)klass;
  gstelement_class = (GstElementClass*)klass;

  g_object_class_install_property(G_OBJECT_CLASS(klass), ARG_FPS,
    g_param_spec_double("frames_per_second","frames_per_second","frames_per_second",
                        G_MINDOUBLE,G_MAXDOUBLE,0,G_PARAM_READABLE)); // CHECKME

  parent_class = g_type_class_ref(GST_TYPE_ELEMENT);

  gobject_class->set_property = gst_mpeg2play_set_property;
  gobject_class->get_property = gst_mpeg2play_get_property;

}

static void
gst_mpeg2play_init (GstMpeg2play *mpeg2play)
{
  /* create the sink and src pads */
  mpeg2play->sinkpad = gst_pad_new_from_template (sink_template, "sink");
  gst_element_add_pad(GST_ELEMENT(mpeg2play),mpeg2play->sinkpad);
  gst_pad_set_chain_function(mpeg2play->sinkpad,gst_mpeg2play_chain);

  mpeg2play->srcpad = gst_pad_new_from_template (src_template, "src");
  gst_element_add_pad(GST_ELEMENT(mpeg2play),mpeg2play->srcpad);

  /* initialize the mpeg2play decoder state */
  mpeg2play->decoder = mpeg2play_new_decoder();
  mpeg2play->parse_state = FIRST_BLOCK;
  mpeg2play->next_time = 0;

  // reset the initial video state
  mpeg2play->format = -1;
  mpeg2play->width = -1;
  mpeg2play->height = -1;
}

static void
gst_mpeg2play_chain (GstPad *pad, GstBuffer *buf)
{
  GstMpeg2play *mpeg2play;
  guchar *data;
  gulong size;
  GstBuffer *outbuf;

  g_return_if_fail(pad != NULL);
  g_return_if_fail(GST_IS_PAD(pad));
  g_return_if_fail(buf != NULL);
  //g_return_if_fail(GST_IS_BUFFER(buf));

  mpeg2play = GST_MPEG2PLAY(GST_OBJECT_PARENT (pad));

  data = (guchar *)GST_BUFFER_DATA(buf);
  size = GST_BUFFER_SIZE(buf);
  GST_DEBUG (0,"gst_mpeg2play_chain: got buffer of %ld bytes in '%s'\n",size,
          GST_OBJECT_NAME (mpeg2play));

  if (GST_BUFFER_FLAG_IS_SET(buf, GST_BUFFER_FLUSH)) {
    mpeg2play->next_time = 0;
  }
  // we have a new block
  mpeg2play->parse_state |= START_BLOCK;
  mpeg2play->parse_state &= ~FINISHED_BLOCK;

  //fprintf(stderr, "gst_mpeg2play_chain: state before loop%04X\n", mpeg2play->parse_state);

  // while we have not finished this block
  while (!(mpeg2play->parse_state & FINISHED_BLOCK)) {
    //fprintf(stderr, "gst_mpeg2play_chain: state in loop %04X\n", mpeg2play->parse_state);
    GST_DEBUG (0,"gst_mpeg2play_chain: about to decode a frame\n");

    // decode this block
    mpeg2play->parse_state = mpeg2play_new_buffer(mpeg2play->decoder, data, size, mpeg2play->parse_state);
    //fprintf(stderr, "gst_mpeg2play_chain: state  after decode %04X\n", mpeg2play->parse_state);

    GST_DEBUG (0,"gst_mpeg2play_chain: decoded  frame %p\n", mpeg2play->decoder->output_frame);

		// the decoding gave a new picture
    if ((mpeg2play->parse_state & NEW_PICTURE) && mpeg2play->decoder->output_frame) {
      mpeg2play->next_time = mpeg2play->next_time+mpeg2play->time_interval;

      if (GST_PAD_CONNECTED(mpeg2play->srcpad)) {
        outbuf = gst_buffer_new();

        // see if we have a new size
        if ((MPEG2PLAY_WIDTH(mpeg2play->decoder) != mpeg2play->width) ||
          (MPEG2PLAY_HEIGHT(mpeg2play->decoder) != mpeg2play->height) ||
          (MPEG2PLAY_FORMAT(mpeg2play->decoder) != mpeg2play->format)) {


	  mpeg2play->format = MPEG2PLAY_FORMAT(mpeg2play->decoder);
	  mpeg2play->width = MPEG2PLAY_WIDTH(mpeg2play->decoder);
	  mpeg2play->height = MPEG2PLAY_HEIGHT(mpeg2play->decoder);

          mpeg2play->time_interval = (guint64)(1000000.0/mpeg2play->decoder->frame_rate);

          gst_info("gst_mpeg2play_chain: MPEG%d %dx%d, %g fps\n", mpeg2play->decoder->base.MPEG2_Flag+1, mpeg2play->width,
			  mpeg2play->height, mpeg2play->decoder->frame_rate);

          mpeg2play->outsize = mpeg2play->width * mpeg2play->height+
			          mpeg2play->width * mpeg2play->height / 2;

        }

        GST_DEBUG (0,"gst_mpeg2play_chain: making buffer with timestamp %lld\n", mpeg2play->next_time);
        GST_BUFFER_TIMESTAMP(outbuf) = mpeg2play->next_time;
        GST_BUFFER_SIZE(outbuf) = mpeg2play->outsize;
        GST_BUFFER_DATA(outbuf) = g_malloc(mpeg2play->outsize);

	// send off the luminance info
        memcpy(GST_BUFFER_DATA(outbuf), mpeg2play->decoder->output_frame, mpeg2play->outsize);
        //GST_BUFFER_DATA(outbuf) =  mpeg2play->decoder->backward_reference_frame[0];
	// keep a reference to the data because we still need it

        gst_pad_push(mpeg2play->srcpad,outbuf);
      }
    }
  }

  gst_buffer_unref(buf);
}

static void
gst_mpeg2play_set_property (GObject *object, guint prop_id, const GValue *value, GParamSpec *pspec)
{
  GstMpeg2play *src;

  /* it's not null if we got it, but it might not be ours */
  g_return_if_fail(GST_IS_MPEG2PLAY(object));
  src = GST_MPEG2PLAY(object);

  switch (prop_id) {
    case ARG_FPS:
      break;
    default:
      break;
  }
}

static void
gst_mpeg2play_get_property (GObject *object, guint prop_id, GValue *value, GParamSpec *pspec)
{
  GstMpeg2play *src;

  /* it's not null if we got it, but it might not be ours */
  g_return_if_fail(GST_IS_MPEG2PLAY(object));
  src = GST_MPEG2PLAY(object);

  switch (prop_id) {
    case ARG_FPS:
      g_value_set_double (value, src->decoder->frame_rate);
      break;
    default:
      break;
  }
}

static gboolean
plugin_init (GModule *module, GstPlugin *plugin)
{
  GstElementFactory *factory;

  /* this filter needs the getbits package */
  if (!gst_library_load("gstgetbits")) {
    gst_info("mpeg2play:: could not load support library: 'gstgetbits'\n");
    return FALSE;
  }
  if (!gst_library_load("gstidct")) {
    gst_info("mpeg2play:: could not load support library: 'gstidct'\n");
    return FALSE;
  }

  /* create an elementfactory for the mpeg2play element */
  factory = gst_elementfactory_new("mpeg2play",GST_TYPE_MPEG2PLAY,
                                   &gst_mpeg2play_details);
  g_return_val_if_fail(factory != NULL, FALSE);

  src_template = src_factory ();
  gst_elementfactory_add_padtemplate (factory, src_template);

  sink_template = sink_factory ();
  gst_elementfactory_add_padtemplate (factory, sink_template);

  gst_plugin_add_factory(plugin,factory);

  return TRUE;
}

GstPluginDesc plugin_desc = {
  GST_VERSION_MAJOR,
  GST_VERSION_MINOR,
  "mpeg2play",
  plugin_init
};
