
/* mpeg2dec.c, main(), initialization, option processing                    */

/* Copyright (C) 1996, MPEG Software Simulation Group. All Rights Reserved. */

/*
 * Disclaimer of Warranty
 *
 * These software programs are available to the user without any license fee or
 * royalty on an "as is" basis.  The MPEG Software Simulation Group disclaims
 * any and all warranties, whether express, implied, or statuary, including any
 * implied warranties or merchantability or of fitness for a particular
 * purpose.  In no event shall the copyright-holder be liable for any
 * incidental, punitive, or consequential damages of any kind whatsoever
 * arising from the use of these programs.
 *
 * This disclaimer of warranty extends to the user of these programs and user's
 * customers, employees, agents, transferees, successors, and assigns.
 *
 * The MPEG Software Simulation Group does not represent or warrant that the
 * programs furnished hereunder are free of infringement of any third-party
 * patents.
 *
 * Commercial implementations of MPEG-1 and MPEG-2 video, including shareware,
 * are subject to royalty fees to patent holders.  Many of these patents are
 * general enough such that they are unavoidable regardless of implementation
 * design.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <fcntl.h>

//#define DEBUG_ENABLED
#include <gst/gst.h>
#include "mpeg2dec.h"


/* zig-zag and alternate scan patterns */
static unsigned char scan_normal[2][64]
=
{
  { /* Zig-Zag scan pattern  */
    0,1,8,16,9,2,3,10,17,24,32,25,18,11,4,5,
    12,19,26,33,40,48,41,34,27,20,13,6,7,14,21,28,
    35,42,49,56,57,50,43,36,29,22,15,23,30,37,44,51,
    58,59,52,45,38,31,39,46,53,60,61,54,47,55,62,63
  },
  { /* Alternate scan pattern */
    0,8,16,24,1,9,2,10,17,25,32,40,48,56,57,49,
    41,33,26,18,3,11,4,12,19,27,34,42,50,58,35,43,
    51,59,20,28,5,13,6,14,21,29,36,44,52,60,37,45,
    53,61,22,30,7,15,23,31,38,46,54,62,39,47,55,63
  }
}
;

/* zig-zag and alternate scan patterns */
static unsigned char scan_transpose[2][64]
=
{
  { /* Zig-Zag scan pattern  */
    0*8+0/* 0*/, 1*8+0/* 1*/, 0*8+1/* 8*/, 0*8+2/*16*/, 1*8+1/* 9*/, 2*8+0/* 2*/, 3*8+0/* 3*/, 2*8+1/*10*/,
    1*8+2/*17*/, 0*8+3/*24*/, 0*8+4/*32*/, 1*8+3/*25*/, 2*8+2/*18*/, 3*8+1/*11*/, 4*8+0/* 4*/, 5*8+0/* 5*/,
    4*8+1/*12*/, 3*8+2/*19*/, 2*8+3/*26*/, 1*8+4/*33*/, 0*8+5/*40*/, 0*8+6/*48*/, 1*8+5/*41*/, 2*8+4/*34*/,
    3*8+3/*27*/, 4*8+2/*20*/, 5*8+1/*13*/, 6*8+0/* 6*/, 7*8+0/* 7*/, 6*8+1/*14*/, 5*8+2/*21*/, 4*8+3/*28*/,
    3*8+4/*35*/, 2*8+5/*42*/, 1*8+6/*49*/, 0*8+7/*56*/, 1*8+7/*57*/, 2*8+6/*50*/, 3*8+5/*43*/, 4*8+4/*36*/,
    5*8+3/*29*/, 6*8+2/*22*/, 7*8+1/*15*/, 7*8+2/*23*/, 6*8+3/*30*/, 5*8+4/*37*/, 4*8+5/*44*/, 3*8+6/*51*/,
    2*8+7/*58*/, 3*8+7/*59*/, 4*8+6/*52*/, 5*8+5/*45*/, 6*8+4/*38*/, 7*8+3/*31*/, 7*8+4/*39*/, 6*8+5/*46*/,
    5*8+6/*53*/, 4*8+7/*60*/, 5*8+7/*61*/, 6*8+6/*54*/, 7*8+5/*47*/, 7*8+6/*55*/, 6*8+7/*62*/, 7*8+7/*63*/
  },
  { /* Alternate scan pattern */
    0*8+0/*0 */, 0*8+1/* 8*/, 0*8+2/*16*/, 0*8+3/*24*/, 1*8+0/* 1*/, 1*8+1/* 9*/, 2*8+0/* 2*/, 2*8+1/*10*/,
    1*8+2/*17*/, 1*8+3/*25*/, 0*8+4/*32*/, 0*8+5/*40*/, 0*8+6/*48*/, 0*8+7/*56*/, 1*8+7/*57*/, 1*8+6/*49*/,
    1*8+5/*41*/, 1*8+4/*33*/, 2*8+3/*26*/, 2*8+2/*18*/, 3*8+0/* 3*/, 3*8+1/*11*/, 4*8+0/* 4*/, 4*8+1/*12*/,
    3*8+2/*19*/, 3*8+3/*27*/, 2*8+4/*34*/, 2*8+5/*42*/, 2*8+6/*50*/, 2*8+7/*58*/, 3*8+4/*35*/, 3*8+5/*43*/,
    3*8+6/*51*/, 3*8+7/*59*/, 4*8+2/*20*/, 4*8+3/*28*/, 5*8+0/* 5*/, 5*8+1/*13*/, 6*8+0/* 6*/, 6*8+1/*14*/,
    5*8+2/*21*/, 5*8+3/*29*/, 4*8+4/*36*/, 4*8+5/*44*/, 4*8+6/*52*/, 4*8+7/*60*/, 5*8+4/*37*/, 5*8+5/*45*/,
    5*8+6/*53*/, 5*8+7/*61*/, 6*8+2/*22*/, 6*8+3/*30*/, 7*8+0/* 7*/, 7*8+1/*15*/, 7*8+2/*23*/, 7*8+3/*31*/,
    6*8+4/*38*/, 6*8+5/*46*/, 6*8+6/*54*/, 6*8+7/*62*/, 7*8+4/*39*/, 7*8+5/*47*/, 7*8+6/*55*/, 7*8+6/*63*/
  }
}
;

static void Initialize_Sequence(mpeg2play_vid_stream *vid_stream);

/* IMPLEMENTAION specific rouintes */
mpeg2play_vid_stream *mpeg2play_new_decoder()
{
  int i,j;
  mpeg2play_vid_stream *new;
  
  /* allocate memoy for the new decode */
  new = (mpeg2play_vid_stream *) malloc (sizeof(mpeg2play_vid_stream));

  /* Clip table */
  if (!(Clip=(unsigned char *)malloc(1024)))
    Error("Clip[] malloc failed\n");

  Clip += 384;

  for (i=-384; i<640; i++)
    Clip[i] = (i<0) ? 0 : ((i>255) ? 255 : i);

  /* IDCT */
  if (new->Reference_IDCT_Flag)
    new->idct = gst_idct_new(GST_IDCT_FAST_INT);
  else
    new->idct = gst_idct_new(GST_IDCT_DEFAULT);

  new->ld = &new->base;
  new->inited = 0;

  if (GST_IDCT_TRANSPOSE(new->idct)) {
    for (i=0; i<2; i++) {
      for (j=0; j<64; j++) {
        new->scan[i][j] = scan_transpose[i][j];
      }
    }
  }
  else {
    for (i=0; i<2; i++) {
      for (j=0; j<64; j++) {
        new->scan[i][j] = scan_normal[i][j];
      }
    }
  }

  mpeg2play_init_predictions();

  new->Verbose_Flag = 0;
  new->Output_Type = 0;
  new->hiQdither  = 0;
  new->Output_Type = 0;
  new->Frame_Store_Flag = 0;
  new->Spatial_Flag = 0;
  new->Reference_IDCT_Flag = 0;
  new->Trace_Flag = 0;
  new->Quiet_Flag = 0;
  new->Ersatz_Flag = 0;
  new->Two_Streams = 0;
  new->Big_Picture_Flag = 0;
  new->Main_Bitstream_Flag = 0;
  new->Verify_Flag = 0;
  new->Stats_Flag  = 0;
  new->User_Data_Flag = 0; 
  gst_getbits_init(&new->ld->gb, NULL, NULL);
  return new;
}

int mpeg2play_new_buffer (mpeg2play_vid_stream *vid_stream, char *inbuf, long inlen, int parse_state)
{
  unsigned long data;
  gboolean found = FALSE;
  gst_getbits_t *gb = &vid_stream->ld->gb;
  static int framenum = 0;
	  
#define DATA_IN_BUFFER ((inlen) - ((gst_getbits_bufferpos(gb)-vid_stream->swapbuf)<<2)) 

  GST_DEBUG (0,"new buffer %p\n", inbuf);

  parse_state &= ~NEW_PICTURE;

  // if we have a new block to parse, setup the buffers
  if (parse_state & START_BLOCK) {
					
    vid_stream->swapbuf = (unsigned long *)inbuf;

    /* Initialize bitstream i/o fields. */
    gst_getbits_newbuf(gb, inbuf, inlen);

    parse_state &= ~START_BLOCK;
  }

  /*
   * If called for the first time, find start code, make sure it is a
   * sequence start code.
   */
  if (parse_state & FIRST_BLOCK) {
    parse_state &= ~FIRST_BLOCK;
    //mpeg2play_next_start_code(vid_stream, DATA_IN_BUFFER);  /* sets curBits */
    mpeg2play_next_start_code(gb);  /* sets curBits */
    data = gst_showbits32(gb);
    GST_DEBUG (0,"next_code data=%08lx\n", data);
    if (data != SEQUENCE_HEADER_CODE) {
      fprintf(stderr, "mpeg_play: This is not an MPEG video stream. (%08lx)\n",data);
      parse_state |= FINISHED_BLOCK;
      return parse_state;
    }
  } 

  //next_start_code(vid_stream, DATA_IN_BUFFER);  
  mpeg2play_next_start_code(gb);  /* sets curBits */

  while (gst_getbits_bitsleft(gb) > 32) {
	  
    if (vid_stream->status == SKIP_PICTURE) {
      int nextbits = gst_showbits32(gb);
      if (nextbits != PICTURE_START_CODE) {
        if (nextbits ==  GROUP_START_CODE)  {
          found = TRUE;
	}
        else if (nextbits == SEQUENCE_END_CODE) {
          found = TRUE;
	}
      }
      else found = TRUE;
      if (!found) {
	gst_flushbitsn(gb, 24);
	goto done;
      }
    }
    
    data = gst_showbits32(gb);
    /* Get next 32 bits (size of start codes). */
//    GST_DEBUG (0,"showing 32 (%08lx) %ld %ld\n", data, DATA_IN_BUFFER, inlen);

    // if we have found something else than e slice and we were parsing it...
    if ((vid_stream->state == IN_SLICE_PARSE) && 
	   ((data < SLICE_START_CODE_MIN) || (data > SLICE_START_CODE_MAX)) &&
	    (data != SEQUENCE_ERROR_CODE)) {

      vid_stream->state = REGULAR_PARSE;
      Decode_Picture_End(vid_stream, framenum++, framenum);
      //DoPictureDisplay(vid_stream, 0);
      parse_state |= NEW_PICTURE | CONTINUE_BLOCK;
      return parse_state;
    }

    /*
     * Process according to start code (or parse macroblock if not a start code
     * at all).
     */

    switch (data) {

      case SEQUENCE_END_CODE:
      case ISO_END_CODE:   /*  handle ISO_11172_END_CODE too */

        /* Display last frame. */
        //if (vid_stream->future != NULL) {
          //vid_stream->current = vid_stream->future;
	  //DoPictureDisplay(vid_stream,0);
	  //parse_state |= NEW_PICTURE;
        //}
        //vid_stream->film_has_ended=TRUE;
	parse_state |= FINISHED_BLOCK;
	return parse_state;
        break;

      case SEQUENCE_HEADER_CODE:

        gst_flushbits32(gb);
        /* Sequence start code. Parse sequence header. */
        if (sequence_header(vid_stream) != PARSE_OK) {
          fprintf(stderr, "mpeg_play: error parsing seqhead !!\n");
          goto error;
        }
        vid_stream->state = IN_SEQ_PARSE;
        goto done;

      case GROUP_START_CODE:

        gst_flushbits32(gb);
        /* Group of Pictures start code. Parse gop header. */
        if (group_of_pictures_header(vid_stream) != PARSE_OK) {
          fprintf(stderr, "mpeg_play: error parsing gophead !!\n");
          goto error;
        }
        vid_stream->state = IN_GOP_PARSE;
        goto done;

      case PICTURE_START_CODE:

        gst_flushbits32(gb);
        /* Picture start code. Parse picture header and first slice header. */
        vid_stream->status = picture_header(vid_stream);

	Decode_Picture(vid_stream, framenum, framenum);
        vid_stream->state = IN_PICTURE_PARSE;

        if (vid_stream->status == SKIP_PICTURE) {
          gst_flushbitsn(gb, 24);
          //fprintf(stderr, "mpeg_play: status == SKIP_PICTURE  %p (%08x) %d %ld\n",vid_stream->buffer, data, (vid_stream->buffer-vid_stream->swapbuf)<<2, inlen);
          goto done;
        } else if (vid_stream->status != PARSE_OK) {
          fprintf(stderr, "mpeg_play: status  parse picture != PARSE_OK %d\n", vid_stream->status);
          goto error;
        }
        goto done;

      case USER_DATA_START_CODE:
      case EXTENSION_START_CODE:
        gst_flushbits32(gb);
	extension_and_user_data(vid_stream, data);
        if (vid_stream->state == IN_SEQ_PARSE && !vid_stream->inited) {
	  Initialize_Sequence(vid_stream);
	  vid_stream->inited = 1;
	}
        goto done;

      case SEQUENCE_ERROR_CODE:
	/* just goto next start code */
        gst_flushbits32(gb);
        goto done;
    
      default:

        /* Check for slice start code. */
        if ((data >= SLICE_START_CODE_MIN) && (data <= SLICE_START_CODE_MAX)) {
 	  int MBAmax;
	  int ret;

	  //* we're parsing slices */
	  vid_stream->state = IN_SLICE_PARSE;
	   
	  MBAmax = vid_stream->mb_width*vid_stream->mb_height;
	  if (vid_stream->picture.picture_structure!=FRAME_PICTURE)
	        MBAmax>>=1; /* field picture has half as mnay macroblocks as frame */

          /* Slice start code. Parse slice header. */
	  ret = slice(vid_stream, framenum, MBAmax);
          GST_DEBUG (0,"done slice %d %d\n", framenum, ret);
        }
        break;

    }
    goto done;

error:
    fprintf(stderr, "Error!!!! skipping..\n");

done:
    GST_DEBUG (0,"done: searching next start code..\n");
    //next_start_code(vid_stream, DATA_IN_BUFFER);
    if (mpeg2play_next_start_code(gb) == NO_STARTCODE) break;  /* sets curBits */
    //(stderr, "done: next start code done..\n");

  } /* end while */

  //(stderr, "mpegVidRsrc end \n");
  //free(vid_stream->swapbuf);
  parse_state |= FINISHED_BLOCK;
  return parse_state;
}

/* mostly IMPLEMENTAION specific rouintes */
static void Initialize_Sequence(mpeg2play_vid_stream *vid_stream)
{
  int cc, sizeY, sizeCrCb, size;
  static int Table_6_20[3] = {6,8,12};


  /* check scalability mode of enhancement layer */
  if (vid_stream->Two_Streams && (vid_stream->enhan.scalable_mode!=SC_SNR) && (vid_stream->base.scalable_mode!=SC_DP))
    Error("unsupported scalability mode\n");

  /* force MPEG-1 parameters for proper decoder behavior */
  /* see ISO/IEC 13818-2 section D.9.14 */
  if (!vid_stream->base.MPEG2_Flag)
  {
    GST_DEBUG (0,"setting MPEG1 defaults\n");
    vid_stream->sequence.progressive_sequence = 1;
    vid_stream->sequence.chroma_format = CHROMA420;
    vid_stream->sequence.matrix_coefficients = 5;
    vid_stream->picture.progressive_frame = 1;
    vid_stream->picture.picture_structure = FRAME_PICTURE;
    vid_stream->picture.frame_pred_frame_dct = 1;
  }
  {
    GST_DEBUG (0,"setting MPEG2 defaults\n");
  }

  /* round to nearest multiple of coded macroblocks */
  /* ISO/IEC 13818-2 section 6.3.3 sequence_header() */
  vid_stream->mb_width = (vid_stream->horizontal_size+15)/16;
  vid_stream->mb_height = (vid_stream->base.MPEG2_Flag && !vid_stream->sequence.progressive_sequence) ? 2*((vid_stream->vertical_size+31)/32)
                                        : (vid_stream->vertical_size+15)/16;

  vid_stream->Coded_Picture_Width = 16*vid_stream->mb_width;
  vid_stream->Coded_Picture_Height = 16*vid_stream->mb_height;

  /* ISO/IEC 13818-2 sections 6.1.1.8, 6.1.1.9, and 6.1.1.10 */
  vid_stream->Chroma_Width = (vid_stream->sequence.chroma_format==CHROMA444) ? vid_stream->Coded_Picture_Width
                                           : vid_stream->Coded_Picture_Width>>1;
  vid_stream->Chroma_Height = (vid_stream->sequence.chroma_format!=CHROMA420) ? vid_stream->Coded_Picture_Height
                                            : vid_stream->Coded_Picture_Height>>1;
  
  /* derived based on Table 6-20 in ISO/IEC 13818-2 section 6.3.17 */
  vid_stream->block_count = Table_6_20[vid_stream->sequence.chroma_format-1];

    sizeY = vid_stream->Coded_Picture_Width*vid_stream->Coded_Picture_Height;
    sizeCrCb = vid_stream->Chroma_Width*vid_stream->Chroma_Height;
    size = sizeY + sizeCrCb*2;

    if (!(vid_stream->backward_reference_frame[0] = (unsigned char *)malloc(size)))
      Error("backward_reference_frame[] malloc failed\n");
    vid_stream->backward_reference_frame[1] = vid_stream->backward_reference_frame[0] + sizeY;
    vid_stream->backward_reference_frame[2] = vid_stream->backward_reference_frame[1] + sizeCrCb;


    if (!(vid_stream->forward_reference_frame[0] = (unsigned char *)malloc(size)))
      Error("forward_reference_frame[] malloc failed\n");
    vid_stream->forward_reference_frame[1] = vid_stream->forward_reference_frame[0] + sizeY;
    vid_stream->forward_reference_frame[2] = vid_stream->forward_reference_frame[1] + sizeCrCb;

    if (!(vid_stream->auxframe[0] = (unsigned char *)malloc(size)))
      Error("auxframe[] malloc failed\n");
    vid_stream->auxframe[1] = vid_stream->auxframe[0] + sizeY;
    vid_stream->auxframe[2] = vid_stream->auxframe[1] + sizeCrCb;

    if(vid_stream->Ersatz_Flag) {
      if (!(vid_stream->substitute_frame[0] = (unsigned char *)malloc(size)))
        Error("substitute_frame[] malloc failed\n");
      vid_stream->substitute_frame[1] = vid_stream->substitute_frame[0] + sizeY;
      vid_stream->substitute_frame[2] = vid_stream->substitute_frame[1] + sizeCrCb;
    }

  for (cc=0; cc<3; cc++) {

    if (vid_stream->base.scalable_mode==SC_SPAT)
    {
      /* this assumes lower layer is 4:2:0 */
      if (!(llframe0[0] = (unsigned char *)malloc((vid_stream->sequence.lower_layer_prediction_horizontal_size*
					      vid_stream->sequence.lower_layer_prediction_vertical_size)/(cc?4:1))))
        Error("llframe0 malloc failed\n");
      if (!(llframe1[0] = (unsigned char *)malloc((vid_stream->sequence.lower_layer_prediction_horizontal_size*
					      vid_stream->sequence.lower_layer_prediction_vertical_size)/(cc?4:1))))
        Error("llframe1 malloc failed\n");
    }
  }

  /* SCALABILITY: Spatial */
  if (vid_stream->base.scalable_mode==SC_SPAT)
  {
    if (!(lltmp = (short *)malloc(vid_stream->sequence.lower_layer_prediction_horizontal_size*
		    ((vid_stream->sequence.lower_layer_prediction_vertical_size*vid_stream->sequence.vertical_subsampling_factor_n)/
		       vid_stream->sequence.vertical_subsampling_factor_m)*sizeof(short))))
      Error("lltmp malloc failed\n");
  }

}

void Error (char *text) 
{
  fprintf(stderr, text);
  exit(-1);
}

static void Deinitialize_Sequence(mpeg2play_vid_stream *vid_stream)
{
  int i;

  /* clear flags */
  vid_stream->base.MPEG2_Flag=0;

  for(i=0;i<3;i++)
  {
    free(vid_stream->backward_reference_frame[i]);
    free(vid_stream->forward_reference_frame[i]);
    free(vid_stream->auxframe[i]);

    if (vid_stream->base.scalable_mode==SC_SPAT)
    {
     free(llframe0[i]);
     free(llframe1[i]);
    }
  }

  if (vid_stream->base.scalable_mode==SC_SPAT)
    free(lltmp);
}

