/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/* gturing.c - a Turing machine simulator.
 * Copyright (C) 1998 The Free Software Foundation
 * Copyright (C) 2001-2002 German Poo-Caaman~o
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>

#include <gnome.h>
#include <libgnomeui/libgnomeui.h>
#include <libgnomeui/gnome-window.h>
#include <libgnomecanvas/libgnomecanvas.h>
#include <gconf/gconf-client.h>

#include "turing.h"
#include "gturing.h"
#include "graph_editor.h"

#define SCROLL_WIDTH 10000

#define STATE 0
#define READ 1
#define WRITE 2
#define MOVE 3
#define NEW_STATE 4
#define ENTRIES_MAX 5

#define BTN_POWER	helper_toolbar[3].widget
#define BTN_STOP	helper_toolbar[4].widget
#define BTN_PLAY	helper_toolbar[5].widget
#define BTN_STEP	helper_toolbar[6].widget
#define MENU_POWER	helper_action_menu[0].widget
#define MENU_STOP	helper_action_menu[1].widget
#define MENU_PLAY	helper_action_menu[2].widget
#define MENU_STEP	helper_action_menu[3].widget

/* A few globals. */
const gchar *font_name = "courier Bold 12";

static char *prog_message = NULL;
static char tape_string[1024] = "";
static char states_fname[1024] = "";

static long speed = -1;
static int stop_flag = 1;

static GtkWidget *rootw;
static GtkWidget *dialog = NULL;
static GtkWidget *entry;

static GtkWidget *vbox;
static GtkWidget *tapelabel;
static GtkWidget *headlabel;
/* static GtkWidget *hbox; */
static GtkTreeView *treeview_editor = NULL;
/* static GtkWidget *graph_editor = NULL; */

static GnomeAppBar *statusline;
/* static GtkWidget *save; */
/*static GtkWidget *edit_save = NULL;*/

GConfClient *gengine;
GError *gerror;


void
set_save_sens (gboolean saves)
{
	/* gtk_widget_set_sensitive (save, saves);

	   if (edit_save)
	   gtk_widget_set_sensitive (edit_save, saves); */
	return;
}

void
set_toolbar_sens (gboolean powers, gboolean stops,
		  gboolean plays, gboolean steps)
{
	g_return_if_fail (GTK_IS_WIDGET (BTN_POWER)
			  && GTK_IS_WIDGET (BTN_STOP));
	g_return_if_fail (GTK_IS_WIDGET (BTN_PLAY)
			  && GTK_IS_WIDGET (BTN_STEP));
	g_return_if_fail (GTK_IS_WIDGET (MENU_POWER)
			  && GTK_IS_WIDGET (MENU_STOP));
	g_return_if_fail (GTK_IS_WIDGET (MENU_PLAY)
			  && GTK_IS_WIDGET (MENU_STEP));

	gtk_widget_set_sensitive (BTN_POWER, powers);
	gtk_widget_set_sensitive (BTN_STOP, stops);
	gtk_widget_set_sensitive (BTN_PLAY, plays);
	gtk_widget_set_sensitive (BTN_STEP, steps);

	gtk_widget_set_sensitive (MENU_POWER, powers);
	gtk_widget_set_sensitive (MENU_STOP, stops);
	gtk_widget_set_sensitive (MENU_PLAY, plays);
	gtk_widget_set_sensitive (MENU_STEP, steps);

	return;
}

void
set_tape (char *str)
{
	int len, i;
	gchar *tmp;

	len = strlen (str);
	tmp = g_malloc (len + 1);

	for (i = 0; i < len; i++)
		if (i == tm->pos)
			tmp[i] = '^';
		else
			tmp[i] = ' ';

	tmp[len] = 0;

	gtk_label_set_text (GTK_LABEL (tapelabel), str);
	gtk_label_set_text (GTK_LABEL (headlabel), tmp);

	g_free (tmp);
}

/* view_comment: Show the comments gotten from file and show them in a message box */
void
view_comment (void)
{
	GtkWidget *dialog;
	gchar *mess;

	if (prog_message && *prog_message && *prog_message != '\n')
		mess = prog_message;
	else
		mess = _("No comment for this program.");

	dialog = gtk_message_dialog_new (GTK_WINDOW (rootw),
					 GTK_DIALOG_MODAL |
					 GTK_DIALOG_DESTROY_WITH_PARENT,
					 GTK_MESSAGE_INFO, GTK_BUTTONS_OK,
					 "%s", mess);
	gtk_dialog_run (GTK_DIALOG (dialog));
	gtk_widget_destroy (dialog);
}

void
set_states_fname (char *str)
{
	gchar *c;

	if ((*states_fname != 0) && (*tape_string != 0))
		power_do ();
	else
		set_toolbar_sens (FALSE, FALSE, FALSE, FALSE);

	if ((c = strrchr (states_fname, '/')) != NULL)
		c++;
	else
		c = states_fname;

	/*state_clist_load_states ();
	   state_clist_select_state (tm); */
	gtk_window_set_title (GTK_WINDOW (rootw), c);

	turing_table_editor_set_model (treeview_editor, tm);
}

void
states_fname_examples_callback (GtkWidget * ok_button, gpointer data)
{
	GtkFileSelection *fsel;

	fsel = GTK_FILE_SELECTION (data);
#define GTURING_EXAMPLES_DIR "foo"
	gtk_file_selection_set_filename (fsel, GTURING_EXAMPLES_DIR);
}

void
states_fname_callback (GtkWidget * ok_button, gpointer data)
{
	gboolean action_save;
	gchar *fname, *comment;
	GtkWidget *text;
	GtkTextBuffer *buffer;
	GtkTextIter start, end;

	fname =
	    g_strdup (gtk_file_selection_get_filename
		      (GTK_FILE_SELECTION (dialog)));

	action_save = (gboolean) data;

	if (action_save) {
		text = g_object_get_data (G_OBJECT (dialog), "text");
		/*
		   comment =
		   gtk_editable_get_chars (GTK_EDITABLE (text), 0, -1); */
		buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (text));
		gtk_text_buffer_get_bounds (buffer, &start, &end);
		comment =
		    gtk_text_buffer_get_text (buffer, &start, &end, FALSE);

		if (turing_fwrite_states (tm->statehead, fname, comment))
			*states_fname = 0;	/*error */
		else
			strncpy (states_fname, fname, 1024);

		if (prog_message)
			free (prog_message);

		prog_message = strdup (comment);
	} else {
		if (turing_fread_states (tm, fname)) {
			*states_fname = 0;	/*error */
		} else {
			strncpy (states_fname, fname, 1024);

			if (prog_message)
				free (prog_message);

			prog_message = turing_fread_comments (fname);
			view_comment ();
			/*
			   set_save_sens (TRUE); 
			 */
		}
	}

	set_states_fname (fname);
	g_free (fname);

	/*gnome_config_set_string ("/gTuring/Options/program", states_fname);*/
	gconf_client_set_string (gengine, "/apps/gturing/program",
				 states_fname, NULL);
	gtk_widget_destroy (dialog);
	dialog = NULL;
}

void
tape_string_callback (GtkWidget * ok_button, gpointer data)
{
	strncpy (tape_string, gtk_entry_get_text (GTK_ENTRY (entry)),
		 1024);
	turing_set_tape (tm, tape_string);
	set_tape (tape_string);

	if (tm->statehead && (*tape_string != 0))
		power_do ();
	else
		set_toolbar_sens (FALSE, FALSE, FALSE, FALSE);

	/*gnome_config_set_string ("/gTuring/Options/tape", tape_string);*/
	gconf_client_set_string (gengine, "/apps/gturing/tape",
				 tape_string, NULL);
}

void
speed_callback (gpointer data)
{
	speed = atol (gtk_entry_get_text (GTK_ENTRY (entry)));

	gconf_client_set_int (gengine, "/apps/gturing/speed", speed, NULL);
	/*gnome_config_set_int ("/gTuring/Options/speed", speed);*/
}

void
cancel_callback (GtkWidget * widget, gpointer data)
{
	/*void *p;

	   if ((p = gtk_object_get_user_data (GTK_OBJECT (widget))) != NULL)
	   g_free (p); */

	gtk_widget_destroy (dialog);
	dialog = NULL;
}

int
next_state (turing * t)
{
	int ret;
	char buff[20];

	ret = turing_run_state (t);

	if (t->actualstate) {
		snprintf (buff, 20, _("State: %d"), t->actualstate->no);
		gnome_appbar_set_status (GNOME_APPBAR (statusline), buff);
	} else {
		gnome_appbar_set_status (GNOME_APPBAR (statusline),
					 _("Stopped"));
	}

	return ret;
}

void
power_do (void)
{
	stop_flag = 1;
	tm->state = 0;
	tm->pos = 0;

	turing_set_tape (tm, tape_string);
	set_tape (tape_string);

	next_state (tm);
	set_toolbar_sens (FALSE, FALSE, TRUE, TRUE);
}


gint
do_play (gpointer data)
{
	char *tmp;
	int cont;

	cont = FALSE;

	if (!stop_flag) {
		tmp = turing_get_tape (tm);
		set_tape (tmp);
		free (tmp);

		cont = next_state (tm);

		if (!cont)
			set_toolbar_sens (TRUE, FALSE, stop_flag,
					  stop_flag);
	}

	return cont;
}

void
play_callback (GtkWidget * play_button, gpointer data)
{
	set_toolbar_sens (TRUE, TRUE, FALSE, FALSE);

	stop_flag = 0;
	gtk_timeout_add (speed, do_play, NULL);
}

void
step_callback (GtkWidget * step_buttton, gpointer data)
{
	char *tmp;

	tmp = turing_get_tape (tm);
	set_tape (tmp);
	free (tmp);

	if (!next_state (tm)) {
		set_toolbar_sens (TRUE, FALSE, FALSE, FALSE);
		gnome_appbar_set_status (GNOME_APPBAR (statusline),
					 _("Stopped"));
	}

	gtk_widget_set_sensitive (BTN_POWER, TRUE);
}

void
stop_callback (GtkWidget * stop_button, gpointer data)
{
	stop_flag = 1;

	set_toolbar_sens (TRUE, FALSE, TRUE, TRUE);
}

void
prompt (char *title, char *msg, GtkSignalFunc callback, char *def)
{
	GtkWidget *vbox, *hbox, *label;
	gint response;

	if (dialog != NULL)
		return;

	dialog = gtk_dialog_new_with_buttons (title, GTK_WINDOW (rootw),
					      GTK_DIALOG_MODAL |
					      GTK_DIALOG_DESTROY_WITH_PARENT,
					      GTK_STOCK_CANCEL,
					      GTK_RESPONSE_CANCEL,
					      GTK_STOCK_OK,
					      GTK_RESPONSE_OK, NULL);

	vbox = GTK_DIALOG (dialog)->vbox;

	hbox = gtk_hbox_new (FALSE, 8);
	gtk_container_set_border_width (GTK_CONTAINER (hbox), 8);
	gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);

	label = gtk_label_new (msg);
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
	entry = gtk_entry_new ();
	gtk_entry_set_text (GTK_ENTRY (entry), def);
	gtk_box_pack_start (GTK_BOX (hbox), entry, TRUE, TRUE, 0);

	gtk_widget_show_all (hbox);
	/*gtk_widget_show (dialog); */
	response = gtk_dialog_run (GTK_DIALOG (dialog));
	if (response == GTK_RESPONSE_OK) {
		/* Call the function to set their respective value
		   It's necessary FIX this code, because the compiler
		   can't chek it if there any error */
		GTK_SIGNAL_FUNC (callback) ();
	}

	gtk_widget_destroy (dialog);
	dialog = NULL;
}

void
save_call (GtkWidget * w, gpointer data)
{
	GtkWidget *fsel;
	GtkWidget *frame, *frame2, *scrolled, *text;
	GtkTextBuffer *buffer;

	dialog = fsel =
	    gtk_file_selection_new (_("Save gTuring Program File"));

	frame2 = gtk_frame_new (_("Comment"));
	gtk_box_pack_start (GTK_BOX (GTK_FILE_SELECTION (fsel)->main_vbox),
			    frame2, TRUE, TRUE, 0);
	frame = gtk_frame_new (NULL);
	gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_NONE);
	gtk_container_set_border_width (GTK_CONTAINER (frame), 8);
	gtk_container_add (GTK_CONTAINER (frame2), frame);
	scrolled = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled),
					GTK_POLICY_AUTOMATIC,
					GTK_POLICY_AUTOMATIC);
	gtk_container_add (GTK_CONTAINER (frame), scrolled);
	buffer = gtk_text_buffer_new (NULL);
	text = gtk_text_view_new_with_buffer (buffer);
	gtk_text_view_set_editable (GTK_TEXT_VIEW (text), TRUE);

	if (prog_message)
		gtk_text_buffer_set_text (buffer, prog_message,
					  strlen (prog_message));

	gtk_container_add (GTK_CONTAINER (scrolled), text);

	gtk_widget_show_all (frame2);

	gtk_file_selection_set_filename (GTK_FILE_SELECTION (fsel),
					 states_fname);
	g_object_set_data (G_OBJECT (fsel), "text", text);
	g_signal_connect (GTK_OBJECT
			  (GTK_FILE_SELECTION (fsel)->ok_button),
			  "clicked",
			  GTK_SIGNAL_FUNC (states_fname_callback),
			  (gpointer) TRUE);
	g_signal_connect (GTK_OBJECT
			  (GTK_FILE_SELECTION (fsel)->cancel_button),
			  "clicked", GTK_SIGNAL_FUNC (cancel_callback),
			  NULL);
	gtk_widget_show (fsel);
}

void
open_call (GtkWidget * w, gpointer data)
{
	GtkWidget *fsel;
	GtkWidget *button;

	dialog = fsel =
	    gtk_file_selection_new (_("Open gTuring Program File"));

	button = gtk_button_new_with_label (_("Examples"));
	gtk_container_add (GTK_CONTAINER
			   (GTK_FILE_SELECTION (fsel)->button_area),
			   button);
	gtk_widget_show (button);

	g_signal_connect (GTK_OBJECT (button), "clicked",
			  GTK_SIGNAL_FUNC
			  (states_fname_examples_callback), fsel);
	g_signal_connect (GTK_OBJECT
			  (GTK_FILE_SELECTION (fsel)->ok_button),
			  "clicked",
			  GTK_SIGNAL_FUNC (states_fname_callback),
			  (gpointer) FALSE);
	g_signal_connect (GTK_OBJECT
			  (GTK_FILE_SELECTION (fsel)->cancel_button),
			  "clicked", GTK_SIGNAL_FUNC (cancel_callback),
			  NULL);
	gtk_widget_show (fsel);
}

void
tape_call (GtkWidget * w, gpointer data)
{
	prompt (_("Tape Setting"), _("Please enter the tape:"),
		GTK_SIGNAL_FUNC (tape_string_callback), tape_string);
}


void
view_comment_call (GtkWidget * w, gpointer data)
{
	view_comment ();
}

void
exit_call (GtkWidget * w, gpointer data)
{
	gtk_main_quit ();
	exit (0);
}

void
playspeed_call (GtkWidget * w, gpointer data)
{
	char buff[1024];

	sprintf (buff, "%ld", speed);
	prompt (_("Animation Speed"), _("Miliseconds between steps:"),
		GTK_SIGNAL_FUNC (speed_callback), buff);
}


void
help_call (GtkWidget * w, gpointer data)
{
	GError *error = NULL;

	gnome_help_display ("gturing.xml", NULL, &error);

	if (error != NULL) {
		g_warning (error->message);
		g_error_free (error);
	}
}

void
about_call (GtkWidget * w, gpointer data)
{
	/* TODO: Add logo */
	GtkWidget *about_box = NULL;
	const gchar *authors[] = { "Arturo Espinosa Aldama",
  		"Germán Poo Caamaño",
		NULL
	};
	const gchar *documentors[] = { "Arturo Espinosa Aldama", 
		"Alexander Kirillov", NULL };

	about_box = gnome_about_new ("gTuring",
				     VERSION,
				     _
				     ("(c) 1997-2001 Free Software Foundation\n(c) 2001-2002 Germán Poo Caamaño"),
				     _("A Turing machine for GNOME"),
				     authors, documentors,
				     _
				     ("This is an untranslated version of gturing"),
				     NULL /* logo pixbuf */ );

	/*gtk_dialog_run (GTK_DIALOG (about_box));

	   gtk_widget_destroy (about_box); */
	gtk_window_present (GTK_WINDOW (about_box));

}

void
create_machine (void)
{
	PangoFontDescription *font_desc = NULL;

	font_desc = pango_font_description_from_string (font_name);
	g_return_if_fail (font_desc != NULL);

	gtk_widget_modify_font (GTK_WIDGET (tapelabel), font_desc);
	gtk_widget_modify_font (GTK_WIDGET (headlabel), font_desc);

	pango_font_description_free (font_desc);
}

	/* Menu definitions for the GnomeAppHelper test */
static GnomeUIInfo helper_file_menu[] = {
	/*{GNOME_APP_UI_ITEM, "_New", "Create a new file", item_activated,
	   "file/new", NULL,
	   GNOME_APP_PIXMAP_STOCK, GTK_STOCK_NEW, 'n',
	   GDK_CONTROL_MASK, NULL}, */
	{GNOME_APP_UI_ITEM, "_Open...", "Open an existing file", open_call,
	 "file/open", NULL,
	 GNOME_APP_PIXMAP_STOCK, GTK_STOCK_OPEN, 'o',
	 GDK_CONTROL_MASK, NULL},
	/*{GNOME_APP_UI_ITEM, "_Save", "Save the current file", save_call,
	   "file/save", NULL,
	   GNOME_APP_PIXMAP_STOCK, GTK_STOCK_SAVE, 's',
	   GDK_CONTROL_MASK, NULL}, */
	{GNOME_APP_UI_ITEM, "Save _as...",
	 "Save the current file with a new name", save_call,
	 "file/save as", NULL,
	 GNOME_APP_PIXMAP_STOCK, GTK_STOCK_SAVE_AS, 0, 0, NULL},

	GNOMEUIINFO_SEPARATOR,

	{GNOME_APP_UI_ITEM, "_Comments...", "View the program's comment.",
	 view_comment_call, "file/comments", NULL,
	 GNOME_APP_PIXMAP_STOCK, GTK_STOCK_UNDO, 0,
	 GDK_CONTROL_MASK, NULL},

/*		{GNOME_APP_UI_ITEM, "_Print...", "Print the current file",
		 item_activated, "file/print", NULL,
		 GNOME_APP_PIXMAP_STOCK, GTK_STOCK_PRINT, 'p',
		 GDK_CONTROL_MASK, NULL},*/

	GNOMEUIINFO_SEPARATOR,

	{GNOME_APP_UI_ITEM, "E_xit", "Exit the program", exit_call,
	 "file/exit", NULL,
	 GNOME_APP_PIXMAP_STOCK, GTK_STOCK_QUIT, 'q',
	 GDK_CONTROL_MASK, NULL},
	GNOMEUIINFO_END
};


static GnomeUIInfo helper_action_menu[] = {
	{GNOME_APP_UI_ITEM, "_Reset",
	 "Reset the machine going to the first state.",
	 power_do, "action/reset", NULL,
	 GNOME_APP_PIXMAP_STOCK, GTK_STOCK_REFRESH, 'r',
	 GDK_CONTROL_MASK, NULL},

	{GNOME_APP_UI_ITEM, "_Pause", "Stop the machine.",
	 stop_callback, "action/stop", NULL,
	 GNOME_APP_PIXMAP_STOCK, GTK_STOCK_STOP, 'g',
	 GDK_CONTROL_MASK, NULL},

	{GNOME_APP_UI_ITEM, "_Play",
	 "Start the machine from the current state.",
	 play_callback, "action/play", NULL,
	 GNOME_APP_PIXMAP_STOCK, GTK_STOCK_GO_FORWARD, 'p',
	 GDK_CONTROL_MASK, NULL},

	{GNOME_APP_UI_ITEM, "_Step",
	 "Allow the machine advance step by step from the current state.",
	 step_callback, "action/step", NULL,
	 GNOME_APP_PIXMAP_STOCK, GTK_STOCK_GOTO_LAST, 't',
	 GDK_CONTROL_MASK, NULL},

	/*{GNOME_APP_UI_ITEM, "Graph editor",
	   "Open a graphical interface to edit the machine's states.",
	   item_activated, "view/graph", NULL,
	   GNOME_APP_PIXMAP_STOCK, GTK_STOCK_CUT, 'x',
	   GDK_CONTROL_MASK, NULL}, */
	GNOMEUIINFO_END
};

static GnomeUIInfo helper_settings_menu[] = {
	{GNOME_APP_UI_ITEM, "_Play speed...", "Set play speed",
	 playspeed_call, "settings/play", NULL,
	 GNOME_APP_PIXMAP_NONE, NULL, 0,
	 GDK_CONTROL_MASK, NULL},
	{GNOME_APP_UI_ITEM, "_Tape...", "Set the tape to play", tape_call,
	 "settings/tape", NULL,
	 GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL},

/*		GNOMEUIINFO_SEPARATOR,
	
		{GNOME_APP_UI_ITEM, "P_references... ",
		 "Personalize your gturing's settings", item_activated,
		 "settings/preferences", NULL,
		 GNOME_APP_PIXMAP_STOCK, GTK_STOCK_REDO, 0, 0, NULL},*/
	GNOMEUIINFO_END
};

static GnomeUIInfo helper_help_menu[] = {
	{GNOME_APP_UI_ITEM, "_Contents",
	 "Open the gTuring's manual", help_call,
	 "help/contents", NULL,
	 GNOME_APP_PIXMAP_STOCK, GTK_STOCK_HELP, 0, 0, NULL},
	{GNOME_APP_UI_ITEM, "_About...",
	 "Displays information about the program", about_call,
	 "help/about", NULL,
	 GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_ABOUT, 0, 0, NULL},
	GNOMEUIINFO_END
};

static GnomeUIInfo helper_main_menu[] = {
	{GNOME_APP_UI_SUBTREE, "_File", "File operations",
	 helper_file_menu, NULL, NULL,
	 GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL},
	{GNOME_APP_UI_SUBTREE, "_Actions", "Make the state machine works",
	 helper_action_menu, NULL, NULL,
	 GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL},
	{GNOME_APP_UI_SUBTREE, "_Settings", "gTuring settings",
	 helper_settings_menu, NULL, NULL,
	 GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL},
	{GNOME_APP_UI_SUBTREE, "_Help", "Help on the program",
	 helper_help_menu, NULL, NULL,
	 GNOME_APP_PIXMAP_NONE, NULL, 0, 0, NULL},
	GNOMEUIINFO_END
};


	/* Toolbar definition for the GnomeAppHelper test */
static GnomeUIInfo helper_toolbar[] = {
	/* New machine's states should be call an machine's editor, graph or
	   table */
	/* {GNOME_APP_UI_ITEM, "New", "Create a new machine's states", item_activated,
	   "toolbar/new", NULL,
	   GNOME_APP_PIXMAP_STOCK, GTK_STOCK_NEW, 0, 0, NULL}, */
	{GNOME_APP_UI_ITEM, "Open", "Open an existing file", open_call,
	 "toolbar/open", NULL,
	 GNOME_APP_PIXMAP_STOCK, GTK_STOCK_OPEN, 0, 0, NULL},
	{GNOME_APP_UI_ITEM, "Save", "Save the current file", save_call,
	 "toolbar/save", NULL,
	 GNOME_APP_PIXMAP_STOCK, GTK_STOCK_SAVE_AS, 0, 0, NULL},

	GNOMEUIINFO_SEPARATOR,

	{GNOME_APP_UI_ITEM, "Reset", "Reset the machine's states",
	 power_do, "toolbar/reset", NULL,
	 GNOME_APP_PIXMAP_STOCK, GTK_STOCK_REFRESH, 0, 0, NULL},
	{GNOME_APP_UI_ITEM, "Stop", "Stop the execution of tape",
	 stop_callback, "toolbar/stop", NULL,
	 GNOME_APP_PIXMAP_STOCK, GTK_STOCK_STOP, 0, 0, NULL},
	{GNOME_APP_UI_ITEM, "Play", "Start the tape", play_callback,
	 "toolbar/play", NULL,
	 GNOME_APP_PIXMAP_STOCK, GTK_STOCK_GO_FORWARD, 0, 0, NULL},
	{GNOME_APP_UI_ITEM, "Step", "Play the tape step by step",
	 step_callback, "toolbar/step", NULL,
	 GNOME_APP_PIXMAP_STOCK, GTK_STOCK_GOTO_LAST, 0, 0, NULL},

	GNOMEUIINFO_END
};


void
init_interface (int argc, char *argv[])
{
	static GtkWidget *hpaned;
	/* GtkWidget *graph; */
	GtkWidget *treeview;	/* treeview with scrolled window */

	rootw = gnome_app_new ("gTuring", _("A Turing machine"));
	gnome_app_create_menus (GNOME_APP (rootw), helper_main_menu);
	gnome_app_create_toolbar (GNOME_APP (rootw), helper_toolbar);

	statusline =
	    GNOME_APPBAR (gnome_appbar_new
			  (FALSE, TRUE, GNOME_PREFERENCES_NEVER));
	gnome_app_set_statusbar (GNOME_APP (rootw),
				 GTK_WIDGET (statusline));

	vbox = gtk_vbox_new (FALSE, GNOME_PAD_SMALL);

	tapelabel = gtk_label_new (_("Welcome to gTuring"));
	headlabel = gtk_label_new (_("^                 "));
	gtk_box_pack_start (GTK_BOX (vbox), GTK_WIDGET (tapelabel), FALSE,
			    FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox), GTK_WIDGET (headlabel), FALSE,
			    FALSE, 0);

	hpaned = gtk_hpaned_new ();

	gtk_container_set_border_width (GTK_CONTAINER (hpaned), 5);

	/*graph = graph_editor_new (&(graph_editor)); */
	treeview = turing_table_editor_new (&treeview_editor, tm);

	/*gtk_paned_add1 (GTK_PANED (hpaned), graph); */
	gtk_paned_add2 (GTK_PANED (hpaned), treeview);

	gtk_box_pack_start (GTK_BOX (vbox), hpaned, TRUE, TRUE, 0);
/*	set_save_sens (FALSE);
	set_toolbar_sens (FALSE, FALSE, FALSE, FALSE);*/

	gnome_app_set_contents (GNOME_APP (rootw), vbox);
	g_signal_connect (GTK_OBJECT (rootw), "destroy",
			  GTK_SIGNAL_FUNC (exit_call), NULL);

	create_machine ();
	
	gtk_window_set_default_size (GTK_WINDOW (rootw), 350, 400);
	
	gtk_widget_show_all (rootw);
}

void
init_globals (gchar * filename)
{
	gchar *tape;

	tm = turing_new ();

	tape =
	    gconf_client_get_string (gengine, "/apps/gturing/tape", NULL);
	
	
	if (tape != NULL) {
		strncpy (tape_string, tape, strlen (tape));
		set_tape (tape);
	}
	
	/*strncpy (tape_string,
	   gconf_client_get_string (gengine, "/gTuring/Options/tape", NULL), 1024); */

	if (speed < 0) {	/* Was it assigned as argument? */
		speed =
		    gconf_client_get_int (gengine, "/apps/gturing/speed",
					  NULL);
	}

	/*strncpy (tape_string,
	   gnome_config_get_string ("/gTuring/Options/tape="), 1024);
	   strncpy (states_fname,
	   gnome_config_get_string ("/gTuring/Options/program="),
	   1024);
	   speed = gnome_config_get_int ("/gTuring/Options/speed=50"); */

	if (turing_fread_states (tm, filename)) {
		*states_fname = 0;	/*error */
	} else {
		strncpy (states_fname, filename, 1024);

		if (prog_message)
			free (prog_message);

		prog_message = turing_fread_comments (filename);
	}
}

/* The main. */
int
main (int argc, char *argv[])
{
	const struct poptOption gturing_options[] = {
		{"play-speed", 'p', POPT_ARG_LONG, &speed, 0,
		 _("Set the animation speed (miliseconds)"), _("ms")},
		{NULL, 0, 0, NULL, 0},
	};
	gchar *filename = NULL;
	poptContext optCon;
	GnomeProgram *program;

	gnome_score_init ("gturing");

#ifdef ENABLE_NLS
	setlocale (LC_ALL, "");
	bindtextdomain (GETTEXT_PACKAGE, GTURING_LOCALEDIR);
	bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
	textdomain (GETTEXT_PACKAGE);
#endif

	program =
	    gnome_program_init (PACKAGE, VERSION, LIBGNOMEUI_MODULE,
				argc, argv, GNOME_PARAM_POPT_TABLE,
				gturing_options, 
				GNOME_PARAM_APP_DATADIR, DATADIR,
				GNOME_PARAM_NONE);

	g_object_get (G_OBJECT (program), GNOME_PARAM_POPT_CONTEXT,
		      &optCon, GNOME_PARAM_NONE);

	gconf_init (argc, argv, NULL);
	gengine = gconf_client_get_default ();

	init_interface (argc, argv);

	filename = (gchar *) poptGetArg (optCon);
	poptFreeContext (optCon);

	init_globals (filename);

	if (*states_fname != 0)
		set_states_fname (states_fname);

	gtk_main ();

	return 0;
}
